/*
 * temperature.c -- code for temprature monitoring
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * Written by Sos Pter <sp@osb.hu>, 2002-2003
 */


#include <linux/config.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/types.h>
#include <linux/init.h>
#include <linux/proc_fs.h>

#include <asm/system.h>
#include <asm/errno.h>
#include <asm/io.h>
#include <asm/uaccess.h>

#ifdef OMNIBOOK_STANDALONE
#include "omnibook.h"
#else
#include <linux/omnibook.h>
#endif

#include "ec.h"

static struct proc_dir_entry *proc_temperature;

int omnibook_get_cpu_temp(void)
{
	u8 temp = 0;
	int retval;

	switch (omnibook_ectype) {
	case XE3GF:
		retval = omnibook_ec_read(XE3GF_CTMP, &temp);
		if (retval)
			return retval;
		retval = temp;
		break;
	case XE3GC:
		retval = omnibook_ec_read(XE3GC_CTMP, &temp);
		if (retval)
			return retval;
		retval = temp;
		break;
	case OB500:
	case OB510:
	case OB6000:
	case OB6100:
	case XE4500:
		retval = omnibook_ec_read(OB500_CTMP, &temp);
		if (retval)
			return retval;
		retval = temp;
		break;
	case OB4150:
		retval = omnibook_ec_read(OB4150_TMP, &temp);
		if (retval)
			return retval;
		retval = temp;
		break;
	default:
		printk(KERN_INFO "%s: Temperature monitoring is unsupported on this machine.\n", MODULE_NAME);
		retval = -ENODEV ;
	}
	return retval;
}

static int omnibook_proc_temperature(char *buffer, char **start, off_t off, int count, int *eof, void *data)
{
	int len = 0;
	char *b = buffer;
	u8 cpu_temp;

	cpu_temp = omnibook_get_cpu_temp();

	b += sprintf(b, "CPU temperature:            %2d C\n", cpu_temp);

	len = b - buffer;
	if (len <= off + count)
		*eof = 1;
	*start = buffer + off;
	len -= off;
	if (len > count)
		len = count;
	if (len < 0)
		len = 0;
	return len;
}

int __init omnibook_temperature_init(void)
{
	mode_t pmode;

	switch (omnibook_ectype) {
	case XE3GF:
	case XE3GC:
	case OB500:
	case OB510:
	case OB6000:
	case OB6100:
	case XE4500:
	case OB4150:
		pmode = S_IFREG | S_IRUGO;
		proc_temperature = create_proc_read_entry("temperature", pmode, omnibook_proc_root, omnibook_proc_temperature, NULL);
		break;
	default:
		printk(KERN_INFO "%s: Temperature monitoring is unsupported on this machine.\n", MODULE_NAME);
		return 0;
	}
	if (! proc_temperature) {
		printk(KERN_ERR "%s: Unable to create /proc/%s/temperature.\n", MODULE_NAME, MODULE_NAME);
		return -ENOENT;
	}
	printk(KERN_INFO "%s: Temperature monitoring is enabled.\n", MODULE_NAME);
	return 0;
}

void __exit omnibook_temperature_cleanup(void)
{
	if (proc_temperature)
		remove_proc_entry("temperature", omnibook_proc_root);
}

EXPORT_SYMBOL(omnibook_get_cpu_temp);

/* End of file */
