/*
 * leap-frog point to allow malloc-debug on/off via relink.
 *
 * Copyright 1993 by the Antaire Corporation
 *
 * This file is part of the malloc-debug package.
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose and without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies, and that
 * the name of Antaire not be used in advertising or publicity pertaining to
 * distribution of the document or software without specific, written prior
 * permission.
 *
 * The Antaire Corporation makes no representations about the suitability of
 * the software described herein for any purpose.  It is provided "as is"
 * without express or implied warranty.
 *
 * The author may be contacted at gray.watson@antaire.com
 */

/*
 * If anyone can think of a better way to do this *please* let me know.
 *
 * The goal of the routines in this file is to allow people to use the
 * debug library during development and then disable and return to the
 * system (or more efficient) malloc functions easily.
 *
 * The problem is that we want to provide the library with file/line
 * information with each call.  Backtracing the stack-frame is neither
 * portable or necessarily easy so the cpp __FILE__ and __LINE__ directives
 * are used instead.  But how to pass this information to the library is
 * the challange.  How can we do this in a graceful and sane manner while
 * still providing the "quick-release" functionality above?
 *
 * I have decided on the leap-frog routines in this file that will have
 * to *always* be compiled in if you include malloc_dbg.h in your source.
 * It has the special _malloc_file and _malloc_line variables and sets
 * them whether malloc-debug routines are linked in to use them or not.
 *
 * This *will* mean an extra function call per memory interaction but on
 * most systems this is pretty cheap.
 *
 * Please mail me with any reasonable ideas.
 */

#define MALLOC_DEBUG_DISABLE

#include "malloc_dbg.h"				/* for MALLOC_SIZE... */
#include "conf.h"				/* for const */

#include "malloc_loc.h"				/* for EXPORT */
#include "malloc_lp.h"

#if INCLUDE_RCS_IDS
static	char	*rcs_id =
  "$Id: malloc_lp.c,v 1.16 1994/03/21 14:58:52 gray Exp $";
#endif

/*
 * exported variables
 */
/* to inform the malloc-debug library from which file the call comes from */
EXPORT	char		*_malloc_file = MALLOC_DEFAULT_FILE;

/* to inform the library from which line-number the call comes from */
EXPORT	unsigned int	_malloc_line = MALLOC_DEFAULT_LINE;

/* pre-set malloc_debug() value before the library is setup */
EXPORT	int		_malloc_debug_preset = DEBUG_PRE_NONE;

/* pointers to shutdown function to allow calls without linked routine */
EXPORT	void		(*_malloc_shutdown_func)() = NULL;

/* pointers to log_heap_map function to allow calls without linked routine */
EXPORT	void		(*_malloc_log_heap_map_func)() = NULL;

/* pointers to log_stats function to allow calls without linked routine */
EXPORT	void		(*_malloc_log_stats_func)() = NULL;

/* pointers to log_unfreed function to allow calls without linked routine */
EXPORT	void		(*_malloc_log_unfreed_func)() = NULL;

/* pointers to verify function to allow calls without linked routine */
EXPORT	int		(*_malloc_verify_func)() = NULL;

/* pointers to debug function to allow calls without linked routine */
EXPORT	void		(*_malloc_debug_func)() = NULL;

/* pointers to debug_current function to allow calls without linked routine */
EXPORT	int		(*_malloc_debug_current_func)() = NULL;

/* pointers to examine function to allow calls without linked routine */
EXPORT	int		(*_malloc_examine_func)() = NULL;

/* pointers to strerror function to allow calls without linked routine */
EXPORT	char		*(*_malloc_strerror_func)() = NULL;

/*
 * leap routine to calloc
 */
EXPORT	void	*_calloc_leap(const char * file, const int line,
			      MALLOC_SIZE elen, MALLOC_SIZE size)
{
  void	*ret;
  
  _malloc_file = (char *)file;
  _malloc_line = line;
  
  ret = calloc(elen, size);
  
  return ret;
}

/*
 * leap routine to free
 */
#if defined(__STDC__) && __STDC__ == 1
EXPORT	void	_free_leap(const char * file, const int line, void * pnt)
#else
EXPORT	int	_free_leap(const char * file, const int line, void * pnt)
#endif
{
  _malloc_file = (char *)file;
  _malloc_line = line;
  
#if defined(__STDC__) && __STDC__ == 1
  free(pnt);
#else
  return free(pnt);
#endif
}

/*
 * leap routine to malloc
 */
EXPORT	void	*_malloc_leap(const char * file, const int line,
			      MALLOC_SIZE size)
{
  void	*ret;
  
  _malloc_file = (char *)file;
  _malloc_line = line;
  
  ret = malloc(size);
  
  return ret;
}

/*
 * leap routine to realloc
 */
EXPORT	void	*_realloc_leap(const char * file, const int line, void * oldp,
			       MALLOC_SIZE new_size)
{
  void	*ret;
  
  _malloc_file = (char *)file;
  _malloc_line = line;
  
  ret = realloc(oldp, new_size);
  
  return ret;
}

/***************************** xmalloc functions *****************************/

/*
 * leap routine to calloc with error checking
 */
EXPORT	void	*_xcalloc_leap(const char * file, const int line,
			       MALLOC_SIZE elen, MALLOC_SIZE size)
{
  void	*ret;
  
  _malloc_file = (char *)file;
  _malloc_line = line;
  
  ret = calloc(elen, size);
  if (ret == NULL) {
    char	str[256];
    (void)sprintf(str,
		  "Out of memory while allocating %d bytes from '%s:%d'\n",
		  size, file, line);
    (void)write(STDERR, str, strlen(str));
    _exit(1);
  }
  
  return ret;
}

/*
 * leap routine to free
 */
#if defined(__STDC__) && __STDC__ == 1
EXPORT	void	_xfree_leap(const char * file, const int line, void * pnt)
#else
EXPORT	int	_xfree_leap(const char * file, const int line, void * pnt)
#endif
{
  _malloc_file = (char *)file;
  _malloc_line = line;
  
#if defined(__STDC__) && __STDC__ == 1
  free(pnt);
#else
  return free(pnt);
#endif
}

/*
 * leap routine to malloc with error checking
 */
EXPORT	void	*_xmalloc_leap(const char * file, const int line,
			      MALLOC_SIZE size)
{
  void	*ret;
  
  _malloc_file = (char *)file;
  _malloc_line = line;
  
  ret = malloc(size);
  if (ret == NULL) {
    char	str[256];
    (void)sprintf(str,
		  "Out of memory while allocating %d bytes from '%s:%d'\n",
		  size, file, line);
    (void)write(STDERR, str, strlen(str));
    _exit(1);
  }
  
  return ret;
}

/*
 * leap routine to realloc with error checking
 */
EXPORT	void	*_xrealloc_leap(const char * file, const int line, void * oldp,
				MALLOC_SIZE new_size)
{
  void	*ret;
  
  _malloc_file = (char *)file;
  _malloc_line = line;
  
  ret = realloc(oldp, new_size);
  if (ret == NULL) {
    char	str[256];
    (void)sprintf(str,
		  "Out of memory while allocating %d bytes from '%s:%d'\n",
		  new_size, file, line);
    (void)write(STDERR, str, strlen(str));
    _exit(1);
  }
  
  return ret;
}

/********************* routines when running malloc_dbg **********************/

/*
 * routine to call malloc_shutdown when linked in
 */
EXPORT	void	malloc_shutdown(void)
{
  if (_malloc_shutdown_func != NULL)
    _malloc_shutdown_func();
}

/*
 * routine to call malloc_verify when linked in
 */
EXPORT	int	malloc_verify(const void * pnt)
{
  if (_malloc_verify_func != NULL)
    return _malloc_verify_func(pnt);
  else
    return MALLOC_VERIFY_NOERROR;
}

/*
 * routine to call malloc_log_unfreed when linked in
 */
EXPORT	void	malloc_log_unfreed(void)
{
  if (_malloc_log_unfreed_func != NULL)
    _malloc_log_unfreed_func();
}

/*
 * routine to call malloc_log_stats when linked in
 */
EXPORT	void	malloc_log_stats(void)
{
  if (_malloc_log_stats_func != NULL)
    _malloc_log_stats_func();
}

/*
 * routine to call malloc_log_heap_map when linked in
 */
EXPORT	void	malloc_log_heap_map(void)
{
  if (_malloc_log_heap_map_func != NULL)
    _malloc_log_heap_map_func();
}

/*
 * routine to call malloc_debug when linked in
 */
EXPORT	void	malloc_debug(const int debug)
{
  if (_malloc_debug_func == NULL)
    _malloc_debug_preset = debug;
  else
    _malloc_debug_func(debug);
}

/*
 * routine to call malloc_debug_current when linked in
 */
EXPORT	int	malloc_debug_current(void)
{
  if (_malloc_debug_current_func != NULL)
    return _malloc_debug_current_func();
  else
    return 0;
}

/*
 * routine to call malloc_examine when linked in
 */
EXPORT	int	malloc_examine(const void * pnt, MALLOC_SIZE * size,
			       char ** file, unsigned int * line,
			       void ** ret_attr)
{
  if (_malloc_examine_func != NULL)
    return _malloc_examine_func(pnt, size, file, line, ret_attr);
  else
    return ERROR;
}

/*
 * routine to call malloc_strerror when linked in
 */
EXPORT	char	*malloc_strerror(const int errnum)
{
  if (_malloc_strerror_func != NULL)
    return _malloc_strerror_func(errnum);
  else
    return "unknown";
}
