/*
 * Xchrom.c -- a Chrom-A-Chron clock widget.
 * Copyright (c) Olaf Heimburger 1990
 * Last edited: Mon Oct 22 14:37:33 1990 by olafh (Olaf Heimburger) on GECKO
 */

#include <stdio.h>
#include <math.h>
#include <X11/IntrinsicP.h>
#include <X11/Xos.h>
#include <X11/StringDefs.h>
#include "XchromP.h"

static void Initialize (), Resize (), Realize (), Destroy (),
    Redisplay (), Timeout (), GetGC (), GetColorGC (), ShowTime ();

static Boolean SetValues ();

static Dimension winwidth = MINWIDTH;
static Dimension winheight = MINHEIGHT;

static int SliceAngles[NUMFIELDS] = {
     75, /* twelve */
     45, /* one */
     15, /* two */
    345, /* three */
    315, /* four */
    285, /* five */
    255, /* six */
    225, /* seven */
    195, /* eight */
    165, /* nine */
    135, /* ten */
    105, /* eleven */
};

static XtResource resources[] = {
    { XtNwidth, XtCWidth, XtRDimension, sizeof(Dimension),
	XtOffset(Widget, core.width), XtRDimension, (caddr_t)&winwidth },
    { XtNheight, XtCHeight, XtRDimension, sizeof(Dimension),
	XtOffset(Widget, core.height), XtRDimension, (caddr_t)&winheight },
    { XtNforeground, XtCForeground, XtRPixel, sizeof(Pixel),
        XtOffset(XchromWidget, xchrom.foreground), XtRString,
	  XtDefaultForeground},
    { XtNtwelveOClock, XtCTwelveOClock, XtRPixel, sizeof(Pixel),
        XtOffset(XchromWidget, xchrom.color[0]), XtRString, "Yellow"},
    { XtNoneOClock, XtCOneOClock, XtRPixel, sizeof(Pixel),
        XtOffset(XchromWidget, xchrom.color[1]), XtRString, "Orange"},
    { XtNtwoOClock, XtCTwoOClock, XtRPixel, sizeof(Pixel),
        XtOffset(XchromWidget, xchrom.color[2]), XtRString, "Pink"},
    { XtNthreeOClock, XtCThreeOClock, XtRPixel, sizeof(Pixel),
        XtOffset(XchromWidget, xchrom.color[3]), XtRString, "Red"},
    { XtNfourOClock, XtCFourOClock, XtRPixel, sizeof(Pixel),
        XtOffset(XchromWidget, xchrom.color[4]), XtRString, "LightPink"},
    { XtNfiveOClock, XtCFiveOClock, XtRPixel, sizeof(Pixel),
        XtOffset(XchromWidget, xchrom.color[5]), XtRString, "Violet"},
    { XtNsixOClock, XtCSixOClock, XtRPixel, sizeof(Pixel),
        XtOffset(XchromWidget, xchrom.color[6]), XtRString, "RoyalBlue"},
    { XtNsevenOClock, XtCSevenOClock, XtRPixel, sizeof(Pixel),
        XtOffset(XchromWidget, xchrom.color[7]), XtRString, "DarkGreen"},
    { XtNeightOClock, XtCEightOClock, XtRPixel, sizeof(Pixel),
        XtOffset(XchromWidget, xchrom.color[8]), XtRString, "Turquoise"},
    { XtNnineOClock, XtCNineOClock, XtRPixel, sizeof(Pixel),
        XtOffset(XchromWidget, xchrom.color[9]), XtRString, "Brown"},
    { XtNtenOClock, XtCTenOClock, XtRPixel, sizeof(Pixel),
        XtOffset(XchromWidget, xchrom.color[10]), XtRString, "LightYellow"},
    { XtNelevenOClock, XtCElevenOClock, XtRPixel, sizeof(Pixel),
        XtOffset(XchromWidget, xchrom.color[11]), XtRString, "Wheat"},
};

XchromClassRec xchromClassRec = {
    { /* core fields */
    /* superclass		*/	&widgetClassRec,
    /* class_name		*/	"Xchrom",
    /* widget_size		*/	sizeof(XchromRec),
    /* class_initialize		*/	NULL,
    /* class_part_initialize	*/	NULL,
    /* class_inited		*/	FALSE,
    /* initialize		*/	Initialize,
    /* initialize_hook		*/	NULL,
    /* realize			*/	Realize,
    /* actions			*/	NULL,
    /* num_actions		*/	0,
    /* resources		*/	resources,
    /* resource_count		*/	XtNumber(resources),
    /* xrm_class		*/	NULL,
    /* compress_motion		*/	TRUE,
    /* compress_exposure	*/	TRUE,
    /* compress_enterleave	*/	TRUE,
    /* visible_interest		*/	FALSE,
    /* destroy			*/	Destroy,
    /* resize			*/	Resize,
    /* expose			*/	Redisplay,
    /* set_values		*/	SetValues,
    /* set_values_hook		*/	NULL,
    /* set_values_almost	*/	XtInheritSetValuesAlmost,
    /* get_values_hook		*/	NULL,
    /* accept_focus		*/	NULL,
    /* version			*/	XtVersion,
    /* callback_private		*/	NULL,
    /* tm_table			*/	NULL,
    /* query_geometry		*/	NULL,
    }
};

WidgetClass xchromWidgetClass = (WidgetClass) &xchromClassRec;

/* ARGSUSED */
static void Initialize (request, new)
    XchromWidget   request;
    XchromWidget   new;
{
    if (new) {
	int i;
	new->xchrom.IntervalId = None;
	new->xchrom.WdwPixmap = new->xchrom.ColorPixmap = None;
	/*
	 * initialize the color table
	 */
	for (i = 0; i < NUMFIELDS; ++i) {
	    GetColorGC (new, i);
	}
	/*
	 * Initialize the angles for the slices.
	 */
	for (i = 0; i < NUMFIELDS; ++i) {
	    new->xchrom.Slices[i].angle1 = (short)(SliceAngles[i] * XARCMAGIC);
	    new->xchrom.Slices[i].angle2 = (short)(SLICEANGLE * XARCMAGIC);
	}
	new->xchrom.Slices[SLICEMASK].angle1 = STARTDEGREES * XARCMAGIC;
	new->xchrom.Slices[SLICEMASK].angle2 = MASKDEGREES * XARCMAGIC;
	new->xchrom.Slices[SLICEMASK].x =
	    new->xchrom.Slices[SLICEMASK].y = STARTCORNER + MASKOFFSET;
    }
    GetGC (new);
}

static void GetColorGC (w, i)
    XchromWidget w;
    int i;
{
    if (w && i >= 0 && i < NUMFIELDS) {
	XGCValues gcv;
	XtGCMask mask = GCForeground | GCBackground | GCFunction;
	gcv.background = WhitePixel (XtDisplay (w),
				     DefaultScreen (XtDisplay (w)));
	gcv.function = GXcopy;
	gcv.foreground = w->xchrom.color[i];
	w->xchrom.Colors[i] = XtGetGC ((Widget)w, mask, &gcv);
    }
}

static void GetGC (w)
    XchromWidget w;
{
    if (w) {
	XGCValues gcv;
	XtGCMask mask = GCForeground | GCBackground | GCFunction;
	gcv.background = w->core.background_pixel;
	gcv.foreground = w->xchrom.foreground;
	gcv.function = GXcopy;
	w->xchrom.Colors [BACKCOLOR] = XtGetGC ((Widget)w, mask, &gcv);
    }
}

static void CreatePixmaps (w)
    XchromWidget w;
{
    if (w) {
	/*
	 * Create the pixmap's and clear 'em.
	 */
	Display *dpy = XtDisplay (w);
	Window win = XtWindow (w);
	GC TmpGC;
	XGCValues gcv;
	XtGCMask mask = GCForeground | GCBackground | GCFunction;
	if (w->xchrom.ColorPixmap != None) {
	    XFreePixmap (dpy, w->xchrom.ColorPixmap);
	}
	if (w->xchrom.WdwPixmap != None) {
	    XFreePixmap (dpy, w->xchrom.WdwPixmap);
	}
	gcv.foreground = WhitePixel (dpy, DefaultScreen (dpy));
	gcv.background = BlackPixel (dpy, DefaultScreen (dpy));
	gcv.function = GXcopy;
	TmpGC = XtGetGC (w, mask, &gcv);
	w->xchrom.ColorPixmap =
	    XCreatePixmap (dpy, win, w->xchrom.ArcSize, w->xchrom.ArcSize,
			   DefaultDepth (dpy, DefaultScreen (dpy)));
	XFillRectangle (dpy, w->xchrom.ColorPixmap, TmpGC, 0, 0,
			w->xchrom.ArcSize, w->xchrom.ArcSize);
	w->xchrom.WdwPixmap =
	    XCreatePixmap (dpy, win, w->core.width, w->core.height,
			   DefaultDepth (dpy, DefaultScreen (dpy)));
	XFillRectangle (dpy, w->xchrom.WdwPixmap, TmpGC, 0, 0, w->core.width,
			w->core.height);
	XtReleaseGC (w, TmpGC);
    }
}

static void Realize (w, valueMask, attrs)
    Widget w;
    XtValueMask *valueMask;
    XSetWindowAttributes *attrs;
{
    *valueMask |= CWBitGravity;
    attrs->bit_gravity = ForgetGravity;
    XtCreateWindow(w, InputOutput, (Visual*)CopyFromParent, *valueMask,
		   attrs);
    Resize(w);
}

static void Destroy (w)
    XchromWidget w;
{
    int n;
    
    if (w->xchrom.IntervalId) {
	XtRemoveTimeOut(w->xchrom.IntervalId);
    }
    for (n = 0; n < NUMFIELDS + 1; ++n) {
	XtReleaseGC (w, w->xchrom.Colors[n]);
    }
    if (w->xchrom.ColorPixmap) {
	XFreePixmap (XtDisplay(w), w->xchrom.ColorPixmap);
    }
    if (w->xchrom.WdwPixmap) {
	XFreePixmap (XtDisplay(w), w->xchrom.WdwPixmap);
    }
}

/* ARGSUSED */
static void Resize  (w)
    XchromWidget    w;
{
    if (XtIsRealized (w)) {
	Display *dpy = XtDisplay (w);
	int Width = w->core.width;
	int Height = w->core.height;
	/*
	 * Get the minimum size of the square
	 */
	w->xchrom.SquareSize =
	    ((Width > Height) ? Height : Width) - (STARTCORNER * 2);
	if (w->xchrom.SquareSize > 0) {
	    XRectangle Rect;
	    int i = 0;
	    int HalfSize = w->xchrom.SquareSize / 2;
	    w->xchrom.ArcSize =
		((int) sqrt ((float) ((HalfSize * HalfSize)
				      + (HalfSize * HalfSize))) * 2) + 3;
	    w->xchrom.Offset =
		(int)((w->xchrom.ArcSize - w->xchrom.SquareSize) / 2);
	    CreatePixmaps (w);
	    /*
	     * Fill the color pixmap
	     */
	    for (i = 0; i < NUMFIELDS; ++i) {
		w->xchrom.Slices[i].x = w->xchrom.Slices[i].y = 0;
		w->xchrom.Slices[i].width = w->xchrom.Slices[i].height =
		    w->xchrom.ArcSize;
		XFillArcs (dpy, w->xchrom.ColorPixmap, w->xchrom.Colors[i],
			   &w->xchrom.Slices[i], 1);
	    }
	    /*
	     * Fill the background
	     */
	    Rect.x = 0;
	    Rect.y = 0;
	    Rect.width = Width;
	    Rect.height = Height;
	    XFillRectangles (dpy, w->xchrom.WdwPixmap,
			     w->xchrom.Colors[BACKCOLOR], &Rect, 1);
	    /*
	     * Set the slice mask size.
	     */
	    w->xchrom.Slices[SLICEMASK].width =
		w->xchrom.Slices[SLICEMASK].height =
		    w->xchrom.SquareSize - (MASKOFFSET * 2);
	}
    }
}

/* ARGSUSED */
static void Redisplay  (w)
    XchromWidget    w;
{
    if (XtIsRealized (w)) {
	long t = time (0);
	if (w->xchrom.IntervalId != None) {
	    XtRemoveTimeOut (w->xchrom.IntervalId);
	    w->xchrom.IntervalId = NULL;
	}
	ShowTime (w);
	w->xchrom.IntervalId =
	    XtAddTimeOut ((unsigned long)(60 - (t % 60)) * 1000, Timeout, w);
    }
}

static void ShowTime(w)
    XchromWidget w;
{
    long t = time(0);
    register struct tm *l_time = localtime(&t);

    /*
     * Clear the previous slice mask.
     */
    XCopyArea (XtDisplay (w), w->xchrom.ColorPixmap, w->xchrom.WdwPixmap,
	       w->xchrom.Colors[BACKCOLOR], w->xchrom.Offset, w->xchrom.Offset,
	       w->xchrom.SquareSize, w->xchrom.SquareSize, STARTCORNER,
	       STARTCORNER);
    /*
     * Calculate current position of slice mask
     */
    if (l_time) {
	int Min = l_time->tm_min;
	int Hour = l_time->tm_hour % NUMFIELDS;
	if (w->xchrom.SavedHour != Hour || w->xchrom.SavedMin != Min) {
	    w->xchrom.Slices[SLICEMASK].angle1 =
		(SliceAngles[Hour] + SLICEANGLE) * XARCMAGIC;
	    /*
	     * Display every minute. (0.5 degrees per minute)
	     */
	    w->xchrom.Slices[SLICEMASK].angle1 -= (Min) * (int)(XARCMAGIC / 2);
	    w->xchrom.SavedHour = Hour;
	    w->xchrom.SavedMin = Min;
	}
    }
    /*
     * Draw slice mask.
     */
    XFillArcs (XtDisplay (w), w->xchrom.WdwPixmap, w->xchrom.Colors[BACKCOLOR],
	       &w->xchrom.Slices[SLICEMASK], 1);
    /*
     * Finally, show the result.
     */
    XCopyArea (XtDisplay (w), w->xchrom.WdwPixmap, XtWindow (w),
	       w->xchrom.Colors[BACKCOLOR], 0, 0, w->core.width,
	       w->core.height, 0, 0);
    XFlush (XtDisplay (w));
}

static void Timeout (w, id)
    XchromWidget w;
    XtIntervalId *id;
{
    ShowTime (w);
    w->xchrom.IntervalId = XtAddTimeOut (60000, Timeout, w);
}

/* ARGSUSED */
static Boolean SetValues (current, request, new)
    XchromWidget current, request, new;
{
    Boolean Redraw = False;
    int i;
    if (new->xchrom.foreground != current->xchrom.foreground
	||  new->core.background_pixel != current->core.background_pixel) {
	XtReleaseGC (current, current->xchrom.Colors[BACKCOLOR]);
	GetGC (new);
	Redraw = True;
    }
    for (i = 0; i < NUMFIELDS; ++i) {
	if (new->xchrom.color[i] != current->xchrom.color[i]) {
	    printf ("color %d changed\n", i);
	    XtReleaseGC (current, current->xchrom.Colors[i]);
	    GetColorGC (new, i);
	    Redraw = True;
	}
    }
    if (Redraw) {
	Resize (new); /* pixmaps need to be redrawn */
    }
    return Redraw;
}
