/*
 * dproc.c - RISCos process access functions for lsof
 */


/*
 * Copyright 1994 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright 1994 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dproc.c,v 1.1 97/02/24 07:43:28 abe Exp $";
#endif

#include "lsof.h"


/*
 * Local static values
 */

static KA_T Kp;				/* kernel's process table address */
static int Knp;				/* number of processes according
					 * to kernel */
static int Np;				/* number of processes read from
					 * kernel */
struct var Var;				/* kernel variables */


_PROTOTYPE(static void get_kernel_access,(void));
_PROTOTYPE(static int mread,(KA_T addr, char *buf, READLEN_T len));


/*
 * gather_proc_info() -- gather process information
 */

void
gather_proc_info()
{
	int err, i, pgrp, pid, px;
	struct proc *p;
	static int p2bsh, p2psh, ps;
	KA_T pa;
	struct proc pbuf;
	short pss, sf;
	static struct user *u = NULL;
	static char *ua;
	char *ub;
	static char *ubl;
	struct file **uf;
	uid_t uid;
/*
 * Allocate user area buffer.
 *
 * Define user area paging parameters.
 */
	if (!u) {
	    MALLOC_S ual;

	    ual = (MALLOC_S) (sizeof(struct user) 
		+ ((Var.v_nofiles - 1) * sizeof(struct file *)));
	    if ((ua = (char *)malloc(ual)) == NULL) {
		(void) fprintf(stderr, "%s: no space for %d byte user area\n",
			Pn, ual);
		Exit(1);
	    }
	    u = (struct user *)ua;
	    ubl = ua + ual;
	    if ((ps = getpagesize()) == 4096)
		p2bsh = p2psh = 12;
	    else if (ps == 16384) {
		p2bsh = 14;
		p2psh = 10;
	    } else {
		(void) fprintf(stderr, "%s: unknown page size (%d)\n", Pn, ps);
		Exit(1);
	    }
	}

#if	defined(HASNCACHE)
/*
 * Read kernel name cache.
*/
        ncache_load();
#endif	/* defined(HASNCACHE) */

/*
 * Examine proc structures and their associated information.
 */
	p = &pbuf;
	for (px = 0; px < Np; px++) {

	/*
	 * Read the proc structure.  Reject uninteresting ones.
	 *
	 * Set the process group ID, Process ID, and User ID.
	 */
		pa = (KA_T)(Kp + (long)(px * sizeof(struct proc)));
		if (kread(pa, (char *)&pbuf, sizeof(pbuf)))
			continue;
		if (p->p_stat == 0 || p->p_stat == SZOMB)
			continue;
		pid = (int)p->p_pid;
		pgrp = (int)p->p_pgrp;
		uid = (uid_t)p->p_uid;
	/*
	 * See if process is excluded.
	 */
		if (is_proc_excl(pid, pgrp, (UID_ARG)uid, &pss, &sf))
			continue;
	/*
	 * Read the user area.
	 */
		for (err = i = 0, ub = ua; ub < ubl; i++, ub += ps) {
			if (mread((KA_T)((p->p_ubptbl[i].pgi.pg_pde >> p2psh)
			   << p2bsh), ub,
			   (READLEN_T)(((ubl - ub) > ps) ? ps : (ubl - ub))))
			{
				err++;
				break;
			}
		}
		if (err || (KA_T)u->u_procp != pa)
			continue;
	/*
	 * Allocate a local process structure.
	 */
		if (is_cmd_excl(u->u_comm, &pss, &sf))
			continue;
		alloc_lproc(pid, pgrp, (int)p->p_ppid, (UID_ARG)uid, u->u_comm,
			(int)pss, (int)sf);
		Plf = NULL;
	/*
	 * Save current working directory information.
	 */
		if (u->u_cdir) {
			alloc_lfile(CWD, -1);
			FILEPTR = (struct file *)NULL;
			process_node((caddr_t)u->u_cdir);
			if (Lf->sf)
				link_lfile();
		}
	/*
	 * Save root directory information.
	 */
		if (u->u_rdir) {
			alloc_lfile(RTD, -1);
			FILEPTR = (struct file *)NULL;
			process_node((caddr_t)u->u_rdir);
			if (Lf->sf)
				link_lfile();
		}
	/*
	 * Print information on the text file.
	 */
		if (u->u_exdata.vp != NULL) {
			alloc_lfile(" txt", -1);
			FILEPTR = (struct file *)NULL;
			process_node((caddr_t)u->u_exdata.vp);
			if (Lf->sf)
				link_lfile();
		}
	/*
	 * Loop through user's files.
	 */
		uf = (struct file **)u->u_ofile;
		for (i = 0; i < Var.v_nofiles; i++) {
			if (uf[i]) {
                                alloc_lfile(NULL, i);
				process_file(uf[i]);
				if (Lf->sf)
					link_lfile();
			}
		}
	/*
	 * Examine results.
	 */
		if (examine_lproc())
			return;
	}
}


/*
 * get_kernel_access() - access the required information in the kernel
 */

static void
get_kernel_access()
{
	unsigned long v;

/*
 * Open access to /dev/mem.
 */
	if ((Mem = open("/dev/mem", O_RDONLY, 0)) < 0) {
		(void) fprintf(stderr, "%s: can't open /dev/mem: %s\n",
			Pn, strerror(errno));
		Exit(1);
	}

#if	defined(WILLDROPGID)
/*
 * If kernel memory isn't coming from KMEM, drop setgid permission
 * before attempting to open the (Memory) file.
 */
	if (Memory)
		(void) dropgid();
#else	/* !defined(WILLDROPGID) */
/*
 * See if the non-KMEM memory file is readable.
 */
	if (Memory && !is_readable(Memory, 1))
		Exit(1);
#endif	/* defined(WILLDROPGID) */

/*
 * Open kernel memory accesses.
 */
	if ((Kd = open(Memory ? Memory : KMEM, O_RDONLY, 0)) < 0) {
		(void) fprintf(stderr, "%s: can't open %s: %s\n", Pn,
			Memory ? Memory : KMEM, strerror(errno));
		Exit(1);
	}

#if	defined(WILLDROPGID)
/*
 * Drop setgid permission, if necessary.
 */
	if (!Memory)
		(void) dropgid();
#else	/* !defined(WILLDROPGID) */
/*
 * See if the name list file is readable.
 */
	if (Nmlst && !is_readable(Nmlst, 1))
		Exit(1);
#endif	/* defined(WILLDROPGID) */

/*
 * Access kernel symbols.
 */
	(void) build_Nl(Drive_Nl);
	if (nlist(Nmlst ? Nmlst : N_UNIX, Nl) < 0) {
                (void) fprintf(stderr, "%s: can't read namelist from %s\n",
			Pn, Nmlst ? Nmlst : N_UNIX);
                Exit(1);
	}
	if (get_Nl_value("proc", Drive_Nl, (unsigned long *)&Kp) < 0 || !Kp) {
		(void) fprintf(stderr, "%s: can't read proc table info\n", Pn);
		Exit(1);
	}
	if (get_Nl_value("nproc", Drive_Nl, &v) < 0 || !v
	||  kread((KA_T)v, (char *)&Knp, sizeof(Knp)) || Knp < 1)
		Knp = PROCDFLT;
	v = (unsigned long)0;
	if (get_Nl_value("var", Drive_Nl, &v) < 0 || !v
	||  kread((KA_T)v, (char *)&Var, sizeof(Var))) {
		(void) fprintf(stderr,
			"%s: can't read kernel var struct from %#x\n", Pn, v);
		Exit(1);
	}
	Np = Var.v_proc;
}


/*
 * initialize() - perform all initialization
 */

void
initialize()
{
	get_kernel_access();
}


/*
 * kread() - read from kernel memory
 */

int
kread(addr, buf, len)
	KA_T addr;			/* kernel memory address */
	char *buf;			/* buffer to receive data */
	READLEN_T len;			/* length to read */
{
	int br;

	if (lseek(Kd, addr, L_SET) == (off_t)-1L)
		return(-1);
	br = read(Kd, buf, len);
	return((br == len) ? 0 : 1);
}


/*
 * mread() -- read from /dev/mem
 */

static int
mread(addr, buf, len)
	KA_T addr;			/* /dev/mem address */
	char *buf;			/* buffer to receive data */
	READLEN_T len;			/* length to read */
{
	int br;

	if (lseek(Mem, addr, L_SET) == (off_t)-1L)
		return(1);
	br = read(Mem, buf, len);
	return((br == len) ? 0 : 1);
}
