//  Module:     File
//  Version:    1.00    18-Jan-1989
//  Language:   C++
//  Purpose:    Defines a generic "file" class
//
//  Written by Scott Robert Ladd. This file is released into the public domain.

#include "string.h"
#include "file.hpp"

//**** CLASS: file ****

// constructor
file::file(char * filename, ACCESS_MODE m, TRANSLATION t)
    {
    name  = strdup(filename);
    mode  = m;
    trans = t;
    error = okay;

    if (trans == binary)
        {
        switch (mode)
            {
            case read_only :
                osdata = fopen(name,"rb");
                break;
            case write_only :
                osdata = fopen(name,"wb");
                break;
            case read_write :
                osdata = fopen(name,"w+b");
                break;
            case append_mode :
                osdata = fopen(name,"a+b");
                break;
            }
        }
    else
        {
        switch (mode)
            {
            case read_only :
                osdata = fopen(name,"r");
                break;
            case write_only :
                osdata = fopen(name,"w");
                break;
            case read_write :
                osdata = fopen(name,"w+");
                break;
            case append_mode :
                osdata = fopen(name,"a+");
                break;
            }
        }

    if (osdata == NULL)
        error = closed;
    else
        error = okay;
    }

// destructor
file::~file()
    {
    fclose(osdata);
    }

// common i/o methods
void file::read(void * rec, unsigned int len)
    {
    int res;

    if (mode == write_only)
        {
        error = access_violation;
        return;
        }

    res = fread(rec,len,1,osdata);

    if (ferror(osdata))
        error = failed;
    else
        error = okay;
    }

void file::write(void * rec, unsigned int len)
    {
    int res;

    if (mode == read_only)
        {
        error = access_violation;
        return;
        }

    res = fwrite(rec,len,1,osdata);

    if (ferror(osdata))
        error = failed;
    else
        error = okay;
    }

// file pointer movement methods
void file::goto_end()
    {
    int res;

    res = fseek(osdata,0L,SEEK_END);

    if (res != 0)
        error = failed;
    else
        error = okay;
    }

void file::goto_begin()
    {
    int res;

    res = fseek(osdata,0L,SEEK_SET);

    if (res != 0)
        error = failed;
    else
        error = okay;
    }

void file::goto_pos(long loc)
    {
    int res;

    res = fseek(osdata,loc,SEEK_SET);

    if (res != 0)
        error = failed;
    else
        error = okay;
    }

// data retrieval methods
void file::get_name(char *n)
    {
    strcpy(n,name);
    }

ACCESS_MODE file::get_mode()
    {
    return mode;
    }

TRANSLATION file::get_trans()
    {
    return trans;
    }

// error checking routines
int file::in_error()
    {
    if (error != okay)
        return 1;
    else
        return 0;
    }

ERROR_CODE file::get_error()
    {
    return error;
    }

int file::at_end()
    {
    if (feof(osdata))
        return 1;
    else
        return 0;
    }

//**** CLASS: text_file ****

// reading & writing methods
void text_file::read(char * s, unsigned int len)
    {
    if (mode == write_only)
        {
        error = access_violation;
        return;
        }

    if (NULL == fgets(s,len,osdata))
        if (feof(osdata))
            error = okay;
        else
            error = failed;
    else
        error = okay;
    }

void text_file::write(char * s)
    {
    if (mode == read_only)
        {
        error = access_violation;
        return;
        }

    if (0 == fputs(s,osdata))
        error = failed;
    else
        error = okay;
    }
