/* Copyright (c) 1998 by Michael J. Roberts.  All Rights Reserved. */
/*
Name
  plantrd.t - road locations
Function
  
Notes
  
Modified
  07/12/98 MJRoberts  - Creation
*/


/* ------------------------------------------------------------------------ */
/*
 *   Lightning daemon.  We fire lightning bolts randomly when in rooms
 *   exposed to the sky.  
 */
lightningDaemon: function(param)
{
    /* only fire lightning randomly, a certain percentage of the time */
    if (rand(100) > 30)
        return;

    /* tell the room to fire off some lightning */
    parserGetMe().location.roomLightning;
}


/* ------------------------------------------------------------------------ */
/*
 *   A barcodeItem is an item that contains a bar code that can be read
 *   with a scanner.  Since this class is based on the root 'object' type,
 *   it should always be multiply inherited with an appropriate base class
 *   (item, fixeditem, distantItem, etc).
 *   
 *   The doScanWithScanner(actor) method can be defined in each instance
 *   to provide specialized results when scanning the object.  By default,
 *   we'll display the scanCode value when scanned.
 *   
 *   A barcodeBaseItem is an item with a bar code but no vocabulary for a
 *   bar code.  This is useful for items that contain bar code items --
 *   this lets the player scan the container and get the contained bar
 *   code, which is reasonable when the container is a piece of paper or
 *   something else obviously exhibiting the bar code.  
 */
class barcodeBaseItem: object
    sdesc = "bar code"
    verDoRead(actor) =
    {
        "The bar code is meant to be scanned by a computer. ";
    }
    verDoReadWith(actor, iobj) = { }
    scanCode = '0.00'
    doScanWithScanner(actor) =
    {
        "You point the scanner at the bar code and pull the trigger.
        Pencil-thin lines of red light vibrate across the bar code
        for a few moments, then the scanner emits a short <i>bip</i>,
        and the display lights up to show:
        <tt><b><<self.scanCode>></b></tt>. ";
        
        portableScanner.displayText := self.scanCode;
    }
    ioShineOn(actor, dobj) =
    {
        if (dobj = portableScanner)
            self.doScanWithScanner(actor);
        else
            inherited.ioShineOn(actor, dobj);
    }
    doSynonym('ReadWith') = 'ScanWith'
    ioSynonym('ShineOn') = 'PointAt'
;

class barcodeItem: barcodeBaseItem
    noun = 'barcode' 'code' 'line' 'lines' 'bar-code'
    adjective = 'bar' 'vertical'
;

/* ------------------------------------------------------------------------ */
/*
 *   Generic add-in class for a location with random distant lightning.
 */
class distantLightningRoom: object
    roomLightning =
    {
        switch(rand(4))
        {
        case 1:
            "\bA lightning flash momentarily lights up the sky in the
            distance. ";
            break;

        case 2:
            "\bLightning flashes for a moment on the horizon. ";
            break;

        case 3:
            "\bA corner of the sky glows briefly with a lightning flash. ";
            break;

        case 4:
            "\bIn the distance, a flash of lightning glows in the sky. ";
            break;
        }
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Generic class for rooms on the road.  This class's main purpose is to
 *   tag road rooms so that the floating items associated with road rooms
 *   know that instances of this class are valid locations; it also
 *   provides some custom room behavior appropriate for road locations.  
 */
class roadRoom: room, distantLightningRoom
    noexit =
    {
        "It's too dangerous to attempt to travel the off-road terrain
        on foot in this darkness";
        if (flashlight.isIn(parserGetMe()) && flashlight.islit)
            ", even with the flashlight";
        ". ";
        return nil;
    }
    up =
    {
        "You seem to have left your wings at home. ";
        return nil;
    }
    down =
    {
        "You're about as close to the ground as you care to go
        just now. ";
        return nil;
    }
;

/*
 *   'road' item for road locations 
 */
class roadItem: fixeditem
    noun = 'road' 'shoulder'
    sdesc = "road"
    verIoPutOn(actor) = { }
    ioPutOn(actor, dobj) = { dobj.doDrop(actor); }
    verIoPutIn(actor) = { }
    ioPutIn(actor, dobj) = { dobj.doDrop(actor); }
;


/*
 *   moon - shows up in any roadRoom location 
 */
road_moon: fixeditem, floatingItem
    sdesc = "moon"
    noun = 'moon' 'moonlight'
    adjective = 'gibbous'
    location =
    {
        if (isclass(parserGetMe().location, roadRoom))
            return parserGetMe().location;
        else
            return nil;
    }
    ldesc = "The gibbous moon provides enough light to see your
             immediate surroundings. "
;


/* ------------------------------------------------------------------------ */
/*
 *   Your temporary ID card 
 */
myIdCard: readable
    isCardSized = true
    location = Me
    sdesc = "your temporary ID card"
    adesc = "your temporary ID card"
    thedesc = "your ID card"
    noun = 'card' 'badge' 'id'
    adjective = 'id' 'my' 'temporary'
    ldesc = "It's your temporary employee identification card; it's just
             a worn piece of cardboard with the Omegatron logo and the
             word <font color=red>TEMPORARY</font> stamped in red letters.
             It's amazing how inefficient the company's bureaucracy is;
             you've worked at Omegatron for three months now, and you're
             <i>still</i> stuck with this temporary badge, which doesn't
             have the little magnetic stripe that opens doors, so you
             constantly have to call Security to come unlock
             doors for you. "
;

/* ------------------------------------------------------------------------ */
/*
 *   Inside the car
 */
inCar: room
    sdesc = "In the car"
    ldesc = "The Toyunchknisk Piglet was imported from Blottnya
             during the brief period between the fall of the old iron-fisted
             regime and the ethnic unrest that divided the
             tiny country into several even tinier countries whose names
             you can't recall, since the press lost interest several years
             ago.  The car is almost comically spartan, so it's just like
             your boss to own one.  The only amenity the dashboard offers
             is a glove compartment; no radio, no air conditioner, no console
             armrest, no cup-holders.
             \n\tYou're sitting in the rather uncomfortable passenger's
             seat.  The driver's door is <<inCar_driverDoor.isopen
             ? "open" : "closed" >>, and your door is <<
             inCar_passengerDoor.isopen ? "open" : "closed" >>. "
    out = inCar_passengerDoor
    teeterwallerPreFollow(rm) =
    {
        /* open the driver's door if necessary */
        if (!inCar_driverDoor.isopen)
        {
            "Mr.\ Teeterwaller opens his door and comes with you.\b";
            inCar_driverDoor.isopen := true;
            inCar_driverDoor.otherside.isopen := true;
        }
        else
        {
            /* show the normal pre-follow message */
            inherited.teeterwallerPreFollow(rm);
        }
    }
    askedGivingUp = nil
    enterRoom(actor) =
    {
        /* do the normal work */
        inherited.enterRoom(actor);

        /* check for possible wait-for-help scenario */
        if (plantHill.sceneSeen && !self.askedGivingUp)
        {
            /* ask if they want to give up */
            "\b<q>Does this mean you're giving up?</q> Teeterwaller
            asks. <q>Are you going to make me figure this out by
            myself, and just wait here to be rescued?</q> ";

            /* set up for teeter to receive the answer */
            teeterwaller.askingGivingUp := true;
            parserGetMe().conversingWith := teeterwaller;
            
            /* note that we've asked */
            self.askedGivingUp := true;
        }
    }
;

inCar_glovebox: fixeditem, openable
    location = inCar
    noun = 'box' 'glovebox' 'compartment'
    adjective = 'glove'
    isopen = nil
    sdesc = "glove compartment"
;

garageOpener: item
    location = inCar_glovebox
    sdesc = "garage opener"
    noun = 'opener'
    adjective = 'garage' 'door'
    ldesc =
    {
        "It's a small plastic box with a large button. On the back
        is a small compartment";
        if (garageOpener_compartment.isopen)
        {
            ". ";
            garageOpener_compartment.ldesc;
        }
        else
            ", which is currently closed. ";
    }
    teeterNoted = nil
    initRadioCode = '55567562'
    radioCode = '55567562'
    doTake(actor) =
    {
        inherited.doTake(actor);
        if (actor = parserGetMe() && !self.teeterNoted
            && teeterwaller.location = actor.location)
        {
            "\bTeeterwaller notices you taking the opener. <q>That isn't
            mine,</q> he offers. <q>I think it belonged to the previous
            owner of the car.  I've never been able to return it, of
            course, since I bought the car at an estate sale.</q> ";
            self.teeterNoted := true;
        }
    }
    doOpen -> garageOpener_compartment
    doClose -> garageOpener_compartment
    doLookin -> garageOpener_compartment
    doSearch -> garageOpener_compartment
    ioPutIn -> garageOpener_compartment
    ioInstallIn -> garageOpener_compartment
;

garageOpener_compartment: fixeditem, openable
    location = garageOpener
    sdesc = "garage opener compartment"
    noun = 'compartment'
    adjective = 'garage' 'door' 'opener'
    isopen = nil
    ldesc =
    {
        "The compartment is currently ";
        if (self.isopen)
        {
            "open";
            if (itemcnt(self.contents) != 0)
            {
                ", and contains ";
                listcont(self);
                ". ";
            }
            else
            {
                ", and is just large enough to hold a battery (which
                is not currently installed).  A
                diagram in the compartment shows the proper way to
                install the battery, and a bank of tiny switches
                is at the back of the compartment behind where the
                battery goes.  Just above the switches, someone
                has written \"<<garageOpener.initRadioCode>>\". ";
            }
        }
        else
            "closed. ";
    }
    doSynonym('Inspect') = 'Lookin' 'Search'
    ioPutIn(actor, dobj) =
    {
        if (dobj != garageOpener_battery)
            "That doesn't fit; the compartment is made to hold
            a battery. ";
        else if (dobj.isIn(self))
            "It's already installed. ";
        else
        {
            "You install the battery, making sure to align it in the
            correct direction according to the diagram. ";
            dobj.moveInto(self);

            garageOpener_diagram.moveInto(nil);
            garageOpener_switch.moveInto(nil);
        }
    }
    ioSynonym('PutIn') = 'InstallIn'
    Grab(obj) =
    {
        if (obj = garageOpener_battery)
        {
            "Behind the battery, you find a diagram showing the proper
            way to install the battery, and a bank of tiny switches.\b";

            garageOpener_diagram.moveInto(self);
            garageOpener_switch.moveInto(self);
        }

        /* perform the default handling */
        inherited.Grab(obj);
    }
;

garageOpener_diagram: fixeditem
    noun = 'diagram'
    adjective = 'battery'
    sdesc = "battery diagram"
    ldesc = "The diagram shows the correct way to install the battery. "
;

garageOpener_switch: fixeditem
    sdesc = "bank of switches"
    noun = 'bank' 'switch' 'switches'
    adjective = 'tiny' 'slide'
    ldesc =
    {
        "Eight tiny slide switches are lined up next to each other.
        On the side of each switch, eight small notches show the
        possible positions; each switch can be set to a number from
        0 to 7.  Currently, the switches are in the following
        positions: << garageOpener.radioCode >>.
        Just above the switches, someone
        has written \"<<garageOpener.initRadioCode>>\". ";
    }
    verDoChange(actor) =
    {
        "If you want to change the position of the switches, you should
        set them to a specific sequence of values.  Currently, the
        switches are set to the following positions: <<
        garageOpener.radioCode >>. ";
    }
    verDoChangeTo(actor, iobj) =
    {
        if (iobj != numObj)
            "You can't set the switches to << iobj.thedesc >>. ";
    }
    doChangeTo(actor, iobj) =
    {
        local str;
        local i;
        local cur;
            
        /* 
         *   Get the actual string of numbers that they used.  (Note that
         *   objwords(2) is the list of words used for the indirect
         *   object; since it should be just a number, it's only one word,
         *   so we can just pick out the first element of the list.)  
         */
        str := objwords(2)[1];

        /* make sure it's the right length */
        if (length(str) != 8)
        {
            "There are eight switches; please be specific about all
            eight new settings. ";
            return;
        }

        /* make sure the values are all in range */
        for (i := 1 ; i <= length(str) ; ++i)
        {
            cur := substr(str, i, 1);
            if (cur < '0' || cur > '7')
            {
                "The switches can only be set to values from 0 to 7. ";
                return;
            }
        }

        /* set the new code */
        garageOpener.radioCode := str;

        /* tell them what happened */
        "You carefully manipulate the tiny switches into the new
        positions. ";
    }
    doSynonym('Change') = 'Set' 'Move' 'Slide' 'Push' 'Pull' 'Switch'
    doSynonym('ChangeTo') = 'SetTo' 'MoveTo' 'SlideTo' 'SwitchTo' 'TurnTo'
;

garageOpener_battery: item
    location = garageOpener_compartment
    sdesc = "battery"
    noun = 'battery' 'cell'
    adjective = 'aaa'
    ldesc = "It's an ordinary AAA cell. "
    verDoLookbehind(actor) =
    {
        if (self.location = garageOpener_compartment)
            "You'll have to remove the battery to see what's behind it. ";
        else
            inherited.verDoLookbehind(actor);
    }
    verDoInstall(actor) =
    {
        if (self.isIn(garageOpener_compartment))
            "It's already installed. ";
        else if (garageOpener_compartment.isIn(actor))
            garageOpener_compartment.verIoPutIn(actor);
        else
            "You'll have to be more specific about where you want
            to put the battery. ";
    }
    doInstall(actor) =
    {
        garageOpener_compartment.ioPutIn(actor, self);
    }
    doSynonym('Lookbehind') = 'Lookunder'
    doSynonym('PutIn') = 'InstallIn'
;

garageOpener_button: fixeditem
    location = garageOpener
    sdesc = "garage opener button"
    noun = 'button'
    adjective = 'garage' 'door' 'opener'
    ldesc = "You presumably push the button to transmit a radio signal
             to open a garage door somewhere. "
    verDoPush(actor) = { }
    doPush(actor) =
    {
        /* if the battery is installed, try sending the radio signal */
        if (garageOpener_battery.location = garageOpener_compartment)
        {
            switch(actor.location)
            {
            case driveway:
                if (driveway_door.tryGarageOpener(self.location.radioCode))
                    return;
                break;
                
            case warehouse:
                if (warehouse_door.tryGarageOpener(self.location.radioCode))
                    return;
                break;

            case plantPathWest:
                if (plantPathWest_ditchGate.
                    tryGarageOpener(self.location.radioCode))
                    return;
                break;

            case ditchAtFence:
            case canalInPlant:
                if (ditchAtFence_gate.
                    tryGarageOpener(self.location.radioCode))
                    return;
                break;

            default:
                break;
            }
        }

        /* if we got here, we didn't do anything special */
        "<q>Click.</q> ";
    }
;

inCar_driverDoor: lockableDoorway
    noun = 'door'
    plural = 'doors'
    adjective = 'driver\'s'
    location = inCar
    sdesc = "driver's door"
    isopen = nil
    doordest = roadAtCar
    otherside = roadAtCar_driverDoor
    islockable = true
    islocked = nil
;

inCar_passengerDoor: lockableDoorway
    noun = 'door'
    plural = 'doors'
    adjective = 'passenger\'s' 'my'
    location = inCar
    sdesc = "passenger's door"
    isopen = nil
    doordest = roadAtCar
    otherside = roadAtCar_passengerDoor
    islockable = true
    islocked = nil
;

inCar_driverSeat: fixeditem
    noun = 'seat'
    adjective = 'driver\'s'
    sdesc = "driver's seat"
    location = inCar
    ldesc =
    {
        if (teeterwaller.location = self.location)
            "Mr.\ Teeterwaller is occupying the driver's seat. ";
        else
            "The driver's seat is empty. ";
    }
    verDoEnter(actor) =
    {
        if (actor = parserGetMe())
            "It's Mr.\ Teeterwaller's car, so you should probably stick
            with the passenger's seat. ";
        else
            inherited.verDoEnter(actor);
    }
    doSynonym('Enter') = 'Board' 'Siton'
;

inCar_passengerSeat: fixeditem
    noun = 'seat'
    sdesc = "passenger's seat"
    adjective = 'passenger\'s'
    location = inCar
    ldesc =
    {
        "You're currently occupying the seat, which is rather
        hard and uncomfortable. ";
    }
    verDoEnter(actor) =
    {
        if (actor = parserGetMe())
            "You're already in the passenger's seat. ";
        else
            inherited.verDoEnter(actor);
    }
    doSynonym('Enter') = 'Board' 'Siton'
;

map: readable
    noun = 'map'
    sdesc = "map"
    location = inCar_backSeat
    ldesc =
    {
        "You find state route 142 on the map.  You don't know exactly
        how far you are from Bakerton, but you'd guess that it's at
        least fifty miles west, which puts you in this big empty expanse
        out in the valley.  The nearest city is Conners City,
        which is perhaps twenty miles east.  ";

        /* add some detail, depending on what they've seen so far */
        if (roadAtDetour.isseen)
            "The map shows a couple of crossroads in this general vicinity,
            but it doesn't indicate that any service stations are nearby. ";
        else
            "The map shows a couple of
            crossroads that could be a short distance to the east; it
            doesn't depict any businesses, but it's
            possible that one of the crossroads has a service station or
            at least someone with a phone. ";
    }
    doInspect(actor) =
    {
        if (actor = parserGetMe() && teeterwaller.location = actor.location)
        {
            "<q>Excellent idea,</q> Teeterwaller says.  The two of you
            unfold the map and inspect it.\b";

            self.ldesc;
        }
        else
            inherited.doInspect(actor);
    }
    doSynonym('Inspect') = 'Read'
;

inCar_backSeat: surface, fixeditem
    noun = 'seat'
    adjective = 'back' 'rear'
    sdesc = "back seat"
    location = inCar
    verDoEnter(actor) =
    {
        "It's too cramped to get into the back seat. ";
    }
    doSynonym('Enter') = 'Board' 'Siton'
;

carKey: keyItem
    noun = 'key'
    adjective = 'car' 'my'
    location = inCar_ignition
    sdesc = "car key"
    ldesc = "It's the key for Mr.\ Teeterwaller's car; this single key
             is used for the ignition and the doors. "
    verDoTurn(actor) =
    {
        if (self.location = inCar_ignition)
            inCar_ignition.verDoTurn(actor);
        else
            inherited.verDoTurn(actor);
    }
    doTurn(actor) =
    {
        if (self.location = inCar_ignition)
            inCar_ignition.doTurn(actor);
        else
            inherited.doTurn(actor);
    }
    verDoTake(actor) =
    {
        if (actor = parserGetMe() && self.isIn(teeterwaller.location))
            "Mr.\ Teeterwaller looks at you sharply, wondering if you've
            forgotten whose car this is. ";
        else
            inherited.doTake(actor);
    }
;

inCar_car: fixeditem
    sdesc = "car"
    noun = 'car' 'piglet'
    adjective = 'toyunchnisk' 'mr' 'mr.' 'teeterwaller\'s' 'boss\'s' 'boss\''
    location = inCar
    verDoTurnon(actor) = { inCar_ignition.verDoTurnon(actor); }
    doTurnon(actor) = { inCar_ignition.doTurnon(actor); }
    ldesc = { location.lookAround(true); }
    verDoUnboard(actor) = {}
    doUnboard(actor) = { actor.travelTo(actor.location.out); }
    carDaemon =
    {
        if (inCar_hazards.isActive && self.batteryLife != 0)
        {
            if (parserGetMe().location = inCar)
                "\bThe hazard lights click on and off in a steady rhythm. ";
            else if (parserGetMe().location = roadAtCar)
                "\bThe hazard lights flash on and off. ";
        }
        
        if (inCar_headlights.isActive)
        {
            self.batteryLife--;
            if (parserGetMe().location = self.location)
            {
                switch(self.batteryLife)
                {
                case 25:
                case 22:
                case 20:
                case 19:
                case 15:
                case 12:
                    if (parserGetMe().location = inCar
                        || parserGetMe().location = roadAtCar)
                        "\bThe headlights seem to be getting
                        a little dimmer. ";
                    break;
                    
                case 6:
                case 5:
                case 4:
                case 3:
                    if (parserGetMe().location = inCar
                        || parserGetMe().location = roadAtCar)
                        "\bThe headlights are growing very dim. ";
                    break;

                case 2:
                case 1:
                    if (parserGetMe().location = inCar
                        || parserGetMe().location = roadAtCar)
                        "\bThe headlights are nearly out. ";
                    break;
                    
                case 0:
                    if (parserGetMe().location = inCar
                        || parserGetMe().location = roadAtCar)
                    {
                        "\bThe headlights go out";
                        if (inCar_hazards.isActive)
                            ", and the hazard lights stop flashing";
                        ". ";
                    }
                    unnotify(self, &carDaemon);
                    break;
                }
            }
        }
    }
    batteryLife = 50
;

inCar_wheel: fixeditem
    sdesc = "steering wheel"
    noun = 'wheel'
    adjective = 'steering'
    location = inCar
    ldesc = "It's about what you'd expect. "
    verDoTurn(actor) = {}
    doTurn(actor) =
    {
        "You can turn it a little, but without the engine running
        this won't do you much good. ";
    }
;

inCar_ignition: fixeditem
    sdesc = "ignition"
    adesc = "an ignition"
    noun = 'ignition'
    location = inCar
    ldesc = "You put the key in the ignition and turn it to
             start the car.  The key is <<
             carKey.location = self ? "" : "not" >> in the ignition. "
    verDoTurn(actor) =
    {
        if (carKey.location != self)
            "The ignition won't turn without a key. ";
        else if (teeterwaller.location = self.location)
            "<q>Excuse me,</q> Mr.\ Teeterwaller says, <q>but I already
            tried that several times.  Let's not run down the battery.</q> ";
    }
    doTurn(actor) =
    {
        "The engine cranks and grinds, but it won't start. ";
    }
    doSynonym('Turn') = 'Turnon'
    verIoPutIn(actor) = { }
    ioPutIn(actor, dobj) =
    {
        if (dobj != carKey)
            "You should really only put the key in the ignition. ";
        else
            dobj.doPutIn(actor, self);
    }
;

inCar_dash: fixeditem
    sdesc = "dashboard"
    noun = 'dash' 'dashboard' 'board'
    adjective = 'dash'
    location = inCar
    ldesc = "The dash is austere but has the necessary controls
             and instruments: speedometer, odometer, fuel gauge,
             warning lights.  There's a switch for the headlights
             (currently <<inCar_headlights.isActive ? "on" : "off" >>),
             and another for the hazard lights (currently <<
             inCar_hazards.isActive ? "on" : "off" >>).  On the passenger
             side is a glove compartment, which is <<
             inCar_glovebox.isopen ? "open" : "closed">>. "
;

inCar_headlights: switchItem
    sdesc = "headlight switch"
    noun = 'switch' 'headlights' 'lights'
    adjective = 'headlight'
    location = inCar
    isActive = nil
    ldesc =
    {
        "The headlight switch is currently << self.isActive ?
        "on" : "off. " >>";
        if (self.isActive)
        {
            if (inCar_car.batteryLife > 0)
                ", and the lights are shining. ";
            else
                ", but the lights are dark; the battery must be dead. ";
        }
    }
    verDoSwitch(actor) = { "You can turn them either on or off. "; }
    doTurnon(actor) =
    {
        if (inCar_car.batteryLife > 0)
            "The headlights come on, illuminating the road outside. ";
        else
            "Nothing happens; the battery must be dead. ";
        self.isActive := true;
    }
;

inCar_hazards: switchItem
    sdesc = "hazard light switch"
    noun = 'switch' 'lights' 'hazards'
    adjective = 'hazard' 'light'
    location = inCar
    isActive = true
    verDoSwitch(actor) = { "You can turn them either on or off. "; }
    ldesc =
    {
        "The hazard light switch is currently << self.isActive ?
        "on" : "off. " >>";
        if (self.isActive)
        {
            if (inCar_car.batteryLife > 0)
                ", and the lights are flashing. ";
            else
                ", but the lights are dark; the battery must be dead. ";
        }
    }
    doTurnon(actor) =
    {
        if (inCar_car.batteryLife > 0)
            "The hazard lights start flashing. ";
        else
            "Nothing happens; the battery must be dead. ";
        self.isActive := true;
    }
;

inCar_lights: fixeditem
    sdesc = "warning light"
    location = inCar
    noun = 'light' 'lights'
    adjective = 'warning'
    ldesc =
    {
        if (inCar_car.batteryLife > 0 && carKey.location = inCar_ignition)
            "The warning lights are all turned on; this usually
            indicates that something may be wrong with the car. ";
        else
            "Everything on the dash is dark. ";
    }
;

inCar_speed: fixeditem
    sdesc = "speedometer"
    location = inCar
    noun = 'speedometer'
    ldesc = "Not surprisingly, it reads 0. "
;

inCar_odometer: fixeditem
    sdesc = "odometer"
    adesc = "an odometer"
    location = inCar
    noun = 'odometer'
    ldesc = "It says 55,377, and holding. "
;

inCar_gauge: fixeditem
    sdesc = "fuel gauge"
    location = inCar
    noun = 'gauge'
    adjective = 'fuel' 'gas'
    ldesc = "It reads around half a tank, so lack of fuel must
             not be the problem. "
;

teeterwallerJacket: item
    noun = 'jacket' 'coat'
    adjective = 'mr' 'mr.' 'teeterwaller\'s' 'boss\'s' 'boss\'' 'suit'
    sdesc = "jacket"
    location = inCar_backSeat
    ldesc = "It's a dull brown suit jacket, matching Teeterwaller's
             pants. "
    verDoWear(actor) =
    {
        if (actor = parserGetMe())
            "The jacket is too small for you. ";
        else
            inherited.verDoWear(actor);
    }
;

dufon: readable
    noun = 'journal' 'magazine'
    adjective = 'dufon'
    location = inCar_backSeat
    isseen = nil
    sdesc = "<<self.isseen ? "DUFON journal" : "magazine">>"
    ldesc =
    {
        "\t<b>DUFON Journal</b>
         \n\t<i>The Official Journal of the Discovery UFO Network</i>
         \bThe cover has a photo of a streak of light in the dusk sky
         above a wooded hilltop, with the caption <i>The Milltown
         Sighting: A New Re-Examination, Part III.</i>
         \bYou thumb through the magazine.  There's an article by an
         expert on official cover-ups, insisting that the government
         open an investigation into a previous investigation of an
         alleged cover-up.  Another article describes the many
         revelations that have come out of Blottnya since the break-up
         of the old government, including photos from Blottnya's old
         space program that show that the <q>face</q> on Mars bears an
         uncanny resemblance a famous Eastern European television star,
         raising the possibility that the ancient Martians not only
         were capable of space travel but time travel as well.
         \bThe interesting thing about the journal is that they're
         obviously quite serious; it's not a supermarket tabloid. ";
        self.isseen := true;
    }
;

tycheSpec: readable
    noun = 'specification' 'spec' 'spiral' 'page' 'pages' 'diagrams'
           'schematics' 'manual'
    adjective = 'note' 'tyche' 'project' 'plastic' 'photocopied'
                'spec' 'specification'
    sdesc = "Project Tyche manual"
    location = inCar_backSeat
    isAskedAbout = nil
    isExtraInfoKnown = nil
    ldesc =
    {
        "It's a stack of photocopied pages, bound with a plastic spiral.
        The cover page reads <q>Project Tyche Specification</q> at
        the top, and <q>Omegatron Corporation - Copyright 1972 -
        Confidential</q> at the bottom.  You glance through it, and find
        page after page of electronic diagrams and schematics
        interspersed with dense blocks of technical information and
        parts lists.  None of it is familiar to you, since you're in
        the software part of the company. ";

        if (!self.isAskedAbout)
            "\bIt occurs to you that you should ask Mr.\ Teeterwaller
            about the project some time, since you're always
            interested in hearing stories about the early days at
            the company. ";
    }
;


/* ------------------------------------------------------------------------ */
/*
 *   Road outside car 
 */

roadAtCar: roadRoom
    sdesc = "Road, near car"
    ldesc =
    {
        "The road stretches off into the dim moonlight to the east and
        west.  Mr.\ Teeterwaller's car sits in the shoulder, parked just
        slightly off the road. ";
    }
    in = roadAtCar_passengerDoor
    west = roadWest
    east = roadEast
    teeterwallerPreFollow(rm) =
    {
        /* open the driver's door if necessary */
        if (rm = inCar && !inCar_driverDoor.isopen)
        {
            "Mr.\ Teeterwaller opens his door and comes with you.\b";
            inCar_driverDoor.isopen := true;
            inCar_driverDoor.otherside.isopen := true;
        }
        else
        {
            /* show the normal pre-follow message */
            inherited.teeterwallerPreFollow(rm);
        }
    }
;

roadAtCar_road: roadItem
    location = roadAtCar
    ldesc = "The road runs east and west. "
;

roadAtCar_driverDoor: lockableDoorway
    noun = 'door'
    plural = 'doors'
    adjective = 'driver\'s'
    location = roadAtCar
    sdesc = "driver's door"
    isopen = nil
    doordest = inCar
    otherside = inCar_driverDoor
    islockable = true
    islocked = nil
    mykey = carKey
    verDoEnter(actor) =
    {
        if (actor = parserGetMe())
            "You should probably get in the passenger's side, since this
            isn't your car. ";
        else
            inherited.verDoEnter(actor);
    }
    doSynonym('Enter') = 'Board'
;

roadAtCar_passengerDoor: lockableDoorway
    noun = 'door'
    plural = 'doors'
    adjective = 'passenger\'s'
    location = roadAtCar
    sdesc = "passenger's door"
    isopen = nil
    doordest = inCar
    otherside = inCar_passengerDoor
    islockable = true
    islocked = nil
    mykey = carKey
    verDoEnter(actor ) = { }
    doEnter(actor) = { actor.travelTo(actor.location.in); }
    doSynonym('Enter') = 'Board'
;

roadAtCar_headlights: fixeditem
    sdesc = "headlight"
    noun = 'headlights' 'lights' 'light'
    location = roadAtCar
    ldesc =
    {
        if (inCar_car.batteryLife > 0 && inCar_headlights.isActive)
            "The headlights illuminate a small part of the road
            in front of the car. ";
        else
            "The headlights are dark. ";
    }
    verDoTurnon(actor) = { "You can only do that from inside the car. "; }
    verDoTurnoff(actor) = { "You can only do that from inside the car. "; }
;

roadAtCar_car: fixeditem
    noun = 'car' 'piglet'
    adjective = 'toyunchnisk' 'mr' 'mr.' 'teeterwaller\'s' 'boss\'s' 'boss\''
    location = roadAtCar
    sdesc = "car"
    ldesc = "The small economy car is parked in the shoulder just off
             the road.  The driver's door is <<
             roadAtCar_driverDoor.isopen ? "open" : "closed">>, and
             the passenger's door is <<
             roadAtCar_passengerDoor.isopen ? "open" : "closed">>. "
    verDoPush(actor) =
    {
        if (teeterwaller.location = actor.location)
            "<q>I don't think we'll be able to push it far enough,</q>
            Mr.\ Teeterwaller says. ";
        else
            "It's much too far to push the car to a service station.  You
            should probably leave the car here and find help. ";
    }
    doSynonym('Push') = 'Pull'
    verDoBoard(actor) = {}
    doBoard(actor) = { actor.travelTo(actor.location.in); }
    doSynonym('Board') = 'Enter'
;

/* ------------------------------------------------------------------------ */
/*
 *   Road to west of car 
 */
class distantCar: distantItem
    sdesc = "car"
    noun = 'car' 'piglet' 'light' 'lights' 'headlights'
    adjective = 'toyunchnisk' 'hazard' 'teeterwaller\'s' 'boss\'s' 'boss\''
    location = roadWest
    ldesc =
    {
        "You can't see much from this distance";
        if (inCar_car.batteryLife > 0
            && (inCar_headlights.isActive || inCar_hazards.isActive))
        {
            ", although you can clearly see its ";
            if (inCar_headlights.isActive)
            {
                "headlights shining";
                if (inCar_hazards.isActive)
                    " and ";
            }
            if (inCar_hazards.isActive)
                "hazard lights blinking";
        }
        ". ";
    }
;

roadWest: roadRoom
    sdesc = "Road"
    ldesc = "The road continues east and west. To the east you can just make
             out Teeterwaller's car. It's too dark to see much to the west,
             although you remember from having driven this stretch that
             there's nothing for a long ways. "
    east = roadAtCar
    west =
    {
        if (teeterwaller.location = self)
            "Teeterwaller stops you. <q>We just came from this direction,</q>
            he says. <q>There's nothing this way for at least twenty
            miles.</q> ";
        else
            "You remember from just having driven from this direction that
            there's nothing for many miles this way; you couldn't possibly
            make it to the nearest town by morning, and the thought of
            walking all night is enough to turn you back. ";
        return nil;
    }
;

roadWest_car: distantCar
    location = roadWest
;

roadWest_road: roadItem
    location = roadWest
    ldesc = "The road continues east and west. "
;

/* ------------------------------------------------------------------------ */
/*
 *   Road east of car 
 */
roadEast: roadRoom
    sdesc = "Road at driveway"
    ldesc = "A gravel driveway leads away from the road to the
             north.  The road continues east and west. To the west you can
             just make out Mr.\ Teeterwaller's car; it's too dark to see
             much to the east."
    west = roadAtCar
    east = roadEast2
    north = driveway
;

roadEast_driveway: fixeditem
    noun = 'driveway'
    adjective = 'gravel'
    sdesc = "gravel driveway"
    location = roadEast
    ldesc = "The driveway leads north. "
;

roadEast_car: distantCar
    location = roadEast
;

roadEast_road: roadItem
    location = roadEast
    ldesc = "The road continues east and west. "
;

/*
 *   Item class for things that are part of the detour.  This class serves
 *   only to mark objects that must be removed from the game when the
 *   detour disappears. 
 */
class detourItem: object
;


/* ------------------------------------------------------------------------ */
/*
 *   Driveway 
 */
driveway: roadRoom
    sdesc = "Driveway"
    ldesc =
    {
        "The driveway ends at <<driveway_door.isopen
        ? "an open" : "a closed" >> roll-up metal door in a large, windowless
        building to the north.  The door looks like it belongs on a
        one-car garage. ";

        "Bolted to the side of the building is a heavy steel chain";
        if (length(driveway_chain.attachmentList) != 0)
            ", attached to which is <<
            driveway_chain.attachmentList[1].adesc>>";
        ". ";
        
        "The driveway leads south, back to the road. ";
    }
    south = { return self.checkLeave(roadEast); }
    north = { return self.checkLeave(driveway_door); }
    in = { return self.checkLeave(driveway_door); }
    checkLeave(dest) =
    {
        if (harness.isIn(parserGetMe())
            && harness.isAttachedTo(driveway_chain))
        {
            "You can't go far with the harness as long as it's
            chained to the wall. ";
            return nil;
        }
        else
            return dest;
    }
;

driveway_driveway: fixeditem
    sdesc = "driveway"
    noun = 'driveway'
    adjective = 'gravel'
    location = driveway
    ldesc = "The driveway ends at the building to the north,
             and continues back to the road to the south. "
;

driveway_road: distantItem
    location = driveway
    sdesc = "road"
    noun = 'road'
    ldesc = "The road is just to the south. "
;

driveway_building: fixeditem
    noun = 'building' 'warehouse'
    sdesc = "building"
    location = driveway
    ldesc = "The building is large, square, and featureless, perhaps
             two stories tall, although it's hard to be sure without any
             windows to establish the scale.  At the end of the driveway
             is a roll-up metal door, suitable for a one-car garage;
             the door is currently << driveway_door.isopen ? "open" : "closed"
             >>. "
    verDoEnter(actor) = { }
    doEnter(actor) =
    {
        if (driveway_door.isopen)
            actor.travelTo(self.location.north);
        else
            "You must open the door before you can enter the building. ";
    }
;

class warehouseDoorClass: doorway
    noun = 'door'
    adjective = 'roll-up' 'metal'
    sdesc = "roll-up door"
    isopen = nil
    noAutoOpen = true
    ldesc =
    {
        if (isopen)
            "The door is open. ";
        else
            "It's a metal door the size and style of a one-car garage
            door.  It's currently closed. ";
    }
    verDoUnlock(actor) = { "You see no locking mechanism. "; }
    verDoLock(actor) = { "You see no locking mechanism. "; }
    verDoOpen(actor) =
    {
        if (self.isopen)
            inherited.verDoOpen(actor);
        else
            "Try as you might, you are unable to move the door. ";
    }
    verDoClose(actor) =
    {
        if (self.isopen)
            "You are unable to move the door. ";
        else
            inherited.verDoClose(actor);
    }
    verDoPull(actor) =
    {
        "You use all your strength, but you are unable to move the door. ";
    }
    doSynonym('Pull') = 'Push' 'Move' 'Lift'
    radioCode = '55567562'
    pointsAwarded = nil
    tryGarageOpener(code) =
    {
        /* if the code matches my code, open/close the door */
        if (self.radioCode = code)
        {
            /* tell them what happened */
            "You press the button. A moment later, you hear a
            <i>thunk</i>, and the door starts << self.isopen
            ? "closing" : "opening" >>.  The door slowly rolls
            << self.isopen ? "down" : "up" >> its tracks,
            and finally stops when fully << self.isopen
            ? "closed" : "open" >>. ";

            /* update my state */
            self.isopen := !self.isopen;
            self.otherside.isopen := !self.otherside.isopen;

            /* score some points if appropriate */
            if (maybeIncscore(2, 'opening the warehouse door', self))
            {
                /* 
                 *   note in the other side of the door that we awarded
                 *   the points 
                 */
                self.otherside.pointsAwarded := true;
            }

            /* indicate that the opener did something interesting */
            return true;
        }
        else
        {
            /* my code doesn't match; we didn't do anything interesting */
            return nil;
        }
    }
;

driveway_door: warehouseDoorClass
    location = driveway
    doordest =
    {
        if (doggy.isGuarding)
        {
            "The dog jumps into your path, snarling and growling, refusing
            to let you pass. ";
            return nil;
        }
        else
            return warehouse;
    }
    otherside = warehouse_door
;


class harnessHookableItem: attachableItem, fixeditem
    verifyAttachTo(actor, otherObj) =
    {
        /* do the default processing */
        if (!inherited.verifyAttachTo(actor, otherObj))
            return nil;
        
        /* only the harness can be attached to a harness hookable item */
        if (otherObj != harness)
        {
            "You see no way to attach <<otherObj.thedesc>> to <<
            self.thedesc>>. ";
            return nil;
        }

        /* looks okay */
        return true;
    }
;

driveway_chain: harnessHookableItem
    sdesc = "chain"
    noun = 'chain' 'clip' 'link' 'links'
    adjective = 'heavy' 'steel'
    location = driveway
    ldesc =
    {
        "It's about ten feet long and made of heavy steel links.
        One end is bolted to the wall of the building; at the other
        end is a clip";

        if (length(self.attachmentList) = 0)
            " for attaching the chain to a collar or harness. ";
        else
            ", which is attached to <<self.attachmentList[1].thedesc>>. ";
    }
    verDoTake(actor) =
    {
        "The chain is securely bolted to the wall. ";
    }
    attachmentList = [harness]

    processAttachTo(actor, otherObj, isFirst) =
    {
        inherited.processAttachTo(actor, otherObj, isFirst);
        "The chain is now clipped onto the harness. ";
    }

    processDetachFrom(actor, dobj, isFirst) =
    {
        /* inherit default behavior */
        inherited.processDetachFrom(actor, dobj, isFirst);
        "You unfasten the chain from the harness. ";
    }
;

ausVerb: deepverb
    verb = 'aus'
    action(actor) =
    {
        if (doggy.isIn(actor.location))
            doggy.actorAction(self, nil, nil, nil);
        else
            "You are being obtuse. ";
    }
;

doggy: Actor
    sdesc = "huge dog"
    noun = 'dog' 'doggy'
    adjective = 'huge' 'moose' 'blottnian'
    location = driveway
    isGuarding = true
    isHim = true
    teeterNoted = nil
    isseen = nil
    ldesc =
    {
        "The dog is bigger than you are. ";

        if (blottnian_chapter_3.isseen)
            "He looks just like the Blottnian Moose Dog from the
            Blottnian textbook";
        else
            "You don't recognize the breed";
        
        "; he has tall, pointy ears, a long muzzle, a long bushy tail,
        and a body like a bear.  His hair is light brown, with a few
        darker markings. ";

        if (harness.isIn(self))
        {
            "He has a harness around his chest";
            if (length(harness.attachmentList) != 0)
                ", which is attached to <<harness.attachmentList[1].adesc>>";
            ". ";
        }

        if (!self.teeterNoted && teeterwaller.isIn(self.location))
        {
            "\b<q>I saw a television program where they explained that
            guard dogs are usually trained using German commands,</q>
            Teeterwaller says.  <q>I think you just have to say <q>aus,</q>
            which is German for <q>out,</q> to call off an attack.</q> ";
  
            self.teeterNoted := true;
        }

        self.isseen := true;
    }
    actorDesc =
    {
        if (harness.isIn(self) && harness.isAttachedTo(driveway_chain))
        {
            "A huge dog chained to the side of the building ";
            if (self.isGuarding)
                "eyes you warily. ";
            else 
                "wags his tail at you. ";
        }
        else
            "A huge dog is here, happily puttering around. ";
    }
    verGrab(obj) =
    {
        if (self.isGuarding)
            "The dog tenses up and makes a long growl so low you can
            more feel it than hear it; you think better of trying
            to take <<obj.thedesc>>. ";
    }
    harnessRemoved = nil
    Grab(obj) =
    {
        /* make a note of the harness when removed for the first time */
        if (obj = harness && !self.harnessRemoved)
        {
            "The dog stands still for you to remove the harness, and
            happily romps around once it's off.\b";
            self.harnessRemoved := true;
        }
        
        /* 
         *   make sure any unlisted objects I'm carrying become listed
         *   once they're removed from me 
         */
        obj.isListed := true;
        inherited.Grab(obj);
    }
    hasSaidAus = nil
    actorAction(v, d, p, i) =
    {
        if (isclass(v, blottVerb))
        {
            local showEnd := true;
            
            switch(v)
            {
            case blottSitVerb:
                "The dog stands up on his hind legs for a few moments";
                break;

            case blottStandVerb:
                "The dog sits down for a moment";
                break;

            case blottSpeakVerb:
                "The dog closes his mouth and becomes completely silent
                for a moment, then ";
                if (self.isGuarding)
                    "resumes his menacing stare and low growl. ";
                else
                    "goes back to puttering about. ";
                showEnd := nil;
                break;
                    
            case blottQuietVerb:
                "The dog barks several times";
                break;
                    
            case blottAttackVerb:
                if (self.isGuarding)
                {
                    "The dog suddenly becomes very friendly.  He wags
                    his tale and whines at you happily. ";

                    if (teeterwaller.isIn(self.location))
                        "\b<q>Whatever you said, it worked!</q> Teeterwaller
                        exclaims.  He cautiously approaches the dog and
                        pets him, getting lots of licking and tail-wagging
                        in return. ";

                    self.isGuarding := nil;
                    incscore(10, 'befriending the dog');
                    showEnd := nil;
                }
                else
                    "The dog comes over and licks you";
                break;
                
            case blottBefriendVerb:
                if (self.isGuarding)
                    "The dog growls and barks even more menacingly at
                    you for several seconds";
                else
                {
                    "The dog looks at you curiously, but seems to decide
                    he'd rather stay friends. ";
                    showEnd := nil;
                }
                break;

            case blottnianNullVerbs:
                "The dog looks at you alertly, as though he partially
                recognizes the words but doesn't know what you're trying
                to say. ";
                showEnd := nil;
                break;
            }

            if (showEnd)
            {
                if (self.isGuarding)
                    ", then resumes his guarding posture. ";
                else
                    ", then goes back to his puttering. ";
            }

            exit;
        }
        else if (self.isGuarding)
        {
            if (v = ausVerb && !self.hasSaidAus)
            {
                "The dog rears up on his hind legs and barks angrily at
                you. ";
                if (teeterwaller.isIn(self.location))
                    "\bTeeterwaller practically jumps into the air at the
                    barking and runs back a ways further from the dog.
                    <q>I guess this must not be a German dog,</q> he
                    says. ";
                "\bThe dog gradually calms down again, and resumes
                his guarding posture. ";

                self.hasSaidAus := true;
            }
            else
                "The dog just snarls at you menacingly. ";
            exit;
        }
        else
        {
            "He looks at you, cocks his head for a moment, then goes
            back to puttering around. ";
            exit;
        }
    }
    verIoTakeOff(actor) = { }
    ioTakeOff(actor, dobj) =
    {
        if (dobj.isIn(self) && dobj.isworn)
        {
            if (dobj = harness)
                dobj.moveInto(actor);
            else
                dobj.doUnwear(self);
        }
        else
            "You see no way to do that. ";
    }
    verIoPutOn(actor) = { }
    ioPutOn(actor, dobj) =
    {
        if (dobj = harness)
        {
            if (dobj.isIn(self) && dobj.isworn)
                "The harness is already on the dog. ";
            else
            {
                dobj.moveInto(self);
                dobj.isworn := true;
                "The dog stands still long enough for you to
                put the harness on him and fasten it up. ";
            }
        }
        else
            "You see no way to do that. ";
    }
    verRemoveHarness =
    {
        if (self.isGuarding)
        {
            "The dog tenses up as you approach and snarls menacingly.
            You decide to keep at a safe distance. ";
            return nil;
        }
        else
            return true;
    }
    verDoPet(actor) = { }
    verDoAttack(actor) = { }
    doAttack(actor) =
    {
        if (self.isGuarding)
            "You can't be serious. ";
        else
            "But why would you want to do that?  He's being so friendly. ";
    }
    verDoAttackWith(actor, iobj) = { }
    doAttackWith(actor, iobj) = { self.doAttack(actor); }
    doPet(actor) =
    {
        if (self.isGuarding)
            "The dog is acting too menacing to pet him right now. ";
        else
            "The dog wags his tail at you happily, and returns to his
            puttering around. ";
    }
    disavow = "The dog does not answer. He must not be a talking dog. "
    verDoUntie(actor) =
    {
        if (harness.isIn(self) && harness.isworn
            && length(harness.attachmentList) != 0)
            "You'll need to be more specific about how to do that.
            Take a closer look at the harness. ";
        else
            "The dog is not tied. ";
    }

    doSynonym('Untie') = 'Free' 'Release'
;

harness: clothingItem, attachableItem
    sdesc = "dog harness"
    location = doggy
    noun = 'harness' 'ring'
    adjective = 'dog'
    isListed = nil
    isworn = true
    ldesc =
    {
        "The harness goes around a dog's chest. Even though it's
        meant for a dog, this one seems like it's big enough
        for a bear. At the top of the harness is a ring that's
        used to attach the harness to a chain or leash";
        if (length(self.attachmentList) != 0)
            "; it's currently attached to <<self.attachmentList[1].thedesc>>";
        ". ";
    }
    attachmentList = [driveway_chain]
    verDoWear(actor) =
    {
        if (length(self.attachmentList) != 0)
            "It's going to be hard to get into it while it's
            attached to <<self.attachmentList[1].thedesc>>. ";
        else
            inherited.verDoWear(actor);
    }
    doWear(actor) =
    {
        inherited.doWear(actor);
        "Tightening the harness securely around your chest,
        you can't help but feel a little silly. ";
    }
    verDoUnwear(actor) =
    {
        if (!self.location.verRemoveHarness)
            return;

        if (length(self.attachmentList) != 0)
            "It's going to be hard to get out of it while it's
            attached to <<self.attachmentList[1].thedesc>>. ";
        else if (self.location != doggy)
            inherited.verDoUnwear(actor);
    }
    doUnwear(actor) =
    {
        if (self.isIn(doggy) && self.isworn)
            self.moveInto(actor);
        else
            inherited.doUnwear(actor);
    }
    
    doTake(actor) =
    {
        if (self.location = doggy)
            self.moveInto(actor);
        else
            inherited.doTake(actor);
    }

    verifyAttachTo(actor, otherObj) =
    {
        /* do the standard tests first */
        if (!inherited.verifyAttachTo(actor, otherObj))
            return nil;

        /* only allow it to be attached to one thing at a time */
        if (length(self.attachmentList) != 0)
        {
            "\^<<self.thedesc>> is already attached
            to <<self.attachmentList[1].thedesc>>. ";
            return nil;
        }

        /* only allow hooking it to harnessHookableItem's */
        if (!isclass(otherObj, harnessHookableItem))
        {
            "You see no way to do that. ";
            return nil;
        }

        /* it looks acceptable */
        return true;
    }

    verifyDetachFrom(actor, otherObj) =
    {
        /* do the standard tests first */
        if (!inherited.verifyDetachFrom(actor, otherObj))
            return nil;

        /* check to see if the harness can be removed from its location */
        if (!self.location.verRemoveHarness)
            return nil;

        /* looks fine */
        return true;
    }

    verDoTakeOff(actor, iobj) =
    {
        /* 
         *   treat it as removing clothing if it's in the indirect object,
         *   otherwise treat it as the default disconnection 
         */
        if (self.isIn(iobj) && self.isworn)
            self.verDoUnwear(iobj);
        else
            inherited.verDoTakeOff(actor, iobj);
    }

    verifyRemove(actor) =
    {
        if (length(self.attachmentList) != 0)
            "You'll have to detach <<self.thedesc>> from <<
            self.attachmentList[1].thedesc>> first. ";
        else
            inherited.verifyRemove(actor);
    }

    doSynonym('Detach') = 'Untie'
;

/* ------------------------------------------------------------------------ */
/*
 *   Road further east of car 
 */
roadEast2: roadRoom
    sdesc = "Road at ditch"
    ldesc =
    {
        "The road continues east and west, rising slightly as it
        crosses a deep ditch.  Metal guard rails run alongside the
        road on both sides to prevent vehicles from running into
        the ditch";
        if (self.pathseen)
            "; a narrow foot path leads northeast from the end of
            the north rail";
        ". ";
        if (roadAtDetour_barricades.location != nil)
            "You can't see anything through
            the darkness to the west, but to the east you can see some
            dimly flashing lights. ";
        else
            "In the darkness, you can't see far down the road
            in either direction. ";
    }
    west = roadEast
    east = roadAtDetour
    roomSwim(actor) =
    {
        "It's too dangerous to jump into the ditch from here. ";
    }
    pathseen = nil
    ne =
    {
        if (self.pathseen)
            return ditchPath;
        else
            return inherited.ne;
    }
    north = { return self.south; }
    south = 
    {
        "It's too dangerous to jump into the ditch from here. ";
        return nil;
    }
;

roadEast2_rails: fixeditem
    noun = 'rail' 'rails' 'guardrails'
    adjective = 'guard' 'metal' 'north' 'south'
    location = roadEast2
    sdesc = "guard rail"
    ldesc =
    {
        "The guard rails are heavily dented from years of collisions. ";
        if (!roadEast2.pathseen)
        {
            "At the right end of the rail on the north side of the road,
            you notice a narrow foot path leading northeast. ";
            incscore(1, 'finding the path along the ditch');
            roadEast2.pathseen := true;
            roadEast2_path.moveInto(roadEast2);
        }
    }
    verDoClimb(actor) =
    {
        "The guard rail isn't very tall; if you want to jump in the
        ditch, just say so. ";
    }
    doSynonym('Climb') = 'Climbover' 'Board' 'Standon'
;

roadEast2_path: fixeditem
    noun = 'path' 'footpath'
    adjective = 'narrow' 'foot'
    sdesc = "path"
    ldesc = "The path is at the right end of the north rail; it leads
             northeast. "
;

class ditchItem: fixeditem
    noun = 'ditch' 'water'
    sdesc = "ditch"
    verDoJumpin(actor) = {}
    doJumpin(actor) =
    {
        "It's too dangerous to jump into the ditch from here. ";
    }
    doSynonym('Jumpin') = 'Swimin'
    verDoEnter(actor) = {}
    doEnter(actor) =
    {
        "It's too dangerous to get into the ditch from here. ";
    }
    doSynonym('Enter') = 'Board'
    verIoPutIn(actor) = {}
    ioPutIn(actor, dobj) =
    {
        "You'd better not; you probably wouldn't be able to find
        <<dobj.thedesc>> again, and even if you could it would
        be soaked in mud. ";
    }
;

roadEast2_ditch: ditchItem
    location = roadEast2
    ldesc = "It's a deep ditch, running north and south under
             the road. "
;

roadEast2_road: roadItem
    location = roadEast2
    ldesc = "The road continues east and west. "
;

roadEast2_lights: distantItem, detourItem
    noun = 'light' 'lights'
    adjective = 'flashing' 'dimly'
    location = roadEast2
    sdesc = "flashing light"
    ldesc = "The lights look like they may be road hazard warning
             lights, but you can't see much from this distance. "
;

/* ------------------------------------------------------------------------ */
/*
 *   path along ditch 
 */
ditchPath: room, distantLightningRoom
    sdesc = "Path"
    south = roadEast2
    sw = roadEast2
    north =
    {
        "The path veers toward the ditch, then slopes down along
        a terrace cut into the ditch's bank until it reaches the
        bottom of the ditch.\b";
        return ditchPathNorth;
    }
    ldesc =
    {
        "The path runs north and south alongside a deep ditch,
        which is to the west.  The road lies to the south. ";
    }
    down = { return self.west; }
    west =
    {
        "The bank is too steep to climb down here. ";
        return nil;
    }
    noexit =
    {
        "It's too dangerous to leave the path in the darkness. ";
        return nil;
    }
;

ditchPath_road: distantItem
    location = ditchPath
    noun = 'road'
    sdesc = "road"
    ldesc = "The road is just to the south. "
;

dithPath_path: fixeditem
    location = ditchPath
    noun = 'path'
    sdesc = "path"
    ldesc = "It's a narrow foot path running north and south. "
;

ditchPath_ditch: ditchItem
    location = ditchPath
    ldesc = "The ditch runs north and south alongside the path.
             The banks are steep here. "
    doEnter(actor) = { "The bank is too steep to climb down here. "; }
;

/* ------------------------------------------------------------------------ */
/*
 *   Ditch path north 
 */
ditchPathRoom: room, distantLightningRoom
    west =
    {
        "You don't have any desire to wade into the stagnant water. ";
        return nil;
    }
    east =
    {
        "The bank is too steep to climb. ";
        return nil;
    }
    roomSwim(actor) =
    {
        "You have no desire to immerse yourself in the stagnant water. ";
    }
;    

ditchPathNorth: ditchPathRoom
    sdesc = "Path in ditch"
    ldesc =
    {
        "The path occupies a narrow ledge just above the stagnant water
        of the ditch.  The path to the south climbs a narrow
        terrace cut into the ditch's bank.  You can't go any
        further north, but the tenuous ledge hugging the bank to
        the southwest looks passable; ";

        if (self.isLightPresent)
            "you may be able to find your way with something to
            light the way. ";
        else
            "without something to light the way, though, it would
            be too dangerous. ";
    }
    sw =
    {
        if (self.isLightPresent)
        {
            "It's difficult, but since you have enough light to see
            what you're doing, you manage to make your way further
            along the ledge.\b";
            return ditchTunnel;
        }
        else
        {
            "The ledge is too narrow for you to continue without
            something to light your way. ";
            return nil;
        }
    }
    teeterwallerPreFollow(rm) =
    {
        if (rm = ditchTunnel)
            "<q>Wait for me!</q> your boss cries from behind.  You
            shine the light back for him, and he nervously negotiates
            the narrow ledge.\b";
        else
            inherited.teeterwallerPreFollow(rm);
    }
    up  = { return self.south; }
    east =
    {
        "The bank is too steep to climb head-on, but you can follow
        the path, which climbs the terrace alongside the bank to
        the south. ";
        return nil;
    }
    south =
    {
        "The path ascends a terrace cut into the ditch's bank
        back to the top of the bank.\b";
        return ditchPath;
    }
;

ditchPathNorth_ledge: fixeditem
    noun = 'ledge'
    adjective = 'narrow'
    sdesc = "ledge"
    ldesc = "It's a narrow strip of ground just above the water level. "
    location = ditchPathNorth
    verDoClimb(actor) =
    {
        "You can simply travel southwest to get to the ledge. ";
    }
    doSynonym('Climb') = 'Board' 'Standon' 'Enter' 'Jump'
;

ditchPathNorth_terrace: fixeditem
    noun = 'terrace'
    location = ditchPathNorth
    sdesc = "terrace"
    ldesc = "It's just wide enough to accommodate the path. "
;

class inDitchPathItem: fixeditem
    noun = 'path'
    sdesc = "path"
    ldesc = "The path leads north and south. "
;

ditchPathNorth_path: inDitchPathItem
    location = ditchPathNorth
    ldesc = "The path to the south ascends a narrow terrace cut
             into the ditch's bank. "
;

class inDitchBankItem: fixeditem
    noun = 'bank' 'banks'
    adjective = 'ditch\'s'
    sdesc = "bank"
    ldesc = "It's a steep wall about ten feet high. "
    verDoClimb(actor) = {}
    doClimb(actor) = { "The bank is too steep to climb. "; }
    doSynonym('Climb') = 'Climbup'
;

ditchPathNorth_bank: inDitchBankItem
    location = ditchPathNorth
;

class inDitchDitchItem: fixeditem
    noun = 'ditch' 'water'
    adjective = 'stagnant'
    sdesc = "ditch"
    ldesc = "The ditch runs north and south. "
    verDoLookin(actor) = {}
    doLookin(actor) =
    {
        "Stagnant water covers the bottom of the ditch. ";
    }
    verDoSwimin(actor) = {}
    doSwimin(actor) =
    {
        "The brackish water doesn't appear to be deep enough
        for swimming, and you'd have no interest in immersing
        yourself in it even if it were deep enough. ";
    }
    verDoEnter(actor) = {}
    doEnter(actor) =
    {
        "You have no desire to immerse yourself in the stagnant water. ";
    }
    doSynonym('Enter') = 'Board'
    verIoPutIn(actor) = {}
    ioPutIn(actor, dobj) =
    {
        "You'd better not; you probably wouldn't be able to find
        <<dobj.thedesc>> again, and even if you could it would
        be soaked in mud. ";
    }
    verDoTake(actor) =
    {
        "You have no desire to take any of the filthy water. ";
    }
    verDoDrink(actor) = { }
    doDrink(actor) = { "It makes you ill just to think of it. "; }
;

ditchPathNorth_ditch: inDitchDitchItem
    location = ditchPathNorth
;
             

/* ------------------------------------------------------------------------ */
/*
 *   Tunnel under road 
 */
ditchTunnel: ditchPathRoom
    sdesc = "Tunnel"
    ldesc = "The ditch passes through a tunnel under the road.
             The path is wide and straight here, continuing
             alongside the ditch to the north and south. "
    north = ditchPathNorth
    south = ditchPathSouth
    ne = ditchPathNorth
;

ditchTunnel_tunnel: fixeditem
    sdesc = "tunnel"
    noun = 'tunnel'
    location = ditchTunnel
    ldesc = { self.location.ldesc; }
;

ditchTunnel_road: distantItem
    sdesc = "road"
    noun = 'road'
    location = ditchTunnel
    ldesc = "The road is at least fifteen feet above you. "
;

ditchTunnel_ditch: inDitchDitchItem
    location = ditchTunnel
;

ditchTunnel_path: inDitchPathItem
    location = ditchTunnel
;

ditchTunnel_bank: inDitchBankItem
    location = ditchTunnel
;


/* ------------------------------------------------------------------------ */
/*
 *   Ditch, south of road 
 */
ditchPathSouth: ditchPathRoom
    sdesc = "Path near ditch"
    ldesc = "The path continues north and south along the ditch.
             To the north, you can just barely make out the road crossing
             over the ditch.  To the south, you see some faint
             lights in the distance. "
    north = ditchTunnel
    south = ditchAtFence
;

ditchPathSouth_road: distantItem
    noun = 'road' 'tunnel'
    sdesc = "road"
    location = ditchPathSouth
    ldesc = "You can just make out the faint outline of the road
             crossing over the ditch some distance to the north. "
;

ditchPathSouth_lights: distantItem
    noun = 'light' 'lights'
    adjective = 'faint' 'distant'
    location = ditchPathSouth
    sdesc = "distant lights"
    adesc = "a light"
    isThem = true
    ldesc = "You can see numerous faint patches of colored lights
             at a great distance to the south, high above the horizon,
             seemingly part of a tall structure.  The distance prevents
             you from seeing any more detail. "
;

ditchPathSouth_ditch: inDitchDitchItem
    location = ditchPathSouth
;

ditchPathSouth_path: inDitchPathItem
    location = ditchPathSouth
;

ditchPathSouth_bank: inDitchBankItem
    location = ditchPathSouth
;


/* ------------------------------------------------------------------------ */
/*
 *   Ditch, at the plant fence 
 */
ditchAtFence: ditchPathRoom
    sdesc = "Ditch, north of fence"
    ldesc =
    {
        "The ditch is deep here&mdash;the banks are perhaps twenty
        feet high.  A chain-link fence runs above the top of the ditch
        to the south, below which is a gate made of closely-spaced
        steel bars; ";

        if (ditchAtFence_gate.isRaised)
            "the gate has been raised up
            several feet above the water to create an opening that you
            could easily get through. ";
        else
            "the gate covers the full width of the ditch, and blocks
            travel to the south. ";
    }
    north = ditchPathSouth
    south =
    {
        if (ditchAtFence_gate.isRaised)
        {
            "You have to bend down slightly to move past the gate. ";
            if (teeterwaller.location = self)
                "Mr.\ Teeterwaller cautiously joins you, looking up to
                make sure the gate isn't going to come crashing down on
                him as he passes underneath. ";
            "As you cross under the fence, the ditch changes to a
            concrete-lined canal.\b";
            return canalInPlant;
        }
        else
        {
            "The gate is blocking the way. ";
            return nil;
        }
    }
;

ditchAtFence_gate: fixeditem
    noun = 'gate' 'bar' 'bars'
    adjective = 'steel' 'closely-spaced' 'spaced'
    sdesc = "steel gate"
    location = ditchAtFence
    isRaised = nil
    ldesc =
    {
        "The gate is made of vertical steel bars, spaced about six inches
        apart.  Rails at either side guide the gate so that it can slide
        up and down, and a pair of cables attached to the top of the gate
        are connected to a winch high above. ";
        if (self.isRaised)
            "The gate is raised up several feet above the water, creating
            an opening underneath that you could easily get through. ";
        else
            "The gate is immersed in the water and covers the full width
            of the ditch; it completely blocks travel to the south. ";
    }

    /* 
     *   Reverse my open/closed state.  This doesn't produce any message,
     *   apart from possibly a score notification, but performs the
     *   essential state changes needed to open the gate.  Since the gate
     *   can be opened from this location or from the path above the
     *   ditch, different messages are needed depending on where the
     *   player is when the gate is opened; the state change code, at
     *   least, is common so is centralized here.  
     */
    pointsAwarded = nil
    reverseOpenState =
    {
        /* reverse my open state */
        self.isRaised := !self.isRaised;

        /* award points for getting the gate open if we haven't already */
        maybeIncscore(5, 'opening the ditch gate', self);
    }
    tryGarageOpener(code) =
    {
        if (plantPathWest_blackBox.radioCode = code)
        {
            if (self.isRaised)
                "From above, you hear the winch start humming.  The gate
                slowly travels down the rails, and after a few moments
                slides into the water.  It continues to travel for a few
                feet, then stops, and the winch becomes quiet. ";
            else
                "From above, you hear the winch make a loud humming
                noise.  The cables become tense, then the gate starts
                slowly rising.  It lifts up out of the water, then
                continues sliding up the rails for a few feet.  When
                it reaches the top of the rails, it stops, and the winch
                becomes silent. ";
            self.reverseOpenState;
            return true;
        }
        else
        {
            /* the code doesn't match; ignore it */
            return nil;
        }
    }
    liftAttempts = 0
    verDoLift(actor) = { }
    doLift(actor) =
    {
        "You brace yourself and pull the bars as hard as you can, but
        the gate is much too heavy to move. ";

        if (teeterwaller.location = actor.location)
        {
            self.liftAttempts++;
            if (self.liftAttempts = 3)
            {
                "\b<q>It's no use trying to lift it ourselves,</q>
                Teeterwaller says.  <q>I don't know why you keep
                trying.</q> ";
                self.liftAttempts := 0;
            }
        }
    }
    doSynonym('Lift') = 'Move' 'Pull' 'Open'
    verDoPush(actor) = { }
    doPush(actor) =
    {
        "You give the gate a good, hard push, but it's extremely heavy,
        and you don't manage to budge it even slightly. ";
    }
    verDoClimb(actor) = { }
    doClimb(actor) =
    {
        "The arrangement of the bars doesn't give you any way to get
        a foot-hold.  Besides, even if you could climb to the top of
        the gate, you'd still have to get over the fence high above
        it, and the razor wire would make that impossible. ";
    }
    doSynonym('Climb') = 'Climbover' 'Climbup'
;

ditchAtFence_rails: fixeditem
    noun = 'rail' 'rails'
    adjective = 'left' 'right'
    sdesc = "rail"
    location = ditchAtFence
    ldesc = "The rails guide the gate so that it can move up and down. "
;

ditchAtFence_winch: distantItem
    noun = 'winch'
    sdesc = "winch"
    location = ditchAtFence
    ldesc = "It's high above; you can't see any detail from here. "
;

ditchAtFence_cables: distantItem
    noun = 'cable' 'cables' 'pair'
    sdesc = "cable"
    location = ditchAtFence
    ldesc = "The cables connect the gate to a winch above the gate. "
;

ditchAtFence_fence: distantItem
    sdesc = "fence"
    noun = 'fence'
    adjective = 'chain' 'link' 'chain-link'
    location = ditchAtFence
    ldesc = "You can't see much detail from here; the fence is above
             the tops of the banks of the ditch. "
;

ditchAtFence_ditch: inDitchDitchItem
    location = ditchAtFence
;

ditchAtFence_path: inDitchPathItem
    location = ditchAtFence
;

ditchAtFence_bank: inDitchBankItem
    location = ditchAtFence
    ldesc = "It's a steep wall about twenty feet high. "
;

/* ------------------------------------------------------------------------ */
/*
 *   road intersection at detour
 */
roadAtDetour: roadRoom
    west = roadEast2
    east =
    {
        if (roadAtDetour_barricades.location != nil)
            "On foot, you have no difficulty slipping around the
            barricades.\b";
        return roadAtConstruction;
    }
    sdesc =
    {
        if (roadAtDetour_barricades.location != nil)
            "Detour";
        else
            "Intersection";
    }
    ldesc =
    {
        if (roadAtDetour_barricades.location != nil)
            "A set of wooden barricades, painted with orange stripes
            and topped with flashing orange lights, stand
            blocking the road to the east to vehicles, and <q>Detour</q>
            signs point to the intersecting road leading south.  Beyond
            the signs, trucks and construction machinery sit idle on
            the road, presumably awaiting the return of workers in
            the morning. ";
        else
            "The main road leads east and west here.  A smaller
            intersecting road leads south. ";
    }
    south = roadSouth
    teeterwallerPreFollow(rm) =
    {
        if (rm = roadAtConstruction
            && roadAtDetour_barricades.location != nil)
            "<q>Are you sure this is a good idea?</q> Mr.\ Teeterwaller
            protests, but carefully follows you past the barricades.\b";
        else
            inherited.teeterwallerPreFollow(rm);
    }
    teeterwallerPostFollow =
    {
        /* comment on the difference, if appropriate */
        if (self.looksDifferent)
        {
            "\b<q>Isn't this where the detour was?</q> Teeterwaller
            mumbles to himself. ";
            
            self.looksDifferent := nil;
        }
        
        inherited.teeterwallerPostFollow;
    }
;

roadAtDetour_barricades: fixeditem, detourItem
    sdesc = "barricade"
    noun = 'set' 'barricades' 'sign' 'signs' 'stripes' 'light' 'lights'
    adjective = 'orange' 'flashing' 'detour'
    location = roadAtDetour
    ldesc = "The barricades block vehicle travel to the east, and
            direct traffic along the road to the south. "
;

roadAtDetour_machinery: distantItem, detourItem
    sdesc = "machinery"
    adesc = "any machinery"
    noun = 'machinery' 'trucks' 'equipment' 'machines'
    adjective = 'construction'
    location = roadAtDetour
    ldesc = "You can't see much detail from here, but all of the equipment
             looks idle. "
;

roadAtDetour_road: roadItem
    location = roadAtDetour
    ldesc =
    {
        if (roadAtDetour_barricades.location != nil)
            "The main road continues east and west, and an intersecting
            road leads south.  Vehicle travel to the east is blocked with
            barricades and <q>Detour</q> signs that direct traffic
            to the intersecting road. ";
        else
            "The main road continues east and west, and an intersecting
            road leads south. ";
    }
;


/* ------------------------------------------------------------------------ */
/*
 *   road at construction site 
 */
roadAtConstruction: roadRoom
    east = roadFarEast
    west =
    {
        if (roadAtDetour_barricades.location != nil)
            "You slip back around the barricades.\b";
        return roadAtDetour;
    }
    sdesc =
    {
        if (roadAtDetour_barricades.location != nil)
            "Road, at construction site";
        else
            "Road";
    }
    ldesc =
    {
        if (roadAtDetour_barricades.location != nil)
            "The road continues east and west.  About a half dozen
            assorted trucks and construction machines are parked
            in the middle of the road.  There's no evidence that
            any road construction has actually taken place here;
            perhaps the equipment has been assembled here in preparation
            for a project that won't be underway until tomorrow
            morning. ";
        else
            "The brush alongside the road is crushed against the
            ground, and deep tracks from heavy machinery are pressed
            into the soil.  The road continues east and west; you can
            see only darkness in either direction. ";
    }
;

roadAtConstruction_tracks: fixeditem
    noun = 'tracks' 'track' 'soil' 'dirt' 'brush'
    sdesc = "tracks"
    adesc = "a set of tracks"
    isThem = true
    isseen = nil
    ldesc =
    {
        "The tracks are from several different types of trucks or
        tractors; they're wide and deep, clearly from heavy equipment.
        They don't lead anywhere, but just meander on and off the
        road for a stretch, as though the equipment was turning around
        at this spot. ";

        if (!self.isseen)
        {
            "\bYou notice a flashlight partially hidden by some of
            the flattened brush.  You reach down and pick up the
            flashlight. ";
            self.isseen := true;
            flashlight.moveInto(Me);
            incscore(1, 'finding the flashlight');
        }
    }
    doSynonym('Inspect') = 'Search' 'Lookin'
;

roadAtConstruction_machines: fixeditem, detourItem
    noun = 'machines' 'truck' 'trucks' 'bulldozers' 'back-hoe' 'equipment'
           'shovel' 'machinery'
    adjective = 'construction' 'dump' 'steam'
    sdesc = "construction equipment"
    adesc = "any construction equipment"
    location = roadAtConstruction
    ldesc = "About a half dozen pieces of equipment are parked in
             the middle of the road: a couple of dump trucks, a back-hoe,
             a bulldozer, a steam shovel. They're all locked. "
    dobjGen(a, v, i, p) =
    {
        if (v != inspectVerb && v != askVerb && v != tellVerb)
        {
            "The trucks and machines are all locked up. ";
            exit;
        }
    }
    iobjGen(a, v, d, p) = { self.dobjGen(a, v, d, p); }
;

roadAtConstruction_barricade: distantItem, detourItem
    sdesc = "barricade"
    noun = 'barricades' 
    location = roadAtConstruction
    ldesc = "The barricades are some distance to the west. "
;

roadAtConstruction_road: roadItem
    location = roadAtConstruction
    ldesc =
    {
        if (roadAtDetour_barricades.location != nil)
            "The road seems strangely unaffected by whatever construction is
            underway here.  The road continues to the east and west, back
            to the barricades. ";
        else
            "The road continues east and west. ";
    }
;

flashlight: lightsource
    noun = 'flashlight' 'torch' 'light'
    sdesc = "flashlight"
    ldesc = "It's a large lantern-style flashlight. It's currently
             turned << self.islit ? "on" : "off" >>. "
    islit = nil
    verDoOpen(actor) = { "There's no reason to open the flashlight. "; }
    verDoTurnon(actor) = {}
    verDoTurnoff(actor) = {}
;

/* ------------------------------------------------------------------------ */
/*
 *   Continuing road to the east of the construction site 
 */
roadFarEast: roadRoom
    sdesc = "Road"
    ldesc =
    {
        if (roadAtDetour_barricades.location != nil)
            "The road continues east and west. It's too dark to see
            much in either direction, although you can make out the
            dim lights of the detour barricades to the west. ";
        else
            "The road continues east and west. It's too dark to
            see much in either direction. ";
    }
    west = roadAtConstruction
    east = roadFarEast2
;

roadFarEast_lights: distantItem, detourItem
    noun = 'light' 'lights' 'barricades'
    adjective = 'dim' 'detour' 'flashing'
    location = roadFarEast
    sdesc = "flashing light"
    ldesc = "You can just barely see the dim flashing lights of the
             detour barricade. "
;

roadFarEast_road: roadItem
    location = roadFarEast
    ldesc = "The road continues east and west. "
;


/* ------------------------------------------------------------------------ */
/*
 *   Even further east 
 */
roadFarEast2: roadRoom
    sdesc = "Road"
    ldesc = "A small, dilapidated building is just north of the road,
             which continues east and west.  In the darkness, you
             can just barely make out a sign by the side of the road. "
    west = roadFarEast
    east =
    {
        if (!roadFarEast2_sign.isseen)
        {
            "Before you go on, you notice the sign by the road:\b";
            roadFarEast2_sign.ldesc;
            "\b";
        }

        if (teeterwaller.location = self)
            "Teeterwaller stops you. <q>Look at the sign!</q> he says,
            flustered. <q>We can't walk twenty-five miles. We'll miss
            our meeting!</q> ";
        else
            "Given that the nearest city is twenty-five miles, you
            are too daunted to continue any further in this direction. ";
        return nil;
    }
    north = schoolhouse
    in = schoolhouse
;

roadFarEast2_building: fixeditem
    sdesc = "dilapidated building"
    noun = 'building' 'school' 'schoolhouse'
    adjective = 'dilapidated' 'small'
    ldesc = "The building is only big enough for a single room, and
             has fallen into a state of considerable disrepair.  The
             entrance is to the north. "
    location = roadFarEast2
    verDoEnter(actor) = { }
    doEnter(actor) = { actor.travelTo(self.location.north); }
;

roadFarEast2_sign: readable, fixeditem
    sdesc = "sign"
    location = roadFarEast2
    noun = 'sign'
    ldesc =
    {
        "<blockquote><font face='TADS-Sans'>Conners
        City&nbsp;&nbsp;&nbsp;&nbsp;<tab id=dist>25 mi
        <br>Knighting<tab to=dist>37 mi
        <br>Milltown<tab to=dist>51 mi
        </font></blockquote>";
        self.isseen := true;
    }
;

roadFarEast2_road: roadItem
    location = roadFarEast2
    ldesc = "The road continues east and west. "
;

/* ------------------------------------------------------------------------ */
/*
 *   Schoolhouse 
 */
schoolhouse: room
    sdesc = "Schoolhouse"
    ldesc = "This one-room schoolhouse has clearly been out of use for
             many years.  A chalkboard is on one wall, and a single
             desk remains.  Exits lead north and south. "
    south = roadFarEast2
    north = schoolyard
;

schoolhouse_chalkboard: fixeditem, readable
    noun = 'chalkboard' 'board' 'blackboard'
    adjective = 'chalk' 'black'
    sdesc = "chalkboard"
    location = schoolhouse
    ldesc = "The blackboard is chipped and warped with age. "
;

schoolhouse_desk: fixeditem, openable
    isopen = nil
    sdesc = "desk"
    noun = 'chair' 'desk'
    location = schoolhouse
    ldesc =
    {
        "It's the type that combines a desk and a chair into one unit.
        The top of the desk can be opened to get at a storage area. ";

        if (self.isopen)
        {
            "The top is open";
            if (itemcnt(self.contents) = 0)
                ", and the desk is empty. ";
            else
                "; the desk contains <<listcont(self)>>. ";
        }
        else
            "The top is currently closed. ";
    }
    verDoSiton(actor) = { "It's too small for you. "; }
;

schoolhouse_desk_top: fixeditem
    sdesc = "desk top"
    noun = 'top' 'desktop'
    adjective = 'desk'
    location = schoolhouse
    ldesc =
    {
        if (schoolhouse_desk.isopen)
            "The desk top is open. ";
        else
            "The desk top is closed. ";
    }
    verDoTake(actor) = { schoolhouse_desk.verDoOpen(actor); }
    doTake(actor) = { schoolhouse_desk.doOpen(actor); }
    doOpen -> schoolhouse_desk
    doClose -> schoolhouse_desk
;

blottnian_textbook: readable
    sdesc =
    {
        if (self.isExamined)
            "Blottnian textbook";
        else
            "book";
    }
    isExamined = nil
    noun = 'textbook' 'book' 'ninshta'
    adjective = 'blottnian' 'blottnisq'
    location = schoolhouse_desk
    ldesc =
    {
        self.isExamined := true;
        
        "\t<i><b>Blottnisq Ninshta</b></i>
        \n\t<i>Blottnian Today</i>
        \bIt seems to be an elementary Blottnian textbook, for learning
        Blottnian as a foreign language.  Despite
        its age (<q>copyright 1967, The Horton, Merfin Company, New
        York</q>), it's still intact.  It has chapters numbered 1 through 5
        that you can read. ";
    }
;

class blottnian_chapter: fixeditem, readable
    location = blottnian_textbook
    noun = 'chapter'
    adjective = 'blottnian' 'textbook'
    sdesc = "blottnian textbook chapter <<self.chapterNum>>"
    adesc = { self.sdesc; }
    thedesc = { self.sdesc; }
;

blottnian_chapter_1: blottnian_chapter
    adjective = '1'
    chapterNum = 1
    ldesc = "<i>Inshna Hojdzunshchk!</i>
             \bWelcome to your introductory Blottnian course!  In this
             book, we will learn not only how to speak and write the
             Blottnian language, but much about the rich history and
             colorful culture of the nation of Blottnya.
             \b(The chapter goes on at length about how to use the
             textbook and accompanying audio tapes; a quick scan of
             the chapter reveals little of note.) "
;

blottnian_chapter_2: blottnian_chapter
    adjective = '2'
    chapterNum = 2
    ldesc = "This chapter covers some basic vocabulary.
             \bCommon conversational verbs:
             \n\t<i>azdrupiq</i> &mdash; to sit
             \n\t<i>yijtgiq</i> &mdash; to stand
             \n\t<i>pozhbriq</i> &mdash; to open
             \n\t<i>oxviq</i> &mdash; to close
             \n\t<i>lloyriq</i> &mdash; to speak
             \n\t<i>chdraliq</i> &mdash; to be silent
             \n\t<i>wurtriq</i> &mdash; to attack
             \n\t<i>shchrudiq</i> &mdash; to befriend "
;

blottnian_chapter_3: blottnian_chapter
    adjective = '3'
    chapterNum = 3
    isseen = nil
    ldesc =
    {
        "This chapter discusses the culture of Blottnya.  You scan
        a few of the passages.

        \bA picture of a dog";

        if (self.isIn(doggy.location))
            ", which looks very much like the guard dog,";
        else if (doggy.isseen)
            ", which looks very much like the guard dog
            you encountered earlier,";

        " is accompanied by a caption:
        \bYears ago, herders used the Blottnian Moose Dog to help control
        their herds.  Centuries of careful breeding have made the Moose Dog
        extremely intelligent and hard-working.  Today, the same herding
        instinct that once served moose herders now serves Blottnian
        government security police, who use the
        Moose Dog to help control the cheerful citizens.  Blottnya even
        trains Moose Dogs for service abroad; guard dogs are the country's
        second largest export. ";

        /* 
         *   if I haven't been read before, have Teeterwaller comment on
         *   all this blottnian stuff 
         */
        if (!self.isseen && teeterwaller.isIn(parserGetMe().location))
            "\bTeeterwaller reads the book over your shoulder.
            <q>I've never heard of anyone in this country studying
            Blottnian before,</q> he says. ";

        self.isseen := true;
    }
;

blottnian_chapter_4: blottnian_chapter
    adjective = '4'
    chapterNum = 4
    ldesc = "This chapter is about the convoluted grammar of Blottnian,
             which is considerably more complicated than any language
             you've studied.  You find a verb conjugation table for
             regular <q>-iq</q> verbs; they seem to use <q>to open</q>
             as the example:
             \bFirst-person singular, prideful: <i>pozhbroq</i>
             \nFirst-person singular, shameful: <i>pozhbronsh</i>
             \nFirst-person singular, uncertain: <i>pozhbrioo</i>
             \nFirst-person singular, emphatic: <i>pozhbrim</i>
             \bSecond-person singular, complimentary: <i>pozhbras</i>
             \nSecond-person singular, accusatory: <i>pozhbrazim</i>
             \nSecond-person singular, commanding: <i>pozhbru</i>
             \bIt goes on and on like that for forty-five different
             cases.  And that's just the present tense. "
;

blottnian_chapter_5: blottnian_chapter
    adjective = '5'
    chapterNum = 5
    ldesc = "This is a chapter on culture, focusing on how children
             in Blottnya are raised.  For example, it points out that
             parents in Blottnya are very fond of using reverse psychology,
             always telling children just the opposite of what they want
             them to do.  Apparently this practice applies to
             road signs as well, making driving in Blottnya extremely
             confusing for foreigners. "
;

/*
 *   Blottnian verb, for talking to the dog 
 */
class blottVerb: deepverb
    sdesc = "do"
    action(actor) =
    {
        if (doggy.isIn(actor.location))
            doggy.actorAction(self, nil, nil, nil);
        else
            "Your Blottnian pronunciation could use some work. ";
    }
;

/* 
 *   add compound words for 'say <blottnian words>' that translate to just
 *   the blottnian words 
 */
blottPreps: Prep
    preposition =
        'azdrupiq'
        'azdrupoq'
        'azdruponsh'
        'azdrupioo'
        'azdrupim'
        'azdrupas'
        'azdrupazim'
        'yijtgiq'
        'yijtgoq'
        'yijtgonsh'
        'yijtgioo'
        'yijtgim'
        'yijtgas'
        'yijtgazim'
        'pozhbriq'
        'pozhbroq'
        'pozhbronsh'
        'pozhbrioo'
        'pozhbrim'
        'pozhbras'
        'pozhbrazim'
        'pozhbru'
        'oxviq'
        'oxvoq'
        'oxvonsh'
        'oxvioo'
        'oxvim'
        'oxvas'
        'oxvazim'
        'oxvu'
        'lloyriq'
        'lloyroq'
        'lloyronsh'
        'lloyrioo'
        'lloyrim'
        'lloyras'
        'lloyrazim'
        'chdraliq'
        'chdraloq'
        'chdralonsh'
        'chdralioo'
        'chdralim'
        'chdralas'
        'chdralazim'
        'wurtriq'
        'wurtroq'
        'wurtronsh'
        'wurtrioo'
        'wurtrim'
        'wurtras'
        'wurtrazim'
        'shchrudiq'
        'shchrudoq'
        'shchrudonsh'
        'shchrudioo'
        'shchrudim'
        'shchrudas'
        'shchrudazim'
        'azdrupu'
        'yijtgu'
        'lloyru'
        'chdralu'
        'wurtru'
        'shchrudu'
;
    
/*
 *   This is the repository of all of the Blottnian verbs we don't care
 *   about.  In particular, all of these verbs are in cases the dog
 *   doesn't understand. 
 */
blottnianNullVerbs: blottVerb
    verb = 'azdrupiq' 'say azdrupiq'
            'azdrupoq' 'say azdrupoq'
            'azdruponsh' 'say azdruponsh'
            'azdrupioo' 'say azdrupioo'
            'azdrupim' 'say azdrupim'
            'azdrupas' 'say azdrupas'
            'azdrupazim' 'say azdrupazim'
            'yijtgiq' 'say yijtgiq'
            'yijtgoq' 'say yijtgoq'
            'yijtgonsh' 'say yijtgonsh'
            'yijtgioo' 'say yijtgioo'
            'yijtgim' 'say yijtgim'
            'yijtgas' 'say yijtgas'
            'yijtgazim' 'say yijtgazim'
            'pozhbriq' 'say pozhbriq'
            'pozhbroq' 'say pozhbroq'
            'pozhbronsh' 'say pozhbronsh'
            'pozhbrioo' 'say pozhbrioo'
            'pozhbrim' 'say pozhbrim'
            'pozhbras' 'say pozhbras'
            'pozhbrazim' 'say pozhbrazim'
            'pozhbru' 'say pozhbru'
            'oxviq' 'say oxviq'
            'oxvoq' 'say oxvoq'
            'oxvonsh' 'say oxvonsh'
            'oxvioo' 'say oxvioo'
            'oxvim' 'say oxvim'
            'oxvas' 'say oxvas'
            'oxvazim' 'say oxvazim'
            'oxvu' 'say oxvu'
            'lloyriq' 'say lloyriq'
            'lloyroq' 'say lloyroq'
            'lloyronsh' 'say lloyronsh'
            'lloyrioo' 'say lloyrioo'
            'lloyrim' 'say lloyrim'
            'lloyras' 'say lloyras'
            'lloyrazim' 'say lloyrazim'
            'chdraliq' 'say chdraliq'
            'chdraloq' 'say chdraloq'
            'chdralonsh' 'say chdralonsh'
            'chdralioo' 'say chdralioo'
            'chdralim' 'say chdralim'
            'chdralas' 'say chdralas'
            'chdralazim' 'say chdralazim'
            'wurtriq' 'say wurtriq'
            'wurtroq' 'say wurtroq'
            'wurtronsh' 'say wurtronsh'
            'wurtrioo' 'say wurtrioo'
            'wurtrim' 'say wurtrim'
            'wurtras' 'say wurtras'
            'wurtrazim' 'say wurtrazim'
            'shchrudiq' 'say shchrudiq'
            'shchrudoq' 'say shchrudoq'
            'shchrudonsh' 'say shchrudonsh'
            'shchrudioo' 'say shchrudioo'
            'shchrudim' 'say shchrudim'
            'shchrudas' 'say shchrudas'
            'shchrudazim' 'say shchrudazim'
;

blottSitVerb: blottVerb verb = 'azdrupu' 'say azdrupu';
blottStandVerb: blottVerb verb = 'yijtgu' 'say yijtgu';
blottSpeakVerb: blottVerb verb = 'lloyru' 'say lloyru';
blottQuietVerb: blottVerb verb = 'chdralu' 'say chdralu';
blottAttackVerb: blottVerb verb = 'wurtru' 'say wurtru';
blottBefriendVerb: blottVerb verb = 'shchrudu' 'say shchrudu';
            

           
/* ------------------------------------------------------------------------ */
/*
 *   School yard 
 */
schoolyard: room, distantLightningRoom
    sdesc = "School yard"
    ldesc = "This school yard must always have been a modest playground
             at best, but is now little more than a dirt patch overgrown
             with weeds.  The schoolhouse building is to the south. "
    south = schoolhouse
    in = schoolhouse
;

handpump: item
    noun = 'pump' 'handpump' 'airpump'
    adjective = 'hand' 'air'
    sdesc = "hand pump"
    location = schoolyard
    ldesc = "It's a small hand-operated pump, probably for inflating
             basketballs or something similar. "

    verIoInflateWith(actor) = { }
    ioInflateWith(actor, dobj) = { dobj.doInflateWith(actor, self); }
    verIoAttachTo(actor) = { }
    ioAttachTo(actor) = { "You see no way to do that. "; }
    verDoAttachTo(actor, iobj) =
    {
        if (iobj = silverBalloon)
            iobj.showUsage;
        else
            "You see no way to do that. ";
    }
    verDoPutIn(actor, iobj) =
    {
        if (iobj = silverBalloon)
            iobj.showUsage;
        else
            inherited.verDoPutIn(actor, iobj);
    }
;

schoolyard_weeds: decoration
    sdesc = "weeds"
    adesc = "some weeds"
    isThem = true
    noun = 'weed' 'weeds'
    location = schoolyard
    ldesc = "They're just ordinary weeds, ragged and thorny. "
;

schoolyard_schoolhouse: fixeditem
    noun = 'schoolhouse' 'house' 'building'
    adjective = 'schoolhouse' 'building'
    location = schoolyard
    sdesc = "schoolhouse"
    ldesc = "An entrance to the dilapidated building is to the south. "
    verDoEnter(actor) = { }
    doEnter(actor) = { actor.travelTo(self.location.in); }
;


/* ------------------------------------------------------------------------ */
/*
 *   Road south of detour 
 */
roadSouth: roadRoom
    sdesc = "Road"
    north = roadAtDetour
    south = roadSouth2
    ldesc =
    {
        if (roadAtDetour_barricades.location != nil)
            "The road runs north and south. To the north, you can just make
            out the dim lights of the detour barricades. ";
        else
            "The road runs north and south. In both directions, you can
            see nothing but darkness. ";
    }
;

roadSouth_barricade: distantItem, detourItem
    sdesc = "barricade"
    location = roadSouth
    noun = 'light' 'lights' 'barricades'
    adjective = 'detour' 'flashing' 'dim'
    ldesc = "You can just make out the dim flashing lights of the
             detour barricade from here. "
;

roadSouth_road: roadItem
    location = roadSouth
    ldesc = "The road runs north and south. "
;


/* ------------------------------------------------------------------------ */
/*
 *   A few road rooms are slightly foggy.  
 */
class foggyRoadRoom: roadRoom
;

roadFog: fixeditem, floatingItem
    sdesc = "fog"
    noun = 'fog'
    adjective = 'light' 'thin'
    location =
    {
        if (isclass(parserGetMe().location, foggyRoadRoom))
            return parserGetMe().location;
        else
            return nil;
    }
    ldesc = "It's a thin fog, not dense enough to obscure nearby objects
             very much, but it gives the night air a damp chill. "
;

/* ------------------------------------------------------------------------ */
/*
 *   Road further south of detour 
 */
roadSouth2: foggyRoadRoom
    sdesc = "Road"
    north = roadSouth
    south = roadAtPlant
    ldesc = "A light fog hangs over the road in the still night air.
             The road continues into the darkness to the north. To
             the south, distant lights glow weakly through the fog. "
;

roadSouth2_lights: distantItem
    sdesc = "distant light"
    noun = 'light' 'lights' 'building' 'structure' 'patch' 'patches'
    adjective = 'distant' 'dim' 'colored' 'faint'
    location = roadSouth2
    ldesc = "The lights are so muted by distance and fog that you can
             only make out faint patches of colored light, but the lights
             are numerous, as though part of some large building or
             structure, although you can see no detail beyond the
             dim lights. "
;


/* ------------------------------------------------------------------------ */
/*
 *   Road in front of the plant 
 */
roadAtPlant: foggyRoadRoom
    north = roadSouth2
    east = plantPath
    west = plantPathWest
    up =
    {
        "The fence is too tall to climb, and the razor wire would
        certainly stop you even if you did reach the top. ";
        return nil;
    }
    sdesc = "Outside Complex"
    ldesc = "The road ends at the gated entrance to a huge industrial
             complex sprawling over the foggy landscape to the south.
             A tall chain-link fence surrounds the complex; the fence is
             interrupted only by a sliding section of fence that
             serves as a gate (currently <<
             roadAtPlant_gate.isopen ? "open" : "closed" >>) across
             the road. Narrow paths run along the outside of the
             fence east and west of the road. The road continues into
             the darkness to the north. "
    teeterwallerPostFollow =
    {
        /* comment on the detour when first we enter */
        if (!self.teeterwallerSeen)
            "\b<q>That's curious,</q> Teeterwaller mumbles. <q>Why would
            they set up a detour that just comes to a dead end?</q> ";

        inherited.teeterwallerPostFollow;
    }
    south =
    {
        "The gate is closed. ";
        return nil;
    }
;

class plantComplexItem: distantItem
    noun = 'complex' 'building' 'smokestacks' 'tank' 'tanks' 'pipe' 'pipes'
           'tower' 'towers' 'cylinders' 'light' 'lights' 'plant' 'factory'
           'refinery'
    adjective = 'huge' 'industrial' 'vertical' 'flashing'
    sdesc = "complex"
    ldesc = "The complex must be a chemical factory, or an oil refinery,
             or perhaps a power plant.  Towering smokestacks and fat
             vertical tanks are wrapped in a thick tangle of pipes,
             which run up the towers, between smokestacks, alongside
             each other, grouped together in wide sets of parallel
             cylinders for hundreds of yards and then suddenly parting
             ways, bending abruptly up, down, around one other, joining
             new groups for a while then bending away again.  Atop the
             towers are flashing lights to warn off aircraft, dozens of
             lights slowly flashing
             to their own rhythm, twinkling sometimes with their neighbors
             and often on their own. "
;

roadAtPlant_complex: plantComplexItem
    location = roadAtPlant
;

class plantFenceItem: fixeditem
    noun = 'fence' 'wire' 'loop' 'loops'
    adjective = 'chain' 'link' 'chain-link' 'razor'
    sdesc = "fence"
    ldesc = "The fence is huge, easily twenty feet high, and topped with
             loops of razor wire. "
    verDoClimb(actor) = { }
    doClimb(actor) = { "It's too tall and too topped with razor wire. "; }
    doSynonym('Climb') = 'Climbover' 'Climbup'
    verDoLookthru(actor) = { }
    doLookthru(actor) =
    {
        "Beyond the fence lies the vast sprawl of the complex. ";
    }
;

roadAtPlant_fence: plantFenceItem
    location = roadAtPlant
;

roadAtPlant_path: fixeditem
    location = roadAtPlant
    noun = 'path' 'paths'
    adjective = 'east' 'west'
    sdesc = "path"
    ldesc = "It's not much of a path; just a narrow band along the fence
             where no vegetation is growing.  One path leads east, and
             another leads west. "
;

roadAtPlant_vegetation: fixeditem
    location = roadAtPlant
    noun = 'vegetation' 'bush' 'bushes'
    sdesc = "vegetation"
    adesc = "vegetation"
    ldesc = "Scrubby bushes are all along the sides of the
             road. "
;

roadAtPlant_gate: fixeditem
    noun = 'gate'
    adjective = 'sliding'
    sdesc = "gate"
    location = roadAtPlant
    ldesc = "The gate is merely a separate section of the fence that
             slides across the road.  It's currently <<
             self.isopen ? "open" : "closed" >>. "
    isopen = nil
    verDoClimb(actor) = { }
    doClimb(actor) = { "The fence is too tall to climb over. "; }
    doSynonym('Climb') = 'Climbover' 'Climbup'
    verDoOpen(actor) =
    {
        "The gate is securely closed. ";
    }
    doSynonym('Open') = 'Move' 'Pull' 'Push'
    verDoClose(actor) = { "It's already closed. "; }
;


/* ------------------------------------------------------------------------ */
/*
 *   Path east of plant entrance 
 */
plantPath: room, distantLightningRoom
    sdesc = "Path outside fence"
    ldesc = "The terrain is clear for a few feet outside the perimeter
             of the fence, creating this path. The road is just to
             the west, and the enormous plant complex is beyond the
             fence to the south. The terrain slopes up to the east. "
    west = roadAtPlant
    east = plantHill
    south =
    {
        "You'd have to climb the fence, which seems out of the question. ";
        return nil;
    }
;

plantPath_fence: plantFenceItem
    location = plantPath
;

plantPath_complex: plantComplexItem
    location = plantPath
;

class plantPathItem: fixeditem
    noun = 'path'
    sdesc = "path"
    ldesc = "It's just a cleared strip running east and west
             along the fence. "
;

plantPath_path: plantPathItem
    location = plantPath
;

class distantRoadItem: distantItem
    noun = 'road'
    sdesc = "road"
    ldesc = "The road ends at the plant entrance, and continues to the
             north. "
;

plantPath_road: distantRoadItem
    location = plantPath
;

class plantGateItem: distantItem
    noun = 'gate'
    adjective = 'front'
    sdesc = "front gate"
    ldesc = { roadAtPlant_gate.ldesc; }
;

plantPath_gate: plantGateItem
    location = plantPath
;

/* ------------------------------------------------------------------------ */
/*
 *   Hilltop east of plant entrance 
 */
plantHill: room, distantLightningRoom
    sdesc = "Hilltop"
    ldesc = "The terrain is considerably elevated above the road,
             which lies some distance to the west.  Scrubby bushes
             grow almost up to the fence here, making it too
             difficult to travel further to the east.
             The elevation gives you a clear view of the road
             through the vegetation.  The sprawling plant complex
             lies to the south, beyond the fence, and the path leads
             west along the fence. "
    sceneInProgress = nil
    sceneJustEnded = nil
    sceneState = 0
    sceneSeen = nil
    roomAction(a, v, d, p, i) =
    {
        /* if they're hiding during the movie, respond sensibly */
        if ((v = getdownVerb || v = hideVerb) && d = nil
            && self.sceneInProgress)
        {
            "You're fine; the bushes and darkness are already giving you
            plenty of cover. ";
            exit;
        }

        /* use the default handling  */
        inherited.roomAction(a, v, d, p, i);
    }
    enterRoom(actor) =
    {
        local wasseen;
        
        /* note whether we've been here before */
        wasseen := self.isseen;

        /* inherit default */
        inherited.enterRoom(actor);

        /* if this is the first time here, start the scene */
        if (!wasseen)
        {
            /* 
             *   note that the scene is in progress, so that the player
             *   doesn't go anywhere 
             */
            self.sceneInProgress := true;

            /* describe the start of the action */
            "\bFrom the direction of the plant, you hear three short, 
            low-pitched electronic beeps, distant and echoing but still
            loud, then a deep male voice, broadcast through a public
            address system in the plant.

            \b<q>Section blue, section green, clear sector S, stand by
            stations.  Section black, target approaching, commence
            operation.</q>

            \bFrom the road to the north, faint lights come into view,
            moving down the road toward the gate.  The lights come closer
            and you see that several large vehicles are approaching.

            \bThe gate in the fence starts slowly sliding open.  A group
            of men wearing dark jumpsuits and black helmets and carrying
            machine guns trot out through the gate; some take
            up positions along the shoulder of the road, others run out
            into the underbrush alongside the road.  The men keep pouring
            out of the gate until at least a couple hundred have emerged.

            \bThe first few of the approaching vehicles rumble to a stop
            near the gate.  You can now see that the vehicles are large
            semi trucks, painted a uniform white, without any markings
            you can see. ";

            /* start notifier for subsequent developments */
            self.sceneState := 0;
            notify(self, &doScene, 0);

            /* move objects for the trucks and men into view */
            plantHill_trucks.moveInto(self);
            plantHill_men.moveInto(self);

            /* the gate is now open */
            roadAtPlant_gate.isopen := true;
        }
    }
    west =
    {
        if (self.sceneInProgress)
        {
            if (teeterwaller.location = parserGetMe().location)
                "Teeterwaller stops you. <q>Let's stay here for a moment,</q>
                he whispers. <q>We need to figure out what's going on before
                we walk into the middle of this.</q> ";
            else
                "As much as you'd like to contact someone for help, you think
                it would be best to stay here, out of sight, until you know
                what's going on. ";
        }
        else 
        {
            /* if we just ended the scene, comment on it */
            if (self.sceneJustEnded)
            {
                "You start back towards the road, staying alert for
                anyone who might still be patrolling.\b";
                self.sceneJustEnded := nil;
            }
            
            /* travel to the path */
            return plantPath;
        }
    }
    south =
    {
        "You'd have to climb the fence, which seems out of the question. ";
        return nil;
    }
    doScene =
    {
        local obj;
        
        /* see where we are */
        switch(self.sceneState)
        {
        case 1:
            "\bThe last of the trucks rolls to a stop up the road;
            you count eight trucks.  Some of the armed men start moving
            in groups towards the trucks, while others remain crouched
            along the sides of the road.

            \bMr.\ Teeterwaller crouches behind the bushes, watching
            the road intently. <q>Get down! Get down!</q> he whispers
            at you urgently.  You move to conceal yourself a little
            better. ";
            break;

        case 2:
            "\bOne by one, the truck drivers emerge from their cabs,
            holding their arms up while the armed men lead them away
            from the trucks, towards the plant gate.

            \bThe fourth truck in line suddenly lurches forward,
            pulls out of the line, and starts picking up speed, sending
            everyone on the road scrambling to get out of its way.
            Gunfire immediately erupts from the men flanking the road,
            and the truck is engulfed in a shower of sparks and flying
            metal.  The cab of the truck suddenly lurches sideways; half
            a moment later a deafening boom sounds.  The truck rolls off
            the road and stops as the engine compartment bursts into
            flames; two figures jump out of the cab and flee in opposite
            directions.

            \bMore gunfire from up the road.  Soldiers in full battle
            gear have come storming out of the backs of two of the
            trucks and are scrambling around the trucks looking for
            cover, and trying to fire back through the hailstorm of
            bullets coming from the sides of the road. ";
            plantHill_drivers.moveInto(self);
            plantHill_soldiers.moveInto(self);
            break;

        case 3:
            "\bThe gunfire starts to trail off.  Soon the soldiers seem
            to be the only ones firing; they quickly notice this and
            stop firing, looking warily around.

            \bA deep rumbling sound draws your attention to the plant
            gate, where you see a tank come rolling out onto the road.
            The tank rumbles to a stop just past the first truck.
            The tank's turret swings slowly around until the protruding
            cylinder of its cannon is pointed at one of the trucks the
            soldiers are using for cover.  Almost in unison, the
            soldiers around the truck bolt from their positions and
            scatter across the road; only a moment later, flames shoot
            out of the tank's cannon and the truck explodes in a huge
            fireball.

            \bThe attackers waste no time rounding up the
            scattered soldiers, and the remaining soldiers quickly
            surrender and are marching toward the plant gate with
            their hands in the air. ";

            plantHill_tank.moveInto(self);
            break;

        case 4:
            "\bThe last of the soldiers march into the plant, a large
            contingent of their attackers escorting them at gunpoint.
            A few dozen of the armed men continue patrolling around
            the edges of the road, but most are heading back towards
            the gate.

            \bElectronic beeps echo from the direction of the plant,
            followed by a voice on the PA system. <q>Section blue,
            section green, all clear.  Section white, initiate
            containment procedures.</q>
            
            \bFrom up the road, << roadAtConstruction.isseen
                ? "the trucks and tractors from the construction site "
                : "several vehicles that look like road construction
                   equipment&mdash;dump trucks,bulldozers,and the like&mdash;"
            >>roll toward the wreckage.  At the same time, a few dozen
            people wearing blue hard hats and tool belts, making them
            look like construction workers, walk out of the complex's
            gate and start heading purposefully toward the trucks. ";

            /* the soldiers are gone */
            plantHill_soldiers.moveInto(nil);
            plantHill_workers.moveInto(self);

            /* the construction machinery is moving here */
            roadAtConstruction_machines.moveInto(nil);
            plantHill_machines.moveInto(self);
            break;

        case 5:
            "\bThe trucks that are still intact start moving into the
            plant, one by one, as the bulldozers clear a path through
            the wreckage. ";
            break;

        case 6:
            "\bThe construction workers continue clearing the wreckage
            out of the road.
            
            \b<q>Did you see that?</q> Teeterwaller says, talking very
            quietly.  <q>What could that have been about?</q> ";
            break;

        case 7:
            "\bThe construction workers continue to clear the wreckage
            from the road.

            \b<q>Look,</q> Teeterwaller says, <q>I know this is going
            to sound a little strange, but I think we should try and
            find out what's going on here.  I think we need to get
            inside there,</q> he says, indicating the industrial complex
            beyond the fence. <q>I think this could be really
            important.</q> ";
            break;

        case 8:
            "\bThe construction workers clear the last of the wreckage
            from the road.  The
            construction equipment and workers withdraw into the complex,
            leaving only a few of the armed men still out on the road.
            \b<q>I could really use your help getting in there,</q>
            Teeterwaller says. <q>I know it could be dangerous, so if
            you don't want to join me, if you just want to return to the
            car and wait for help, I'll understand.  But I'm not sure I
            can do it alone.</q> ";

            plantHill_machines.moveInto(nil);
            plantHill_tank.moveInto(nil);
            plantHill_drivers.moveInto(nil);
            plantHill_workers.moveInto(nil);
            plantHill_trucks.moveInto(nil);

            /* 
             *   get rid of everything related to the detour - all objects
             *   of class detourItem are relevant only to the detour, and
             *   can be removed when the detour is removed 
             */
            for (obj := firstobj(detourItem) ; obj != nil ;
                 obj := nextobj(obj, detourItem))
                obj.moveInto(nil);

            /* existify the tracks left by the construction equipment */
            roadAtConstruction_tracks.moveInto(roadAtConstruction);

            /* 
             *   show full descriptions again next time we enter the rooms
             *   affected by the removal of the detour; even though we've
             *   seen them before, we haven't seen them yet in their new
             *   state 
             */
            roadAtDetour.isseen := nil;
            roadAtDetour.looksDifferent := true;
            roadAtConstruction.isseen := nil;
            break;

        case 9:
            "\bYou hear the rumbling of approaching trucks, this time from
            the complex. Out of the gate comes a large, white truck, exactly
            like the ones that arrived before the fighting. The truck
            accelerates up the road to the north.  Another truck follows,
            then another, until the trucks number eight.  These obviously
            <i>aren't</i> the original eight, though, since several of
            those were badly damaged in the fighting.

            \b<q>I used to fancy myself something of an amateur sleuth,</q>
            Teeterwaller tells you, looking a little sheepish and little
            smug at the same time.  <q>This will give me a chance to get
            back in practice.</q>
            
            \bThe trucks travel away to the north until they're out of
            sight.  The remaining men on the road withdraw into the complex,
            and the gate slides shut. ";

            /* the gate is now closed again */
            roadAtPlant_gate.isopen := nil;

            /* 
             *   we're done playing out the scene - let the player move
             *   about freely again, but make a note that we just finished
             *   the scene, so that we can comment when the player does
             *   travel back toward the road 
             */
            self.sceneInProgress := nil;
            self.sceneJustEnded := true;

            /* note that the scene has been seen */
            self.sceneSeen := true;

            /* the trucks and men are gone */
            plantHill_men.moveInto(nil);
            break;
        }

        /* advance to the next scene state */
        self.sceneState++;
    }
    teeterIdleAction =
    {
        if (self.sceneInProgress)
            "\bTeeterwaller nervously watches the road through the
            bushes. ";
        else
            inherited.teeterIdleAction;
    }
;

plantHill_bushes: fixeditem, seethruItem
    noun = 'bush' 'bushes' 'vegetation'
    adjective = 'scrubby'
    sdesc = "bushes"
    adesc = "any bushes"
    isThem = true
    location = plantHill
    ldesc = "The bushes are thick here, growing almost up to the fence. "
    thrudesc = "You have a clear view of the road and the complex's
                entrance gate through the bushes. "
;

plantHill_path: fixeditem
    location = plantHill
    noun = 'path'
    sdesc = "path"
    ldesc = "The path leads west along the fence. "
;

plantHill_road: distantRoadItem
    location = plantHill
;

plantHill_fence: plantFenceItem
    location = plantHill
;

plantHill_complex: plantComplexItem
    location = plantHill
;

plantHill_gate: plantGateItem
    location = plantHill
;

plantHill_trucks: distantItem
    noun = 'truck' 'trucks' 'semi'
    sdesc = "truck"
    ldesc = "The trucks are large, white semis, without any obvious
             markings. "
;

plantHill_men: distantItem
    noun = 'men' 'man' 'jumpsuits' 'helmets' 'gun' 'guns'
    adjective = 'black' 'dark' 'machine' 'armed'
    sdesc = "armed men"
    isThem = true
    adesc = "any armed men"
    ldesc = "The men are dressed in dark jumpsuits and black helmets, and
             they're armed with machine guns. "
;

plantHill_drivers: distantItem
    noun = 'drivers' 'uniforms' 'slacks' 'shirts' 'cap' 'caps'
    adjective = 'truck' 'blue' 'light' 'white'
    sdesc = "driver"
    ldesc = "You can't see a lot of detail from here, but the drivers
             look like they're wearing simple uniforms: light blue slacks,
             white shirt, blue cap. "
;

plantHill_soldiers: distantItem
    noun = 'soldiers' 'gear'
    adjective = 'battle' 'full'
    sdesc = "soldier"
    ldesc = "The soldiers are wearing full battle gear, and are
             scrambling about the trucks, trying to find cover. "
;

plantHill_tank: distantItem
    noun = 'tank' 'turret'
    sdesc = "armored tank"
    adesc = "an armored tank"
    adjective = 'armored' 'huge' 'dark' 'green'
    ldesc = "You don't know much about military equipment, but it
             looks like something from cold war surplus: huge, dark
             green, the enormous cylinder of its cannon pointed
             straight out of the turret. "
;

plantHill_workers: distantItem
    noun = 'workers' 'hat' 'hardhat' 'hard-hat' 'belt' 'toolbelt' 'people'
    adjective = 'blue' 'tool' 'construction'
    sdesc = "construction workers"
    adesc = "any construction workers"
    isThem = true
    ldesc = "They look like construction workers with their blue hard
             hats and tool belts.  They seem to be busy cleaning up
             the wreckage of the battle. "
;


plantHill_machines: distantItem
    noun = 'machines' 'truck' 'trucks' 'bulldozers' 'back-hoe' 'equipment'
           'shovel' 'machinery'
    adjective = 'construction' 'dump' 'steam'
    sdesc = "construction equipment"
    adesc = "any construction equipment"
    ldesc = "About a half dozen pieces of equipment are busily clearing
             the wreckage from the battle off the road. "
;

/* ------------------------------------------------------------------------ */
/*
 *   path west of plant gate 
 */

plantPathWest: room, distantLightningRoom
    sdesc = "Path outside fence"
    ldesc = "The path ends abruptly to the west with a wall of tall,
             thorny bushes growing out of a deep ditch.  The road
             is just to the east, and the plant complex is on the
             other side of the fence to the south.
             \n\tA metal framework under the fence inside the ditch
             is visible through the bushes. "
    east = roadAtPlant
    west =
    {
        "The bushes are too thick. ";
        return nil;
    }
    down = { return self.west; }
    south =
    {
        "You'd have to climb the fence, which seems out of the question. ";
        return nil;
    }
;

plantPathWest_bushes: fixeditem
    noun = 'bush' 'bushes' 'plant' 'plants'
    adjective = 'tall' 'thorny'
    sdesc = "bush"
    pluraldesc = "bushes"
    location = plantPathWest
    ldesc = "The plants are growing abundantly out of the steep bank
             of the ditch. "
    verDoLookthru(actor) = { }
    doLookthru(actor) = { "A deep ditch lies beyond the plants. "; }
;

plantPathWest_ditch: fixeditem
    noun = 'ditch' 'bank' 'water'
    adjective = 'steep' 'deep'
    location = plantPathWest
    sdesc = "ditch"
    ldesc = "The ditch looks like it goes under the fence into the plant.
             It looks deep, although you can't tell from here exactly
             how deep, since you can't see the bottom through the bushes
             lining the bank.
             \bA metal framework is positioned inside the ditch
             directly under the fence. "
    verDoEnter(actor) = { "The bushes are too thick. "; }
    doSynonym('Enter') = 'Board'
;

plantPathWest_framework: distantItem
    noun = 'framework'
    adjective = 'metal' 'steel'
    location = plantPathWest
    sdesc = "metal framework"
    ldesc =
    {
        "The framework is inside the ditch, under the fence, so you
        can't easily see the whole thing through the bushes.
        Two vertical rails, one on each side of the ditch, are
        connected at the top by a crossbar.  A winch is in the center
        of the crossbar, and a pair of cables lead down from the winch. ";
        if (ditchAtFence_gate.isRaised)
            "A steel gate is just below the crossbars, raised up a few feet
            from the ditch. ";
    }
;

plantPathWest_rails: distantItem
    noun = 'rail' 'rails'
    sdesc = "rail"
    location = plantPathWest
    ldesc = "The rails look like they go to the bottom of the ditch,
             although you don't have a clear view from here. "
;

plantPathWest_crossbar: distantItem
    noun = 'crossbar' 'i-beam' 'beam'
    adjective = 'steel'
    sdesc = "crossbar"
    location = plantPathWest
    ldesc = "The crossbar is a sturdy-looking steel I-beam running across
             the rails and anchored on the banks of the ditch.  A winch
             is attached to the crossbar. "
;

plantPathWest_winch: distantItem
    noun = 'winch'
    sdesc = "winch"
    location = plantPathWest
    ldesc = "It looks like a heavy-duty winch capable of lifting an
             extremely large load.
             \bNext to the winch is a small black box. "
    verDoTurnon(actor) = { "There's no obvious way to do that from here. "; }
;

plantPathWest_blackBox: distantItem
    radioCode = '42325515'
    noun = 'box'
    adjective = 'black' 'small'
    location = plantPathWest
    sdesc = "black box"
    ldesc =
    {
        local lights;

        lights := self.location.getLightSources;
        if (length(lights) = 0)
            "It's too dark to see much; you'll need some more light
            before you'll be able to make out any detail. ";
        else
        {
            "You shine << lights[1].thedesc >> on the black box, which
            gives you enough light to make out some detail.  Several wires
            run from the box to the winch, and a short antenna sticks up
            out of the top of the box.  On the side of the box is a white
            adhesive label with what looks like a bar code printed on it. ";

            plantPathWest_antenna.moveInto(self.location);
            plantPathWest_wires.moveInto(self.location);
            plantPathWest_barcode.moveInto(self.location);
        }
    }
    verIoShineOn(actor) = { }
    ioShineOn(actor, dobj) =
    {
        if (dobj.islamp && dobj.islit)
            self.ldesc;
        else if (dobj = portableScanner)
            self.verDoScanWith(actor, dobj);
        else
            inherited.ioShineOn(actor, dobj);
    }
    ioSynonym('ShineOn') = 'PointAt'
    verDoScanWith(actor, iobj) =
    {
        if (iobj = portableScanner)
        {
            if (plantPathWest_barcode.isIn(self.location))
                plantPathWest_barcode.doScanWithScanner(actor);
            else
                "You point the scanner at the black box, and red lights
                vibrate over its surface.  It takes many seconds before
                the scanner gives up with a long <i>buzz</i>, as though
                it's on the verge of picking up some information but
                isn't quite on target. ";
        }
        else
            "You can't scan the box with that. ";
    }
    doSynonym('ScanWith') = 'ReadWith'
;

plantPathWest_barcode: barcodeItem, distantItem
    sdesc = "black box bar code"
    noun = 'label'
    adjective = 'white' 'adhesive' 'black' 'box'
    ldesc = "The label is printed with a series of black lines of varying
             widths, similar to the ubiquitous bar codes stamped on the
             sides of cereal boxes and laundry detergent.  You see no
             numbers or other symbols printed on it, though&mdash;just
             the bar code. "
    pointsAwarded = nil
    scanCode = '4.71'
    doScanWithScanner(actor) =
    {
        /* use the default display code */
        inherited.doScanWithScanner(actor);

        /* award points if we haven't already */
        maybeIncscore(1, 'reading the bar code on the black box', self);
    }
;

plantPathWest_antenna: distantItem
    noun = 'antenna'
    sdesc = "antenna"
    adesc = "an antenna"
    ldesc = "It's just a short bare wire sticking up out of the top
             of the black box. "
;

plantPathWest_wires: distantItem
    noun = 'wire' 'wires'
    adjective = 'small-gauged' 'small' 'gauged'
    sdesc = "wire"
    ldesc = "Four small-gauged wires run between the black box and the
             winch. "
;

plantPathWest_cables: distantItem
    noun = 'cable' 'cables'
    sdesc = "cable"
    location = plantPathWest
    ldesc = "The cables are connected to the winch; the winch is
             connected to its load via the cables. "
;

plantPathWest_ditchGate: distantItem
    noun = 'gate' 'bar' 'bars'
    adjective = 'steel' 
    sdesc = "steel gate"
    location = plantPathWest
    ldesc =
    {
        if (ditchAtFence_gate.isRaised)
            "The gate is made of closely-spaced steel bars.  It's lifted
            up so that it is just below the top of the framework. ";
        else
            "You don't have a clear view from here. ";
    }
    tryGarageOpener(code) =
    {
        if (plantPathWest_blackBox.radioCode = code)
        {
            "The winch makes a loud humming noise, ";
            if (ditchAtFence_gate.isRaised)
                "and starts unwinding the cable to lower the gate.
                After a few moments, the gate drops out of sight
                below, and the winch stops moving. ";
            else
                "and starts winding up the cable.  From below, the
                gate lifts slowly up, sliding along the rails.
                After a few moments, the gate reaches the top of
                the rails, then stops, and the winch becomes silent. ";
            ditchAtFence_gate.reverseOpenState;
            return true;
        }
        else
        {
            /* the code doesn't match; ignore it */
            return nil;
        }
    }
;

plantPathWest_fence: plantFenceItem
    location = plantPathWest
;

plantPathWest_complex: plantComplexItem
    location = plantPathWest
;

plantPathWest_path: plantPathItem
    location = plantPathWest
;

plantPathWest_road: distantRoadItem
    location = plantPathWest
;

plantPathWest_gate: plantGateItem
    location = plantPathWest
;

portableScanner: item
    noun = 'scanner' 'grip' 'handle' 'device' 'body'
    adjective = 'portable' 'hand-held' 'plastic' 'small' 'boxy' 'pistol'
    location = warehouse_workbench
    sdesc =
    {
        if (self.wasInspected)
            "portable hand-held scanner";
        else
            "small plastic device";
    }
    displayText = nil
    wasInspected = nil
    ldesc =
    {
        if (self.wasInspected)
            "The scanner is shaped vaguely like a gun, with the large plastic
            body mounted on a pistol grip, at the top of which is a trigger.
            At the front of the gun is a little red window, and on top is
            a display. ";
        else
        {
            self.wasInspected := true;
            "The device appears to be a portable hand-held scanner of some
            kind.  The boxy plastic body is mounted on top of a pistol grip
            with a trigger; at
            the front of the body is a little red window, and on top is
            an electronic display.  Clearly, you are meant to hold the
            scanner by the grip, point it at an object to be scanned, and
            pull the trigger. ";
        }
        portableScanner_display.ldesc;
    }
    verIoScanWith(actor) = { }
    ioScanWith(actor, dobj) =
    {
        if (defined(dobj, &doScanWithScanner))
            dobj.doScanWithScanner(actor);
        else
            "You point the scanner at <<dobj.thedesc>> and pull the
            trigger.  Thin lines of red light vibrate and dance
            around <<dobj.thedesc>> for several seconds, then the
            scanner emits a long <i>buzz</i>. ";
    }
    verDoPointAt(actor, iobj) = { }
    doPointAt(actor, iobj) = { self.ioScanWith(actor, iobj); }
    ioSynonym('ScanWith') = 'ReadWith'
    doSynonym('PointAt') = 'ShineOn'
;

portableScanner_window: fixeditem
    noun = 'window'
    adjective = 'red' 'little' 'small' 'scanner'
    sdesc = "scanner window"
    location = portableScanner
    ldesc = "It's a small red window. "
    verDoLookin(actor) =
    {
        "You can't see anything through the window; presumably there
        the scanner's electronics are within. ";
    }
    doSynonym('Lookin') = 'Lookthru'
;

portableScanner_display: fixeditem
    noun = 'display'
    adjective = 'scanner'
    sdesc = "scanner display"
    location = portableScanner
    ldesc =
    {
        if (portableScanner.displayText = nil)
            "The display is currently dark. ";
        else
            "The display currently reads <tt><b>
            <<portableScanner.displayText>></b></tt>. ";
    }
;

portableScanner_trigger: fixeditem
    noun = 'trigger'
    adjective = 'scanner'
    location = portableScanner
    sdesc = "scanner trigger"
    verDoPull(actor) = { }
    doPull(actor) =
    {
        "You need to point the scanner at something before you
        pull the trigger. ";
    }
;

