#ifndef lint
static char sccsid[] = "@(#)erf.c	1.3	(ucb.beef)	10/2/89";
#endif	/* !defined(lint) */
/* 
 * This packet evaluates erf(x), erfc(x), and exp(x*x)*erfc(x)
 *   for a "float" argument x.  It contains three subprograms:
 *   erff(), erfcf(), and erfcxf() and * one "static" subprogram,
 *   calerf.  The calling statements for the primary entries are:
 * 
 *                   y = erf(x),
 * 
 *                   y = erfc(x),
 *   and
 *                   y = erfcx(x).
 * 
 *   The routine calerf() is intended for internal packet use only,
 *   all computations within the packet being concentrated in this
 *   routine.  The 3 primary subprograms invoke calerf with the
 *   statement
 * 
 *          result = calerf(arg,jint);
 * 
 *   where the parameter usage is as follows
 * 
 *      Function                     Parameters for calerf
 *       call              arg                  result          jint
 * 
 *     erf(arg)      ANY "double" ARGUMENT      erf(arg)          0
 *     erfc(arg)     fabs(arg) < XBIG           erfc(arg)         1
 *     erfcx(arg)    XNEG < arg < XMAX          erfcx(arg)        2
 * 
 *   The main computation evaluates near-minimax approximations
 *   from "Rational Chebyshev approximations for the error function"
 *   by W. J. Cody, Math. Comp., 1969, PP. 631-638.  This
 *   transportable program uses rational functions that theoretically
 *   approximate  erf(x)  and  erfc(x)  to at least 18 significant
 *   decimal digits.  The accuracy achieved depends on the arithmetic
 *   system, the compiler, the intrinsic functions, and proper
 *   selection of the machine-dependent constants.
 * 
 ********************************************************************
 ********************************************************************
 * 
 * Explanation of machine-dependent constants
 * 
 *   XMIN   = the smallest positive floating-point number.
 *   XINF   = the largest positive finite floating-point number.
 *   XNEG   = the largest negative argument acceptable to erfcx;
 *            the negative of the solution to the equation
 *            2*exp(x*x) = XINF.
 *   XSMALL = argument below which erf(x) may be represented by
 *            2*x/sqrt(pi)  and above which  x*x  will not underflow.
 *            A conservative value is the largest machine number x
 *            such that   1.0 + x = 1.0   to machine precision.
 *   XBIG   = largest argument acceptable to erfc;  solution to
 *            the equation:  W(x) * (1-0.5/x**2) = XMIN,  where
 *            W(x) = exp(-x*x)/[x*sqrt(pi)].
 *   XHUGE  = argument above which  1.0 - 1/(2*x*x) = 1.0  to
 *            machine precision.  A conservative value is
 *            1/[2*sqrt(XSMALL)]
 *   XMAX   = largest acceptable argument to erfcx; the minimum
 *            of XINF and 1/[sqrt(pi)*XMIN].
 * 
 *   Approximate values for some important machines are:
 * 
 *                          XMIN       XINF        XNEG     XSMALL
 * 
 *  CDC 7600      (S.P.)  3.13E-294   1.26E+322   -27.220  7.11E-15
 *  CRAY-1        (S.P.)  4.58E-2467  5.45E+2465  -75.345  7.11E-15
 *  IEEE (IBM/XT,
 *    SUN, etc.)  (S.P.)  1.18E-38    3.40E+38     -9.382  5.96E-8
 *  IEEE (IBM/XT,
 *    SUN, etc.)  (D.P.)  2.23D-308   1.79D+308   -26.628  1.11D-16
 *  IBM 195       (D.P.)  5.40D-79    7.23E+75    -13.190  1.39D-17
 *  UNIVAC 1108   (D.P.)  2.78D-309   8.98D+307   -26.615  1.73D-18
 *  VAX D-Format  (D.P.)  2.94D-39    1.70D+38     -9.345  1.39D-17
 *  VAX G-Format  (D.P.)  5.56D-309   8.98D+307   -26.615  1.11D-16
 * 
 * 
 *                          XBIG       XHUGE       XMAX
 * 
 *  CDC 7600      (S.P.)  25.922      8.39E+6     1.80X+293
 *  CRAY-1        (S.P.)  75.326      8.39E+6     5.45E+2465
 *  IEEE (IBM/XT,
 *    SUN, etc.)  (S.P.)   9.194      2.90E+3     4.79E+37
 *  IEEE (IBM/XT,
 *    SUN, etc.)  (D.P.)  26.543      6.71D+7     2.53D+307
 *  IBM 195       (D.P.)  13.306      1.90D+8     7.23E+75
 *  UNIVAC 1108   (D.P.)  26.582      5.37D+8     8.98D+307
 *  VAX D-Format  (D.P.)   9.269      1.90D+8     1.70D+38
 *  VAX G-Format  (D.P.)  26.569      6.71D+7     8.98D+307
 * 
 ********************************************************************
 ********************************************************************
 * 
 * Error returns
 * 
 *  The program returns  erfc = 0      for  arg .GE. XBIG;
 * 
 *                       erfcx = XINF  for  arg .LT. XNEG;
 *      and
 *                       erfcx = 0     for  arg .GE. XMAX.
 * 
 * 
 * Intrinsic functions required are:
 * 
 *     fabs, exp
 * 
 * 
 *  Author: W. J. Cody
 *          Mathematics and Computer Science Division
 *          Argonne National Laboratory
 *          Argonne, IL 60439
 * 
 *  Latest modification: June 16, 1988
 */

#include "fpumath.h"

					/* Machine-dependent constants */
#if defined(vax) || defined(tahoe)
#define	XINF	(double)1.70e38
#define	XNEG	(double)(-9.345e0)
#define	XSMALL	(double)1.39e-17
#define	XBIG	(double)9.269e0
#define	XHUGE	(double)1.90e8
#define	XMAX	(double)1.70e38
#else	/* defined(vax) || defined(tahoe) */
#define	XINF	MAXFLOAT
#define	XNEG	(double)(-26.628e0)
#define	XSMALL	(double)1.11e-16
#define	XBIG	(double)26.543e0
#define	XHUGE	(double)6.71e7
#define	XMAX	(double)2.53e307
#endif	/* defined(vax) || defined(tahoe) */
					/* Mathematical constants */
#define	FOUR	(double)4
#define	ONE	(double)1
#define	HALF	(double)0.5
#define	TWO	(double)2
#define	ZERO	(double)0
#define	SQRPI	(double)5.6418958354775628695e-1
#define	THRESH	(double)0.46875
#define	SIXTEN	(double)16

/*
 * Coefficients for approximation to  erf  in first interval
 */
static double A[] = {
	3.16112374387056560e00,
	1.13864154151050156e02,
	3.77485237685302021e02,
	3.20937758913846947e03,
	1.85777706184603153e-1,
};
static double B[] = {
	2.36012909523441209e01,
	2.44024637934444173e02,
	1.28261652607737228e03,
	2.84423683343917062e03,
};

/*
 * Coefficients for approximation to  erfc  in second interval
 */
static double C[] = {
	5.64188496988670089e-1,
	8.88314979438837594e0,
	6.61191906371416295e01,
	2.98635138197400131e02,
	8.81952221241769090e02,
	1.71204761263407058e03,
	2.05107837782607147e03,
	1.23033935479799725e03,
	2.15311535474403846e-8,
};
static double D[] = {
	1.57449261107098347e01,
	1.17693950891312499e02,
	5.37181101862009858e02,
	1.62138957456669019e03,
	3.29079923573345963e03,
	4.36261909014324716e03,
	3.43936767414372164e03,
	1.23033935480374942e03,
};

/*
 * Coefficients for approximation to  erfc  in third interval
 */
static double P[] = {
	3.05326634961232344e-1,
	3.60344899949804439e-1,
	1.25781726111229246e-1,
	1.60837851487422766e-2,
	6.58749161529837803e-4,
	1.63153871373020978e-2,
};
static double Q[] = {
	2.56852019228982242e00,
	1.87295284992346047e00,
	5.27905102951428412e-1,
	6.05183413124413191e-2,
	2.33520497626869185e-3,
};

static double
#if defined(__STDC__) || defined(__GNUC__)
calerf(double x,int jint)
#else
calerf(x,jint)
double x;
int jint;
#endif
{
	register i,skip;
	double y,ysq,xnum,xden,result;

	y = fabs(x);
	if (y <= THRESH) {	/* Evaluate erf for |x| <= 0.46875 */
		ysq = y > XSMALL ? y*y : ZERO;
		xnum = A[4]*ysq;
		xden = ysq;
		for (i = 0; i <= 2; i++) {
			xnum = (xnum+A[i])*ysq;
			xden = (xden+B[i])*ysq;
		}
		result = x*(xnum+A[3])/(xden+B[3]);
		if (jint)
			result = ONE-result;
		if (jint == 2)
			result *= exp(ysq);
		return result;
	}
	else if (y <= FOUR) {	/* Evaluate erfc for 0.46875 <= |x| <= 4.0 */
		ysq = y*y;
		xnum = C[8]*y;
		xden = y;
		for (i = 0; i <= 6; i++) {
			xnum = (xnum+C[i])*y;
			xden = (xden+D[i])*y;
		}
		result = (xnum+C[7])/(xden+D[7]);
		if (jint != 2) {
			i = (int)(y*SIXTEN); ysq = (double)i/SIXTEN;
			result *= exp(-ysq*ysq)*exp(-(y-ysq)*(y+ysq));
		}
	}
	else {			/* Evaluate erfc for |x| > 4.0 */
		result = ZERO;
		skip = 0;
		if (y >= XBIG) {
			if (jint != 2 || y >= XMAX)
				skip++;
			else if (y >= XHUGE) {
				result = SQRPI/y;
				skip++;
			}
		}
		if (!skip) {
			ysq = ONE/(y*y);
			xnum = P[5]*ysq;
			xden = ysq;
			for (i = 0; i <= 3; i++) {
				xnum = (xnum+P[i])*ysq;
				xden = (xden+Q[i])*ysq;
			}
			result = ysq*(xnum+P[4])/(xden+Q[4]);
			result = (SQRPI-result)/y;
			if (jint != 2) {
				i = (int)(y*SIXTEN); ysq = (double)i/SIXTEN;
				result *= exp(-ysq*ysq)*exp(-(y-ysq)*(y+ysq));
			}
		}
	}
	if (jint == 0) {	/* Fix up for negative argument, erf, etc. */
		result = HALF-result; result += HALF;
		if (x < ZERO)
			result = -result;
	}
	else if (jint == 1) {
		if (x < ZERO)
			result = TWO-result;
	}
	else if (x < ZERO) {
		if (x < XNEG)
			result = XINF;
		else {
			i = (int)(x*SIXTEN); ysq = (double)i/SIXTEN;
			y = exp(ysq*ysq)*exp((x-ysq)*(x+ysq));
			result = -result; result += y+y;
		}
	}
	return result;
}

/* 
 *  This subprogram computes approximate values for erf(x).
 *    (see comments heading calerf()).
 * 
 *    Author/date: W. J. Cody, January 8, 1985
 */
float
erff(float x)
{
	return ((float)calerf(x,0));
}

/* 
 *  This subprogram computes approximate values for erfc(x).
 *    (see comments heading calerf()).
 * 
 *    Author/date: W. J. Cody, January 8, 1985
 */
float
erfcf(float x)
{
	return ((float)calerf(x,1));
}

/* 
 *  This subprogram computes approximate values for exp(x*x) * erfc(x).
 *    (see comments heading calerf()).
 * 
 *    Author/date: W. J. Cody, March 30, 1987
 */
float
erfcxf(float x)
{
	return ((float)calerf(x,2));
}
