/*
 *-----------------------------------------------------------------------------
 *  Copyright (c) 1993-1995 European Synchrotron Radiation Facility
 *
 *  Permission to use, copy, modify, and distribute this software and its
 *  documentation for any purpose and without fee is hereby granted, provided
 *  that the "Terms and Conditions of Distribution", given in the documentation
 *  to this software, are applicable and the above copyright message appears
 *  in each copy.
 *
 *-----------------------------------------------------------------------------
 *
 *
 *  CTAXT:      Combine Tcl/Tk with arbitrary X Toolkits 
 *              (into a single application)
 *
 *  Module:	ctaxtEvent.c
 *
 *  Purpose:    Functions for event handling
 *
 *  22.10.1993
 * -25.10.1993	ds/hp	first version
 *  26.10.1993  hp      CTAXT_CreateTopLevelWindow ():
 *                      Calling Tk_CreateMainWindow for initialization of
 *                      the Tk-Stuff but causing it to produce an error before
 *                      any windows are created.
 *  27.10.1993  hp      CTAXT_CreateMainWindow ():
 *                      Saving the XContext before exiting the function
 *                      Calling TkWmMapWindow for initialization of the Tk-
 *                      Stuff for the WindowManager
 *  28.10.1993  hp      Got rid of call to TkWmMapWindow since it slowed down
 *                      the initialization-phase a lot and didn't seem to have
 *                      any useful effects
 *  03.11.1993  hp      since Tk considers its main-window to be the
 *                      application-main-window we have to compute the absolute
 *                      x- and y-coordinates of the window
 *  04.11.1993  hp      Positioning and Resizing is now done by the Generic-
 *                      Event-Handler, so there's no more need for explicit
 *                      Call-Back-Functions for Resizing and therefore no more
 *                      need for using the Motif-DrawingArea as the placeholder
 *                      for the Tk-Main-Window.
 *                      Functions for Explicit-Event-Handling
 *  08.11.1993  hp      Hiding the Tcl-related stuff
 *                      Preparing the Interface for handling multiple Tk-Main-
 *                      Windows and multiple Tcl-Interpreters
 *  10.11.1993  hp      Put in some debug-outputs (conditional compilation)
 *                      Made the Interface independent from the Xt-Library by
 *                      removing the XtDispatch()-Call. If application
 *                      designers want their part of the application
 *                      to response to user interaction (or even to appear on
 *                      the screen) they have to install an explicit event
 *                      handler and dispatch the events their by calling their
 *                      favorite event dispatcher
 *  15.11.1993  hp      Mechanism for adding and freeing extensions to a Tcl 
 *                      interpreter
 *  29.11.1993  hp      changes to make compatible to Tk3.6
 *  06.12.1993  hp      changes to make the TK_LIBRARY define auto-configurable
 *  09.12.1993  hp      changed CTAXT_GetScreen(), so that it doesn't allocate
 *                      a new display structure, if there already exists one
 *                      for the given display.
 *  10.12.1993  hp      changed the generic event handlers to keep track of the
 *                      focus
 *                      Moved all the stuff related to the generic event 
 *                      handlers into a new module
 *                      Fixed the bug, that Tcl interpreters weren't deleted
 *                      within the generic event handlers. This bug fix caused
 *                      a new bug: deleting the last Tcl interpreter causes the
 *                      application to crash. Removed the deletion of Tcl
 *                      interpreters from the generic event handlers. Instead,
 *                      the generic event handler installs a doWhenIdle-handler
 *                      for the deletion of the Tcl interpreter. That should
 *                      be the safest way, since one can be sure, that there
 *                      are no events in the event queue, that might cause
 *                      (somewhere in CTAXT or Tcl/Tk) an interpreter to be
 *                      used, when the application is idling.
 *  14.12.1993  hp      Moved the calls to the functions for tidying up Tcl
 *                      extensions to CTAXT_DeleteInterpreter().
 *                      Removing event handlers CTAXT_KeepTrackOfFocus(), when
 *                      the Tk window is destroyed.
 *  14.01.1994  hp      Defined forward references for functions 
 *                      CTAXT_KeepTrackOfFocus() and CTAXT_DeleteInterpreter()
 *                      Fixed some bugs of incorrect casts like
 *                        (Tk_EventProc) proc --> (Tk_EventProc *) proc
 *                      Fixed the bug in the parameter definition of 
 *                      CTAXT_GetAbsCoords(): 
 *                         Tk_Window *tkwin; --> Tk_Window tkwin;
 *                      Corrected the forward references of 
 *                      CTAXT_GetAbsCoords() and CTAXT_MoveResizeMainWindow()
 *
 */




#include <tkInt.h>
#include "ctaxtInt.h"




/*
 *  Types internally needed by this module
 */

typedef struct {
  Tcl_Interp *interp;
  Tk_Window tkWin;
  CTAXT_GlobalData *data;
} CTAXT_DelInfo;




/*
 *  Prototypes for forward references
 */

void CTAXT_KeepTrackOfFocus _ANSI_ARGS_ ((ClientData clientData,
                                          XEvent *eventPtr));
static void CTAXT_DeleteInterpreter _ANSI_ARGS_ ((ClientData clientData));
static void CTAXT_GetAbsCoords _ANSI_ARGS_ ((Tk_Window tkwin));
int CTAXT_MoveResizeMainWindow _ANSI_ARGS_ ((Tk_Window tkwin));
static void CTAXT_DeleteInterpreter _ANSI_ARGS_ ((ClientData clientData));




/*
 *----------------------------------------------------------------------
 *
 * CTAXT_KeepTrackOfFocus --
 *
 *      This function is a Tk event handler to keep track of the focus.
 *      Whenever a Tk-Main-Window is entered, it creates a FocusIn event
 *      for that window, whenever a Tk-Main-Window is left, it creates
 *      a FocusOut event for that window.
 *
 * Side effects:
 *
 *----------------------------------------------------------------------
 */

void
CTAXT_KeepTrackOfFocus (clientData,eventPtr)
  ClientData clientData;
  XEvent *eventPtr;
{
  static XEvent focusEvent;
  static TkWindow tkwin;


  if (eventPtr->type == EnterNotify) {
    if (eventPtr->xcrossing.detail == NotifyInferior) {

      /*  We're staying within the Tk-Main-Window, nothing to do  */
	
#if CTAXT_DEBUG_LEVEL > 0
      fprintf (stderr,"\nEnterNotify in CTAXT_KeepTrackOfFocus(): "
	              "Nothing to do");
#endif

      return;
    }

#if CTAXT_DEBUG_LEVEL > 0
      fprintf (stderr,
	       "\nEnterNotify in CTAXT_KeepTrackOfFocus(): Send FocusIn "
	       "to Tk-Main-Window 0x%lX",eventPtr->xany.window);
#endif

    focusEvent.xfocus.type = focusEvent.xany.type = focusEvent.type = 
      FocusIn;
  } else {
      if (eventPtr->xcrossing.detail == NotifyInferior) {

	/*  We're staying within the Tk-Main-Window, nothing to do  */
	
#if CTAXT_DEBUG_LEVEL > 0
	fprintf (stderr,"\nLeaveNotify in CTAXT_KeepTrackOfFocus(): "
		        "Nothing to do");
#endif

	return;
      }

#if CTAXT_DEBUG_LEVEL > 0
      fprintf (stderr,
	       "\nLeaveNotify in CTAXT_KeepTrackOfFocus(): Send "
	       "FocusOut to Tk-Main-Window 0x%lX",eventPtr->xany.window);
#endif

      focusEvent.xfocus.type = focusEvent.xany.type = focusEvent.type = 
	FocusOut;

    }  /*  Matches  else  */

  focusEvent.xany.serial = focusEvent.xfocus.serial = eventPtr->xany.serial;
  focusEvent.xany.send_event = focusEvent.xfocus.send_event = False;
  focusEvent.xany.display = focusEvent.xfocus.display = eventPtr->xany.display;
  focusEvent.xany.window = focusEvent.xfocus.window = eventPtr->xany.window;
  focusEvent.xfocus.mode = NotifyNormal;
  focusEvent.xfocus.detail = NotifyVirtual;
  Tk_HandleEvent (&focusEvent);
}


/*
 *----------------------------------------------------------------------
 *
 * CTAXT_DeleteInterpreter --
 *
 *      This function is a Tk idle callback. It gets installed by the
 *      generic event handlers, when a Tcl interpreter is to be deleted.
 *      Tk will invoke the function as soon as there are no events in
 *      the application's event queue pending. The purpose of this event
 *      handler is to delete a Tcl interpreter, to tidy up Tcl extensions
 *      and to remove the event handler to keep track of the focus for the
 *      destroyed window.
 *
 * Side effects:
 *
 *      The Tcl Interpreter will be deleted, it should never be used
 *      again in the application.
 *
 *----------------------------------------------------------------------
 */

static void
CTAXT_DeleteInterpreter (clientData)
  ClientData clientData;
{
  CTAXT_DelInfo *info = (CTAXT_DelInfo *) clientData;
  int k;


  /*  Remove the event handler for keeping track of the focus  */

#if CTAXT_DEBUG_LEVEL > 0
  fprintf (stderr,
	   "\nApplication is idling.\n"
	   "Removing the event handler for window 0x%lX in "
	   "CTAXT_DeleteInterpreter()",
	   ((TkWindow *) info->tkWin)->window);
#endif

  Tk_DeleteEventHandler (info->tkWin,
			 EnterWindowMask | LeaveWindowMask,
			 (Tk_EventProc *) CTAXT_KeepTrackOfFocus,
			 (ClientData) info->data);

#if CTAXT_DEBUG_LEVEL > 0
  fprintf (stderr," ...removed.");
#endif

  /*  Call the functions for cleaning up Tcl extensions  */

  for (k = 0;k < info->data->extensions;k++) {
    if (info->data->extension[k].freeTclExtension != NULL) {

#if CTAXT_DEBUG_LEVEL > 0
  fprintf (stderr,
	   "Calling function 0x%p for tidying up Tcl extension",
	   info->data->extension[k].freeTclExtension);
#endif

      info->data->extension[k].freeTclExtension (
        info->interp,
	info->tkWin,
	info->data->extension[k].clientData
      );

#if CTAXT_DEBUG_LEVEL > 0
  fprintf (stderr," ...done.");
#endif

    }  /*  Matches  if (info->data->extension[k]->freeTclExtension != NULL)  */

  }  /*  Matches  for (k = 0;....)  */


#if CTAXT_DEBUG_LEVEL > 0
  Tcl_Eval (info->interp,"puts stderr \"\nThe interpreter is still usable\""); 
  fprintf (stderr,
	   "\nDeleting now Tcl interpreter 0x%p in CTAXT_DeleteInterpreter()",
	   info->interp);
#endif

  Tcl_DeleteInterp (info->interp);

#if CTAXT_DEBUG_LEVEL > 0
  fprintf (stderr," ...deleted.");
#endif

  /*  free the memory allocated for the info structure  */

  free (info);
}


/*
 *----------------------------------------------------------------------
 *
 * CTAXT_FirstEventHandler --
 *
 *      During the creation of Tk-Windows we already get Events for the
 *      Application. These events have to be passed to the application,
 *      otherwise it will loose some events for e.g. popping up windows.
 *      Since I tried to make the CTAXT_EventHandler as fast as possible,
 *      I didn't want to slow it down by putting in a condition like
 *              if (data->winList == NULL) { ... }
 *      This would have had the same result but the condition will only be
 *      true for the first events. Therefore CTAXT_PutTclTkInWindow() registers
 *      this event handler as a generic one before it calls 
 *      CTAXT_CreateMainWindow() and will replace it by CTAXT_EventHandler()
 *      as soon as the CTAXT_MainLoop() is entered.
 *      In a multiple Tk-Window application it might (although it should not)
 *      be possible that we already get events to destroy previous created
 *      Tk-Windows. Therefore this handler has to provide the same 
 *      functionality as CTAXT_EventHandler(). Hence, the only difference
 *      between the two is the additional condition.
 *
 * Results:
 *      Should return 0 for all events that should be processed by Tk too,
 *      otherwise non-zero
 *
 * Side effects:
 *      depends on the application
 *
 *----------------------------------------------------------------------
 */

int
CTAXT_FirstEventHandler (clientData,event)
  ClientData clientData;
  XEvent *event;
{
  /*  
   *  To speed up the event handler make the functions' local variables
   *  static.
   *
   *  !!!!!!!!!!!!!!!!!!!!!!!!!!!!  Be careful  !!!!!!!!!!!!!!!!!!!!!!!
   *
   *  When we get a Configure-Event we eventually call 
   *  CTAXT_MoveResizeMainWindow() which itself calls Tk_HandleEvent() which
   *  itself invokes this function. However, if we get invoked due to a call
   *  to CTAXT_MoveResizeMainWindow(), we just get to the point where the
   *  function will return without having found an appropriate CTAXT-Main-
   *  Window.
   *  Since "data" is just set to clientData, clientData will not change and
   *  points to CTAXT_Global in CTAXT_PutTclTkInWindow.c, a structure which is
   *  static there, it is safe to declare data as static
   *
   *  !!!!!!!!!!!  Consider carefuly which locals to make static  !!!!!!!!!!!
   */

  static CTAXT_GlobalData *data;
  static CTAXT_MainWindow *w;
  static CTAXT_DelInfo *info;
  static int n,k;
  int i,found;




  data = (CTAXT_GlobalData *) clientData;


  if (data->windows == 0) {
    if (data->handleEventExplicit != NULL) {
      data->handleEventExplicit (event);
    }
    return (0);
  }


  /*
   *  IMPORTANT!!!
   *
   *  Incoming events first have to be handled by the application, 
   *  otherwise e.g. the call to CTAXT_MoveResizeMainWindow() will not have 
   *  the intended effect since the window actually isn't moved or 
   *  resized!
   *  However, if we get a destroy-event for a Tk-Main-Window, don't pass it
   *  to the application, otherwise the window will already be destroyed when
   *  Tk tries to do that.
   *  The application designer has to care for event dispatching by
   *  registering an explicit event handler before entering the CTAXT-Main-Loop
   */

  if ((event->type == ConfigureNotify) || 
       (event->type == ConfigureRequest) ||
       (event->type == DestroyNotify)) {

#if CTAXT_DEBUG_LEVEL > 0
    fprintf (stderr,"\nGot Configure- or Destroy-Event in "
	            "CTAXT_FirstEventHandler()");
#endif

    /*  Event for the Window held in the Cache?  */

    found = data->cache;
    if (((event->xany.window != data->winList[found].mainWinFrame) &&
	 (event->xany.window != data->winList[found].mainWin)) ||
	(event->xany.display != data->winList[found].display)) {

      /*  Window isn't present in cache, look up the dynamic array
	  and put it in  */

#if CTAXT_DEBUG_LEVEL > 0
      fprintf (stderr,"\nCouldn't use the Cache");
#endif

      i = 0;
      found = -1;
      while (i < data->windows) {
	if (((event->xany.window == data->winList[i].mainWinFrame) ||
	     (event->xany.window == data->winList[i].mainWin)) &&
	    (event->xany.display == data->winList[i].display)) {
	  found = i;
	  data->cache = found;
	  break;
	}
	i += data->winList[i].sameMainWin;
      }

      /*  Event for a CTAXT-Main-Window?  */

      if (found == -1) {

	/*  No CTAXT-Main-Window  */

#if CTAXT_DEBUG_LEVEL > 0
	fprintf (stderr,"\nEvent wasn't for a CTAXT-Main-Window");
#endif

        if (data->handleEventExplicit != NULL) {
	  data->handleEventExplicit (event);
        }
        return (0);
      }

    }  /*  End of Event for a Window not held in the Cache  */


    n = data->winList[found].sameMainWin;


    /*  check for DestroyNotify  */

    if (event->type == DestroyNotify) {

      /*  
       *  give memory, allocated for the internal structure, back to the 
       *  system. Free up all the array elements which have as mainWin
       *  the window given in the event
       */

#if CTAXT_DEBUG_LEVEL > 0
      fprintf (stderr,
	       "\nHandle Destroy-Event, number of affected Tk-Windows: %d",
	       n);
#endif

      /*
       *  Don't delete the Tcl interpreter within the generic event handler.
       *  It's probably safer to delete it outside the handler, since there
       *  might be still events in the event queue, causing the interpreter
       *  to be used.
       *  Since it should be safe to delete the Tcl interpreter, when the
       *  application is idling, install an event handler to do the job.
       *  Note: Tk_DoWhenIdle() removes the handler as soon as it was invoked
       *  once.
       */

      (data->windows) -= n;
      for (i = 0;i < n;i++) {

	if ((info = (CTAXT_DelInfo *) calloc (1,
					      sizeof(CTAXT_DelInfo))) != NULL){
	  info->tkWin = data->winList[found+i].tkWin;
	  info->interp = data->winList[found+i].interp;
	  info->data = data;

#if CTAXT_DEBUG_LEVEL > 0
	  fprintf (stderr,
		   "\nInstalling DoWhenIdleEventHandler for tidying up "
		   "window %d",
		   i+1);
#endif

	  Tk_DoWhenIdle ((Tk_IdleProc *) CTAXT_DeleteInterpreter,
			 (ClientData) info);

	}  /*  Matches  if ((info = (CTAXT_DelInfo *) calloc (...)) != NULL) */

	memcpy (&(data->winList[found+i]),
		&(data->winList[found+i+n]),
		sizeof (CTAXT_MainWindow));

      }  /*  Matches  for (i = 0;i < n;i++)  */


      for (i = found+n;i < data->windows;i++) {
	memcpy (&(data->winList[i]),
		&(data->winList[i+n]),
		sizeof (CTAXT_MainWindow));
      }
      if (((w = (CTAXT_MainWindow *) 
	           realloc (data->winList,
			    data->windows*
			    sizeof(CTAXT_MainWindow))) == NULL) &&
	  (data->windows != 0)) {
        data->cache = 0;
	return (0);
      }
      data->winList = w;
      data->cache = 0;

#if CTAXT_DEBUG_LEVEL > 0
      fprintf (stderr,"\nNumber of (CTAXT-)Tk-Main-Windows left: %d",
	       data->windows);
#endif

      return (0);

    }  /*  Matches "if (event->type == DestroyNotify) {"  */


    /*  Pass the event to the application  */

    if (data->handleEventExplicit != NULL) {
      data->handleEventExplicit (event);
    }


    /*
     *  If we got a Configure-Event for the frame of an application-main-
     *  or -toplevel-window (created by the Window-Manager) or for such a
     *  window created directly by the application (Toplevels without any
     *  decoration) compute the new windows' position and size.
     *  Don't worry about recursive calls! Although CTAXT_MoveResizeMainWindow
     *  generates an event by calling Tk_HandleEvent, which will be passed to
     *  this generic-handler too, it will not get to this point again since the
     *  window-id isn't identical to mainWinFrame nor to mainWin.
     *  Since the interface supports multiple Tk-Main-Windows in 
     *  applications we have to call CTAXT_MoveResizeMainWindow for each
     *  CTAXT-Main-Window which is a child of the Main-Window, we got
     *  the event for
     */

#if CTAXT_DEBUG_LEVEL > 0
    fprintf (stderr,"\nMoving/Resizing %d CTAXT-Main-Windows",n);
#endif

    for (i = 0;i < n;i++) {

#if CTAXT_DEBUG_LEVEL > 0
      fprintf (stderr,"\nBefore CTAXT_MoveResizeMainWindow");
#endif

      CTAXT_MoveResizeMainWindow (data->winList[found+i].tkWin);

#if CTAXT_DEBUG_LEVEL > 0
      fprintf (stderr,"\nAfter CTAXT_MoveResizeMainWindow");
#endif

    }

    return (0);

  } /*  Matches "if (event->type ....) {"  */

  if (data->handleEventExplicit != NULL) {
    data->handleEventExplicit (event);
  }
  return (0);
}


/*
 *----------------------------------------------------------------------
 *
 * CTAXT_EventHandler --
 *
 *      Each Event that is to be processed by the Tk is mirrored to this
 *      function to be passed to the application
 *
 * Results:
 *      Should return 0 for all events that should be processed by Tk too,
 *      otherwise non-zero
 *
 * Side effects:
 *      depends on the application
 *      If a ConfigureNotify-Event is retrieved for one of the Main-Windows,
 *      all Tk-Main-Windows that are children of the Main-Window will update
 *      their position and size. If a DestroyNotify-Event is retrieved for 
 *      one of the CTAXT-Main-Windows the internal structures allocated for 
 *      that window will be given back to the system and the Tcl-Interpreter 
 *      attached to that window will be destroyed if no other 
 *      CTAXT-Main-Window refers to it.
 *
 *----------------------------------------------------------------------
 */

int
CTAXT_EventHandler (clientData,event)
  ClientData clientData;
  XEvent *event;
{
  /*  
   *  To speed up the event handler make the functions' local variables
   *  static.
   *
   *  !!!!!!!!!!!!!!!!!!!!!!!!!!!!  Be careful  !!!!!!!!!!!!!!!!!!!!!!!
   *
   *  When we get a Configure-Event we eventually call 
   *  CTAXT_MoveResizeMainWindow() which itself calls Tk_HandleEvent() which
   *  itself invokes this function. However, if we get invoked due to a call
   *  to CTAXT_MoveResizeMainWindow(), we just get to the point where the
   *  function will return without having found an appropriate CTAXT-Main-
   *  Window.
   *  Since "data" is just set to clientData, clientData will not change and
   *  points to CTAXT_Global in CTAXT_PutTclTkInWindow.c, a structure which is
   *  static there, it is safe to declare data as static
   *
   *  !!!!!!!!!!!  Consider carefuly which locals to make static  !!!!!!!!!!!
   */

  static CTAXT_GlobalData *data;
  static CTAXT_MainWindow *w;
  static CTAXT_DelInfo *info;
  static int n,k;
  int i,found;




  data = (CTAXT_GlobalData *) clientData;


  /*
   *  IMPORTANT!!!
   *
   *  Incoming events first have to be handled by the application, 
   *  otherwise e.g. the call to CTAXT_MoveResizeMainWindow() will not have 
   *  the intended effect since the window actually isn't moved or 
   *  resized!
   *  However, if we get a destroy-event for a Tk-Main-Window, don't pass it
   *  to the application, otherwise the window will already be destroyed when
   *  Tk tries to do that.
   *  The application designer has to care for event dispatching by
   *  registering an explicit event handler before entering the CTAXT-Main-Loop
   */

  if ((event->type == ConfigureNotify) || 
       (event->type == ConfigureRequest) ||
       (event->type == DestroyNotify)) {

#if CTAXT_DEBUG_LEVEL > 0
    fprintf (stderr,"\nGot Configure- or Destroy-Event in "
	            "CTAXT_EventHandler()");
#endif

    /*  Event for the Window held in the Cache?  */

    found = data->cache;
    if (((event->xany.window != data->winList[found].mainWinFrame) &&
	 (event->xany.window != data->winList[found].mainWin)) ||
	(event->xany.display != data->winList[found].display)) {

      /*  Window isn't present in cache, look up the dynamic array
	  and put it in  */

#if CTAXT_DEBUG_LEVEL > 0
      fprintf (stderr,"\nCouldn't use the Cache");
#endif

      i = 0;
      found = -1;
      while (i < data->windows) {
	if (((event->xany.window == data->winList[i].mainWinFrame) ||
	     (event->xany.window == data->winList[i].mainWin)) &&
	    (event->xany.display == data->winList[i].display)) {
	  found = i;
	  data->cache = found;
	  break;
	}
	i += data->winList[i].sameMainWin;
      }

      /*  Event for a CTAXT-Main-Window?  */

      if (found == -1) {

	/*  No CTAXT-Main-Window  */

#if CTAXT_DEBUG_LEVEL > 0
	fprintf (stderr,"\nEvent wasn't for a CTAXT-Main-Window");
#endif

        if (data->handleEventExplicit != NULL) {
	  data->handleEventExplicit (event);
        }
        return (0);
      }

    }  /*  End of Event for a Window not held in the Cache  */


    n = data->winList[found].sameMainWin;


    /*  check for DestroyNotify  */

    if (event->type == DestroyNotify) {

      /*  
       *  give memory, allocated for the internal structure, back to the 
       *  system. Free up all the array elements which have as mainWin
       *  the window given in the event
       */

#if CTAXT_DEBUG_LEVEL > 0
      fprintf (stderr,
	       "\nHandle Destroy-Event, number of affected Tk-Windows: %d",
	       n);
#endif

      /*
       *  Don't delete the Tcl interpreter within the generic event handler.
       *  It's probably safer to delete it outside the handler, since there
       *  might be still events in the event queue, causing the interpreter
       *  to be used.
       *  Since it should be safe to delete the Tcl interpreter, when the
       *  application is idling, install an event handler to do the job.
       *  Note: Tk_DoWhenIdle() removes the handler as soon as it was invoked
       *  once.
       */

      (data->windows) -= n;
      for (i = 0;i < n;i++) {

	if ((info = (CTAXT_DelInfo *) calloc (1,
					      sizeof(CTAXT_DelInfo))) != NULL){
	  info->tkWin = data->winList[found+i].tkWin;
	  info->interp = data->winList[found+i].interp;
	  info->data = data;

#if CTAXT_DEBUG_LEVEL > 0
	  fprintf (stderr,
		   "\nInstalling DoWhenIdleEventHandler for tidying up "
		   "window %d",
		   i+1);
#endif

	  Tk_DoWhenIdle ((Tk_IdleProc *) CTAXT_DeleteInterpreter,
			 (ClientData) info);

	}  /*  Matches  if ((info = (CTAXT_DelInfo *) calloc (...)) != NULL) */

	memcpy (&(data->winList[found+i]),
		&(data->winList[found+i+n]),
		sizeof (CTAXT_MainWindow));

      }  /*  Matches  for (i = 0;i < n;i++)  */


      for (i = found+n;i < data->windows;i++) {
	memcpy (&(data->winList[i]),
		&(data->winList[i+n]),
		sizeof (CTAXT_MainWindow));
      }
      if (((w = (CTAXT_MainWindow *) 
	           realloc (data->winList,
			    data->windows*
			    sizeof(CTAXT_MainWindow))) == NULL) &&
	  (data->windows != 0)) {
        data->cache = 0;
	return (0);
      }
      data->winList = w;
      data->cache = 0;

#if CTAXT_DEBUG_LEVEL > 0
      fprintf (stderr,"\nNumber of (CTAXT-)Tk-Main-Windows left: %d",
	       data->windows);
#endif

      return (0);

    }  /*  Matches "if (event->type == DestroyNotify) {"  */


    /*  Pass the event to the application  */

    if (data->handleEventExplicit != NULL) {
      data->handleEventExplicit (event);
    }


    /*
     *  If we got a Configure-Event for the frame of an application-main-
     *  or -toplevel-window (created by the Window-Manager) or for such a
     *  window created directly by the application (Toplevels without any
     *  decoration) compute the new windows' position and size.
     *  Don't worry about recursive calls! Although CTAXT_MoveResizeMainWindow
     *  generates an event by calling Tk_HandleEvent, which will be passed to
     *  this generic-handler too, it will not get to this point again since the
     *  window-id isn't identical to mainWinFrame nor to mainWin.
     *  Since the interface supports multiple Tk-Main-Windows in 
     *  applications we have to call CTAXT_MoveResizeMainWindow for each
     *  CTAXT-Main-Window which is a child of the Main-Window, we got
     *  the event for
     */

#if CTAXT_DEBUG_LEVEL > 0
    fprintf (stderr,"\nMoving/Resizing %d CTAXT-Main-Windows",n);
#endif

    for (i = 0;i < n;i++) {

#if CTAXT_DEBUG_LEVEL > 0
      fprintf (stderr,"\nBefore CTAXT_MoveResizeMainWindow");
#endif

      CTAXT_MoveResizeMainWindow (data->winList[found+i].tkWin);

#if CTAXT_DEBUG_LEVEL > 0
      fprintf (stderr,"\nAfter CTAXT_MoveResizeMainWindow");
#endif

    }

    return (0);

  } /*  Matches "if (event->type ....) {"  */

  if (data->handleEventExplicit != NULL) {
    data->handleEventExplicit (event);
  }
  return (0);
}


/*
 *--------------------------------------------------------------
 *
 * CTAXT_GetAbsCoords --
 *
 *	This procedure calculates the absolute coordinates for the top-left-
 *      corner of the Tk-Main-Window by processing the XWindow-Hierarchy.
 *
 * Results:
 *      Updates the Tk-Window-Structure to the new window-position
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

static void
CTAXT_GetAbsCoords(tkwin)
  Tk_Window  tkwin;
{
  TkWindow               *winPtr = ((TkWindow *) tkwin);
  XWindowAttributes      xattrbs;
  Window                 root,parent,window,*children;
  unsigned int           nchildren;
  int                    x,y;

    
  x = y = 0;
  parent = winPtr->window;
  do {
    window = parent;
    if (XGetWindowAttributes (winPtr->display,window,&xattrbs) == 0) {
      break;
    }
    x += xattrbs.x;
    y += xattrbs.y;
    if (XQueryTree (winPtr->display,window,&root,&parent,
		    &children,&nchildren) == 0) {
      XFree ((caddr_t) children);
      break;
    }
    XFree ((caddr_t) children);
  } while (window != root);
  winPtr->changes.x = x;
  winPtr->changes.y = y;
}


/*
 *----------------------------------------------------------------------
 *
 * CTAXT_MoveResizeMainWindow --
 *
 *      This function is called by the Generic-Event-Handler whenever it
 *      handles a Configure-Event.
 *
 * Results:
 *
 * Side effects:
 *      The Size of each Tk-Children of the Tk-MainWindow is updatet on
 *      the screen.
 *
 *----------------------------------------------------------------------
 */

int
CTAXT_MoveResizeMainWindow (tkwin)
  Tk_Window tkwin;
{
  XWindowAttributes    attrbs;
  XEvent               event;
  TkWindow             *winPtr = (TkWindow *) tkwin;


  if (tkwin == NULL) {
    return TCL_ERROR;
  }
  XGetWindowAttributes (winPtr->display,winPtr->window,&attrbs);
  winPtr->changes.width = attrbs.width;
  winPtr->changes.height = attrbs.height;
  CTAXT_GetAbsCoords (tkwin);
  event.type = ConfigureNotify;
  event.xconfigure.serial = LastKnownRequestProcessed(winPtr->display);
  event.xconfigure.send_event = False;
  event.xconfigure.display = winPtr->display;
  event.xconfigure.event = winPtr->window;
  event.xconfigure.window = winPtr->window;
  event.xconfigure.x = winPtr->changes.x;
  event.xconfigure.y = winPtr->changes.y;
  event.xconfigure.width = winPtr->changes.width;
  event.xconfigure.height = winPtr->changes.height;
  event.xconfigure.border_width = winPtr->changes.border_width;
  if (winPtr->changes.stack_mode == Above) {
    event.xconfigure.above = winPtr->changes.sibling;
  } else {
      event.xconfigure.above = None;
    }
  event.xconfigure.override_redirect = winPtr->atts.override_redirect;
  Tk_HandleEvent(&event);

  return TCL_OK;
}
