/* Bos_Basic.c - basic bos operations */

/* 
 *
 * /afs/cs/project/edrc/ndim/source/bos/libbos/Bos_Basic.c,v 1.2 1992/07/14 03:01:26 snl Exp
 *
 * HISTORY
 *
 * Bos_Basic.c,v
 * Revision 1.2  1992/07/14  03:01:26  snl
 * Added evanescence everywhere, plus small changes to Storage
 *
 * Revision 1.1.1.1  1992/05/08  19:45:31  snl
 * bos 1.2
 *
 * Revision 1.1  92/03/06  22:02:55  snl
 * Initial revision
 * 
 * Revision 1.2  92/01/27  16:19:18  snl
 * Port to new TCL
 * 
 * Revision 1.1  91/12/16  20:14:38  snl
 * Initial revision
 * 
 */

#ifndef lint
static char *_RCSId =
 "/afs/cs/project/edrc/ndim/source/bos/libbos/Bos_Basic.c,v 1.2 1992/07/14 03:01:26 snl Exp";
#endif /* lint */

#include "bosInt.h"
#include "bosSearch.h"

#define _Bos_SEARCH_FOUND 0x01

static Bos_Object *bosCreate();
static void bosDestroy();

Bos_Object *Bos_Find(world, obj_name)
     Bos_World *world;
     char *obj_name;
{
  Tcl_HashEntry *e, *_Bos_Find();

  e = _Bos_Find(Bos_Objects(world), obj_name);
  return e? (Bos_Object *)Tcl_GetHashValue(e): (Bos_Object *)0;
}

Bos_Object *Bos_CreateNewObject(world, obj_name)
     Bos_World *world;
     char *obj_name;
{
  return bosCreate(world, obj_name);
}

Bos_Object *Bos_Copy(world, obj_name, new_name)
     Bos_World *world;
     char *obj_name;
     char *new_name;
{
  Bos_Object *orig, *obj;

  orig = Bos_Find(world, obj_name);
  if (orig == (Bos_Object *)0)
    obj = (Bos_Object *)0;
  else if (_Bos_Find(Bos_Objects(world), new_name) != (Tcl_HashEntry *)0)
    obj = (Bos_Object *)0;
  else {
    void copySlots();

    obj = bosCreate(world, new_name);
    copySlots(obj, orig);
  }
  return obj;
}

Bos_Object *Bos_Destroy(world, obj_name)
     Bos_World *world;
     char *obj_name;
{
  Tcl_HashEntry *entry;
  Bos_Object *obj;

  entry = _Bos_Find(Bos_Objects(world), obj_name);
  if (entry == (Tcl_HashEntry *)0)
    obj = (Bos_Object *)0;
  else {
    obj = (Bos_Object *)Tcl_GetHashValue(entry);
    bosDestroy(world, entry);
  }
  return obj;
}

void Bos_DestroySearchResults(table)
     Tcl_HashTable *table;
{
  Bos_World fake_world;
  Tcl_HashSearch search;
  Tcl_HashEntry *entry;

  if (table == (Tcl_HashTable *)0)
    return;
  fake_world.objects = table;
  fake_world.methods = (Tcl_Interp *)0;
  for (entry = Tcl_FirstHashEntry(table, &search);
       entry != (Tcl_HashEntry *)0;
       entry = Tcl_NextHashEntry(&search)) {
    Bos_Object *obj;

    obj = (Bos_Object *)Tcl_GetHashValue(entry);
    bosDestroy(&fake_world, entry);
  }
  ckfree(table);
}

Tcl_HashTable *Bos_Search1(search_space, results, slot_name, boolean, value)
     Tcl_HashTable *search_space;
     Tcl_HashTable *results;
     char *slot_name;
     Bos_Search_Boolean boolean;
     char *value;
{
  Tcl_HashSearch search;
  Tcl_HashEntry *entry;
  Tcl_HashTable *sweepSearchResults();
  unsigned n_matched;

  n_matched = 0;
  for (entry = Tcl_FirstHashEntry(search_space, &search);
       entry != (Tcl_HashEntry *)0;
       entry = Tcl_NextHashEntry(&search)) {
    Bos_Object *obj;
    Tcl_HashSearch slot_search;
    Tcl_HashEntry *s;
    char matched;

    obj = (Bos_Object *)Tcl_GetHashValue(entry);
    matched = 0;
    for (s = Tcl_FirstHashEntry(obj->slots, &slot_search);
	 s != (Tcl_HashEntry *)0;
	 s = Tcl_NextHashEntry(&slot_search)) {
      Bos_Slot *slot;

      slot = (Bos_Slot *)Tcl_GetHashValue(s);
      if (!strcmp(slot->name, slot_name) &&
	  value_matches(slot, boolean, value)) {
	/* If search_space == results, then this is an AND, which means
	   that we're going to remove any objects that don't match from
	   results (and also from search_space). So, we must mark the
	   ones that do match somehow, so we know what to delete when
	   we're done. */
	if (search_space == results) {
	  obj->flags |= _Bos_SEARCH_FOUND;
	  matched = 1;
	  break;
	} else {
	  void copySlots();
	  Bos_Object *internObject();
	  
	  if (results == (Tcl_HashTable *)0) {
	    results = (Tcl_HashTable *)ckalloc(sizeof(Tcl_HashTable));
	    Tcl_InitHashTable(results, TCL_STRING_KEYS);
	  }
	  copySlots(internObject(results, obj->name), obj);
	  matched = 1;
	  break;
	}
      }
    }
    if (matched)
      n_matched++;
    else if (search_space == results)
      /* If this object did not match, and we are going to sweep results
	 of non-marked objects, then make sure that the flags for this
	 object are clear. */
      obj->flags = 0;
  }
  if (search_space == results) {
    if (n_matched > 0)
      results = sweepSearchResults(results);
    else {
      Bos_DestroySearchResults(results);
      results = (Tcl_HashTable *)0;
    }
  }
  return results;
}

/* _Bos_Find - Semi-Public function
 */

Tcl_HashEntry *_Bos_Find(table, obj_name)
     Tcl_HashTable *table;
     char *obj_name;
{
  Tcl_HashEntry *entry;

  if (table == (Tcl_HashTable *)0)
    entry = (Tcl_HashEntry *)0;
  else
    entry = Tcl_FindHashEntry(table, obj_name);
  return entry;
}

/* Local procedures */

static Bos_Object *bosCreate(world, obj_name)
     Bos_World *world;
     char *obj_name;
{
  Bos_Object *obj, *internObject();
  Tcl_HashTable *table = Bos_Objects(world);

  if (table == (Tcl_HashTable *)0)
    obj = (Bos_Object *)0;
  else
    obj = internObject(table, obj_name);
  return obj;
}

static Bos_Object *internObject(table, obj_name)
     Tcl_HashTable *table;
     char *obj_name;
{
  Boolean new_p = FALSE;
  Tcl_HashEntry *he;
  Bos_Object *obj;

  he = Tcl_CreateHashEntry(table, obj_name, &new_p);
  if (!new_p)
    obj = (Bos_Object *)Tcl_GetHashValue(he);
  else {
    obj = (Bos_Object *)ckalloc(sizeof(Bos_Object));
    obj->flags = 0;
    obj->name = (char *)ckalloc(strlen(obj_name) + 1);
    strcpy(obj->name, obj_name);
    obj->slots = (Tcl_HashTable *)ckalloc(sizeof(Tcl_HashTable));
    if (obj->slots == (Tcl_HashTable *)0)
      Bos_FatalError("bosCreate(): could not initialize hash table.\n");
    Tcl_InitHashTable(obj->slots, TCL_STRING_KEYS);
    Tcl_SetHashValue(he, (ClientData)obj);
  }
  return obj;
}

static void copySlots(obj, orig)
     Bos_Object *obj;
     Bos_Object *orig;
{
  Tcl_HashSearch search;
  Tcl_HashEntry *entry;
  
  for (entry = Tcl_FirstHashEntry(orig->slots, &search);
       entry != (Tcl_HashEntry *)0;
       entry = Tcl_NextHashEntry(&search)) {
    Bos_Slot *slot;
    
    slot = (Bos_Slot *)Tcl_GetHashValue(entry);
    (void) Bos_CopySlot(orig, slot->name, obj);
  }
}

static void bosDestroy(world, entry)
     Bos_World *world;
     Tcl_HashEntry *entry;
{
  Bos_Object *obj;
  Tcl_HashEntry *s;
  Tcl_HashSearch search;
  Tcl_HashTable *table = Bos_Objects(world);

  obj = (Bos_Object *)Tcl_GetHashValue(entry);
  ckfree(obj->name);
  for (s = Tcl_FirstHashEntry(obj->slots, &search); s != (Tcl_HashEntry *)0;
       s = Tcl_NextHashEntry(&search)) {
    Bos_Slot *slot;

    slot = (Bos_Slot *)Tcl_GetHashValue(s);
    ckfree(slot->name);
    switch (Bos_PlainSlotType(slot->type)) {
    case Bos_SLOT_METHOD:
      if (slot->value != (_VoidPtr)0) {
        Bos_Method *method = (Bos_Method *)slot->value;

        if (method->flags & Bos_METHOD_INTERNED &&
	    (Bos_Methods(world) != (Tcl_Interp *)0)) {
          (void) Tcl_DeleteCommand(Bos_Methods(world), method->proc_name);
          ckfree(method->proc_name);
	}
	if (method->body != (char *)0)
	  ckfree(method->body);
      }
      /* Fall through to the rest of the slot deletion. */
    case Bos_SLOT_NORMAL:
    case Bos_SLOT_OBJECT:
    case Bos_SLOT_REFERENCE:
      if (slot->value != (_VoidPtr)0)
        ckfree(slot->value);
      break;
    default:
      break;
    }
  }
  Tcl_DeleteHashTable(obj->slots);
  Tcl_DeleteHashEntry(entry);
}

static int value_matches(slot, bool, val)
     Bos_Slot *slot;
     Bos_Search_Boolean bool;
     char *val;
{
  int matches = 0, m, v;

  if (slot->value == (_VoidPtr)0) {
    if  (bool == Bos_SEARCH_TYPE_EQ || bool == Bos_SEARCH_TYPE_NE) {
      Bos_Slot_Type type, plain_type, _Bos_ParseSlotType();

      plain_type = Bos_PlainSlotType(slot->type);
      type = _Bos_ParseSlotType(val);
      if (type == Bos_SLOT_ILLEGAL)
	matches = 0;
      else if (bool == Bos_SEARCH_TYPE_EQ)
	matches = (plain_type == type);
      else
	matches = (plain_type != type);
    } else
      matches = 0;
  } else
    switch (Bos_PlainSlotType(slot->type)) {
    case Bos_SLOT_NORMAL:
    case Bos_SLOT_OBJECT:
    case Bos_SLOT_REFERENCE:
      switch (bool) {
      case Bos_SEARCH_STRING_EQ: 
      case Bos_SEARCH_STRING_NE:
      case Bos_SEARCH_STRING_GT:
      case Bos_SEARCH_STRING_LT:
      case Bos_SEARCH_STRING_GE:
      case Bos_SEARCH_STRING_LE:
	m = strcmp(slot->value, val);
	switch (bool) {
	case Bos_SEARCH_STRING_EQ: 
	  matches = (m == 0);
	  break;
	case Bos_SEARCH_STRING_NE:
	  matches = (m != 0);
	  break;
	case Bos_SEARCH_STRING_GT:
	  matches = (m > 0);
	  break;
	case Bos_SEARCH_STRING_LT:
	  matches = (m < 0);
	  break;
	case Bos_SEARCH_STRING_GE:
	  matches = (m >= 0);
	  break;
	case Bos_SEARCH_STRING_LE:
	  matches = (m <= 0);
	}
	break;
      case Bos_SEARCH_INT_EQ:
      case Bos_SEARCH_INT_NE:
      case Bos_SEARCH_INT_GT:
      case Bos_SEARCH_INT_LT:
      case Bos_SEARCH_INT_GE:
      case Bos_SEARCH_INT_LE:
	if (sscanf(slot->value, "%d", &m) != 1)
	  matches = 0;
	else if (sscanf(val, "%d", &v) != 1)
	  matches = 0;
	else
	  switch (bool) {
	  case Bos_SEARCH_INT_EQ:
	    matches = (m == v);
	    break;
	  case Bos_SEARCH_INT_NE:
	    matches = (m != v);
	    break;
	  case Bos_SEARCH_INT_GT:
	    matches = (m > v);
	    break;
	  case Bos_SEARCH_INT_LT:
	    matches = (m < v);
	    break;
	  case Bos_SEARCH_INT_GE:
	    matches = (m >= v);
	    break;
	  case Bos_SEARCH_INT_LE:
	    matches = (m <= v);
	    break;
	  }
	break;
      case Bos_SEARCH_TYPE_EQ:
      case Bos_SEARCH_TYPE_NE:
	{ Bos_Slot_Type type, _Bos_ParseSlotType();
	
	  type = _Bos_ParseSlotType(val);
	  if (type == Bos_SLOT_ILLEGAL)
	    matches = 0;
	  else if (bool == Bos_SEARCH_TYPE_EQ)
	    matches = (slot->type == type);
	  else
	    matches = (slot->type != type);
	}
	break;
      default:
	Bos_FatalError("Bos_Tcl.c:valueMatches() impossible condition #1");
	break;
      }
      break;
    case Bos_SLOT_METHOD:
    case Bos_SLOT_CMETHOD:
    case Bos_SLOT_FOREIGN:
      matches = 0;
      break;
    default:
      Bos_FatalError("Bos_Tcl.c:valueMatches() impossible condition #2");
      break;
    }
  return matches;
}

static Tcl_HashTable *sweepSearchResults(results)
     Tcl_HashTable *results;
{
  Tcl_HashSearch search;
  Tcl_HashEntry *entry;
  Bos_World fake_world;

  if (results == (Tcl_HashTable *)0)
    return;
  fake_world.objects = results;
  fake_world.methods = (Tcl_Interp *)0;
  for (entry = Tcl_FirstHashEntry(results, &search);
       entry != (Tcl_HashEntry *)0;
       entry = Tcl_NextHashEntry(&search)) {
    Bos_Object *obj;

    obj = (Bos_Object *)Tcl_GetHashValue(entry);
    if (!(obj->flags & _Bos_SEARCH_FOUND))
      bosDestroy(&fake_world, entry);
  }
  return results;
}
