#############################################################################
#
# folders.tcl: This file contains all procedures related to folders
#
# CreateFol {Fol}
#       Create the Folder with some checks. returns List of folders it created
#
# GetMsgNum {Fol maxnum}
#       Get the message number from the selected message in the Folder 
#       list and return it. maxnum is the maximum number of messages,
#       defaults to 1.
#
# OpenFol {Fol force}
#       Create a window in which the headers from messages in $Fol can
#       be displayed, use ScanFol to read the messages. if force is 1,
#	a folder rescan is forced
#
# PackFol {Fol}
#       Renumber the files in the folder so that they will go from
#       1-nummessages, also delete files starting with a , which are
#       backup files from deleted messages.
#       TODO: implement it
#
# ScanFol {Fol}
#       Fill the Folder window with the message headers, using the MH scan 
#       command.
#
# RemoveFol {Fol}
#       Remove the folder, ask for confirmation
#
# RenameFol {Fol}
#       Rename $Fol.
#
# SortFol {Fol}
#       Sort a folder according to the sort options in the tkmh_profile
#       and the settings in the configuration box
#       TODO: write this all, including configuration box entries, and so
#       temporarily use sortm from mh to sort by date
#
# dofolders { dir lstname {path {}}}
#	approximate substitute for the MH folders command, thanks to
#	Mark Moraes (moraes@deshaw.com)
#
#############################################################################

proc dofolders {dir lstname {path {}}} {
    upvar 1 $lstname lst 
    # puts "debug: dofolders $dir $lstname $path"
    set f [open "|/bin/ls $dir" r]
    while {[gets $f line] >= 0} {
        if {[regexp {^,?[0-9]+$} $line]} {
            # puts "debug: skipping $line"
            continue
        }
        catch {file stat $dir/$line st}
        # puts "debug: $line, $path$line, $st(type)"
        if {$st(type) == "directory"} {
            lappend lst "$path$line"
            if {$st(nlink) > 2} {
                dofolders $dir/$line lst $path$line/
            }
        }
    }
    # puts "debug: finishing $dir $path"
    close $f
}

proc CreateFol Fol {
    global MH cmd FolList FolTree

    if {[string first { } $Fol] != -1} {
        MsgWindow "No spaces allowed in foldernames"
        return
    }
    set dirs {}
    set dummy {}
    foreach name [split $Fol "/"] {
        if {$dummy == {}} {set dummy $name} \
        else {set dummy "$dummy/$name"}
        lappend dirs $dummy
    }
    foreach dir $dirs {
        if [file isfile [RF $MH(Path)/$dir]] {
            MsgWindow "$dir already exists as a file"
            return
        }
        if [file isdirectory [RF $MH(Path)/$dir]] {continue}
        eval exec $cmd(mkdir) [RF $MH(Path)/$dir]
        lappend FolList $dir
    }
    set FolList [lsort $FolList]
    Cache:Write [RF $MH(Path)/.folders] $FolList
    set FolTree [BuildFolTree $FolList]
    AdaptFolMenus
    DisplayFols $FolList
}

proc GetMsgNum {Fol {maxnum 1}} {
    set Result {}
    if [conv_listbox_seq $Fol selmsgs] {
        set msgs [conv_seq $Fol selmsgs]
        del_seq $Fol selmsgs
        return [lrange $msgs 0 [expr $maxnum - 1]]
    } else {return {}}
}

proc OpenFol {Fol {force 0}} {
    global FolTree Aliases
    global FONT PREF BM

    set w [FW $Fol]
    if [winfo exists $w] {
      set NumMsg [ScanFol $Fol]
      if $PREF(LastMsg) {
        set height [lindex [split [lindex [$w.t configure -geometry] 4] "x"] 1]
        $w.t yview [expr $NumMsg - $height]
      }
      wm deiconify $w
      return
    }
    toplevel $w -class Folder
    wm title $w $Fol
    wm iconbitmap $w @$BM(folder)

    scrollbar $w.s -command "$w.t yview"
    listbox $w.t -yscroll "$w.s set" -font $FONT(FixB)

    frame $w.m -borderwidth 2 -relief raised
    checkbutton $w.m.backup -variable PREF(BackupMsg) \
      -text {Backup on delete}
    menubutton $w.m.folder -text Folder -menu $w.m.folder.m -width 8
    menu $w.m.folder.m
      $w.m.folder.m add command -label Rescan -command "ScanFol $Fol 1"
      $w.m.folder.m add command -label Pack -command "PackFol $Fol; ScanFol $Fol"
#      $w.m.folder.m add command -label {Pick Sequence} -command "Pick $Fol"
      $w.m.folder.m add command -label {Sort by Date} -command "SortFol $Fol; ScanFol $Fol"

    frame $w.b
    button $w.b.done -text Done -command "destroy $w"
    button $w.b.help -text Help -command {Help Folder}
    button $w.b.show -text Show -command \
     "set Msg \[GetMsgNum $Fol\]; if {\$Msg != {}} {ShowMsg $Fol \$Msg}"
    button $w.b.print -text Print -command \
     "set Msgs \[GetMsgNum $Fol 100\]
      if {\$Msgs != {}} {PrintMsg $Fol \$Msgs}"
    button $w.b.delete -text Delete -command \
     "if \[conv_listbox_seq $Fol delmsgs\] {DeleteMsg $Fol}"
    button $w.b.reply -text Reply -command \
     "if {\[set Msg \[GetMsgNum $Fol\]\] != {}} {ReplyMsg $Fol \$Msg}"
    menubutton $w.b.move -text {Move To} -menu $w.b.move.m
    menubutton $w.b.copy -text {Copy To} -menu $w.b.copy.m

    MakeMenu $w.b.move.m $FolTree 1 {} -command \
      "if \[conv_listbox_seq $Fol movemsgs\] {MoveMsg $Fol %EntryPath}"
    MakeMenu $w.b.copy.m $FolTree 1 {} -command \
      "if \[conv_listbox_seq $Fol copymsgs\] {CopyMsg $Fol %EntryPath}"

    if $PREF(AliMenu) {
      menubutton $w.b.forw -text "Forward To" -menu $w.b.forw.m
      menu $w.b.forw.m
        $w.b.forw.m add command -label *NEW* -command \
          "set Msg \[GetMsgNum $Fol\]
           if {\$Msg != {}} {ForwardMsg $Fol \$Msg}"
       MakeMenu $w.b.forw.m $Aliases 0 {} -command  \
          "set Msg \[GetMsgNum $Fol\]
           if {\$Msg != {}} {
             if \$PREF(ExpandAlias) {ForwardMsg $Fol \$Msg \"\[SearchAlias %Entry 1\]\"} \
             else {ForwardMsg $Fol \$Msg %Entry}
           }"
    } else {
      button $w.b.forw -text "Forward" -command \
          "set Msg \[GetMsgNum $Fol\]
           if {\$Msg != {}} {ForwardMsg $Fol \$Msg}"
    }

    pack append $w.m \
      $w.m.folder {left} \
      $w.m.backup {right}

    pack append $w.b \
      $w.b.show {top expand fill} \
      $w.b.print {top expand fill} \
      $w.b.reply {top expand fill} \
      $w.b.forw {top expand fill} \
      $w.b.copy {top expand fill} \
      $w.b.move {top expand fill} \
      $w.b.delete {top expand fill} \
      $w.b.help {top expand fill} \
      $w.b.done {top expand fill} 

    pack append $w \
      $w.m {top fillx} \
      $w.t {left expand fill} \
      $w.s {left filly} \
      $w.b {left filly} 

    bind $w.t <Double-1> "FlashSel $w.t; $w.b.show invoke"
    bind $w.t <3> \
     "$w.t select clear
      $w.t select from \[$w.t nearest %y\]
      update; FlashSel $w.t; $w.b.reply invoke"
    bind $w.t <Control-d> "$w.b.delete invoke"
    bind $w.t <Control-a> {%W select from 0; %W select to end}

    set NumMsg [ScanFol $Fol]
    if $PREF(LastMsg) {
      set height [lindex [split [lindex [$w.t configure -geometry] 4] "x"] 1]
      $w.t yview [expr $NumMsg - $height]
    }
    focus $w.t
}

proc PackFol {Fol} {
    global MH
    catch {exec $MH(bindir)/folder -pack +$Fol}
}

proc ScanFol {Fol {force 0}} {
    global MH
    
    set w [FW $Fol]
#     if $force {puts "dbg: forced rescan"}
#     if {![file isfile [RF $MH(Path)/$Fol/.xmhcache]]} {puts "dbg: no cache"} \
#     else {
#     if {[file mtime [RF $MH(Path)/$Fol]] > \
#         [file mtime [RF $MH(Path)/$Fol/.xmhcache]]} \
#         {puts "dbg: cache out of time"}
#     }

    if {$force || ![file isfile [RF $MH(Path)/$Fol/.xmhcache]]
               || [file mtime [RF $MH(Path)/$Fol]] > \
                  [file mtime [RF $MH(Path)/$Fol/.xmhcache]]} {
         if [catch {exec $MH(bindir)/scan +$Fol} Msgs] {
#            puts "dbg: scan error: $Msgs"
            Cache:Write [RF $MH(Path)/$Fol/.xmhcache] {}
            del_seq $Fol allmsgs
            if [winfo exists $w] {$w.t delete 0 end}
            return 0
        }
        set Msgs [split $Msgs "\n"]
#        puts "dbg: writing cache"
        set Msgnums [exec $MH(bindir)/pick]
        Cache:Write [RF $MH(Path)/$Fol/.xmhcache] $Msgs
        write_seq $Fol allmsgs
    } else {
#        puts "dbg: Reading cache"
        set Msgs [Cache:Read [RF $MH(Path)/$Fol/.xmhcache]]
        write_seq $Fol allmsgs
    }  

    if [winfo exists $w] {
        set curYview [lindex [$w.s get] 2]
        $w.t delete 0 end
        foreach Msg $Msgs {$w.t insert end $Msg}
        $w.t yview $curYview
    }
    return [llength Msgs]
}

proc RemoveFol Fol {
    global MH cmd FolList FolTree

    if {$Fol == $MH(Inbox) || $Fol == $MH(Draft-Folder)} {
        MsgWindow "You can't delete your drafts or inbox folders!\n"
        return 0
    }
    if [Confirm "Remove $Fol?"] {
        eval exec $cmd(rmdir) [RF $MH(Path)/$Fol]
        set i [lsearch -exact $FolList "$Fol"]
        set FolList [lreplace $FolList $i $i]
        while {[set i [lsearch -regexp $FolList "^$Fol/"]] != -1} {
          set FolList [lreplace $FolList $i $i]
        }
        Cache:Write [RF $MH(Path)/.folders] $FolList
        DisplayFols $FolList
        set FolTree [BuildFolTree $FolList]
        AdaptFolMenus
        if [winfo exists [FW $Fol]] {destroy [FW $Fol]}
        return 1       
    }
    return 0
}

proc RenameFol Fol {
    global MH cmd FolList FolTree
    
    set NewName [Change Folder $Fol "Rename to" {} 25]
    if {$NewName == {}} {return 0}
    if {[lsearch -exact $FolList "$NewName"] != -1} {
        if {![Confirm "$NewName already exists!
            \n\nMove this folder to $NewName?\n"]} {
           return 0
        }
    }
    set Tree [string range $NewName 0 [expr [string last / $NewName] - 1]]
    CreateFol $Tree
    eval exec $cmd(mv) [RF $MH(Path)/$Fol] [RF $MH(Path)/$NewName]
    set i [lsearch -exact $FolList "$Fol"]
    set FolList [lreplace $FolList $i $i]
    while {[set i [lsearch -regexp $FolList "^$Fol/"]] != -1} {
      regsub "^$Fol\(.*\)$" [lindex $FolList $i] "$NewName\\1" REGSUB
      set FolList [lreplace $FolList $i $i $REGSUB]
    }
    lappend FolList $NewName
    set FolList [lsort $FolList]
    Cache:Write [RF $MH(Path)/.folders] $FolList
    DisplayFols $FolList
    set FolTree [BuildFolTree $FolList]
    AdaptFolMenus
    if [winfo exists [FW $Fol]] {destroy [FW $Fol]; OpenFol $NewName}
    return 1
}

proc SortFol {Fol} {
    global MH

    catch {exec $MH(bindir)/sortm +$Fol -noverbose}
}
