/* 
 * Copyright (c) 1994 Open Software Foundation, Inc.
 * 
 * Permission is hereby granted to use, copy, modify and freely distribute
 * the software in this file and its documentation for any purpose without
 * fee, provided that the above copyright notice appears in all copies, and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.  Further, provided that the name of Open
 * Software Foundation, Inc. ("OSF") not be used in advertising or
 * publicity pertaining to distribution of the software without prior
 * written permission from OSF.  OSF makes no representations about the
 * suitability of this software for any purpose.  It is provided "AS IS"
 * without express or implied warranty.
 */ 

#define MAINPROGRAM

static char * rcsid =
 "$RCSfile: otqm.c,v $";


#include <string.h>
#include <stdio.h>
#include <limits.h>
#include <sys/types.h>
#include <dirent.h>
/* #include <sys/dir.h> */
#include <sys/stat.h>
#include <tcl.h>
#include <tclInt.h>
#include "ot.h"
#include "otInt.h"
#include <errno.h>

#include <sys/socket.h>
#include <sys/un.h>


/* command line options for getopt() */
#define optString "p:"     /* for the "otqm" command */


static int compar();	  /* Compare modification times */
OTErr	qmParseArgs();	  /* Parse command line arguments */
OTErr	qmChkArgs();	  /* Check arguments */
OTErr	qmLockQueue();	  /* Create the lock file */
OTErr	qmProcessQueue(); /* Process the queue */
OTErr	qmOneItem();	  /* Process one item (file) from the list */
OTErr	qmChkNumber();	  /* Check the CR number */
OTErr	qmUpdateHdr();	  /* Update headdb file */
OTErr	qmUpdateHist();	  /* Update histdb file */
OTErr	qmUnlockQueue();  /* Remove the lock file */
OTErr	qmChkDBfile();	  /* Create dbfile if doesn't exist */
OTErr	qmUpdateDBfile(); /* Update headdb/histdb file */
OTErr   qmRefreshKwikPix();

int	qmGetList();	  /* Get a sorted list of queue items */
bool	qmFileExist();	  /* Check for an existence of the lock file */

void	qmSaveItem();
void	qmMail();	  /* Mail to "ot" fatal error message */

struct queue		
{             /* structure for building a sorted list of files */
    time_t  q_time;		    /* modification time */
    char    q_name[_POSIX_NAME_MAX+1];     /* control file name */
};



/* 
 * otqm is the otd queue manager. 
 * otd invokes otqm after successful COMMIT operation.
 *
 * 	otqm  -p<OTproj>
 *
 *    where:
 *	    -p    specifies the OT project name
 *
 * otqm does the following:
 *  - checks that there is no other otqm is running for the project
 *  - creates the otqm.lok file
 *  - checks the queue 
 *  - sorts queue items by modification time
 *  - process each file from the sorted queue 
 *  - removes the otqm.lok file
 *  
 * 
 * The input:
 *
 *    Queue items (files) have system generated names started with the "qi"
 *
 *    Each "qi" file contains the following:
 *        crNumber
 *	  headerDBline
 *	  historyDBline
 *
 * The output:
 * 
 * Header and history information ("headdb" and "histdb" files) for the
 * project are updated with the header and history lines from the queue files.
 * Processed queue files are removed from the queue.
 *
 */

main ( argc, argv, envp )
int argc;
char ** argv;
char ** envp;
{
    register int i;
    char *cp1;
    char tmp[COMMAND];
    char logFile[NAMELEN];
    char lock[PATHLEN];
    char funcName[NAMELEN];

    OTErr errCode = OT_SUCCESS;
    OTErr tmpErr = OT_SUCCESS;

    funcName[0] = 0;

    if ( errCode = otInitialInit(&argc, argv) ) {
	cp1 = otGetPostedMessage();
	fprintf(stderr, "otInitialInit: %s\n", cp1);
	qmMail(cp1, "otInitialInit");
	exit(1);
    }
    DBUG_MIN((stderr, "otqm: running for %s project\n", otCB->cb_project));

    /* Open up the log file */

    sprintf(logFile, "%s/serverV3.log", BASE);
    sprintf(otCB->cb_pcb->pcb_logPid, "%ld", (long) getpid());

    if ((otCB->cb_pcb->pcb_efp = fopen(logFile, "a")) == NULL)    {
	sprintf(tmp, "can't append to log file (%s)", logFile);
        fprintf(stderr,"otqm: %s\n", tmp);
	qmMail(tmp, "otqm");
        exit(1);
    }
    chmod(logFile, 0666);

    /* Parse passed arguments */

    if (errCode = qmParseArgs(argc, argv)) {
	otPutPostedMessage(OT_OTQM_USAGE);
	cp1 = otGetPostedMessage();
	fprintf(stderr, "qmParseArgs: %s\n", cp1);
	qmMail(cp1, "qmParseArgs");
	logError("%s", cp1);
    }

    /* Check arguments (project have to be passed) */

    if (errCode = qmChkArgs()) {
	cp1 = otGetPostedMessage();
	fprintf(stderr, "qmChkArgs: %s\n", cp1);
	qmMail(cp1, "qmChkArgs");
	logError("%s", cp1);
    }

    if (errCode = otReadProject(otCB->cb_project)) {
	cp1 = otGetPostedMessage();
	fprintf(stderr, "otReadProject: %s\n", cp1);
	qmMail(cp1, "otReadProject");
	logError("%s", cp1);
    }

    sprintf(lock, "%s/%s/otqm.lok", 
	otCB->cb_pcb->pcb_project->proj_dir, otCB->cb_project);

    /* Return if there is a running otqm for the project */

    if (qmFileExist(lock))  {
	DBUG_MIN((stderr, "otqm: sleeping..\n"));
	sleep(10);
	if (qmFileExist(lock)) {
	    DBUG_MIN((stderr, "otqm: the queue is locked - return\n"));
	    return errCode;
	}
    }

    if (errCode = qmLockQueue(lock)) {
	cp1 = otGetPostedMessage();
	fprintf(stderr, "qmLockQueue: %s\n", cp1);
	qmMail(cp1, "qmLockQueue");
	logError("%s", cp1);
    }

    if (errCode = qmProcessQueue(funcName)) {
	cp1 = otGetPostedMessage();
	fprintf(stderr, "qmProcessQueue: %s\n", cp1);
	if (funcName[0])
	    qmMail(cp1, funcName);
	logWarn("%s", cp1);
    }

    if (tmpErr = qmUnlockQueue(lock)) {
	cp1 = otGetPostedMessage();
	fprintf(stderr, "qmUnlockQueue: %s\n", cp1);
	qmMail(cp1, "qmUnlockQueue");
	logWarn("%s", cp1);
    }

    if (errCode || tmpErr)
	exit(1);
    else
	exit(0);
}




OTErr
qmParseArgs(argc, argv)
int argc;
char * argv[];
{

    register int  opt;
    OTErr errCode = OT_SUCCESS;

    while ( (opt = getopt(argc, argv, optString)) != EOF )  {
	switch ( opt )  {

	case 'p':
	    otPutPostedMessage(OT_INTERNAL_ERROR,
			"otPreparseInput while extracting OT projectname");
	    errCode = OT_INTERNAL_ERROR;
	    break;

	case '?':
	    errCode = OT_OTQM_USAGE;
	    break;
	}
    }

    if (optind != argc)         /* there are remaining args */
	errCode = OT_OTQM_USAGE;

    return errCode;
}



OTErr
qmChkArgs()
{
    char dir[PATHLEN];
    struct stat stat_buf;
    OTErr errCode = OT_SUCCESS;

    if (!otCB->cb_project || !*otCB->cb_project) {
	otPutPostedMessage(OT_NEED_PROJECT);
	return OT_NEED_PROJECT;
    }

    return errCode;
}



bool
qmFileExist(fname)
char *fname;
{
    struct stat stat_buf;
    bool ret;

    if ( stat(fname, &stat_buf) < 0) 
	ret = FALSE;
    else 
	ret = TRUE;
    
    DBUG_MED((stderr, "qmFileExist for %s returns %d\n", fname, ret));
    return ret;
}



OTErr
qmLockQueue(lock)
char *lock;
{

    if (!makeLock(otCB->cb_pcb->pcb_uName, lock, 1)) {
	otPutPostedMessage(OT_CREATE_LOCK_FILE, "otqm queue", lock);
	return OT_CREATE_LOCK_FILE;
    }

    return OT_SUCCESS;
}



OTErr
qmProcessQueue(funcName)
char * funcName;
{
    register int nitems;
    register struct queue *q, **qp;
    struct queue **queue;
    struct stat stat_buf;
    long  topNumber;
    char *cp1;
    char qdir[PATHLEN];
    char qitem[PATHLEN];
    OTErr errCode = OT_SUCCESS;

    sprintf(qdir, "%s/%s/otdqueue", 
	otCB->cb_pcb->pcb_project->proj_dir, otCB->cb_project);

    /* Check if the otd queue exist */

    if ( stat(qdir, &stat_buf) < 0 || !S_ISDIR(stat_buf.st_mode) ) {
	otPutPostedMessage(OT_NO_DIR, qdir);
	strcpy(funcName, "qmProcessQueue");
	return OT_NO_DIR;
    }

    /* Sort the otd queue */

    if ((nitems = qmGetList(&queue, qdir)) < 0) {
	otPutPostedMessage(OT_OTQM_QLIST, qdir);
	strcpy(funcName, "qmGetList");
	return OT_OTQM_QLIST;
    }
    if (nitems == 0)                /* no work to do */
	return errCode;

again:

    /* Get project's top number */

    if (errCode = otReadTopNumber(otCB->cb_pcb->pcb_project, &topNumber)) {
	strcpy(funcName, "otReadTopNumber");
	return errCode;
    }

    DBUG_MIN((stderr, "qmProcessQueue: topNumber=%d\n", topNumber));

    /* Process each file from the queue */

    for (qp = queue; nitems--; free((char *) q)) {
	q = *qp++;
	DBUG_MIN((stderr, "... next file is %s\n", q->q_name));

	sprintf(qitem, "%s/%s", qdir, q->q_name);
	if (stat(qitem, &stat_buf) < 0)
	    continue;

	if (errCode = qmOneItem(qitem, stat_buf.st_size, topNumber)) {
	    if (errCode == OT_OTQM_FATAL_ERROR) {
		otPutPostedMessage(OT_OTQM_FATAL_ERROR);
		strcpy(funcName, "qmOneItem");
		return errCode;
	    }
	    cp1 = otGetPostedMessage();
	    fprintf(stderr, "%s\n", cp1);
	    qmMail(cp1, "qmOneItem");
	    logWarn("%s", cp1);
	}
	else	
	    (void) unlink(qitem);
    }
    free((char *) queue);

    /* Check otd queue for more work */

    if ((nitems = qmGetList(&queue, qdir)) < 0) {
	otPutPostedMessage(OT_OTQM_QLIST, qdir);
	strcpy(funcName, "qmGetList");
	return OT_OTQM_QLIST;
    }

    if (nitems == 0) {              /* no more work to do */
	/* final clean up  do we need to free any memory ??? */

	return errCode;
    }
    goto again;
}



/*
 * Scan directory and make a list of files sorted by creation time.
 * Return the number of entries and a pointer to the list.
 */

int
qmGetList(namelist, qdir)
struct queue *(*namelist[]);
char *qdir;
{
    register struct dirent *d;
    register struct queue *q, **queue;
    register int nitems;
    struct stat stat_buf;
    
    char *cp1;
    char qitem[PATHLEN];
    int arraysz;
    DIR *dirp;

    if ((dirp = opendir(qdir)) == NULL) {
	otPutPostedMessage(OT_OPENDIR_LOCATION, qdir);
	cp1 = otGetPostedMessage();
	fprintf(stderr, "qmGetList: %s\n", cp1);
	logWarn("%s", cp1);
	return(-1);
    }

    arraysz = BUFSIZ / sizeof(struct queue *);
    queue = (struct queue **)malloc(arraysz * sizeof(struct queue *));
    if (!queue)
	goto errdone;

    nitems = 0;
    while ((d = readdir(dirp)) != NULL) {
	if (d->d_name[0] != 'q' || d->d_name[1] != 'i')
	    continue;       /* not a queue item file */

        sprintf(qitem, "%s/%s", qdir, d->d_name);
	if (stat(qitem, &stat_buf) < 0)
	    continue;       /* doesn't exist */

	q = (struct queue *)malloc(sizeof(time_t)+strlen(d->d_name)+1);
	if (!q)
	    goto errdone;

	q->q_time = stat_buf.st_mtime;
	strcpy(q->q_name, d->d_name);

	/* Check to make sure the array has space left and realloc 
	   the maximum size 
	 */
	if (++nitems > arraysz) {
	    arraysz += BUFSIZ / sizeof(struct queue *);
	    queue = (struct queue **)realloc((char *)queue, 
					arraysz * sizeof(struct queue *));
	    if (!queue)
		goto errdone;
	}
	queue[nitems-1] = q;
    }
    closedir(dirp);

    if (nitems)
	qsort(queue, nitems, sizeof(struct queue *), compar);

    *namelist = queue;
    return nitems;


errdone:

    otPutPostedMessage(OT_MALLOC_LOCATION, "qmProcessQueue");
    closedir(dirp);
    return (-1);

}



/*
 * Compare modification times.
 */
static int
compar(p1, p2)
	register struct queue **p1, **p2;
{
#define    NICELET    '0'

	int v1,v2;

	/* compar priority letter in queue name */
	v1 = (*p1)->q_name[3];
	if (!islower(v1)) v1 = NICELET;
	v2 = (*p2)->q_name[3];
	if (!islower(v2)) v2 = NICELET;
	if (v1 < v2)
		return(-1);
	if (v1 > v2)
		return(1);
	if ((*p1)->q_time < (*p2)->q_time)
		return(-1);
	if ((*p1)->q_time > (*p2)->q_time)
		return(1);
	return(0);
}



OTErr
qmOneItem(item, len, topNumber)
char *item;
int   len;
long topNumber;
{
    register int ch, i;
    FILE *fp;		/* file pointer */
    char *hdrline, *histline;
    int  crnum;
    char cr[SHORTSTR];
    OTErr ignore;

    OTErr errCode = OT_SUCCESS;
    OTErr tmpErr = OT_SUCCESS;

    hdrline = histline = 0;

    DBUG_MIN((stderr, "qmOneItem: queue item %s len=%d\n", item, len));

    if ( !(fp = fopen(item, "r")) ) {
	otPutPostedMessage(OT_FILE_OPEN, item, "qmOneItem");
	goto skip;
    }

    if ( (hdrline = (char *)malloc(len)) == 0) {
	otPutPostedMessage(OT_MALLOC_LOCATION, "qmOneItem");
	goto skip;
    }

    if ( (histline = (char *)malloc(len)) == 0) {
        otPutPostedMessage(OT_MALLOC_LOCATION, "qmOneItem");
	goto skip;
    }

    /* Get CR number, header and history lines from the qitem file */

    cr[0] = hdrline[0] = histline[0] = 0;

    if ( fgets(cr, SHORTSTR - 1, fp) == NULL) {
	otPutPostedMessage(OT_OTQM_CORRUPT_QI, item, "CR line");
	goto skip;
    }
    DBUG_MIN((stderr, "qmOneItem: CR = %s\n", cr));

    if ( fgets(hdrline, len - 1, fp) == NULL) {
	otPutPostedMessage(OT_OTQM_CORRUPT_QI, item, "hdrline");
	goto skip;
    }

    if ( fgets(histline, len - 1, fp) == NULL) {
	otPutPostedMessage(OT_OTQM_CORRUPT_QI, item, "histline");
	goto skip;
    }

    /* Check CR number */

    if (errCode = qmChkNumber(cr, topNumber)) {
	goto skip;
    }

    fclose(fp);
    crnum = atoi(cr);

    errCode = qmUpdateHdr(crnum, hdrline, topNumber);
    tmpErr = qmUpdateHist(crnum, histline, topNumber);

    /* ignore qmRefreshKwikPix errors */

    if (!errCode && !tmpErr)
	ignore = qmRefreshKwikPix(crnum, hdrline, histline);

    free(hdrline);
    free(histline);

    /* terminate otqm upon OT_OTQM_FATAL_ERROR error */

    if ((errCode == OT_OTQM_FATAL_ERROR) || (tmpErr == OT_OTQM_FATAL_ERROR) )
	return OT_OTQM_FATAL_ERROR;

    if (errCode || tmpErr)
	return OT_GENERAL_ERROR;

    return errCode;

skip:

    if (hdrline)
	free(hdrline);
    if (histline)
	free(histline);
    if (fp)
	fclose(fp);

    otPutPostedMessage(OT_OTQM_QI_ERROR, item);
    qmSaveItem(item);
    return OT_OTQM_QI_ERROR;
}

void
qmSaveItem(item)
char  *item;
{
    char *cp1;
    char fname[NAMELEN];
    char errname[NAMELEN];
    char tmp[COMMAND];

    strcpy(tmp, item);
    if (cp1 = strrchr(tmp, '/')) 
        strcpy(fname, ++cp1);
    else
	strcpy(fname, item);

    sprintf(errname, "%s/%s/otdqueue/err_%s", 
	otCB->cb_pcb->pcb_project->proj_dir, otCB->cb_project, fname);

    if (rename(item, errname)) {
	otPutPostedMessage(OT_RENAME_FILE, item, errname, "qmSaveItem");
	cp1 = otGetPostedMessage();
	fprintf(stderr, "%s\n", cp1);
	qmMail(cp1, "qmSaveItem");
	logWarn("%s", cp1);
    }
    return;
}



OTErr
qmUpdateHdr(crnum, line, topNumber)
int   crnum;
char *line;
long  topNumber;
{
    char dbfile[PATHLEN];

    OTErr errCode = OT_SUCCESS;

    DBUG_MED((stderr, "qmUpdateHdr crnum=%d line='%s'\n", crnum, line));

    sprintf(dbfile, "%s/%s/%s", 
	otCB->cb_pcb->pcb_project->proj_dir, otCB->cb_project, HEADERDB);

    if (errCode = qmChkDBfile(dbfile, crnum))
	return errCode;

    errCode = qmUpdateDBfile(dbfile, crnum, line, topNumber);

    return errCode;
}



OTErr
qmUpdateHist(crnum, line, topNumber)
int   crnum;
char *line;
long  topNumber;
{
    char dbfile[PATHLEN];

    OTErr errCode = OT_SUCCESS;

    DBUG_MED((stderr, "qmUpdateHist crnum=%d line='%s'\n", crnum, line));

    sprintf(dbfile, "%s/%s/%s", 
	otCB->cb_pcb->pcb_project->proj_dir, otCB->cb_project, HISTORYDB);

    if (errCode = qmChkDBfile(dbfile, crnum))
	return errCode;

    errCode = qmUpdateDBfile(dbfile, crnum, line, topNumber);

    return errCode;
}



OTErr
qmChkDBfile(dbfile, crnum)
char *dbfile;
int  crnum;
{
    register int i;
    FILE  *fp;

    if (! qmFileExist(dbfile)) { 	/* create it */
	umask(0);
	if ((fp = fopen(dbfile, "w")) == NULL) {
	    otPutPostedMessage(OT_CREATE_FILE, "dbfile", dbfile);
	    return OT_OTQM_FATAL_ERROR;
	}

	for (i=crnum; i>0; i--)
	    fprintf(fp, "%d \n", i);

	fclose(fp);
    }

    return OT_SUCCESS;
}



OTErr
qmUpdateDBfile(dbfile, crnum, line, topNumber)
char *dbfile;
int   crnum;
char *line;
long  topNumber;
{
    register int i, j, ch;
    int  dbCRnum, curr_num;
    char tmpname[PATHLEN];
    char *tmp_dbfile;
    char dbfile_cr[SHORTSTR];
    char path [PATHLEN];
    FILE  *fp;
    FILE  *tfp;

    tfp = 0;
    tmp_dbfile = tmpname;

    /* Open dbfile (headdb/histdb) */

    if ((fp = fopen(dbfile, "r")) == NULL) {
	otPutPostedMessage(OT_FILE_OPEN, dbfile, "qmUpdateDBfile");
	goto errdone;
    }

    /* Get CR number from the top line of the dbfile */

    for (i=0, ch=getc(fp); (ch != ' ') && (ch != '\n') && (ch != EOF) && (i<SHORTSTR-2); i++)  {
	dbfile_cr[i] = ch;
	ch = getc(fp);
    }
    dbfile_cr[i] = 0;

    if ( (i > MAXNUMCRDIGITS) || ((dbCRnum = atoi(dbfile_cr)) <= 0) ||
	 (dbCRnum > topNumber) )  {
	otPutPostedMessage(OT_DBFILE_TOP_NUM,
		otCB->cb_pcb->pcb_project->object_name, dbfile_cr, dbfile);
	goto errdone;
    }

    if (i)  { 		/* restore the top line of the dbfile */
        ungetc(ch, fp);
        for (i--; i>=0;  i--)	
	    ungetc(dbfile_cr[i], fp);
    }

    /* Make unique filename in the right place and open temp dbfile */

    sprintf(path, "%s/%s", 
	otCB->cb_pcb->pcb_project->proj_dir, otCB->cb_project);
    tmp_dbfile = tempnam(path, "dbf.");   

    if ((tfp = fopen(tmp_dbfile, "w")) == NULL) {
	otPutPostedMessage(OT_CREATE_FILE, "temporary db", tmp_dbfile);
	goto errdone;
    }

    if (crnum > dbCRnum) { 	
	fputs(line, tfp);

	/* put stabs for non-existent dbfile lines */

	for (i = crnum-dbCRnum-1, j=1; i>0; i--,j++)
	    fprintf(tfp, "%d \n", crnum-j);
    }

    /* copy dbfile lines up to the CR number */

    for ( curr_num = dbCRnum; crnum < curr_num; curr_num--) {
	while ( ((ch=fgetc(fp)) != EOF) && (ch != '\n') ) 
	    fputc(ch, tfp);
	if (ch == '\n')
	    fputc('\n', tfp);
	else { 	
	    otPutPostedMessage(OT_OTQM_DB_EOF, dbfile);
	    goto errdone;
        }
    }

    /* put the new dbfile line instead of old one */

    if (crnum == curr_num) {
	fputs(line, tfp);
	while ( ((ch=fgetc(fp)) != EOF) && (ch != '\n') ) ;
    }

    if ( ch != EOF) {
        while ((ch = fgetc(fp)) != EOF) /* copy the rest of the dbfile */
	    fputc(ch, tfp);
    }

    fclose(fp);
    fclose(tfp);

    /* Move temp dbfile that we just created into the real thing */

    if (rename(tmp_dbfile, dbfile)) {
	otPutPostedMessage(OT_RENAME_FILE, tmp_dbfile, dbfile, 
		"qmUpdateDBfile");
	goto errdone;
    }

    return OT_SUCCESS;

errdone:

    if (fp)
	fclose(fp);
    if (tfp) {
	fclose(tfp);
	unlink(tmp_dbfile);
    }
    return OT_OTQM_FATAL_ERROR;
}



OTErr
qmChkNumber(cr, topNumber)
char *cr;
long topNumber;
{
    long  crnum;

    OTProject * proj = otCB->cb_pcb->pcb_project;
    OTErr errCode = OT_SUCCESS;

    if ( (crnum = atol(cr)) <= 0) {
	otPutPostedMessage(OT_ILLEGAL_ID_NUMBER, proj->object_name, cr);
	return OT_ILLEGAL_ID_NUMBER;
    }

    if (crnum >= topNumber) {
	otPutPostedMessage(OT_NUMBER_HIGH, proj->object_name, crnum, 
				otCB->cb_project);
	return OT_NUMBER_HIGH;
    }
    return errCode;
}



OTErr
qmUnlockQueue(lock)
char *lock;
{

    if (!removeLock(lock)) {
	otPutPostedMessage(OT_REMOVE_LOCK_FILE, lock, "qmUnlockQueue");
	return OT_REMOVE_LOCK_FILE;
    }
    return OT_SUCCESS;
}


void
qmMail(msg, place)
char * msg;
char * place;
{
    char mailcmd[NAMELEN];
    char command[COMMAND];
    struct stat stat_buf;
    FILE *mail_fp;


    if (!stat("/usr/bin/mailx", &stat_buf))
	strcpy(mailcmd, "mailx");
    else
	strcpy(mailcmd, "Mail");

    sprintf(command, "%s -s 'otqm: FATAL error' ot", mailcmd);

    if ( (mail_fp = popen(command, "w")) != NULL ) {
	fprintf(mail_fp, "%s:\n", place);
        fprintf(mail_fp, "%s", msg);
        fputc(EOF, mail_fp);
        pclose(mail_fp);
    }
    return;
}



OTErr
qmRefreshKwikPix(crnum, hdrline, histline)
int crnum;		/* should be long? */
char *hdrline;
char *histline;
{
    OTErr errCode = OT_SUCCESS;
    OTProject *proj = otCB->cb_pcb->pcb_project;
    int s, n, wr;
    struct sockaddr_un sunSock;
    char *updateStr;
    char buf[SHORTSTR];

    /*
     * On updates, otqm connects via a UNIX socket to the otKwikPix server
     * and sends a single command with the new header and history information.
     * It waits for a single acknowledgement character.
     */
    if ( (s = socket(AF_UNIX, SOCK_STREAM, 0)) < 0 )
	errCode = OT_SOCKET;
    else {

	sprintf(sunSock.sun_path, "%s/%s/socket", proj->proj_dir, proj->name);
	sunSock.sun_family = AF_UNIX;
	if (connect(s, (struct sockaddr *)&sunSock, strlen(sunSock.sun_path) + 2)
	    < 0) {
	    otPutPostedMessage(OT_CONNECT, errno);
	    errCode = OT_CONNECT;
	} else {

	    if ( !(updateStr = malloc( strlen(hdrline) + strlen(histline) + 
		SHORTSTR )) ) {
		otPutPostedMessage(OT_MALLOC_LOCATION, "qmRefreshKwikPix()");
		errCode = OT_MALLOC_LOCATION;
	    } else {
	        sprintf(updateStr, "delta %d\n%s%s", crnum, hdrline,
		    histline);
		n = strlen(updateStr);
		wr = write(s, updateStr, n);
		n = read(s, buf, 1);
		free(updateStr);
	    }
	}

    }
    (void)close(s);

    return errCode;

}
