/*
 *=============================================================================
 *                                  tSippPBM.c
 *-----------------------------------------------------------------------------
 * Tcl commands to manipulate and render to PBMPlus PPM and PBM files.
 *-----------------------------------------------------------------------------
 * Copyright 1992-1993 Mark Diekhans
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies.  Mark Diekhans makes
 * no representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 *-----------------------------------------------------------------------------
 * $Id: tSippPBM.c,v 5.0 1994/09/05 01:23:01 markd Rel $
 *=============================================================================
 */

#include "tSippInt.h"
#include "sipp_pixmap.h"
#include "sipp_bitmap.h"

/*
 * Type used to hold a file handle "fileNNN".
 */
typedef char fileHandle_t [10];


/*
 * PBM file table entry.
 */
typedef struct {
    fileHandle_t  handle;
    FILE         *filePtr;
    int           perms;
    bool          raw;
} pbmFile_t, *pbmFile_pt;


/*
 * Client-data for pixel setting call back used to hold a single row of data.
 * The data will be outputed when the last pixel is written.
 */
typedef struct {
    FILE *filePtr;
    int   xSize;
    bool  raw;
} renderData_t, *renderData_pt;

/*
 * Internal prototypes.
 */
static pbmFile_pt
PBMHandleToPtr _ANSI_ARGS_((tSippGlob_pt    tSippGlobPtr,
                            char           *handle));

static bool
ClosePBMFile _ANSI_ARGS_((tSippGlob_pt    tSippGlobPtr,
                          char           *command,
                          pbmFile_pt      pbmFilePtr));

static void
OutputComments _ANSI_ARGS_((pbmFile_pt   pbmFilePtr,
                            char       **comments));

static void *
PBMOutputStart _ANSI_ARGS_((tSippGlob_pt          tSippGlobPtr,
                            tSippOutputParms_pt   outputParmsPtr,
                            char                 *handle,
                            char                **comments));

static void
PBMOutputLine _ANSI_ARGS_((renderData_pt   renderDataPtr,
                           int             y,
                           u_char         *rowPtr));

static void
PBMOutputBitMap _ANSI_ARGS_((renderData_pt   renderDataPtr,
                             Sipp_bitmap    *bitMapPtr));

static bool
PBMOutputEnd _ANSI_ARGS_((tSippGlob_pt          tSippGlobPtr,
                          tSippOutputParms_pt   outputParmsPtr,
                          renderData_pt         renderDataPtr));

/*=============================================================================
 * PBMHandleToPtr --
 *   Utility procedure to convert a PBM file handle to a Tcl OpenFile
 * structure.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o handle (I) - A PBM handle.
 * Returns:
 *   A pointer to the PBM file entry, or NULL if an error occured.
 *-----------------------------------------------------------------------------
 */
static pbmFile_pt
PBMHandleToPtr (tSippGlobPtr, handle)
    tSippGlob_pt    tSippGlobPtr;
    char           *handle;
{
    pbmFile_pt  pbmFilePtr;

    pbmFilePtr = (pbmFile_pt)
        Tcl_HandleXlate (tSippGlobPtr->interp,
                         tSippGlobPtr->pbmTblPtr, handle);
    if (pbmFilePtr == NULL)
        return NULL;
    return pbmFilePtr;

}

/*=============================================================================
 * ClosePBMFile --
 *   Close an PBMPlus file, deleting the handle entry.
 * 
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o command (I) - The command that called this function (argv [0]).
 *   o PBMFilePtr (I) - Pointer to the file entry to close.
 * Returns:
 *   TRUE if all is ok, FALSE if an error occured.
 *-----------------------------------------------------------------------------
 */
static bool
ClosePBMFile (tSippGlobPtr, command, pbmFilePtr)
    tSippGlob_pt    tSippGlobPtr;
    char           *command;
    pbmFile_pt      pbmFilePtr;
{
    int             result;
    char           *closeArgv [2];

    closeArgv [0] = command;
    closeArgv [1] = pbmFilePtr->handle;

    result = Tcl_CloseCmd (NULL, tSippGlobPtr->interp, 2, closeArgv);

    Tcl_HandleFree (tSippGlobPtr->pbmTblPtr, pbmFilePtr);
    return (result == TCL_OK);

}

/*=============================================================================
 * SippPBMOpen --
 *   Implements the command:
 *     SippPBMOpen [-plain|-raw] filename access
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.  We cheat by calling
 * the Tcl "open" command executor to actually do the open, that way we get the
 * full semantics of the open command.
 *-----------------------------------------------------------------------------
 */
static int
SippPBMOpen (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt    tSippGlobPtr = (tSippGlob_pt) clientData;
    fileHandle_t    fileHandle;
    pbmFile_pt      pbmFilePtr;
    int             argIdx, openArgc;
    char           *openArgv [4];
    bool            raw = TRUE;
    bool            gotModeFlag = FALSE;
    FILE           *filePtr;
    int             perms;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    /*
     * Parse off argument flags (only one allowed right now).
     */
    for (argIdx = 1; (argIdx < argc) && (argv [argIdx][0] == '-'); argIdx++) {
        if (STREQU (argv [argIdx], "-plain")) {
            if (gotModeFlag)
                goto wrongArgs;
            raw = FALSE;
            gotModeFlag = TRUE;
            continue;
        }
        if (STREQU (argv [argIdx], "-raw")) {
            if (gotModeFlag)
                goto wrongArgs;
            raw = TRUE;
            gotModeFlag = TRUE;
            continue;
        }
        Tcl_AppendResult (interp, "invalid flag \"", argv [argIdx],
                          "\", expected one of \"-plain\" or \"-raw\"",
                          (char *) NULL);
        return TCL_ERROR;
    }

    /*
     * Validate the number of remaining arguments and set up a vector to pass
     * to the open command.
     */
    openArgc = argc - argIdx + 1;
    if (openArgc != 3)
         goto wrongArgs;

    openArgv [0] = argv [0];
    openArgv [1] = argv [argIdx];
    openArgv [2] = argv [argIdx + 1];
    openArgv [3] = NULL;
    
    /*
     * Use the open command to actually open the file.
     */

    if (Tcl_OpenCmd (NULL, interp, openArgc, openArgv) != TCL_OK)
        return TCL_ERROR;
    strcpy (fileHandle, interp->result);
    Tcl_ResetResult (interp);

    if (Tcl_GetOpenFile (interp, fileHandle, FALSE, FALSE, &filePtr) != TCL_OK)
        return TCL_ERROR;  /* Should never happen */

    perms = Tcl_FilePermissions (filePtr);

    /*
     * We don't currently support reading of PBMPlus files because the
     * PBMPlus libraries exit on error rather than returning status!!!
     */
    if (perms & TCL_FILE_READABLE) {
        ClosePBMFile (tSippGlobPtr, argv [0], pbmFilePtr);
        Tcl_ResetResult (interp);
        Tcl_AppendResult (interp, "read access to PBMPlus files is not ",
                          "supported", (char *) NULL);
        return TCL_ERROR;
    }

    /*
     * Setup the PBM file entry.
     */
    pbmFilePtr = (pbmFile_pt) Tcl_HandleAlloc (tSippGlobPtr->pbmTblPtr,
                                               interp->result);
    strcpy (pbmFilePtr->handle, fileHandle);
    pbmFilePtr->perms = perms;
    pbmFilePtr->filePtr = filePtr;
    pbmFilePtr->raw = raw;

    return TCL_OK;

  wrongArgs:
    Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                      " [-plain|-raw] filename access", (char *) NULL);
    return TCL_ERROR;

}

/*=============================================================================
 * SippPBMClose --
 *   Implements the command:
 *     SippPBMClose pbmhandle
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippPBMClose (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt    tSippGlobPtr = (tSippGlob_pt) clientData;
    pbmFile_pt      pbmFilePtr;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 2) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " pbmhandle", (char *) NULL);
        return TCL_ERROR;
    }

    pbmFilePtr = PBMHandleToPtr (tSippGlobPtr, argv [1]);
    if (pbmFilePtr == NULL)
        return TCL_ERROR;

    if (ClosePBMFile (tSippGlobPtr, argv [0], pbmFilePtr))
        return TCL_OK;
    else
        return TCL_ERROR;

}

/*=============================================================================
 * OutputComments --
 *   Write comments from a startard comment table to a PBMPlus file.  Since
 * comments are terminated by newlines in PBMPlus, these comments containing
 * newlines must be split.
 *
 * Parameters:
 *   o pbmFilePtr (I) - A pointer to the PBM file structure.
 *   o comments (I) - Comments to write.  If NULL, none are written.
 *-----------------------------------------------------------------------------
 */
static void
OutputComments (pbmFilePtr, comments)
    pbmFile_pt   pbmFilePtr;
    char       **comments;
{
    int   idx;
    char *strPtr;

    if (comments == NULL)
        return;

    fprintf (pbmFilePtr->filePtr, "#\n");

    /*
     * Output comments, being careful to split comments containing '\n'.
     */
    for (idx = 0; comments [idx] != NULL; idx++) {
        strPtr = comments [idx];
        fputs ("#@ ", pbmFilePtr->filePtr);
        while (*strPtr != '\0') {
            if (*strPtr == '\n')
                fputs ("\n#  ", pbmFilePtr->filePtr);
            else
                fputc (*strPtr, pbmFilePtr->filePtr);
            strPtr++;
        }
        fputc ('\n', pbmFilePtr->filePtr);
    }
    fprintf (pbmFilePtr->filePtr, "#\n");

}

/*=============================================================================
 * PBMOutputStart --
 *   Start output to an PPM or PBM file.  This routine is pointed to by the
 * PBM image storage class table.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o outputParmsPtr (I) - The parameters describing the image to output.
 *   o handle (I) - Pointer to the PBM handle for the file to write.
 *   o comments (I) - If the source image has comments associated with it, they
 *     are passed here.  If the destination object can store them, it possible.
 *     NULL if no comments available.
 * Returns:
 *   A pointer to be passed back into the other PBM output routines or NULL
 *   an a message in tSippGlobPtr->interp->result if an error occurs.
 *-----------------------------------------------------------------------------
 */
static void *
PBMOutputStart (tSippGlobPtr, outputParmsPtr, handle, comments)
    tSippGlob_pt          tSippGlobPtr;
    tSippOutputParms_pt   outputParmsPtr;
    char                 *handle;
    char                **comments;
{
    pbmFile_pt     pbmFilePtr;
    renderData_pt  renderDataPtr;

    pbmFilePtr = PBMHandleToPtr (tSippGlobPtr, handle);
    if (pbmFilePtr == NULL)
        return NULL;

    if ((pbmFilePtr->perms & TCL_FILE_WRITABLE) == 0) {
        TSippNotWritable (tSippGlobPtr->interp, handle);
        return NULL;
    }

    /*
     * Output the PBM or PPM header, using the plain or raw magic number.
     */
    if (outputParmsPtr->bitMapOutput) {
        fprintf (pbmFilePtr->filePtr, "%s\n",
                 pbmFilePtr->raw ? "P4" : "P1");

        fprintf (pbmFilePtr->filePtr,
                 "#Image rendered with Tcl-SIPP %s\n",  TSIPP_VERSION);
        OutputComments (pbmFilePtr, comments);

        fprintf (pbmFilePtr->filePtr, "%d\n%d\n",
                 outputParmsPtr->imgData.xSize,
                 outputParmsPtr->imgData.ySize);
    } else {
        fprintf (pbmFilePtr->filePtr, "%s\n",
                 pbmFilePtr->raw ? "P6" : "P3");
        fprintf (pbmFilePtr->filePtr, "#Image rendered with Tcl-SIPP %s\n",
                 TSIPP_VERSION);
        OutputComments (pbmFilePtr, comments);

        switch (outputParmsPtr->imgData.field) {
          case BOTH:
            fprintf (pbmFilePtr->filePtr, "%d\n%d\n255\n",
                     outputParmsPtr->imgData.xSize,
                     outputParmsPtr->imgData.ySize);
            break;

          case EVEN:
            fprintf (pbmFilePtr->filePtr,
                     "#Image field containing EVEN lines\n");
            fprintf (pbmFilePtr->filePtr, "%d\n%d\n255\n",
                     outputParmsPtr->imgData.xSize, 
                     (outputParmsPtr->imgData.ySize & 1) ?
                         (outputParmsPtr->imgData.ySize >> 1) + 1 :
                          outputParmsPtr->imgData.ySize >> 1);
            break;

          case ODD:
            fprintf (pbmFilePtr->filePtr,
                     "#Image field containing ODD lines\n");
            fprintf (pbmFilePtr->filePtr, "%d\n%d\n255\n",
                     outputParmsPtr->imgData.xSize, 
                     outputParmsPtr->imgData.ySize >> 1);
            break;
        }
    }

    /*
     * Setup the clientdata to pass back to the other rendering routines.
     */
    renderDataPtr = (renderData_pt) smalloc (sizeof (renderData_t));

    renderDataPtr->filePtr = pbmFilePtr->filePtr;
    renderDataPtr->xSize   = outputParmsPtr->imgData.xSize;
    renderDataPtr->raw     = pbmFilePtr->raw;

    return renderDataPtr;

}

/*=============================================================================
 * PBMOutputLine --
 *   Output a rendered line to an PPM file.  This routine is pointed to by the
 * PBM output target class table.
 *
 * Parameters:
 *   o renderDataPtr (I) - A pointer to the PBM rendering clientdata.
 *   o y (I) - The scan line that was just rendered.
 *   o rowPtr (I) - The pixels for the scanline that was just rendered.
 *-----------------------------------------------------------------------------
 */
static void
PBMOutputLine (renderDataPtr, y, rowPtr)
    renderData_pt  renderDataPtr;
    int            y;
    u_char        *rowPtr;
{
    int  x, outCnt;

    /*
     * Output either in binary or ASCII.  ASCII is limit to 70 characters
     * per line.  Upto 12 characters are outputed per pixel.
     */
    if (renderDataPtr->raw) {
        fwrite (rowPtr, sizeof (u_char), renderDataPtr->xSize * 3,
                renderDataPtr->filePtr);
    } else {
        outCnt = 0;
        for (x = 0; x < renderDataPtr->xSize; x++) {
            fprintf (renderDataPtr->filePtr, "%d %d %d",
                     rowPtr [TSIPP_RED], rowPtr [TSIPP_GREEN],
                     rowPtr [TSIPP_BLUE]);
            rowPtr += 3;

            outCnt++;
            if (outCnt < 5) {
                fputc (' ', renderDataPtr->filePtr);
            } else {
                fputc ('\n', renderDataPtr->filePtr);
                outCnt = 0;
            }
        }
        if (outCnt != 0)
            fputc ('\n', renderDataPtr->filePtr);
    }

}

/*=============================================================================
 * PBMOutputBitMap --
 *   Output a SIPP bit map to an PBM file.  This routine is pointed to by the
 * PBM output target class table.
 *
 * Parameters:
 *   o renderDataPtr (I) - A pointer to the PBM rendering clientdata.
 *   o bitMapPtr (I) - Pointer to the SIPP bit map structure.
 *-----------------------------------------------------------------------------
 */
static void
PBMOutputBitMap (renderDataPtr, bitMapPtr)
    renderData_pt   renderDataPtr;
    Sipp_bitmap    *bitMapPtr;
{
    int        x, y, outCnt;
    u_char    *rowPtr;
    unsigned   bit;

    rowPtr = bitMapPtr->buffer;

    /*
     * Output either in binary or ASCII.  ASCII is limit to 70 characters
     * per line. Two characters are outputed per pixel.
     */
    if (renderDataPtr->raw) {
        for (y = 0; y < bitMapPtr->height; y++) {
            fwrite (rowPtr, 1, bitMapPtr->width_bytes, renderDataPtr->filePtr);
            rowPtr += bitMapPtr->width_bytes;
        }
    } else {
        outCnt = 0;
        for (y = 0; y < bitMapPtr->height; y++) {
            for (x = 0; x < bitMapPtr->width; x++) {
                bit = (rowPtr [x >> 3] >> (7-(x & 7))) & 1;

                fputc (bit ? '1' : '0', renderDataPtr->filePtr);

                outCnt++;
                if (outCnt < 34) {
                    fputc (' ', renderDataPtr->filePtr);
                } else {
                    fputc ('\n', renderDataPtr->filePtr);
                    outCnt = 0;
                }
            }
            rowPtr += bitMapPtr->width_bytes;
        }
        if (outCnt != 0)
            fputc ('\n', renderDataPtr->filePtr);
    }

}

/*=============================================================================
 * PBMOutputEnd --
 *   Finish up output to an PBM file.  This routine is pointed to by the
 * PBM image storage class table.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o outputParmsPtr (I) - The parameters describing the image to output.
 *   o renderDataPtr (I) - A pointer to the PBM rendering clientdata.
 * Returns:
 *   Always returns TRUE, since no errors are currently returned.
 *-----------------------------------------------------------------------------
 */
static bool
PBMOutputEnd (tSippGlobPtr, outputParmsPtr, renderDataPtr)
    tSippGlob_pt          tSippGlobPtr;
    tSippOutputParms_pt   outputParmsPtr;
    renderData_pt         renderDataPtr;
{
    fflush (renderDataPtr->filePtr);

    sfree (renderDataPtr);
    return TRUE;
}

/*=============================================================================
 * TSippPBMInit --
 *   Initialized the pbm and PBM commands.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - Pointer to the top level global data structure.
 *     (currently unused).
 *-----------------------------------------------------------------------------
 */
void
TSippPBMInit (tSippGlobPtr)
    tSippGlob_pt    tSippGlobPtr;
{
    static tSippTclCmdTbl_t cmdTable [] = {
        "SippPBMOpen",   (Tcl_CmdProc *) SippPBMOpen,
        "SippPBMClose",  (Tcl_CmdProc *) SippPBMClose,
        {NULL,           NULL}
    };

    static tSippStorageClass_t storageClass = {
        "pbmfile",                  /* handlePrefix  */
        7,                          /* prefixSize    */
        TSIPP_SCAN_TOP_DOWN,        /* scanDirection */
        TRUE,                       /* bitMapOptimal */
        PBMOutputStart,
        PBMOutputLine,
        PBMOutputBitMap,
        PBMOutputEnd,
        NULL
    };

    tSippGlobPtr->pbmTblPtr =
        Tcl_HandleTblInit ("pbmfile", sizeof (pbmFile_t), 4);

    TSippInitCmds (tSippGlobPtr, cmdTable);
    TSippAddStorageClass (tSippGlobPtr, &storageClass);

}

/*=============================================================================
 * TSippPBMCleanUp --
 *   Close all PBMPlus files and release all associated resources.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
void
TSippPBMCleanUp (tSippGlobPtr)
    tSippGlob_pt  tSippGlobPtr;
{
    int         walkKey = -1;
    pbmFile_pt  pbmFilePtr;

    while (TRUE) {
        pbmFilePtr = Tcl_HandleWalk (tSippGlobPtr->pbmTblPtr, &walkKey);
        if (pbmFilePtr == NULL)
            break;
        if (!ClosePBMFile (tSippGlobPtr, "TSippPBMCleanUp", pbmFilePtr))
            Tcl_ResetResult (tSippGlobPtr->interp);
    }
    Tcl_HandleTblRelease (tSippGlobPtr->pbmTblPtr);
    tSippGlobPtr->pbmTblPtr = NULL;

}

