/*		Structured stream to Rich hypertext converter
**		============================================
**
**	This generates of a hypertext object.  It converts from the
**	structured stream interface fro HTMl events into the style-
**	oriented iunterface of the HText.h interface.  This module is
**	only used in clients and shouldnot be linked into servers.
**
**	Override this module if making a new GUI browser.
**
*/
#include "HTMLDTD.h"
#include "HTML.h"
#include <ctype.h>
#include <stdio.h>

#include "HTAtom.h"
#include "HTextDef.h"

#include "HTAlert.h"
#include "HTMLGen.h"
#include "HTParse.h"

#define "TkWWWCmds.h"

/*		HTML Object
**		-----------
*/

struct _HTStructured {
    CONST HTStructuredClass * 	isa;
    HTParentAnchor * 		node_anchor;
    HText * 			text;

    HTStream*			target;			/* Output stream */
    HTStreamClass		targetClass;		/* Output routines */

    HTChunk 			title;		/* Grow by 128 */
    
    char *			comment_start;	/* for literate programming */
    char *			comment_end;

    BOOL                        in_title;
    BOOL                        in_comment;
    BOOL			in_word;  /* Have just had a non-white char */
    BOOL                        in_verbatim;
};

struct _HTStream {
    CONST HTStreamClass *	isa;
    /* .... */
};


/* 	Entity values -- for ISO Latin 1 local representation
**
**	This MUST match exactly the table referred to in the DTD!
*/
static char * ISO_Latin1[] = {
  	"\306",	/* capital AE diphthong (ligature) */ 
  	"\301",	/* capital A, acute accent */ 
  	"\302",	/* capital A, circumflex accent */ 
  	"\300",	/* capital A, grave accent */ 
  	"\305",	/* capital A, ring */ 
  	"\303",	/* capital A, tilde */ 
  	"\304",	/* capital A, dieresis or umlaut mark */ 
  	"\307",	/* capital C, cedilla */ 
  	"\320",	/* capital Eth, Icelandic */ 
  	"\311",	/* capital E, acute accent */ 
  	"\312",	/* capital E, circumflex accent */ 
  	"\310",	/* capital E, grave accent */ 
  	"\313",	/* capital E, dieresis or umlaut mark */ 
  	"\315",	/* capital I, acute accent */ 
  	"\316",	/* capital I, circumflex accent */ 
  	"\314",	/* capital I, grave accent */ 
  	"\317",	/* capital I, dieresis or umlaut mark */ 
  	"\321",	/* capital N, tilde */ 
  	"\323",	/* capital O, acute accent */ 
  	"\324",	/* capital O, circumflex accent */ 
  	"\322",	/* capital O, grave accent */ 
  	"\330",	/* capital O, slash */ 
  	"\325",	/* capital O, tilde */ 
  	"\326",	/* capital O, dieresis or umlaut mark */ 
  	"\336",	/* capital THORN, Icelandic */ 
  	"\332",	/* capital U, acute accent */ 
  	"\333",	/* capital U, circumflex accent */ 
  	"\331",	/* capital U, grave accent */ 
  	"\334",	/* capital U, dieresis or umlaut mark */ 
  	"\335",	/* capital Y, acute accent */ 
  	"\341",	/* small a, acute accent */ 
  	"\342",	/* small a, circumflex accent */ 
  	"\346",	/* small ae diphthong (ligature) */ 
  	"\340",	/* small a, grave accent */ 
  	"\046",	/* ampersand */ 
  	"\345",	/* small a, ring */ 
  	"\343",	/* small a, tilde */ 
  	"\344",	/* small a, dieresis or umlaut mark */ 
  	"\347",	/* small c, cedilla */ 
  	"\351",	/* small e, acute accent */ 
  	"\352",	/* small e, circumflex accent */ 
  	"\350",	/* small e, grave accent */ 
  	"\360",	/* small eth, Icelandic */ 
  	"\353",	/* small e, dieresis or umlaut mark */ 
  	"\076",	/* greater than */ 
  	"\355",	/* small i, acute accent */ 
  	"\356",	/* small i, circumflex accent */ 
  	"\354",	/* small i, grave accent */ 
  	"\357",	/* small i, dieresis or umlaut mark */ 
  	"\074",	/* less than */ 
  	"\361",	/* small n, tilde */ 
  	"\363",	/* small o, acute accent */ 
  	"\364",	/* small o, circumflex accent */ 
  	"\362",	/* small o, grave accent */ 
  	"\370",	/* small o, slash */ 
  	"\365",	/* small o, tilde */ 
  	"\366",	/* small o, dieresis or umlaut mark */ 
  	"\337",	/* small sharp s, German (sz ligature) */ 
  	"\376",	/* small thorn, Icelandic */ 
  	"\372",	/* small u, acute accent */ 
  	"\373",	/* small u, circumflex accent */ 
  	"\371",	/* small u, grave accent */ 
  	"\374",	/* small u, dieresis or umlaut mark */ 
  	"\375",	/* small y, acute accent */ 
  	"\377",	/* small y, dieresis or umlaut mark */ 
};


/* 	Entity values -- for NeXT local representation
**
**	This MUST match exactly the table referred to in the DTD!
**
*/
static char * NeXTCharacters[] = {
  	"\341",	/* capital AE diphthong (ligature) 	*/ 
  	"\202",	/* capital A, acute accent		*/ 
  	"\203",	/* capital A, circumflex accent 	*/ 
  	"\201",	/* capital A, grave accent 		*/ 
  	"\206",	/* capital A, ring 			*/ 
  	"\204",	/* capital A, tilde 			*/ 
  	"\205",	/* capital A, dieresis or umlaut mark	*/ 
  	"\207",	/* capital C, cedilla 			*/ 
  	"\220",	/* capital Eth, Icelandic 		*/ 
  	"\211",	/* capital E, acute accent 				*/ 
  	"\212",	/* capital E, circumflex accent 			*/ 
  	"\210",	/* capital E, grave accent 				*/ 
  	"\213",	/* capital E, dieresis or umlaut mark 			*/ 
  	"\215",	/* capital I, acute accent 				*/ 
  	"\216",	/* capital I, circumflex accent 	these are	*/ 
  	"\214",	/* capital I, grave accent		ISO -100 hex	*/ 
  	"\217",	/* capital I, dieresis or umlaut mark			*/ 
  	"\221",	/* capital N, tilde 					*/ 
  	"\223",	/* capital O, acute accent 				*/ 
  	"\224",	/* capital O, circumflex accent 			*/ 
  	"\222",	/* capital O, grave accent 				*/ 
  	"\351",	/* capital O, slash 		'cept this */ 
  	"\225",	/* capital O, tilde 					*/ 
  	"\226",	/* capital O, dieresis or umlaut mark			*/ 
  	"\234",	/* capital THORN, Icelandic */ 
  	"\230",	/* capital U, acute accent */ 
  	"\231",	/* capital U, circumflex accent */ 
  	"\227",	/* capital U, grave accent */ 
  	"\232",	/* capital U, dieresis or umlaut mark */ 
  	"\233",	/* capital Y, acute accent */ 
  	"\326",	/* small a, acute accent */ 
  	"\327",	/* small a, circumflex accent */ 
  	"\361",	/* small ae diphthong (ligature) */ 
  	"\325",	/* small a, grave accent */ 
  	"\046",	/* ampersand */ 
  	"\332",	/* small a, ring */ 
  	"\330",	/* small a, tilde */ 
  	"\331",	/* small a, dieresis or umlaut mark */ 
  	"\333",	/* small c, cedilla */ 
  	"\335",	/* small e, acute accent */ 
  	"\336",	/* small e, circumflex accent */ 
  	"\334",	/* small e, grave accent */ 
  	"\346",	/* small eth, Icelandic 	*/ 
  	"\337",	/* small e, dieresis or umlaut mark */ 
  	"\076",	/* greater than */ 
  	"\342",	/* small i, acute accent */ 
  	"\344",	/* small i, circumflex accent */ 
  	"\340",	/* small i, grave accent */ 
  	"\345",	/* small i, dieresis or umlaut mark */ 
  	"\074",	/* less than */ 
  	"\347",	/* small n, tilde */ 
  	"\355",	/* small o, acute accent */ 
  	"\356",	/* small o, circumflex accent */ 
  	"\354",	/* small o, grave accent */ 
  	"\371",	/* small o, slash */ 
  	"\357",	/* small o, tilde */ 
  	"\360",	/* small o, dieresis or umlaut mark */ 
  	"\373",	/* small sharp s, German (sz ligature) */ 
  	"\374",	/* small thorn, Icelandic */ 
  	"\363",	/* small u, acute accent */ 
  	"\364",	/* small u, circumflex accent */ 
  	"\362",	/* small u, grave accent */ 
  	"\366",	/* small u, dieresis or umlaut mark */ 
  	"\367",	/* small y, acute accent */ 
  	"\375",	/* small y, dieresis or umlaut mark */ 
};

/* 	Entity values -- for IBM/PC Code Page 850 (International)
**
**	This MUST match exactly the table referred to in the DTD!
**
*/
/* @@@@@@@@@@@@@@@@@ TBD */



/*		Set character set
**		----------------
*/

PRIVATE char** p_entity_values = ISO_Latin1;	/* Pointer to translation */

PUBLIC void HTMLUseCharacterSet ARGS1(HTMLCharacterSet, i)
{
    p_entity_values = (i == HTML_NEXT_CHARS) ? NeXTCharacters
    					     : ISO_Latin1;
}

/*_________________________________________________________________________
**
**			A C T I O N 	R O U T I N E S
*/

/*	Character handling
**	------------------
*/
PRIVATE void HTML_put_character ARGS2(HTStructured *, me, char, c)
{

  if (me->in_comment)
    return;
	
  if (me->in_title) {
    HTChunkPutc(&me->title, c);
    return;
  }

  if (me->in_verbatim) {
    HText_appendCharacter(me->text, c);
    return;
  }

  if (isspace(c)) {
    if (me->in_word) {
      HText_appendCharacter(me->text, ' ');
      me->in_word = NO;
    }
  } else {
    HText_appendCharacter(me->text, c);
    me->in_word = YES;
  }

}



/*	String handling
**	---------------
**
**	This is written separately from put_character becuase the loop can
**	in some cases be promoted to a higher function call level for speed.
*/
PRIVATE void HTML_put_string ARGS2(HTStructured *, me, CONST char*, s)
{
  CONST char *p;

  if (me->in_comment)    
    return;
	
  if (me->in_title) {
    HTChunkPuts(&me->title, s);
    return;
  }

  if (me->in_verbatim) {
    HText_appendText(me->text, s);
    return;
  }

  for(p=s; *p; p++) {
    if (isspace(*p)) {
      if (me->in_word) {
	HText_appendCharacter(me->text, ' ');
	me->in_word = NO;
      }
    } else {
      HText_appendCharacter(me->text, *p);
      me->in_word = YES;
    }
  } /* for */
}


/*	Buffer write
**	------------
*/
PRIVATE void HTML_write ARGS3(HTStructured *, me, CONST char*, s, int, l)
{
    CONST char* p;
    CONST char* e = s+l;
    for (p=s; s<e; p++) HTML_put_character(me, *p);
}

/*
**
*/

PRIVATE void beginAnchor ARGS3(HTStructured *, me, CONST char*, name, 
			       CONST char *, href)
{
  HText_puts(me->text, "\"\ntkW3HtBeginAnc \"");
  HText_puts(me->text, name);
  HText_puts(me->text, "\" \"");
  HText_puts(me->text, href);
  HText_puts(me->text, "\"\ntkW3HtAdd \"");
}


PRIVATE void endAnchor ARGS1(HTStructured *, me)
{
  HText_puts(me->text, "\"\ntkW3HtEndAnc\ntkW3HtAdd \"");
}

PRIVATE void setImage ARGS3(HTStructured *, me, CONST char*, name,
			    BOOL, is_map) {
  HText_puts(me->text, "\"\ntkW3HtSetImg ");
  HText_puts(me->text, name);
  HText_puts(me->text, " ");
  HText_puts(me->text, is_map ? "1" : "0");
  HText_puts(me->text, "\ntkW3HtAdd \"");
}

PRIVATE void addCommand ARGS3(HTStructured *, me, CONST char *, command,
			      int, elem) {
  HText_puts(me->text, "\"\n");
  HText_puts(me->text, command);
  HText_putc(me->text, ' ');
  HText_puts(me->text, HTML_dtd.tags[elem].name);  
  HText_puts(me->text, "\ntkW3HtAdd \"");
}

#define beginParStyle(me, elem)  addCommand((me), "tkW3HtBeginPar", (elem));
#define endParStyle(me, elem)    addCommand((me), "tkW3HtEndPar", (elem));
#define appendBullet(me, elem)   addCommand((me), "tkW3HtAddBul", (elem));
#define beginCharStyle(me, elem) addCommand((me), "tkW3HtBeginChar", (elem));
#define endCharStyle(me, elem)   addCommand((me), "tkW3HtEndChar", (elem));

/*	Start Element
**	-------------
*/
PRIVATE void HTML_start_element ARGS4(
	HTStructured *, 	me,
	int,		element_number,
	CONST BOOL*,	 	present,
	CONST char **,	value)
{
    switch (element_number) {
    case HTML_A:
	{
	    char * href = NULL;
	    if (present[HTML_A_HREF]) {
	    	StrAllocCopy(href, value[HTML_A_HREF]);
		HTSimplify(href);
	      }
	    beginAnchor(me, 
			present[HTML_A_NAME] ? value[HTML_A_NAME] : "",
			present[HTML_A_HREF] ? href: "");

	  }
    	break;
    case HTML_IMG:
	if (present[HTML_IMG_SRC]) 
	  setImage(me, value[HTML_IMG_SRC], present[HTML_IMG_ISMAP]);
	break;
    case HTML_TITLE:
	me->in_title = YES;
        HTChunkClear(&me->title);
	break;
	
    case HTML_NEXTID:
    	break;
	
    case HTML_ISINDEX:
   	HTAnchor_setIndex(me->node_anchor);
	HText_puts(me->text, "\"\ntkW3HtSetInd\ntkW3HtAdd \"");
	break;
	
    case HTML_DL:
	beginParStyle(me, present && present[DL_COMPACT]
    		? HTML_DLC : HTML_DL);
	break;
	
    case HTML_DT:
    case HTML_DD:
    case HTML_LI:
    case HTML_P:
	appendBullet(me, element_number);
	me->in_word = NO;
	break;

    case HTML_UL:
    case HTML_OL:
    case HTML_MENU:
    case HTML_DIR:
	beginParStyle(me, element_number);
	break;
	
    case HTML_LISTING:				/* Litteral text */
    case HTML_XMP:
    case HTML_PLAINTEXT:
    case HTML_PRE:
	me->in_verbatim = YES;
	beginParStyle(me, element_number);
	break;

    case HTML_HTML:			/* Ignore these altogether */
    case HTML_HEAD:
    case HTML_BODY:
	break;

    
    case HTML_TT:			/* Physical character highlighting */
    case HTML_B:
    case HTML_I:
    case HTML_U:
    
    case HTML_EM:			/* Logical character highlighting */
    case HTML_STRONG:
    case HTML_CODE:
    case HTML_SAMP:
    case HTML_KBD:
    case HTML_VAR:
    case HTML_DFN:
    case HTML_CITE:
	beginCharStyle(me, element_number);
	break;

    case HTML_H1:
    case HTML_H2:
    case HTML_H3:
    case HTML_H4:
    case HTML_H5:
    case HTML_H6:
    case HTML_H7:
    case HTML_ADDRESS:
      case HTML_BLOCKQUOTE:
    	beginParStyle(me, element_number);
	break;

      case HTML_COMMENT:
	me->in_comment = YES;

    } /* end switch */
}


/*		End Element
**		-----------
**
*/

PRIVATE void HTML_end_element ARGS2(HTStructured *, me, int , element_number)
{
  switch(element_number) {

  case HTML_A:
    endAnchor(me);
    break;

  case HTML_TITLE:
    HTChunkTerminate(&me->title);
    HTAnchor_setTitle(me->node_anchor, me->title.data);
    me->in_title = NO;
    break;

  case HTML_COMMENT:
    me->in_comment = NO;
    break;

  case HTML_TT:			/* Physical character highlighting */
  case HTML_B:
  case HTML_I:
  case HTML_U:

  case HTML_EM:			/* Logical character highlighting */
  case HTML_STRONG:
  case HTML_CODE:
  case HTML_SAMP:
  case HTML_KBD:
  case HTML_VAR:
  case HTML_DFN:
  case HTML_CITE:
    endCharStyle(me, element_number);
    break;

  case HTML_LISTING:				/* Litteral text */
  case HTML_XMP:
  case HTML_PLAINTEXT:
  case HTML_PRE:
    me->in_verbatim = NO; /* @@@@@@@ Assumes these do not nest */

  default:
    endParStyle(me, element_number);
    break;
  } /* switch */
}


/*		Expanding entities
**		------------------
*/
/*	(In fact, they all shrink!)
*/

PRIVATE void HTML_put_entity ARGS2(HTStructured *, me, int, entity_number)
{
    HTML_put_string(me, ISO_Latin1[entity_number]);	/* @@ Other representations */
}



/*	Free an HTML object
**	-------------------
**
** If the document is empty, the text object will not yet exist.
   So we could in fact abandon creating the document and return
   an error code.  In fact an empty document is an important type
   of document, so we don't.
**
**	If non-interactive, everything is freed off.   No: crashes -listrefs
**	Otherwise, the interactive object is left.	
*/
PUBLIC void HTML_free ARGS1(HTStructured *, me)
{
  char *address;
  CONST char *title;
  if (me->comment_end)
    HTML_put_string(me,me->comment_end);
  HText_puts(me->text, "\"\ntkW3HtSetName \"");
  address = HTAnchor_address((HTAnchor *) me->node_anchor);
  HText_appendText(me->text, address);
  free(address);
  HText_puts(me->text, "\" \"");
  title = HTAnchor_title(me->node_anchor);
  if (title)
    HText_appendText(me->text, title);
  HText_endAppend(me->text);


  if (me->target) {
    (*me->targetClass.end_document)(me->target);
    (*me->targetClass.free)(me->target);
  }
  free(me);
}


PRIVATE void HTML_end_document ARGS1(HTStructured *, me)

{			/* Obsolete */
}

/*				P U B L I C
*/

/*	Structured Object Class
**	-----------------------
*/
PUBLIC CONST HTStructuredClass HTMLPresentation = /* As opposed to print etc */
{		
	"text/html",
	HTML_free,
	HTML_end_document,
	HTML_put_character, 	HTML_put_string,  HTML_write,
	HTML_start_element, 	HTML_end_element,
	HTML_put_entity
}; 


/*		New Structured Text object
**		--------------------------
**
**	The strutcured stream can generate either presentation,
**	or plain text, or HTML.
*/
PUBLIC HTStructured* HTML_new ARGS3(
	HTParentAnchor *, 	anchor,
	HTFormat,		format_out,
	HTStream*,		stream)
{

    HTStructured * me;
    
    if (format_out != WWW_PLAINTEXT && format_out != WWW_PRESENT) {
        HTStream * intermediate = HTStreamStack(WWW_HTML, format_out,
		stream, anchor);
	if (intermediate) return HTMLGenerator(intermediate);
        fprintf(stderr, "** Internal error: can't parse HTML to %s\n",
       		HTAtom_name(format_out));
	exit (-99);
    }

    me = (HTStructured*) malloc(sizeof(*me));
    if (me == NULL) outofmem(__FILE__, "HTML_new");

    me->isa = &HTMLPresentation;
    me->node_anchor =  anchor;
    me->title.size = 0;
    me->title.growby = 128;
    me->title.allocated = 0;
    me->title.data = 0;

    
    me->comment_start = NULL;
    me->comment_end = NULL;
    me->target = stream;
    me->in_verbatim = NO;
    me->in_word = NO;
    me->in_comment = NO;
    me->in_title = NO;
    if (stream) me->targetClass = *stream->isa;	/* Copy pointers */
    me->text = HText_new2(anchor, stream);
    HText_beginAppend(me->text);
    return (HTStructured*) me;
}


/*	HTConverter for HTML to plain text
**	----------------------------------
**
**	This will convert from HTML to presentation or plain text.
*/
PUBLIC HTStream* HTMLToPlain ARGS3(
	HTPresentation *,	pres,
	HTParentAnchor *,	anchor,	
	HTStream *,		sink)
{
    return SGML_new(&HTML_dtd, HTML_new(anchor, pres->rep_out, sink));
}


/*	HTConverter for HTML to C code
**	------------------------------
**
**	C copde is like plain text but all non-preformatted code
**	is commented out.
**	This will convert from HTML to presentation or plain text.
*/
PUBLIC HTStream* HTMLToC ARGS3(
	HTPresentation *,	pres,
	HTParentAnchor *,	anchor,	
	HTStream *,		sink)
{
    
    HTStructured * html;
    
    (*sink->isa->put_string)(sink, "/* ");	/* Before even title */
    html = HTML_new(anchor, WWW_PLAINTEXT, sink);
    html->comment_start = "/* ";
    html->comment_end = " */\n";	/* Must start in col 1 for cpp */
/*    HTML_put_string(html,html->comment_start); */
    return SGML_new(&HTML_dtd, html);
}


/*	Presenter for HTML
**	------------------
**
**	This will convert from HTML to presentation or plain text.
**
**	Override this if you have a windows version
*/
#ifndef GUI
PUBLIC HTStream* HTMLPresent ARGS3(
	HTPresentation *,	pres,
	HTParentAnchor *,	anchor,	
	HTStream *,		sink)
{
    return SGML_new(&HTML_dtd, HTML_new(anchor, WWW_PRESENT, NULL));
}
#endif


/*	Record error message as a hypertext object
**	------------------------------------------
**
**	The error message should be marked as an error so that
**	it can be reloaded later.
**	This implementation just throws up an error message
**	and leaves the document unloaded.
**	A smarter implementation would load an error document,
**	marking at such so that it is retried on reload.
**
** On entry,
**	sink 	is a stream to the output device if any
**	number	is the HTTP error number
**	message	is the human readable message.
**
** On exit,
**	returns	a negative number to indicate lack of success in the load.
*/



PUBLIC int HTLoadError ARGS3(
	HTStream *, 	sink,
	int,		number,
	CONST char *,	message)
{
  strncpy(HtErrorBuffer, message, ERROR_BUFSIZ);
  return -number;
} 
