/* tkWWW.c Adds World Wide Web commands to a Tcl interpreter
** ===============
**
** Authors:
** Joseph Wang, Department of Astronomy, University of Texas at Austin
** (joe@astro.as.utexas.edu)
**
** Copyright:
** Copyright (C) 1992-1993
** Usenet University -- New Network Academy
** Macvicar Institute for Educational Software Development
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/* Include c stuff
** ---------------
*/

#include <assert.h>
#include <stdio.h>
#include <ctype.h>

#ifdef __STDC__
#include <sys/types.h>
#include <stdlib.h>
#endif 

/* Include World Wide Web stuff
** ----------------------------
*/

#include <HTAnchor.h>
#include <HTParse.h>
#include <HTAccess.h>
#include <HTML.h>
#include <HText.h>
#include <HTTCP.h>
#include <HTStream.h>
#include <HTFile.h>
#include <HTFWriter.h>
#include <HTextDef.h>
#include <HTAtom.h>
#include <HTPlain.h>
#include <HTMLGen.h>
#include <HTFormat.h>
#include <HTMIME.h>
#include <HTInit.h>
#include <HTAlert.h>
#include <TkWWWCmds.h>
#include <TkWWWVersion.h>

/* Some Global Variables
** ---------------------
*/

PUBLIC char * HTAppName = "tkWWW";	/* Application name */
PUBLIC char * HTAppVersion = SERVER_VERSION; 	/* Application version */
PUBLIC Tcl_Interp* HtTclInterp = 0; /* @@@@@@@@ */
PUBLIC int HtTclErrorCode = TCL_OK;
PUBLIC HText* HtTclExecText = 0;

EXTERN HTPresentation* default_presentation;
					   
/* Macro to check arguments
** ------------------------
** Assumes interpreter is in interp and number of arguments is in argc
*/

#define HtCheckArgc(min,max,function_name) \
    if ((argc < (min)) || (argc > (max))) { \
       Tcl_AppendResult(interp, (function_name), \
			": Incorrect number of arguments" , NULL); \
       return (TCL_ERROR); \
       }

/* Procedures called by tkWWW functions
** ------------------------------------
*/

PRIVATE int HtLoadCmd(dummy, interp, argc, argv)
     ClientData dummy;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
  HtCheckArgc(2,2, "HtLoad");
  HtTclInterp = interp;
  HtTclErrorCode = TCL_OK;
  HtTclExecText = NULL;

  if (!HTLoadAbsolute(argv[1]))
    return (TCL_ERROR);

  if (HtTclExecText)
    HtTclErrorCode = Tcl_Eval(interp, HtTclExecText->output->data);

  return (HtTclErrorCode);
}

PRIVATE int HtParseNameCmd(dummy, interp, argc, argv)
     ClientData dummy;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
  char *current_address = NULL;
  
  HtCheckArgc(2,3,"HtParseName");
  if (argc == 3 && argv[2] && *(argv[2])) 
    StrAllocCopy(current_address, argv[2]);
  else {
    StrAllocCopy(current_address, "file:");

/* The following mess is to get the current working directory in a manner
 * that will work on most platforms 
 */

#ifndef MAXPATHLEN
#define NO_GETWD		/* Assume no  getwd() if no MAXPATHLEN */
#endif

#ifdef NO_GETWD  		/* No getwd() on this machine */
#ifdef HAS_GETCWD		/* System V variant SIGN CHANGED TBL 921006 !! */

    {
      char wd[1024];			/*!! Arbitrary*/
      extern char * getcwd();
      char * result = getcwd(wd, sizeof(wd)); 
      if (result) {

#ifdef vms  /* convert directory name to Unix-style syntax */
	char * disk = strchr (wd, ':');
	char * dir = strchr (wd, '[');
	if (disk) {
	  *disk = '\0';
	  StrAllocCat (current_address, "/");  /* needs delimiter */
	  StrAllocCat (current_address, wd);
	}
	if (dir) {
	  char *p;
	  *dir = '/';  /* Convert leading '[' */
	  for (p = dir ; *p != ']'; ++p)
	    if (*p == '.') *p = '/';
	  *p = '\0';  /* Cut on final ']' */
	  StrAllocCat (current_address, dir);
	}
#else  /* not VMS */
	StrAllocCat (current_address, wd);
#endif  /* not VMS */
      } else {
	fprintf(stderr,
		"HTBrowse: Can't read working directory (getcwd).\n");
      }
    }  /* end if good getcwd result */
	
#else   /* has NO getcwd */

    if (TRACE) 
      fprintf(stderr,
	      "HTBrowse: This platform does not support getwd() or getcwd()\n");
#endif	/* has no getcwd */

#else   /* has getwd */
    {
      char wd[MAXPATHLEN];
      extern char * getwd();
      char * result = getwd(wd);
      if (result) {
	StrAllocCat(current_address, wd);
      } else {
	fprintf(stderr, "HTBrowse: Can't read working directory.\n");
      }
    }
#endif
		
#ifdef vms
    StrAllocCat(current_address, "default.html");
#else
    StrAllocCat(current_address, "/default.html");
#endif
  }

  Tcl_SetResult(interp, HTParse(argv[1], current_address, PARSE_ALL), 
		TCL_DYNAMIC);
  free(current_address);
  
  return (TCL_OK);
}

PRIVATE int HtUncacheCmd (dummy, interp, argc, argv) 
     ClientData dummy;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
  HTParentAnchor *parent_anchor;
  HTAnchor *anchor;
  HText *document;
  HtCheckArgc(2, 2, "HtUncache");

  anchor = HTAnchor_findAddress(argv[1]);
  if (anchor) {
    parent_anchor = HTAnchor_parent(anchor);
    document = (HText *)HTAnchor_document(parent_anchor);
    if (document) 
      HText_free(document);
  }
  return (TCL_OK);
}

  
/* HtVersion
** ------------
** returns the version of the tkWWW server
*/

PRIVATE int HtVersionCmd(dummy, interp, argc, argv)
     ClientData dummy;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
  
  Tcl_SetResult(interp, SERVER_VERSION, TCL_STATIC);
  return (TCL_OK);
}

PUBLIC void HTTkSetOutputFile(format, anchor, fnam)
     HTFormat format;
     HTParentAnchor *anchor;
     CONST char *fnam;
 {
  char *address;
  HText *text = HText_new(anchor);
  HTChunkPuts(text->output, "tkW3ConfigDisplayFile ");
  HTChunkPuts(text->output, HTAtom_name(format));
  HTChunkPuts(text->output, " ");
  HTChunkPuts(text->output, fnam);
  HTChunkPuts(text->output, " ");
  address = HTAnchor_address((HTAnchor *) anchor);
  HTChunkPuts(text->output, address);
  HTChunkPuts(text->output, "\n");
  free(address);
  HTChunkTerminate(text->output);
  HtTclExecText = text;
}

PRIVATE HTStream * HTTkDisplay(pres, anchor, sink) 
     HTPresentation *pres;
     HTParentAnchor *anchor;
     HTStream *sink;

{
  char *fnam;
  CONST char * suffix;
  HTStream* this;
  FILE *fp;
  /* Save the file under a suitably suffixed name */

  suffix = HTFileSuffix(pres->rep);

  fnam = tempnam (NULL, "www");
  if (suffix) strcat(fnam, suffix);
    
  fp = fopen (fnam, "w");
  if (!fp) {
    HTAlert("Can't open temporary file!");
    free(fnam);
    return NULL;
  }

  this = HTFWriter_new(fp);
  HTTkSetOutputFile(pres->rep, anchor, fnam);
  free (fnam);
  return this;
}

PUBLIC BOOL HTTkUseInPlace(format_in, format_out)
     HTFormat format_in;
     HTFormat format_out;
{
  int n = HTList_count(HTPresentations);
  int i;
  HTPresentation *pres;
  for (i=0; i<n; i++) {
    pres = HTList_objectAt(HTPresentations, i);
    if (pres->rep == format_in && pres->rep_out == format_out) 
      return (pres->converter == HTTkDisplay) ? TRUE : FALSE;
  }
  if (default_presentation && 
      default_presentation->converter == HTTkDisplay)
    return TRUE;
  return FALSE;

}

PRIVATE int HtSetSuffixCmd(dummy, interp, argc, argv)
     ClientData dummy;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
  float priority;
  HtCheckArgc(4, 5, "HtSetSuffix");
  priority  = (argc==4) ? 1.0 : atof(argv[4]);
  HTSetSuffix(argv[1], argv[2], argv[3], priority);
  return (TCL_OK);
}

/* Add tkWWW commands to a tcl interpreter init_tkWWW(interp)
** -------------
*/

int WWW_AppInit(interp)
     Tcl_Interp *interp;
{
  Tcl_CreateCommand(interp, "HtSetSuffix", HtSetSuffixCmd,
		    (ClientData) NULL, (void (*)()) NULL);
  Tcl_CreateCommand(interp, "HtLoad", HtLoadCmd, 
		    (ClientData) NULL, (void (*)()) NULL);
  Tcl_CreateCommand(interp, "HtParseName", HtParseNameCmd,
		    (ClientData) NULL, (void (*)()) NULL);
  Tcl_CreateCommand(interp, "HtUncache", HtUncacheCmd,
		    (ClientData) NULL, (void (*)()) NULL);
  Tcl_CreateCommand(interp, "HtVersion", HtVersionCmd,
		    (ClientData) NULL, (void (*)()) NULL);
  HTSetConversion("www/mime",  "*",           HTMIMEConvert, 	1.0, 0.0, 0.0);
  HTSetConversion("text/html", "text/x-c",    HTMLToC, 	        0.5, 0.0, 0.0);
  HTSetConversion("text/html", "text/plain",  HTMLToPlain, 	0.5, 0.0, 0.0);
  HTSetConversion("text/html", "www/present", HTMLPresent, 	1.0, 0.0, 0.0);
  HTSetConversion("text/plain", "text/html",  HTPlainToHTML,	1.0, 0.0, 0.0);
  HTSetConversion("*",          "www/present",HTTkDisplay,      0.5, 0.0, 0.0);
  return (TCL_OK);
}
