/* util.c -- utilities and misc. functions */

/*
 * Copyright 1990 the Regents of the University of California.  All
 * rights reserved.  Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without fee is
 * hereby granted, provided that this copyright notice appear in all
 * copies.  See the file copyright.h for more information.
 *
 */

#include <stdio.h>
#include <sgtty.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/dir.h>
#include <errno.h>
#include "debug.h"
#include "global.h"
#include "util.h"
#include <X11/IntrinsicP.h>
#include <X11/ShellP.h>  /* for checking of a shell is already popped up */
static int dtty;
static struct {
	int lined;
	struct sgttyb sg;
	struct tchars tc;
	int local;
	struct winsize ws;
} termset;   /* global for terminal settings. */

static int termsaved = False;
static int termchanged = False;

int mystrcmp(a,b)
     char *a,*b;
     /* makes sure a and b are non-null before calling strcmp */
{
	if ((!a)&&(!b))
	  return 0;
	if ((!a)&&b)
	  return -1;
	if (a&&(!b))
	  return 1;
	return strcmp(a,b);
}

void EmptyEventLoop()
     /* process any events pending */
{
	XEvent event;
	entering("EmptyEventLoop");
	while (XtAppPending(AppCon)==XtIMXEvent) {
		XtAppNextEvent(AppCon,&event);
		XtDispatchEvent(&event);
	}
	myreturn;
}

Widget MyFindWidgetByName(name)
     char *name;
     /* returns a widget by name.  If the widget is being destroyed, then
	returns NULL */
{
	Widget w;
	entering("FindWidgetByName");
	w = XtNameToWidget(TopLevel,name);
	if (w&&(!w->core.being_destroyed)) {
	    	myreturn w;
	} else {
		myreturn NULL;
	}
}

void SafePopupWidget(widget,dest_widget)
     Widget widget,dest_widget;
     /* pops up widget and moves it if necessary to fit on screen.
	Tries to center the widget around the pointer. Then places the
	pointer near the top of the dest_widget */
{
	Window root,child;
	Dimension width, height, borderwidth;
	int scr_width,scr_height,xpos,ypos,x,y,mask,changed;

	entering("SafePopupWidget");
	XQueryPointer(XtDisplay(TopLevel),XtWindow(TopLevel),
		      &root,&child,&xpos,&ypos,&x,&y,&mask);
	XtVaGetValues(widget,XtNheight,&height,XtNwidth,&width,
		      XtNborderWidth,&borderwidth,NULL);
	if (height==0)
	  height = 100;
	if (width==0)
	  width = 100;
	/* center the widget around the pointer */
	xpos = xpos-(int)((width+borderwidth*2)/2);
	ypos = ypos-(int)((height+borderwidth*2)/2);
	if (!((ShellWidget)widget)->shell.popped_up) {
		XtVaSetValues(widget,XtNx,xpos,XtNy,ypos,NULL);
	}
	XtPopup(widget,XtGrabNone);
	/* see if widget fits on screen */
	XtVaGetValues(widget,XtNheight,&height,XtNwidth,&width,
		      XtNborderWidth,&borderwidth,NULL);
	scr_width = WidthOfScreen(XtScreen(widget));
	scr_height = HeightOfScreen(XtScreen(widget));
	changed=False;
	if (xpos<0) {
		xpos=0;
		changed=True;
	} else {
		if (xpos+width>scr_width) {
			xpos=scr_width-(width+borderwidth*2);
			changed=True;
		}
	}
	if (ypos<0) {
		ypos=0;
		changed=True;
	} else {
		if (ypos+height>scr_height) {
			ypos = scr_height-(height+borderwidth*2);
			changed=True;
		}
	}
	if (changed) {
		XtVaSetValues(widget,XtNx,xpos,XtNy,ypos,NULL);
	}
	EmptyEventLoop();
	if (!dest_widget)
	  dest_widget=widget;
	XWarpPointer(XtDisplay(TopLevel),XtWindow(TopLevel),
		     XtWindow(dest_widget),
		     0,0,  /* src_x, src_y */
		     WidthOfScreen(XtScreen(widget)),
		     HeightOfScreen(XtScreen(widget)),
		     /*0,0,*/15,15);
	myreturn;
}

int ListFiles(dirname,argvPtr)
     char *dirname;
     char*** argvPtr;
     /* returns an array of filenames in argvPatr for the directory
	dirname.  The number of filenames (i.e. size of files)
	is returned. */
{
	char **files;
	DIR *dirp;
	struct direct *dp;
	int numfiles;
	entering("filelist");
	dirp = opendir(dirname);
	if (!dirp) {
		files=NULL;
		myreturn 0;
	}
	numfiles=0;
	for (dp=readdir(dirp); dp!=NULL; dp=readdir(dirp))
	  numfiles++;
	files = (char**)XtMalloc((numfiles+1)*(sizeof(char*)));
	rewinddir(dirp);
	numfiles=0;
	for (dp=readdir(dirp); dp!=NULL; dp=readdir(dirp))
	  files[numfiles++]=XtNewString(dp->d_name);
	closedir(dirp);
	*argvPtr=files;
	myreturn numfiles;
}

void SaveTerminal()
     /* saves the terminal settings */
{
	entering("SaveTerminal");
	dtty = 0;
	/* get line discipline */
	/* get basic ioctls */
	/* get tchars */
	/* get local mode */
	/* get window sizes */
	ioctl(dtty,TIOCGETD,&termset.lined);
	ioctl(dtty,TIOCGETP,&termset.sg);
	ioctl(dtty,TIOCGETC,&termset.tc);
	ioctl(dtty,TIOCLGET,&termset.local);
	ioctl(dtty,TIOCGWINSZ,&termset.ws);
	termsaved = True;  /* terminal has been saved. */
	myreturn;
}

void RestoreTerminal()
     /* restores the terminal settings */
{
	entering("RestoreTerminal");
	if (termsaved && termchanged) {
		dprint("restoring terminal...");
		/* set line discipline */
		/* set basic ioctls */
		/* set tchars */
		/* set local mode */
		/* set window sizes */
		ioctl(dtty,TIOCSETD,&termset.lined);
		ioctl(dtty,TIOCSETP,&termset.sg);
		ioctl(dtty,TIOCSETC,&termset.tc);
		ioctl(dtty,TIOCLSET,&termset.local);
		ioctl(dtty,TIOCSWINSZ,&termset.ws);
	}
	myreturn;
}

void AssertTermChanged()
     /* used to assert that the controlling tty has somehow changed and
	needs to be restored.  This routine is necessary so that 
	RestoreTerminal can check before doing an ioctl.  If the controlling
	tty doesn't change, then the program can be put into background. */
{
	entering("AssertTermChanged");
	termchanged = True;
	myreturn;
}

int ptrstrcmp(a,b)
     char **a,**b;
{
	return strcmp(*a,*b);
}

void Quit()
     /* destroys the TopLevel widget */
{
	entering("Quit");
	XtDestroyApplicationContext(AppCon);
	RestoreTerminal();
	exit(0);
	myreturn;
}
