#!/usr/bin/env perl

##############################################################################
#      $URL: http://perlcritic.tigris.org/svn/perlcritic/trunk/Perl-Critic/t/generate_without_optional_dependencies_wrappers.PL $
#     $Date: 2007-10-21 03:46:24 -0500 (Sun, 21 Oct 2007) $
#   $Author: clonezone $
# $Revision: 1991 $
##############################################################################

use strict;
use warnings;

use lib 'inc';

use English qw{-no_match_vars};

use Carp qw{ confess };
use Fatal qw{ open close };

use Perl::Critic::BuildUtilities qw{ recommended_module_versions };

our $VERSION = '1.079_002';

print "\n\nGenerating tests that hide modules and then run other tests.\n";

my $this_program = __FILE__;

my @modules_to_hide = sort keys %{ { recommended_module_versions } };
my $modules_to_hide = join "\n" . q< > x 4, @modules_to_hide;

foreach my $test_program_name (@ARGV) {
    my ($wrapped_test_name) =
        $test_program_name =~ m{
            \A
            ( t/ [\w.]+ \.t )                  # test to be wrapped
            _without_optional_dependencies\.t  # suffix the new test will have
            \z
        }xms;
    if (not $wrapped_test_name) {
        confess
              'Could not figure out the name of the test to wrap from "'
            . $test_program_name
            . q{".};
    }

    print "Generating $test_program_name.\n";
    open my $test_program, '>', $test_program_name
        or die "Could not open $test_program_name: $ERRNO";

    print {$test_program} <<"END_TEST_PROGRAM";
#!/usr/bin/env perl

# Do not edit!!!  This program generated by $this_program.

use strict;
use warnings;
use English qw{-no_match_vars};

our \$VERSION = $VERSION;

use lib 't/tlib';

use English qw(-no_match_vars);

use Perl::Critic::TestUtilitiesWithMinimalDependencies qw{
    should_skip_author_tests
    get_author_test_skip_message
    get_skip_all_tests_tap
};

#-----------------------------------------------------------------------------

if ( should_skip_author_tests() ) {
    print get_skip_all_tests_tap(), get_author_test_skip_message(), "\\n";
    exit 0;
}

eval <<'END_HIDE_MODULES';
use Test::Without::Module qw{
    $modules_to_hide
};
END_HIDE_MODULES

if ( \$EVAL_ERROR ) {
    print
        get_skip_all_tests_tap(),
        'Test::Without::Module required to test with the ',
        "absence of optional modules\\n";
    exit 0;
}


require '$wrapped_test_name';
END_TEST_PROGRAM

    close $test_program;
}

print "Done.\n\n";

__END__

#-----------------------------------------------------------------------------

=pod

=for stopwords 

=head1 NAME

generate_without_additional_dependencies_wrappers.PL - generate tests that are wrappers around other tests but which hide the existence of modules first.


=head1 SYNOPSIS

  generate_without_additional_dependencies_wrappers.PL \
      t/00_modules.t_without_optional_dependencies.t \
      t/01_config.t_without_optional_dependencies.t \
      t/13_bundled_policies.t_without_optional_dependencies.t


=head1 DESCRIPTION

Release 1.07 of Perl::Critic was an embarrassment because there were no tests
of core without the presence of optional modules.  This program generates
wrappers for other tests that hide those optional modules.


=head1 AUTHOR

Elliot Shank C<< <perl@galumph.org> >>


=head1 COPYRIGHT

Copyright (c) 2007 Elliot Shank.  All rights reserved.

This program is free software; you can redistribute it and/or modify it under
the same terms as Perl itself.  The full text of this license can be found in
the LICENSE file included with this module.


=cut

##############################################################################
# Local Variables:
#   mode: cperl
#   cperl-indent-level: 4
#   fill-column: 78
#   indent-tabs-mode: nil
#   c-indentation-style: bsd
# End:
# ex: set ts=8 sts=4 sw=4 tw=78 ft=perl expandtab :
