package OpenInteract::Handler::Security;

# $Id: Security.pm,v 1.15 2003/08/12 03:29:24 lachoy Exp $

use strict;
use Data::Dumper             qw( Dumper );
use SPOPS::Secure            qw( :level :scope );
use SPOPS::Secure::Hierarchy qw( $ROOT_OBJECT_NAME );
use OpenInteract::Handler::GenericDispatcher qw( DEFAULT_SECURITY_KEY );

@OpenInteract::Handler::Security::ISA     = qw( OpenInteract::Handler::GenericDispatcher SPOPS::Secure );
$OpenInteract::Handler::Security::VERSION = sprintf("%d.%02d", q$Revision: 1.15 $ =~ /(\d+)\.(\d+)/);

$OpenInteract::Handler::Security::author            = 'chris@cwinters.com';
$OpenInteract::Handler::Security::default_method    = 'listing';
@OpenInteract::Handler::Security::forbidden_methods = qw( deprecate_me );
%OpenInteract::Handler::Security::security          = ( DEFAULT_SECURITY_KEY() => SEC_LEVEL_WRITE );

use constant MAIN_SCRIPT => '/Security';

my $SECURE_CLASS = 'SPOPS::Secure';


# Display the object classes and handler classes currently used by
# this website and track those that are using security

sub listing {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    my $CONF = $R->CONFIG;
    my ( @object_class, @action_class );
    my ( %classes_visited );

    foreach my $key ( sort keys %{ $CONF->{SPOPS} } ) {
        $R->scrib( 1, "Found object key ($key)" );
        next if ( $key =~ /^_/ );
        my $spops_class = $CONF->{SPOPS}{ $key }{class};
        next if ( ! $spops_class or $classes_visited{ $spops_class } );
        $R->scrib( 1, "Object key ($key) matches with class ($spops_class)" );
        push @object_class, { name   => $key,
                              class  => $spops_class,
                              secure => $spops_class->isa( 'SPOPS::Secure' ),
                              hierarchy_secure => $spops_class->isa( 'SPOPS::Secure::Hierarchy' ) };
        $classes_visited{ $spops_class }++;
    }

    foreach my $key ( sort keys %{ $CONF->{action} } ) {
        $R->scrib( 1, "Found handler key ($key)" );
        next if ( $key =~ /^_/ );
        my $action_class = $CONF->{action}{ $key }{class};
        next if ( ! $action_class or $classes_visited{ $action_class } );
        $R->scrib( 1, "Handler key ($key) matches with class ($action_class)" );
        push @action_class, { name   => $key,
                              class  => $action_class,
                              secure => $action_class->isa( 'SPOPS::Secure' ),
                              hierarchy_secure => $action_class->isa( 'SPOPS::Secure::Hierarchy' ) };
        $classes_visited{ $action_class }++;
    }
    $R->{page}{title} = 'Security Listing';
    return $R->template->handler( {}, { object_list => \@object_class,
                                        action_list => \@action_class },
                                  { name => 'base_security::object_class_list' });
}



sub show {
    my ( $class, $p ) = @_;
    my $error_msg = $p->{error_msg};

    my $R = OpenInteract::Request->instance;

    my ( $object_class, $object_id ) = $class->_find_object_info( $p );
    unless ( $object_class ) {
        return "<h1>Error</h1>\n<p>Could not find an object class for " .
               "which I should modify security.</p>";
    }

    if ( $object_class->isa( 'SPOPS::Secure::Hierarchy' ) ) {
        my $drilldown = $R->apache->param( 'drilldown' );
        return $class->hierarchy_show( $p ) unless ( $drilldown );
    }
    unless ( $object_class->isa( 'SPOPS::Secure' )  ) {
        $error_msg .= _cannot_secure_msg( $object_class );
    }

    $R->DEBUG && $R->scrib( 1, "Editing security for $object_class ($object_id)" );

    my ( $type, $desc, $url ) = $class->_fetch_description({
                                            object       => $p->{object},
                                            object_id    => $object_id,
                                            object_class => $object_class });
    my $params = { main_script        => MAIN_SCRIPT,
                   object_class       => $object_class,
                   object_id          => $object_id,
                   error_msg          => $error_msg,
                   object_description => $desc,
                   object_type        => $type,
                   object_url         => $url };

    # Now fetch the security info -- we want to see who already has
    # security set so we can display that information next to the name
    # of the group/user or world in the listing

    my $security = eval { $R->security_object->fetch_by_object(
                                 undef, { class     => $object_class,
                                          object_id => $object_id,
                                          group     => 'all' } ) };
    $R->DEBUG && $R->scrib( 1, "Security fetched: ", Dumper( $security ) );;
    if ( $@ ) {
        my $ei = OpenInteract::Error->set( SPOPS::Error->get );
        $R->scrib( 0, "Error fetching security for ($object_class)",
                      "($object_id)\n", Dumper( $ei ) );
        $R->throw({ code => 308 });
        return "<h2>Error!</h2><p>Error retrieving security: $ei->{system_msg}</p>";
    }

    # First item in the scope is the WORLD setting

    my $world_level = $security->{ SEC_SCOPE_WORLD() };
    my @scopes = ({ scope => SEC_SCOPE_WORLD,
                    name  => 'World',
                    level => $world_level });

    push @scopes, $class->_get_group_scopes( $security );

    # NOTE: We do not fetch user-level security unless specifically
    # requested

    if ( $R->apache->param( 'include_user' ) ) {
        push @scopes, $class->_get_user_scopes( $security );
    }

    $params->{scope_list} = \@scopes;
    $R->{page}->{title} = 'Edit Object Security';
    return $R->template->handler( {}, $params,
                                  { name => 'base_security::assign_object_security' } );
}


sub hierarchy_show {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;

    my $object = $p->{object};
    my ( $object_class, $object_id ) = $class->_find_object_info( $p );

    # Retrieve the security levels so we can display them -- 'user'
    # and 'group' aren't really necessary here and are just passed to
    # keep SPOPS::Secure from doing lots of work...

    my ( $track, $first, $check_list ) =
            SPOPS::Secure::Hierarchy->get_hierarchy_levels({
                class                 => $object_class,
                object_id             => $object_id,
                security_object_class => $R->security_object,
		        user                  => $R->{auth}->{user},
                group                 => $R->{auth}->{group} });

    my @check_list_items = map { { object_id        => $_,
                               security_defined => $track->{ $_ } } }
                           @{ $check_list };

    my ( $type, $desc, $url ) = $class->_fetch_description({ 
                                            object       => $object,
                                            object_id    => $object_id,
                                            object_class => $object_class });
    my $params = { main_script        => MAIN_SCRIPT,
                   object_class       => $object_class,
                   object_id          => $object_id,
                   check_list         => \@check_list_items,
                   ROOT_OBJECT_NAME   => $ROOT_OBJECT_NAME,
                   object_description => $desc,
                   object_type        => $type,
                   object_url         => $url };
    $R->{page}->{title} = 'Object Hierarchy Security';
    return $R->template->handler( {}, $params,
                                { name => 'base_security::hierarchy_security' } );
}



# Edit security for a particular object or class -- note that the
# widget currently only supports setting one level for many scopes at
# one time.

sub edit {
    my ( $class, $p ) = @_;
    my $R   = OpenInteract::Request->instance;
    my $apr = $R->apache;

    my $level = $apr->param( 'level' );
    my ( $object_class, $object_id ) = $class->_find_object_info( $p );
    my @raw_scope = $apr->param( 'scope' );

    # A link with this information exists on the hierarchical security
    # editing screen and clears out all security for a given class and
    # ID so that ID will inherit from its parents

    if ( $raw_scope[0] eq 'all' and $level eq 'clear' ) {
        $class->clear_all_security({ object_class => $object_class,
                                     object_id    => $object_id });
        return $class->show({ object_class => $object_class,
                              object_id    => $object_id });
    }
    my @scope = map { [ split /;/ ] } @raw_scope;

    # Cycle through each scope specification (scope + scope_id) and
    # set its security for the given object class and ID

    my ( $total, $success ) = ( 0, 0 );
    my ( @status_ok );
    my $security_object_class = $R->security_object;
    foreach my $info ( @scope ) {
        $total++;
        $R->DEBUG && $R->scrib( 1, "Trying level ($level) for ($object_class)",
                                   "($object_id) in scope ($info->[0]) ($info->[1])" );
        my $security_params = { security_object_class => $security_object_class,
                                class          => $object_class,
                                object_id      => $object_id,
                                scope          => $info->[0],
                                scope_id       => $info->[1],
                                security_level => $level };
        my $method = ( $level eq 'clear' )
                       ? 'remove_item_security' : 'set_item_security';
        eval { $SECURE_CLASS->$method( $security_params ) };
        if ( $@ ) {
            my $ei = OpenInteract::Error->set( SPOPS::Error->get );
            $R->scrib( 0, "Error trying to set security!\n$@\n", Dumper( $ei ) );
            $R->throw({ code => 406 });
        }
        else {
            push @status_ok, $object_id;
        }
    }
    my ( $error_msg );
    if ( $total != scalar @status_ok ) {
        $error_msg = "Attempted to set ($total); set ($success). Some " .
                     "attempts had errors.";
    }
    my $status_msg = "Security for $object_class set for the following IDs: " .
                     join( ', ', @status_ok );

    my $return_url = $apr->param( 'return_url' );
    if ( $return_url ) {
        my ( $action_class, $action_method ) = $R->lookup_action( 'redirect' );
        return $action_class->$action_method({ url        => $return_url,
                                               error_msg  => $error_msg,
                                               status_msg => $status_msg });
    }
    return $class->show({ object_class => $object_class,
                          object_id    => $object_id,
                          status_msg   => $status_msg,
                          error_msg    => $error_msg });
}


# Clear all security for a particular object class and ID

sub clear_all_security {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    my $security_list = eval { $R->security_object->fetch_group({
                                      object_id => $p->{object_id},
                                      class     => $p->{object_class} }) };
    foreach my $security_obj ( @{ $security_list } ) {
        eval { $security_obj->remove };
    }
}


sub _get_group_scopes {
    my ( $class, $security ) = @_;
    my $R = OpenInteract::Request->instance;

    # Retrieve groups and match with security level

    my $group_list = eval { $R->group->fetch_group({ order => 'name' }) };
    if ( $@ ) {
        OpenInteract::Error->set( SPOPS::Error->get );
        $R->throw( { code => 403 } );
        $group_list = [];
    }

    my @s = ();
    foreach my $group ( @{ $group_list } ) {
        my $gid = $group->{group_id};
        my $level = $security->{ SEC_SCOPE_GROUP() }->{ $gid };
        push @s, { scope    => SEC_SCOPE_GROUP,
                   scope_id => $gid,
                   name     => $group->{name},
                   level    => $level };
    }
    return @s;
}


sub _get_user_scopes {
    my ( $class, $security ) = @_;
    my $R = OpenInteract::Request->instance;

    my $user_list = eval { $R->user->fetch_group({ order => 'login_name' }) };
    if ( $@ ) {
        OpenInteract::Error->set( SPOPS::Error->get );
        $R->throw({ code => 403 });
        $user_list = [];
    }

    my ( @s );
    foreach my $user ( @{ $user_list } ) {
        my $uid = $user->{user_id};
        my $level = $security->{ SEC_SCOPE_USER() }->{ $uid };
        push @s, { scope    => SEC_SCOPE_USER,
                   scope_id => $uid,
                   name     => $user->{login_name},
                   level    => $level };
    }
    return @s;
}


# Get the object class and object ID either from the subroutine
# parameters, object passed in, or GET/POST parameters

sub _find_object_info {
    my ( $class, $p ) = @_;
    my ( $object_class, $object_id );
    if ( ref $p->{object} ) {
        $object_class = ref $p->{object};
        $object_id    = $p->{object}->id;
    }
    else {
        my $apr = OpenInteract::Request->instance->apache;
        $object_class = $p->{object_class} ||
                        $apr->param( 'object_class' ) ||
                        $p->{handler_class} ||
                        $apr->param( 'handler_class' );
        $object_id    = $p->{object_id} ||
                        $apr->param( 'oid' ) ||
                        $apr->param( 'object_id' ) ||
                        '0';
    }
    return ( $object_class, $object_id );
}


# Get the title of an object given an object or an object class and ID

sub _fetch_description {
    my ( $class, $p ) = @_;
    my $object = $p->{object};
    if ( ! $object and ! $p->{object_class} and ! $p->{object_id} ) {
        return ( 'n/a', 'n/a', undef );
    }
    my ( $name );
    unless ( $object ) {
        unless ( $p->{object_class}->isa( 'SPOPS' ) ) {
            return ( 'Handler', undef, undef );
        }
        $name = $p->{object_class}->CONFIG->{object_name} || 'unknown';
        $object = eval { $p->{object_class}->fetch( $p->{object_id} ) };
        return ( $name, undef, undef ) if ( $@ or ! $object );
    }
    my $oi = $object->object_description;
    return ( $oi->{name}, $oi->{title}, $oi->{url} );
}


# Message for when a class isn't derived from SPOPS::Secure

sub _cannot_secure_msg {
    my ( $object_class ) = @_;
    return <<CANNOTSECURE;
<h2>Object Not Securable</h2>
<p>This class ($object_class) is not currently a subclass of the
Security implementation, meaning it is not under security
protection. You can edit the security and the entries will be put into
the security table, but they will not have any effect until you put
<tt>SPOPS::Secure</tt> in the 'isa' of the class.</p>
CANNOTSECURE
}

# Deprecated methods

sub simple_show { deprecate_me( 'simple_show', 'show' ); return show( @_ ) }
sub simple_edit { deprecate_me( 'simple_edit', 'edit' ); return edit( @_ ) }


sub deprecate_me {
    my ( $old, $new ) = @_;
    warn "the method ", __PACKAGE__, "::$old has been deprecated.",
         "Please change your code to use ", __PACKAGE__, "::$new.\n";
}

1;

__END__

=pod

=head1 NAME

OpenInteract::Handler::Security - Process changes to security made by users

=head1 SYNOPSIS

 # List the object and handler classes
 /Security/listing/

 # Display security settings for a particular object
 /Security/show/?object_id=13;object_class=MySite::Contact

=head1 DESCRIPTION

Handler to display and process the results of object-level security
setting.

=head1 METHODS

B<show>

Feeds the widget that allows users to edit security on a single object
or item.

B<hierarchy_show>

Feeds the widget that displays the parents of a particular object and
whether each one has security currently defined or not.

B<edit>

Processes the results of the 'show' page.

=head1 NOTES

B<Deprecated: simple_*>

The C<simple_show> and C<simple_edit> methods are now deprecated. Use
C<show> and C<edit> instead.

=head1 TO DO

=head1 BUGS

=head1 COPYRIGHT

Copyright (c) 2001-2002 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut
