/* $Id: viewwld.c,v 1.7 89/09/20 18:27:43 mbp Exp $
 *
 * viewwld.c: display a three-dimensional "world" of vectors
 */

/************************************************************************
 *		Copyright (C) 1989 by Mark B. Phillips                  *
 * 									*
 * Permission to use, copy, modify, and distribute this software and    *
 * its documentation for any purpose and without fee is hereby granted, *
 * provided that the above copyright notice appear in all copies and    *
 * that both that copyright notice and this permission notice appear in *
 * supporting documentation, and that the name of Mark B. Phillips or   *
 * the University of Maryland not be used in advertising or publicity   *
 * pertaining to distribution of the software without specific, written *
 * prior permission.  This software is provided "as is" without express *
 * or implied warranty.                                                 *
 ************************************************************************/

/* This file must be compiled with -DLGD_HEADER=<pathname of file lgd.h> */

#include	<stdio.h>
#include	<math.h>
#include	<string.h>
#include	<ctype.h>
#include	LGD_HEADER

#define	YES		1
#define	NO		0
#define	BUFLEN		120

/* input file types: */
#define	ASCII		1
#define	WORLD		2
#define	UNKNOWN		3

#define NEXTARG --argc; ++argv

#define COMMENT_CHAR	'%'

static	char	*copyright = "Copyright (C) 1989 by Mark Phillips";
static	char	*rcs_revision = "$Revision: 1.7 $"; /* RCS puts rev # in */
static	char	*revision_string();
static	int	xflg, yflg, zflg, lineno;
static	double	xmin,xmax,ymin,ymax,zmin,zmax;
static	int	ifiletype=UNKNOWN;
static	char	*ifile=NULL, *ofile=NULL;
static	int	box_segment=1, vector_segment=2;
static	int	toggle_box(), save_world(), load_world(),
		  read_vector_file();

static lgd_Opt_pair main_menu_list[] = {
  "Toggle Box on/off",				toggle_box,
  "Save world (with current view) in file",	save_world,
  "Load a new world file",			load_world,
  "Read a new vector file",			read_vector_file
  };
static lgd_Menu main_menu = {
  "viewwld",
  sizeof( main_menu_list) / sizeof( lgd_Opt_pair ),
  main_menu_list
  };

#ifdef DEVICE_SUNVIEW
#include        <suntool/sunview.h>
static short icon_data[] = {
#include "viewwld.icon"
};
DEFINE_ICON_FROM_IMAGE(viewwld_icon, icon_data);
#include	GR_HEADER
#endif DEVICE_SUNVIEW

main(argc,argv)
     int argc;
     char *argv[];
{
  /* parse command line */
  if (!parse_args(argc, argv)) {
    fprintf(stderr,
      "usage: viewwld [-o ofile [-x x1 x2] [-y y1 y2] [-z z1 z2] ] file\n");
    exit(1);
  }

  initialize();

  switch (ifiletype=filetype(ifile)) {
  case ASCII:
    makewld(ifile);
    break;
  case WORLD:
    if (xflg) fprintf(stderr, "viewwld: ignoring -x option\n");
    if (yflg) fprintf(stderr, "viewwld: ignoring -y option\n");
    if (zflg) fprintf(stderr, "viewwld: ignoring -z option\n");
    lgd_load_world( ifile);
    if (lgd_error) {
      fprintf(stderr, "viewwld: %s\n", lgd_error);
      lgd_error = NULL;
    }
    break;
  default:
    /* error */
    exit(1);
  }

  if (ofile != NULL) {
    lgd_save_world( ofile );
    if (lgd_error) {
      fprintf(stderr, "viewwld: %s\n", lgd_error);
      lgd_error = NULL;
    }
  }

  lgd_error = NULL;
  lgd_main_loop();

  exit(0);

}

/*-----------------------------------------------------------------------
 * Function:	save_world
 * Description:	save the current picture in a world file
 * Notes:	called by menu
 */
static
  save_world()
{
  char fname[BUFLEN];
  char msg[256];

  lgd_put_string("Enter name of file to save world in:");
  lgd_get_string(fname);
  lgd_save_world( fname );
  if (lgd_error==NULL) {
    sprintf(msg, "World saved in file '%s'", fname);
    lgd_put_string( msg );
  }
  else {
    sprintf(msg, "Error: %s", lgd_error);
    lgd_put_string(msg);
    lgd_error = NULL;
  }
}

/*-----------------------------------------------------------------------
 * Function:	load_world
 * Description:	load a world file
 * Notes:	called by menu
 */
static
  load_world()
{
  char fname[BUFLEN];
  char msg[256];

  lgd_put_string("Enter name of world file to load:");
  lgd_get_string(fname);
  lgd_load_world( fname );
  if (lgd_error==NULL) {
    lgd_update_display();
    sprintf(msg, "World loaded from file '%s'", fname);
    lgd_put_string( msg );
  }
  else {
    sprintf(msg, "Error: %s", lgd_error);
    lgd_put_string(msg);
    lgd_error = NULL;
  }
}

/*-----------------------------------------------------------------------
 * Function:	read_vector_file
 * Description:	read a vector file, replacing current picture by
 *		  contents of file
 * Notes:	called by menu
 */
static
  read_vector_file()
{
  char fname[BUFLEN], msg[BUFLEN];

  sprintf(msg, "Enter vector file name (\"%s\"):",
	  ifiletype==ASCII ? ifile : "");
  lgd_put_string(msg);
  lgd_get_string(fname);
  if (ifiletype==ASCII && fname[0]=='\0')
    strcpy(fname, ifile);
  lgd_delete_all();
  makewld(fname);
  lgd_update_display();
}

/*-----------------------------------------------------------------------
 * Function:	toggle_box
 * Description:	toggle the visibility of the bounding box
 * Notes:	called by menu
 */
static
  toggle_box()
{
  int visibility;

  lgd_inquire_segment_visibility( box_segment, &visibility );
  lgd_set_segment_visibility( box_segment, !visibility );
  lgd_update_display();
  if (visibility==YES)
    lgd_put_string("Box turned off");
  else
    lgd_put_string("Box turned on");
}

/*-----------------------------------------------------------------------
 * Function:	makewld
 * Description:	read a vector file, creating a new picture
 * Args  IN:	vectfile: name of vector file to read
 * Returns:	1 if successful, 0 if not
 * Notes:	Does not display anything; just does the computations
 *		and makes the LGD calls to create the world.
 */
static
  makewld(vectfile)
char *vectfile;
{
  FILE *fp;
  double vec[3];
  double wbox_low[3], wbox_high[3];
  int mvdr;

#define	x	vec[0]
#define y	vec[1]
#define	z	vec[2]

  fp = fopen( vectfile, "r" );
  if (fp==NULL) {
    fprintf(stderr,"viewwld: can't open file '%s'\n", vectfile );
    return(0);
  }
  lineno = 0;

  /* If any of the -x, -y, -z options were missing,
     read thru file once to determine world size	*/
  if ( (!xflg) || (!yflg) || (!zflg) ) {
    /* Read first line separately to initialize extreme vals */
    if (readline(fp, vec, &mvdr)) {
      if (!xflg) xmin = xmax = x;
      if (!yflg) ymin = ymax = y;
      if (!zflg) zmin = zmax = z;
    }
    else {
      fprintf(stderr, "viewwld: can't read vectors from file %s\n",
	      vectfile);
      return(0);
    }
    /* Read remaining lines to determine data range */
    while (readline(fp, vec, &mvdr)) {
      if (!xflg) {
	if (x<xmin) xmin = x;
	if (x>xmax) xmax = x;
      }
      if (!yflg) {
	if (y<ymin) ymin = y;
	if (y>ymax) ymax = y;
      }
      if (!zflg) {
	if (z<zmin) zmin = z;
	if (z>zmax) zmax = z;
      }
    }
  }

  /* Define the world to be ~10% larger than the data in each direction
     for which values were not explicity given */
  wbox_low[0] =  xflg ? xmin : xmin - .05*(xmax-xmin);
  wbox_high[0] = xflg ? xmax : xmax + .05*(xmax-xmin);
  wbox_low[1]  = yflg ? ymin : ymin - .05*(ymax-ymin);
  wbox_high[1] = yflg ? ymax : ymax + .05*(ymax-ymin);
  wbox_low[2]  = zflg ? zmin : zmin - .05*(zmax-zmin);
  wbox_high[2] = zflg ? zmax : zmax + .05*(zmax-zmin);
  printf("viewwld: using coordinate system: %f <= x <= %f\n",
	 wbox_low[0], wbox_high[0]);
  printf("                                  %f <= y <= %f\n",
	 wbox_low[1], wbox_high[1]);
  printf("                                  %f <= z <= %f\n",
	 wbox_low[2], wbox_high[2]);

  lgd_define_world( 3, wbox_low, wbox_high );

  /* Put box in a segment by itself; this will be segment 1 */
  lgd_begin_segment( &box_segment );
  drawbox( wbox_low, wbox_high );
  lgd_end_segment();

  /*  Now read thru file again, putting vectors	in another segment;
   *  this will be segment 2 */
  rewind(fp);
  lineno = 0;
  lgd_begin_segment( &vector_segment );
  while (readline(fp, vec, &mvdr)) {
    switch (mvdr) {
    case 0:
      lgd_move_abs( vec );
      break;
    case 1:
      lgd_draw_abs( vec );
      break;
    case 2:
      lgd_move_abs( vec );
      lgd_point();
      break;
    default:
      break;
    }
  }

  lgd_end_segment();
  fclose(fp);
  lgd_update_display();

  return(1);

#undef	x
#undef	y
#undef	z
}

/*-----------------------------------------------------------------------
 * Function:	readline
 * Description:	read a line from a file and return the vector and
 *		  move/draw flag on that line
 * Args  IN:	fp: file to read from
 *      OUT:	vec: the vector
 *		*mvdr: the move/draw flag
 * Returns:	1 if successful, 0 if not (i.e. if eof)
 * Notes:	skips past lines containing less than 4 numbers,
 *		  printing out a warning about them.
 */
static
  readline(fp, vec, mvdr)
FILE *fp;
double vec[3];
int *mvdr;
{
  char inbuf[BUFLEN];

  while (!feof(fp)) {
    if (fgets(inbuf, BUFLEN, fp) != NULL) {
      ++lineno;
      removecomments(inbuf);
      if (!blankline(inbuf)) {
	if (parseline(inbuf, vec,mvdr)) {
	  return(1);
	}
	else {
	  fprintf(stderr, "viewwld: error parsing line %1d: %s",
		  lineno, inbuf);
	  fprintf(stderr, "  (ignoring this line and continuing)\n");
	}
      }
    }
  }
  return(0);
}

/*-----------------------------------------------------------------------
 * Function:	parseline
 * Description:	parse a line from the vector file
 * Args  IN:	inbuf: the line (a null-terminated string)
 *      OUT:	vec: the vector on the line
 *		*mvdr: the move/draw flag on the line
 * Returns:	1 if successful, 0 if not
 * Notes:	ignores anything on the line after the 4th number
 */
static
  parseline(inbuf, vec, mvdr)
char *inbuf;
double vec[3];
int *mvdr;
{
  static char *seps = " ,\t";
  char *c;

  c = strtok(inbuf, seps);
  if (c == NULL) return(0);
  vec[0] = atof(c);

  c = strtok((char*)NULL, seps);
  if (c == NULL) return(0);
  vec[1] = atof(c);

  c = strtok((char*)NULL, seps);
  if (c == NULL) return(0);
  vec[2] = atof(c);

  c = strtok((char*)NULL, seps);
  if (c == NULL) return(0);
  *mvdr= atoi(c);

  return(1);
}

/*-----------------------------------------------------------------------
 * Function:	drawbox
 * Description:	draw the bounding box
 * Args  IN:	wbox_low, wbox_high: box bounds
 * Returns:	nothing
 */
static
  drawbox( wbox_low, wbox_high)
double wbox_low[], wbox_high[];
{
  double v[3];
#define	x	v[0]
#define y	v[1]
#define	z	v[2]
#define XL	wbox_low[0]
#define	YL	wbox_low[1]
#define	ZL	wbox_low[2]
#define XH	wbox_high[0]
#define YH	wbox_high[1]
#define ZH	wbox_high[2]

  x = XL;
  y = YL;
  z = ZL; lgd_move_abs( v );
  x = XH; lgd_draw_abs( v );
  y = YH; lgd_draw_abs( v );
  x = XL; lgd_draw_abs( v );
  y = YL; lgd_draw_abs( v );
  z = ZH; lgd_draw_abs( v );
  x = XH; lgd_draw_abs( v );
  y = YH; lgd_draw_abs( v );
  x = XL; lgd_draw_abs( v );
  y = YL; lgd_draw_abs( v );
  x = XH; lgd_move_abs( v );
  z = ZL; lgd_draw_abs( v );
  y = YH; lgd_move_abs( v );
  z = ZH; lgd_draw_abs( v );
  x = XL; lgd_move_abs( v );
  z = ZL; lgd_draw_abs( v );


#undef	XL
#undef	YL
#undef	ZL
#undef	XH
#undef	YH
#undef	ZH

}

/*-----------------------------------------------------------------------
 * Function:     revision_string
 * Description:  Get the revision number (in string form)
 * Arguments:    (none)
 * Returns:      nothing
 * Notes:	 The revision number is extracted from the
 *               global variable rcs_revision, which is a
 *               string containing the RCS revision number.  This
 *               process destroys the contents of rcs_revision;
 */
static char *
  revision_string()
{
  char *s, *rev;

  /* We extract the version number by looking for the first
   * space-separated token which begins with a digit */
  s = rcs_revision;
  do {
    rev = strtok( s, " " );
    s = NULL;
  } while ( (rev!=NULL) && (!isdigit(*rev)) );
  if (rev==NULL) rev = "";
  return(rev);
}

static
  initialize()
{
  char frame_label[30];

  lgd_initialize();
  lgd_set_menu( &main_menu );

#ifdef DEVICE_SUNVIEW
  {
    extern int gr_redraw();
    char helpfile[256];

    sprintf(frame_label,"viewwld - version %s", revision_string());
    gr_set_frame_label( frame_label );
    gr_set_frame_icon( &viewwld_icon );
    gr_print_button(1, gr_redraw, "Viewwld");
    construct_help_file_name(helpfile);
    gr_set_help_file(helpfile);
  }
#endif

}

#ifdef DEVICE_SUNVIEW
static
  construct_help_file_name(fname)
char *fname;
{
  extern char *getenv();
  char *viewwld_lib;

  /* First look in VIEWWLD_LIB */
  viewwld_lib = getenv("VIEWWLD_LIB");
  if (viewwld_lib != NULL) {
    sprintf(fname,"%s%s%s",
	    viewwld_lib, (viewwld_lib[strlen(viewwld_lib)-1]=='/') ? "" : "/",
	    "viewwld.help");
    if (GR_file_openable(fname, "r")) return;
  }

  /* If not there, then use default */
  strcpy(fname, HELPFILE);
}
#endif

/*-----------------------------------------------------------------------
 * Function:	parse_args
 * Description:	parse the command-line arguments, setting various
 *		  global variables according to them
 * Args  IN:	argc, argv: from main
 * Returns:	1 if all goes well
 *		0 if error in parsing
 * Notes:	usage:
 *		  viewwld [-o ofile      \
 *		            [-x xmin xmax]   \
 *			    [-y ymin ymax]   \
 *			    [-z zmin zmax] ] \
 *			  file
 *
 *		global variables:
 *		  xflg: whether -x option is present
 *		  yflg: whether -y option is present
 *		  zflg: whether -z option is present
 *		  ofile: name of output file
 *		  ifile: name of input file (argument 'file')
 *		  xmin,xmax: x bounds for coord sys
 *		  ymin,ymax: y bounds for coord sys
 *		  zmin,zmax: z bounds for coord sys
 */
static
  parse_args(argc, argv)
int argc;
char *argv[];
{
  xflg = yflg = zflg = NO;
  ifile = ofile = NULL;
  NEXTARG;
  while (argc) {
    if (ifile != NULL) {
      fprintf(stderr, "viewwld: input file must be last argument.\n");
      return(0);
    }
    if ((*argv)[0] == '-')
      /* this arg is an option */
      switch ((*argv)[1]) {
      case 'o':
	if (argc < 2) {
	  fprintf(stderr, "viewwld: bad -o option.\n");
	  return(0);
	}
	NEXTARG; ofile = *argv;
	break;
      case 'x':
	if (argc < 3) {
	  fprintf(stderr, "viewwld: bad -x option\n");
	  return(0);
	}
	xflg = YES;
	NEXTARG; xmin = atof(*argv);
	NEXTARG; xmax = atof(*argv);
	break;
      case 'y':
	if (argc < 3) {
	  fprintf(stderr, "viewwld: bad -y option\n");
	  return(0);
	}
	yflg = YES;
	NEXTARG; ymin = atof(*argv);
	NEXTARG; ymax = atof(*argv);
	break;
      case 'z':
	if (argc < 3) {
	  fprintf(stderr, "viewwld: bad -z option\n");
	  return(0);
	}
	zflg = YES;
	NEXTARG; zmin = atof(*argv);
	NEXTARG; zmax = atof(*argv);
	break;
      default:
	fprintf(stderr, "viewwld: -%c is not a valid option\n",(*argv)[1]);
	return(0);
	break;
      }
    else {
      /* this arg is not an option; it must be the 'file' argument */
      ifile = *argv;
    }
    NEXTARG;
  }
  if (ifile == NULL) {
    fprintf(stderr, "viewwld: no input file specified\n");
    return(0);
  }
  return(1);
}

/*-----------------------------------------------------------------------
 * Function:	filetype
 * Description:	determine the type of a file from its contents
 * Args  IN:	file: name of file to examine
 * Returns:	one of:
 *		  ASCII
 *		  WORLD
 *		  UNKNOWN
 * Notes:	file opened to look at data, and then closed before
 *		returning. If file can't be opened, UNKNOWN is returned.
 */
static
  filetype(file)
char *file;
{
  FILE *fp;
  char buf[80];
  int items, ascii, byte1, type;
  /* The following must be changed to the current format number
     when updating to a new version of LGD: */
  static int format_number = 3;

  type = UNKNOWN;
  fp = fopen(file, "r");
  if (fp != NULL) {
    /* Test for ascii by examining first 80 chars of file */
    items = fread(buf, 1, 80, fp);
    ascii = YES;
    while (items && ascii)
      ascii = isascii(buf[--items]);
    if (ascii)
      type = ASCII;
    else {
      /* Test for world by checking 1st byte.  The first byte of
	 a world file is the LGD format number. */
      rewind(fp);
      if (fread( (char*)&byte1, sizeof(int), 1, fp ) == 1)
	if (byte1 == format_number) type = WORLD;
    }
  }
  fclose(fp);
  return(type);
}

/*-----------------------------------------------------------------------
 * Function:	removecomments
 * Description:	remove any comments from a line by replacing the
 *		  first comment char in the line by the null char
 * Args  IN:	line: the original line
 *      OUT:	line: the line with comments removed
 * Returns:	nothing
 * Notes:	the comment char is defined by the macro COMMENT_CHAR
 */
static
  removecomments(line)
char *line;
{
  while (*line != '\0' && *line != COMMENT_CHAR) ++line;
  if (*line == COMMENT_CHAR) *line = '\0';
}

/*-----------------------------------------------------------------------
 * Function:	blankline
 * Description:	determine whether a line consists of just blanks
 * Args  IN:	line: the line to inspect
 * Returns:	1 if it's all whitespace (as defined by isspace(3)),
 *		0 otherwise
 */
static
  blankline(line)
char *line;
{
  while (*line != '\0')
    if (!isspace(*(line++))) return(NO);
  return(YES);
}
