/*
Initialize all parameters for a given dynamical system to be installed
Parameters are assigned to the default values before this program is called.
-----------------------------------------------------------------------
This is a GENERIC subroutine. The string "generic" should be
GLOBALLY substituted with the proper name string for a given dynamical system
to be installed.
*/

/*
Example 1: 4-d vector field with polar coords and no periodic variable
*/
int generic_init()
{
	/* title label */
	title_label = "Undefined GENERIC case";

	/* mapping toggle: 1=map, 0=vector field */
	mapping_on = 0;
	/* inverse toggle: vector field: always 1,
	   maps: 0=explicit inverse is defined, 1=otherwise  */
	inverse_on = 1;
	/* jacobian toggle: 1=Jacobian is explicitly given, 0=otherwise */
	fderiv_on = 0;
	/* polar toggle: 1=enable polar coordinate feature, 0=otherwise */
	enable_polar = 1;
	/* period toggle: 1=enable periodicity of phase space, 0=otherwise */
	enable_period = 0;

	/* phase space dimension */
	var_dim = 4;
	/* parameter space dimension */
	param_dim = 2;
	/* function space dimension */
	func_dim = 2;

	(void) malloc_init();

	/*periodicity of phase space variables (DIM=var_dim)*/
	/* do not have to be defined if enable_period = 0 */
	period_len[0] =0;
	period_len[1] =0;
	period_len[2] =0;
	period_len[3] =0;

	/* primary phase space variable label (DIM=var_dim)*/
	var_label[0] = "x";
	var_label[1] = "y";
	var_label[2] = "z";
	var_label[3] = "w";
	/* secondary phase space variable label (usually polar coord variable)
	(DIM=var_dim) */
	/* do not have to be defined if enable_polar = 0 */
	var_polar_label[0] = "r";
	var_polar_label[1] = "rp";
	var_polar_label[2] = "theta";
	var_polar_label[3] = "thetap";
	/* parameter variable label (DIM=param_dim)*/
	param_label[0] = "mu";
	param_label[1] = "delta";
	/* function variable label (DIM=func_dim)*/
	func_label[0] = "Energy";
	func_label[1] = "AngMom";

	/* starting parameter values (DIM=param_dim)*/
	param[0] = 2;
	param[1] = 1;

	/* starting primary phase space variable values (DIM=param_dim)*/
	var_i[0] = 0;
	var_i[1] = 0;
	var_i[2] = 0;
	var_i[3] = 0;
	/* starting seconsary phase space variable values (DIM=param_dim)*/
	/* do not have to be defined if enable_polar = 0 */
	var_polar_i[0] = 0;
	var_polar_i[1] = 0;
	var_polar_i[2] = 0;
	var_polar_i[3] = 0;

	/* starting bounds of parameter window box */
	param_min[0]= -5; param_max[0]= 5;
	param_min[1]= -5; param_max[1]= 5;

	/* starting bounds of primary phase space window box */
	var_min[0]= -5; var_max[0]= 5;
	var_min[1]= -5; var_max[1]= 5;
	var_min[2]= -5; var_max[2]= 5;
	var_min[3]= -5; var_max[3]= 5;

	/* starting bounds of secondary phase space window box */
	/* do not have to be defined if enable_polar = 0 */
	var_polar_min[0]= -5; var_polar_max[0]= 5;
	var_polar_min[1]= -5; var_polar_max[1]= 5;
	var_polar_min[2]= -pi; var_polar_max[2]= pi;
	var_polar_min[3]= -5; var_polar_max[3]= 5;

	/* dynamical system and function pointer assignments */
	f_p = generic_f;
	func_p = generic_func;
}

/*
Example 2: 2-d torus map with periodic variables and no polar coords  
*/
int userds0_init()
{
	extern int (*f_p)(),userds0_f(),(*func_p)(),userds0_func();
	extern int var_dim,func_dim,param_dim;
	extern int mapping_on,inverse_on,fderiv_on,enable_polar,enable_period;
	extern double *param,*param_min,*param_max,*func,*func_min,*func_max;
	extern double *var_i,*var_polar_i,*var_min,*var_max,*var_polar_min,*var_polar_max;
	extern double *period_len,pi;
	extern char *title_label,**var_label,**var_polar_label,**param_label,**func_label;

	/* title label */
	title_label = "Undefined GENERIC case";

	/* mapping toggle: 1=map 0=vector field */
	mapping_on = 1;
	/* inverse toggle: vector field: always 1,
	   maps: 0=explicit inverse is defined, 1=otherwise  */
	inverse_on = 0;
	/* jacobian toggle: 1=Jacobian is explicitly given, 0=otherwise */
	fderiv_on = 0;
	/* polar toggle: 1=enable polar coordinate feature, 0=otherwise */
	enable_polar = 0;
	/* period toggle: 1=enable periodicity of phase space, 0=otherwise */
	enable_period = 1;

	/* phase space dimension */
	var_dim = 2;
	/* parameter space dimension */
	param_dim = 2;

	/*periodicity of phase space variables (DIM=var_dim)*/
	/* do not have to be defined if enable_period = 0 */
	period_len[0] =1;
	period_len[1] =1;

	/* primary phase space variable label (DIM=var_dim)*/
	var_label[0] = "x";
	var_label[1] = "y";
	/* seconsary phase space variable label (DIM=param_dim)*/
	/* do not have to be defined if enable_polar = 0 */
	
	/* parameter variable label (DIM=param_dim)*/
	param_label[0] = "a";
	param_label[1] = "b";
	/* function variable label (DIM=func_dim)*/
	func_label[0] = "F1";
	func_label[1] = "F2";

	/* starting parameter values (DIM=param_dim)*/
	param[0] = 2;
	param[1] = 1;

	/* starting primary phase space variable values (DIM=param_dim)*/
	var_i[0] = 0;
	var_i[1] = 0;
	/* starting seconsary phase space variable values (DIM=param_dim)*/
	/* do not have to be defined if enable_polar = 0 */

	/* starting bounds of parameter window box */
	param_min[0]= -5; param_max[0]= 5;
	param_min[1]= -5; param_max[1]= 5;

	/* starting bounds of primary phase space window box */
	var_min[0]= 0; var_max[0]= 1;
	var_min[1]= 0; var_max[1]= 1;

	/* starting bounds of secondary phase space window box */
	/* do not have to be defined if enable_polar = 0 */

	/* dynamical system and function pointer assignments */
	f_p = userds0_f;
	func_p = userds0_func;
}
/*
-----------------------------------------------------------------
user definition of an ordinary differential equation or a mapping
-----------------------------------------------------------------
FILES NEED TO CHANGED: generic_func.c
	(optional) generic_choose_algorithm.c
-----------------------------------------------------------------
Input:	x[n]: coords
	p[n]: parameters
	t: time (used for ode only)
	dim: dimension of coords
	index:	(ode) 0: all equations
		(Hamiltonian ode only) 1: momentum part, 2: coord part
		(map) 1: forward, 0: backward
Output:	f[n]: (ode) values of vector fields, (map) an image
	(values of the right hand side of dynamical system equation)
-----------------------------------------------------------------
Useful system parameters which have been already declared and can be used here:
-----------------------------------------------------------------
	(double) pi:	Pi
	(double) twopi:	2 * Pi
-----------------------------------------------------------------
Useful system parameters which can be externally declared and used here:
-----------------------------------------------------------------
	(int) var_dim: dimension of the phase space
	(int) param_dim: dimension of the system parameters
	(int) model_dim: dimension of a list of all models installed
	(int) model: currently selected model
	(int) cur_color: currently selected color (sometimes overriden)
	(int) symbol_type: currently selected symbol (sometimes overriden)
	(int) symbol_size: size of an symbol (sometimes overriden)
-----------------------------------------------------------------
*/
	
/*
Example 1: mapping without an explicit inverse
*/
int generic_f(f,index,x,p,t,dim)
int index,dim;
double f[],x[],p[],t;
{
	f[0] = p[0] * ( x[1] - x[0] );
	f[1] = p[1] * x[0] - x[1] - x[0] * x[2];
	f[2] = - p[2] * x[2] + x[0] * x[1];
}

/*
Example 2: ODE (non-Hamiltonian)
Note that there is an explicit dependence on time t
and sin and cos are declared as double
*/
int generic_f(f,index,x,p,t,dim)
int index,dim;
double f[],x[],p[],t;
{
	f[0] = x[1];
	f[1] = -p[2] * x[1] - (p[0] * p[0] + p[1] * cos(t) * sin(x[0]);
}

/*
Example 3: mapping with an explicit inverse
*/
int generic_f(f,index,x,p,t,dim)
int index,dim;
double f[],x[],p[],t;
{
	if(index ==1) {
		/* forward mapping */
		f[0] = 1. + x[1] - p[0] * x[0] * x[0];
		f[1] = p[1] * x[0]; 
	}
	else if(index == 0){
		/* backward mapping */
		f[0] = 1. / p[1] * x[1];
		f[1] = -1. + x[0] + p[0] / p[1] /p[1] * x[1] * x[1]; 
	}
}

/*
Example 4: Hamiltonian vector field (when symplectic integration is not used)
Note: Often introduction of new variables reduces the number of
multiplcations, and therefore computation time
*/
int generic_f(f,index,x,p,t,dim)
int index,dim;
double f[],x[],p[],t;
{
	double v0sq,v2sq;

	v0sq = x[0] * x[0];
	v2sq = x[2] * x[2];
	f[0] = x[1];
	f[2] = x[3];
	f[1] = x[0] * (p[0] - (v0sq + v2sq)) + p[1] * x[0] * v2sq;
	f[3] = x[2] * (p[0] - (v0sq + v2sq)) + p[1] * x[2] * v0sq; 
}

/*
Example 5: Hamiltonian vector field (when symplectic integration is used)
it should be divided into two sets of equations,
momentum part: dx/dt = dH/dp, coordinate part: dp/dt = -dH/dx
*/
int generic_f(f,index,x,p,t,dim)
int index,dim;
double f[],x[],p[],t;
{
	double v0sq,v2sq;

	if(index !=2) {
		/* momenturm part dH/dp */
		f[0] = x[1];
		f[2] = x[3];
	}
	if(index !=1){
		/* coordinate part -dH/dx */
		v0sq = x[0] * x[0];
		v2sq = x[2] * x[2];
		f[1] = x[0] * (p[0] - (v0sq + v2sq)) + p[1] * x[0] * v2sq;
		f[3] = x[2] * (p[0] - (v0sq + v2sq)) + p[1] * x[2] * v0sq; 
	}
}

/*
Example 6: Mapping without an explicit inverse (use of pi,twopi)
*/
int generic_f(f,index,x,p,t,dim)
int index,dim;
double f[],x[],p[],t;
{
	/* forward map */
	f[0] = x[0] + p[0] - p[2] * p[3] / twopi * sin(twopi * x[1]);
	f[1] = x[1] + p[1] - p[2] / p[3] / twopi * sin(twopi * x[0]);
}
/*
Definitions of functions
Use:	(1)fuctions can be regarded as parts of generalized coordinates,
	allowing them to be manipulated as other coordinates.
	For example, one can plot energy versus angular momentum.
	(2)values of functions can be monitored and shown on panels 
	(3)functions can be used to define a general surface for a Poincare
	section.
	(4)functions can be used to define coordinate transformations
	such as polar coordinates (this feature is duplicated by a set
	of predefined coordinate transformation subroutines)
	(5)functions can be colorcoded according to their values 
Note:	functions should be defined in terms of primary coordinates,
	the coordinate system with which you define the dynamical system
*/

/*
Example 1: no definition
*/
int generic_func(f,x,p,t,dim)
double f[],x[],t,p[];
int dim;
{
}

/*
Example 2: Energy and angular momentum
*/
int generic_func(f,x,p,t,dim)
double f[],x[],p[],t;
int dim;
{
        double v0sq,v1sq,v2sq,v3sq;

	coordinates */
	v0sq = x[0] * x[0];
	v2sq = x[2] * x[2];
	f[0] = 0.5 * (( x[1] * x[1] + x[3] * x[3]) - p[0]*(v0sq+v2sq) + 0.5 *
	(v0sq+v2sq)*(v0sq+v2sq) - p[1] * v0sq* v2sq);
	f[1] =  x[1] * x[2] - x[0] * x[3];
}

/*
Example 3: in polar coordinates
*/
int func_1(f,x,p,t,dim)
double f[],x[],t,p[];
int dim;
{
	double v0sq,v1sq,v2sq,v3sq,sin(),cos(),vsin,vcos;
	extern double *v;

	/* polar coordinate to euclidean coordinate transformation */
	euclid_to_polar(v,x);
	
	v0sq = v[0] * v[0];
	v1sq = v[1] * v[1];
	v2sq = v[2] * v[2];
	v3sq = v[3] * v[3];
	vsin = sin(v[2]);
	vcos = cos(v[2]);

	/* functions are defined in terms of polar coordinates */
	f[0] = 0.5 * ((v1sq + v0sq * v3sq) - p[0]*v0sq + 0.5 * v0sq * v0sq - p[1] * v0sq * v0sq * vsin * vsin * vcos * vcos);
	f[1] =  v0sq * v[3];
}

/*
Example 4: more sophistigated use
*/
int generic_func(f,x,p,t,dim)
double f[],x[],t,p[];
int dim;
{
	int i;
	extern int forward_toggle;
	extern double *v;
	extern int (*f_p)();
	
	(int) f_p(v,forward_toggle,x,p,t,dim);
	f[0] = 0;
	for(i=0;i<dim;i++)
		f[0] += (v[i]-x[i])*(v[i]-x[i]);
}

/*
Example 5: using time as a function (useful for a conventional time series plot)
*/
int generic_func(f,x,p,t,dim)
double f[],x[],t,p[];
int dim;
{
	f[0] = t;
}

/*
Example 6: motinor a modulus
*/
int generic_func(f,x,p,t,dim)
double f[],x[],t,p[];
int dim;
{
	f[0] = x[0]*x[0] + x[1]*x[1];
}

/*
Example 7: partial polar coordinate transformation 
NOTE: By default, some coord transformations are supported by the system.
*/
int generic_func(f,x,p,t,dim)
double f[],x[],t,p[];
int dim;
{
	f[0] = x[0]*x[0] + x[1]*x[1];
	if(x[0] != 0)
		f[1] = atan(x[1]/x[0]);
	else
		f[1] = twopi/4.;
}
