/***************************************
  $Revision: 1.21 $

  Authentication utilities 

  Status: NOT REVIEWED, TESTED

  Author(s):      Engin Gunduz 

  ******************/ /******************
  Modification History:
        engin (05/04/2000) Created.
  ******************/ /******************
  Copyright (c) 2000,2001,2002                    RIPE NCC
 
  All Rights Reserved
  
  Permission to use, copy, modify, and distribute this software and its
  documentation for any purpose and without fee is hereby granted,
  provided that the above copyright notice appear in all copies and that
  both that copyright notice and this permission notice appear in
  supporting documentation, and that the name of the author not be
  used in advertising or publicity pertaining to distribution of the
  software without specific, written prior permission.
  
  THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS; IN NO EVENT SHALL
  AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY
  DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
  OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 ***************************************/

#include <md5.h>
#include "AU_util.h"
#include "dbupdate.h"
#include "crypt.h"

extern int tracing;
extern char *authmethods[];
extern char *crypt(const char *key, const char *salt);

/* AU_crypt is a wrapper around crypt(3) */
char * AU_crypt(const char *key, const char *setting){

  return crypt(key, setting);

}

/* takes a list of passwords and a crypted password. If any
   of the passwords in the list is the plaintext of crypted
   text, then it immediately returns 1. Otherwise, it returns
   0 */
int au_check_password(char * crypted_password, GSList * password_list)
{
  GSList * password_item = NULL;

  if (tracing )
  {
    printf("TRACING: au_check_password is running" );
  }

  for (password_item = password_list; password_item != NULL; password_item = g_slist_next(password_item ))
  {
    /* if the password is correct, return 1 */
    if (strcmp(crypt((char *)password_item->data, crypted_password), crypted_password) == 0)
    {
      if (tracing )
      {
        printf("TRACING: au_check_password returning 1\n");
      }
      return(1);  /* auth OK */
    }
  }
  /* we couldn't find any correct password. So, return 0 */
  if (tracing )
    printf("TRACING: au_check_password returning 0\n");  
  return(0);      /* auth failed */
}



/* takes a list of passwords and a MD5 password. If any
   of the passwords in the list is the plaintext of MD5 
   digest, then it immediately returns 1. Otherwise, it returns
   0 */
int au_check_MD5_password(char * MD5_password, GSList * password_list)
{
  MD5_CTX md5ctx;
  unsigned char md5_value[16];
  int i;
  GSList * password_item = NULL;

  for (password_item = password_list; password_item != NULL; password_item = g_slist_next(password_item))
  {
    /* if the password is correct, return 1 */

    /* check MD5 password */
    MD5Init(&md5ctx);
    MD5Update(&md5ctx, (unsigned char *)password_item->data, (unsigned int)strlen((char *)password_item->data));
    MD5Final(md5_value, &md5ctx);
    
    /* md5_calc((unsigned char *)md5_value, (unsigned char *)password_item->data, (unsigned int)strlen(password_item->data)); */

    if (tracing )
    {
      printf("TRACING: Digest is: ");
      for (i = 0; i < 32; i++) printf("%02x", md5_value[i]);
      printf("\n");
    }

    if ( strncmp((char *)md5_value, MD5_password, 32) == 0)
    {
      if (tracing )
        printf("TRACING: au_check_MD5_password returning 1\n");
      return(1);  /* auth OK */
    }
  }
  /* we couldn't find any correct password. So, return 0 */
  if (tracing )
    printf("TRACING: au_check_MD5_password returning 0\n");
  return(0);    /* auth failed */
}



/* takes a list of plain text passwords and an MD5 crypt password. If any
   of the passwords in the list is the plaintext of the crypt MD5 
   then it immediately returns 1. Otherwise, it returns 0 */

int au_check_crypt_MD5_password(char * MD5_password, GSList * password_list)
{
  int i;
  char *pw, *cryptpw;
  char *passwd;
  char salt[9];
  GSList * password_item = NULL;

  if (tracing )
  {
    printf("TRACING: au_check_crypt_MD5_password is running" );
  }

  /* find salt in MD5_password ($1$salt$password) */
  passwd = strdup(MD5_password);
  pw = strtok(passwd,"$");  /* points to start of string */
  pw = strtok(NULL,"$");  /* points to start of salt */
  if ( pw )
  {
    strncpy(salt, pw, 8);
    salt[8] = '\0';
  }
  else
    salt[0] = '\0';
  free(passwd);
  
  for (password_item = password_list; password_item != NULL; password_item = g_slist_next(password_item))
  {
    /* encrypt plain text password using the salt from the crypted password*/
    cryptpw = crypt_md5((const char *)password_item->data, (const char *)salt );

    if (tracing )
    {
      printf("TRACING: cryptpw is: [%s]\n", cryptpw );
    }

    if ( strcmp((char *)cryptpw, (char *)MD5_password) == 0)
    {
      if (tracing )
        printf("TRACING: au_check_crypt_MD5_password returning 1\n");

      return(1);  /* auth OK */
    }
  }
  /* we couldn't find any correct password. So, return 0 */
  if (tracing )
    printf("TRACING: au_check_crypt_MD5_password returning 0\n");

  return(0);    /* auth failed */
}



/* simply compares auth_pgpkeyID & mesg_pgpkeyID and
   returns 1 if they are the same. */
int au_check_PGPkey(char * auth_pgpkeyID, /*char * mesg_pgpkeyID*/GSList * mesg_pgpkeyIDs){

  GSList * next = NULL;

  for(next = mesg_pgpkeyIDs; next != NULL; next = g_slist_next(next)){
    /* if auth_pgpkeyID & mesg_pgpkeyID are the same, return 1 */
    if(strcmp(auth_pgpkeyID, (char *)next->data) == 0){
      return(1);
    }
  }
  /* If we reached here, we couldn't find a matching keyID, so return 0 */
  return(0);
}



/* Compares the 'From' address of the message to the regular
   expression in the 'auth' attribute of the maintainer*/
int au_check_from_address(char * regexp, char * from_address){
       
   int status;
   regex_t re;

   if(from_address == NULL){
     return(0);
   }
   if (regcomp(&re, regexp, REG_EXTENDED|REG_NOSUB|REG_ICASE) != 0) {
     //printf("DEBUG: au_check_from_address  returns 0 (couldn't compile)\n");
     return(0);      /* couldn't compile the regexp, return false */
   }
   
   status = regexec(&re, from_address, (size_t) 0, NULL, 0);
   regfree(&re);
   if (status != 0) {
     //printf("DEBUG: au_check_from_address returns 0 (regexp doesn't match)\n\t[regexp:%s][from:%s]\n",
     //       regexp, from_address);
     return(0);      /* failed */
   }
   /* OK, the regexp matches */
   //printf("DEBUG: au_check_from_address returns 1\n");
   return(1);
}


int au_is_valid_authmethod( int type )
{
  char *authtype = NULL;
  int i;
  
  switch (type)
  {
    case AU_NONE:      authtype = strdup("NONE");
                       break;
    case AU_MAIL_FROM: authtype = strdup("MAIL-FROM");
                       break;
    case AU_CRYPT_PW:  authtype = strdup("CRYPT-PW");
                       break;
    case AU_MD5_PW:    authtype = strdup("MD5-PW");
                       break;
    case AU_PGP:       authtype = strdup("PGPKEY");
                       break;
    default:           authtype = strdup("");
                       break;
  }
  
  i=0;
  while ( authmethods[i] != NULL )
  {
    /* check if the authtype from the mntners is in the list of valid authmethods */
    if ( ! strcmp(authmethods[i++], authtype) )
    {
      free(authtype);
      return 1;
    }
  }
  
  /* authtype is not a valid authmethod */
  return 0;
}




/* Gets a auth_vector, and  credentials_struct (which is extracted
   from the update message) and returns 0 if all of the auth
   methods fail, and returns the index of the succeeding auth_struct in the auth_vector 
   if any one of them succeeds. */
int AU_authorise(GSList * auth_vector, credentials_struct credentials)
{
  GSList *auth_item = NULL;
  auth_struct * temp = NULL;
  int result = 0;

  /* if the linked list contains no members, then return 1*/
  if (g_slist_length(auth_vector) == 0)
  {
    return(1);
  }

  for (auth_item = auth_vector; auth_item != NULL; auth_item = g_slist_next(auth_item))
  {
    temp = (auth_struct *)auth_item->data;
    if ( temp != NULL )
    {
      if ( au_is_valid_authmethod(temp->type) )
      {
        switch (temp->type)
        {
          case AU_NONE:      return temp->index; /* NONE, immediately returns true */
                             break;
          case AU_MAIL_FROM: 
                             if (au_check_from_address(temp->auth, credentials.from))
                             {
                               result = temp->index;
                             }
                             break;
          case AU_CRYPT_PW:  if (au_check_password(temp->auth, credentials.password_list))
                             { 
                               result = temp->index;
                             }
                             break;
          case AU_MD5_PW:    if (au_check_crypt_MD5_password(temp->auth, credentials.password_list))
                             { 
                               result = temp->index;
                             }
                             break;
          case AU_PGP:       //printf("DEBUG: AU_authorise: will call au_check_PGPkey\n");
                             //printf("DEBUG: AU_authorise:   with temp->auth=[%s]\n", temp->auth);
                             //printf("DEBUG: AU_authorise:   and credentials.pgp_struct=[%s]\n", credentials.pgp_struct);
                             if (au_check_PGPkey(temp->auth, credentials.pgp_key_list))
                             {
                               result = temp->index;
                             }
                             break;
          default:           ;/* this mustn't happen */
                             break;
        }
      }
      if (result > 0)
      {
        return(result);
      }
    }
  }


  return 0;
}
