// Converted to Java from:

/*      idea.c - C source code for IDEA block cipher.
 *      IDEA (International Data Encryption Algorithm), formerly known as 
 *      IPES (Improved Proposed Encryption Standard).
 *      Algorithm developed by Xuejia Lai and James L. Massey, of ETH Zurich.
 *      This implementation modified and derived from original C code 
 *      developed by Xuejia Lai.  
 *      Zero-based indexing added, names changed from IPES to IDEA.
 *      CFB functions added.  Random number routines added.
 *
 *  Optimized for speed 21 Oct 92 by Colin Plumb.
 *  Very minor speedup on 23 Feb 93 by Colin Plumb.
 *  idearand() given a separate expanded key on 25 Feb 93, Colin Plumb.
 *
 *      The IDEA(tm) block cipher is covered by a patent held by ETH and a
 *      Swiss company called Ascom-Tech AG.  The Swiss patent number is
 *      PCT/CH91/00117.  International patents are pending. IDEA(tm) is a
 *      trademark of Ascom-Tech AG.  There is no license fee required for
 *      noncommercial use.  Commercial users may obtain licensing details
 *      from Dieter Profos, Ascom Tech AG, Solothurn Lab, Postfach 151, 4502
 *      Solothurn, Switzerland, Tel +41 65 242885, Fax +41 65 235761.
 *
 *      The IDEA block cipher uses a 64-bit block size, and a 128-bit key 
 *      size.  It breaks the 64-bit cipher block into four 16-bit words
 *      because all of the primitive inner operations are done with 16-bit 
 *      arithmetic.  It likewise breaks the 128-bit cipher key into eight 
 *      16-bit words.
 *
 *      For further information on the IDEA cipher, see these papers:
 *      1) Xuejia Lai, "Detailed Description and a Software Implementation of 
 *         the IPES Cipher", Institute for Signal and Information
 *         Processing, ETH-Zentrum, Zurich, Switzerland, 1991
 *      2) Xuejia Lai, James L. Massey, Sean Murphy, "Markov Ciphers and 
 *         Differential Cryptanalysis", Advances in Cryptology- EUROCRYPT'91
 *
 *      This code assumes that each pair of 8-bit bytes comprising a 16-bit 
 *      word in the key and in the cipher block are externally represented 
 *      with the Most Significant Byte (MSB) first, regardless of the
 *      internal native byte order of the target CPU.
 */

/* The internal data representation of the values, which are in the multiplicative
 * group mod the prime 2**16+1, is as the values themselves, except that 2**16
 * is represented as 0.  (0 is not in the group.)
 */

public class Idea extends BlockTransformation {
public static final int BS = 8;                 /* block size in bytes */
public static final int KS = 16;                /* key size in bytes */
    
public static final int ROUNDS = 8;            /* Don't change this value, should be 8 */
public static final int KEYLEN = 6*ROUNDS+4;   /* length of key schedule */

short Z[] = new short[KEYLEN];  /* Key */

/* For debugging */
public void dump()
{
    int i, j;
    for(j=0; j<ROUNDS+1; j++)
    {
        System.out.print("round "+(j+1)+":   ");
        if (j==ROUNDS)
            for(i=0; i<4; i++)
                System.out.print(" "+((int)Z[j*6+i]&0xffff));
        else
            for(i=0; i<6; i++)
                System.out.print(" "+((int)Z[j*6+i]&0xffff));
        System.out.print("\n");
    }
}

/*
 *      Multiplication, modulo (2**16)+1
 * Note that this code is structured like this on the assumption that
 * untaken branches are cheaper than taken branches, and the compiler
 * doesn't schedule branches.
 */
/* For the Java version, the difficulty is to do a 16x16=32 unsigned
 * multiply.  This is done by promoting the shorts to ints and then
 * anding with 0xffff.  The "b<a" test must also be done with ints to
 * properly simulate the unsigned test.
 */
static protected short mul(int a, int b)
{
        int p;

        a &= 0xffff;
        b &= 0xffff;
        if (a != 0)
        {       if (b != 0)
                {       p = a * b;
                        b = p & 0xffff;
                        a = p>>>16;
                        return (short)(b - a + ((b < a)?1:0));
                }
                else
                {       return (short)(1-a);
                }
        }
        else
        {       return (short)(1-b);
        }
}        /* mul */

/*
 *      Compute multiplicative inverse of x, modulo (2**16)+1,
 *      using Euclid's GCD algorithm.  It is unrolled twice to
 *      avoid swapping the meaning of the registers each iteration,
 *      and some subtracts of t have been changed to adds.
 */
/* For Java, ints are used throughout to avoid the problems of sign
 * extension of shorts.  However, x, q, and y will always be 16 bits
 * or less, after the initial x masking.  t0 and t1 may get bigger,
 * but we only return the low 16 bits of t1 so that won't hurt.
 */
static protected short inv(int x)     
{
    int t0, t1;
    int q, y;

    x &= 0xffff;
    if (x <= 1)
        return (short)x;        /* 0 and 1 are self-inverse */
    t1 = 0x10001 / x;           /* Since x >= 2, this fits into 16 bits */
    y = 0x10001 % x;
    if (y == 1)
        return (short)(1-t1);
    t0 = 1;
    do
    {       q = x / y;
            x = x % y;
            t0 += q * t1;
            if (x == 1)
                return (short)t0;
            q = y / x;
            y = y % x;
            t1 += q * t0;
        } while (y != 1);
    return (short)(1-t1);
}                               /* inv */

/*      Compute IDEA encryption subkeys Z */

public Idea(byte userkey[], int encdec)
{
    this(userkey, 0, encdec);
}

public Idea(byte userkey[], int off, int encdec)
{
    short uks[] = new short[KS/2];
    bytes.BArrToArrBig(userkey, off, uks, 0, KS);
    en_key_Idea(uks, 0, encdec);
}

/* userkey should be 8 shorts */
public Idea(short userkey[], int encdec)
{
    en_key_Idea(userkey,0,encdec);
}

/* userkey should be 8 shorts */
public Idea(short userkey[], int off, int encdec)
{
    en_key_Idea(userkey, off, encdec);
}
    
/* Note that right shift below can't be >>>, Z gets sign extended to
 * 16 bit int before the shift.
 */
protected void en_key_Idea(short userkey[], int off, int encdec)
{
        int i,j;

        /*
         * shifts
         */
        for (j=0; j<8; j++)
                Z[j] = userkey[j+off];

        int zoff = 0;
        for (i=0; j<KEYLEN; j++)
        {       i++;
                Z[i+7+zoff] = (short)((Z[(i & 7)+zoff] << 9) |
                                      ((Z[((i+1)&7)+zoff] >> 7)&0x1ff));
                zoff += i & 8;
                i &= 7;
        }
        if (encdec==DECRYPTION)
                decryptionmode();
}        /* en_key_idea */


/* Switch this object to decryption mode */
protected void decryptionmode()
{
        int i, j;
        short t1, t2, t3;
        short T[] = new short[KEYLEN];
        int toff = KEYLEN;
        int zoff = 0;

        t1 = inv(Z[zoff++]);
        t2 = (short)-Z[zoff++];
        t3 = (short)-Z[zoff++];
        T[--toff] = inv(Z[zoff++]);
        T[--toff] = t3;
        T[--toff] = t2;
        T[--toff] = t1;

        for (j = 1; j < ROUNDS; j++)
        {
                t1 = Z[zoff++];
                T[--toff] = Z[zoff++];
                T[--toff] = t1;

                t1 = inv(Z[zoff++]);
                t2 = (short)-Z[zoff++];
                t3 = (short)-Z[zoff++];
                T[--toff] = inv(Z[zoff++]);
                T[--toff] = t2;
                T[--toff] = t3;
                T[--toff] = t1;
        }
        t1 = Z[zoff++];
        T[--toff] = Z[zoff++];
        T[--toff] = t1;

        t1 = inv(Z[zoff++]);
        t2 = (short)-Z[zoff++];
        t3 = (short)-Z[zoff++];
        T[--toff] = inv(Z[zoff++]);
        T[--toff] = t3;
        T[--toff] = t2;
        T[--toff] = t1;
        System.arraycopy(T,0,Z,0,KEYLEN);
        for (i=0; i<KEYLEN; ++i)
            T[i] = 0;
} /* de_key_idea */


/*      IDEA encryption/decryption algorithm */
/* Note that in and out can be the same buffer */ 
/* Buffer length is 4 shorts */
void cipher_idea(short in[], short out[])
{
    short x1, x2, x3, x4, s2, s3;
    int zoff = 0;
    int r = ROUNDS;

    x1 = in[0];  x2 = in[1];
    x3 = in[2];  x4 = in[3];
    while (r-- > 0) {
        x1 = mul(x1,Z[zoff++]);
        x2 += Z[zoff++];
        x3 += Z[zoff++];
        x4 = mul(x4, Z[zoff++]);

        s3 = x3;
        x3 ^= x1;
        x3 = mul(x3, Z[zoff++]);
        s2 = x2;
        x2 ^= x4;
        x2 += x3;
        x2 = mul(x2, Z[zoff++]);
        x3 += x2;

        x1 ^= x2;
        x4 ^= x3;

        x2 ^= s3;
        x3 ^= s2;
    }
    x1 = mul(x1, Z[zoff++]);
    out[0] = x1;
    out[1] = (short)(x3 + Z[zoff++]);
    out[2] = (short)(x2 + Z[zoff++]);
    x4 = mul(x4, Z[zoff]);
    out[3] = x4;
}                               /* cipher_idea */

public void processBlock(byte inOutBlock[], int inOutOff)
{
    processBlock(inOutBlock, inOutOff, inOutBlock, inOutOff);
}

public void processBlock(byte inBlock[], int inOff, byte outBlock[], int outOff)
{
    short ins[] = new short[BS];
    short outs[] = new short[BS];
    bytes.BArrToArrBig(inBlock, inOff, ins, 0, BS);
    cipher_idea(ins, outs);
    bytes.ArrToBArrBig(outs, 0, outBlock, outOff, BS);
}

public int blockSize()
{
    return BS;
}

public int keySize()
{
    return KS;
}

public static int ideaKeySize()
{
    return KS;
}

public static int ideaBlockSize()
{
    return BS;
}


}
