/*____________________________________________________________________________
	Copyright (C) 1999 Network Associates, Inc.
	All rights reserved.

	$Id: CPGPException.h,v 1.9 1999/04/02 09:20:40 jason Exp $
____________________________________________________________________________*/

#pragma once 

#include <Errors.h>
#if PGP_DEBUG
#	include <TextUtils.h>
#endif
#include <string>
#include <exception>
#include "pgpPFLErrors.h"
#include "MacErrors.h"


class CPGPException : public exception {
public:
	explicit				CPGPException(PGPInt32 inError)
									: mPGPError(inError)
								{
#ifdef Included_pgpClientErrors_h
									PGPGetClientErrorString(inError, kPGPErrorStringSize,
										mPGPErrorString);
#else
#	ifdef Included_pgpErrors_h
									PGPGetErrorString(inError, kPGPErrorStringSize,
										mPGPErrorString);
#	else
#		ifdef Included_pgpPFLErrors_h
									PGPGetPFLErrorString(inError, kPGPErrorStringSize,
										mPGPErrorString);
#		else
									sprintf(mPGPErrorString, "%ld", mPGPError);
#		endif
#	endif
#endif
								}

	explicit				CPGPException(PGPError inError, const char * inString)
									: mPGPError(inError)
								{
									strcpy(mPGPErrorString, inString);
								}

	explicit				CPGPException(PGPError inError, ConstStringPtr inString)
									: mPGPError(inError)
								{
									memcpy(mPGPErrorString, &inString[1], inString[0]);
								 	mPGPErrorString[inString[0]] = 0;
								}

	virtual const char *	what() const MSIPL_THROW { return mPGPErrorString; }
	virtual PGPInt32		GetError() { return mPGPError; }
	const char *			GetPGPErrorString() { return mPGPErrorString; }
	PGPError				GetPGPError() { return mPGPError; }
	PGPBoolean				IsCancelError() { return mPGPError == kPGPError_UserAbort; }

#if PGP_DEBUG	
	virtual void			DebugMessage(const char * inFile, int inLine)
								{
									if (! IsCancelError()) {
										char temp[128];
										sprintf(temp, "CPGPException: %ld (%s) File: %s Line: %d",
											mPGPError, mPGPErrorString, inFile, inLine);
										c2pstr(temp);
										DebugStr(reinterpret_cast<StringPtr>(temp));
									}
								}
#endif
				
protected:
	enum { kPGPErrorStringSize = 128 };
	
	PGPInt32				mPGPError;
	char					mPGPErrorString[kPGPErrorStringSize];
};



class CPGPOSException : public CPGPException {
public:
	explicit			CPGPOSException(OSStatus inError)
							: CPGPException(MacErrorToPGPError(inError)),
							mOSError(inError) { }

	OSStatus			GetOSError() { return mOSError; }

	virtual PGPInt32	GetError() { return mOSError; }

#if PGP_DEBUG
	virtual void		DebugMessage(const char * inFile, int inLine)
							{
								if (! IsCancelError()) {
									char temp[128];
									sprintf(temp, "CPGPOSException: %ld File: %s Line: %d",
										mOSError, inFile, inLine);
									c2pstr(temp);
									DebugStr(reinterpret_cast<StringPtr>(temp));
								}
							}
#endif

protected:
	OSStatus			mOSError;
};


#if PGP_DEBUG
#	define PGPThrowPGPError_(inError)									\
		do {															\
			CPGPException	__theException(inError);					\
			__theException.DebugMessage(__FILE__, __LINE__);			\
			throw __theException;										\
		} while (false)
#	define PGPThrowPGPErrorString_(inError, inString)					\
		do {															\
			CPGPException	__theException((inError), (inString));		\
			__theException.DebugMessage(__FILE__, __LINE__);			\
			throw __theException;										\
		} while (false)
#	define PGPThrowOSError_(inError)									\
		do {															\
			CPGPOSException	__theException(inError);					\
			__theException.DebugMessage(__FILE__, __LINE__);			\
			throw __theException;										\
		} while (false)
#else
#	define PGPThrowPGPError_(inError)					throw CPGPException(inError)
#	define PGPThrowPGPErrorString_(inError, inString)	throw CPGPException((inError), (inString))
#	define PGPThrowOSError_(inError)					throw CPGPOSException(inError)
#endif



#define PGPThrowIfPGPError_(inError)									\
	if ((inError) != kPGPError_NoErr) {									\
		PGPThrowPGPError_(inError);										\
	}



#define PGPThrowIfOSError_(inError)										\
	if ((inError) != noErr) {											\
		PGPThrowOSError_(inError);										\
	}



#define PGPThrowIfNULL_(inPointer)										\
	if ((inPointer) == 0) {											\
		PGPThrowPGPError_(kPGPError_BadMemAddress);						\
	}



#define	PGPThrowIfResError_()	PGPThrowIfOSError_(ResError())
#define	PGPThrowIfMemError_()	PGPThrowIfOSError_(MemError())



#define PGPThrowIfResFail_(inHandle)									\
	if ((inHandle) == 0) {											\
		OSStatus __theError = ResError();								\
		if (__theError == noErr) {										\
			__theError = resNotFound;									\
		}																\
		PGPThrowOSError_(__theError);									\
	}



#define PGPThrowIfMemFail_(inPointer)									\
	if ((inPointer) == 0) {											\
		OSStatus __theError = MemError();								\
		if (__theError == noErr) {										\
			__theError = memFullErr;									\
		}																\
		PGPThrowOSError_(__theError);									\
	}



#define	PGPThrowIf_(inTest, inString)									\
		if (inTest) {													\
			PGPThrowPGPErrorString_(kPGPError_AssertFailed,				\
								static_cast<const char *>(inString));	\
		}



#define	PGPThrowIfNot_(inTest, inString)								\
		if (! (inTest)) {												\
			PGPThrowPGPErrorString_(kPGPError_AssertFailed,				\
								static_cast<const char *>(inString));	\
		}
