/*____________________________________________________________________________
	Copyright (C) 1997 Network Associates Inc. and affiliated companies.
	All rights reserved.
	
	$Id: PGPclientLibUtils.cp,v 1.66 1999/05/24 07:50:45 heller Exp $
____________________________________________________________________________*/

#include <Folders.h>
#include <LowMem.h>
#include <fp.h>
#include <string.h>

#include <LActiveScroller.h>
#include <LCaption.h>
#include <LGAFocusBorder.h>
#include <LGrafPortView.h>
#include <LIconPane.h>
#include <LRadioGroupView.h>
#include <LTabGroup.h>
#include <PP_Messages.h>
#include <UAttachments.h>
#include <UDrawingState.h>
#include <UGAColorRamp.h>
#include <UEnvironment.h>
#include <UGraphicUtils.h>
#include <URegistrar.h>

// Appearance classes
#include <LBevelButton.h>
#include <LCheckBox.h>
#include <LEditText.h>
#include <LIconControl.h>
#include <LLittleArrows.h>
#include <LMultiPanelView.h>
#include <LPopupButton.h>
#include <LProgressBar.h>
#include <LPushButton.h>
#include <LRadioButton.h>
#include <LStaticText.h>
#include <LTabsControl.h>
#include <LTextGroupBox.h>
#include <LWindowHeader.h>

// Appearance Manager Implementations
#include <LAMControlImp.h>
#include <LAMPopupButtonImp.h>
#include <LAMPushButtonImp.h>
#include <LAMStaticTextImp.h>
#include <LAMTabsControlImp.h>
#include <LAMTrackActionImp.h>

// Grayscale Implementations
#include <LGABevelButtonImp.h>
#include <LGACheckBoxImp.h>
#include <LGAEditTextImp.h>
#include <LGAIconControlImp.h>
#include <LGALittleArrowsImp.h>
#include <LGAPopupButtonImp.h>
#include <LGAProgressBarImp.h>
#include <LGAPushButtonImp.h>
#include <LGARadioButtonImp.h>
#include <LGAStaticTextImp.h>
#include <LGATabsControlImp.h>
#include <LGATextGroupBoxImp.h>
#include <LGAWindowHeaderImp.h>

#include "MacBasics.h"
#include "MacCursors.h"
#include "MacDebug.h"
#include "MacDesktop.h"
#include "MacDialogs.h"
#include "MacEnvirons.h"
#include "MacErrors.h"
#include "MacFiles.h"
#include "MacIcons.h"
#include "MacResources.h"
#include "MacStrings.h"

#include "pflPrefs.h"
#include "pgpBase.h"
#include "pgpDebug.h"
#include "pgpMem.h"
#include "pgpSDKPrefs.h"
#include "pgpFeatures.h"

#include "CPGPActiveScroller.h"
#include "CPGPAMEditTextImp.h"
#include "CPGPModalGrafPortView.h"
#include "CServerStatusDialog.h"
#include "CShareholderList.h"
#include "PGPOpenPrefs.h"
#include "pgpAdminPrefs.h"
#include "pgpClientPrefs.h"
#include "pgpHashWords.h"
#include "pflPrefTypes.h"
#include "PGPclientLibUtils.h"
#include "pgpClientLib.h"
#include "pgpClientLibDialogs.h"
#include "pgpMemoryIO.h"
#include "pgpKeys.h"
#include "pgpRandomPool.h"
#include "pgpVersionHeader.h"
#include "pgpWordWrap.h"
#include "SignatureStatusMessage.h"
#include "WarningAlert.h"
#include "CComboError.h"
#include "PassphraseCache.h"
#include "PowerPlantLeaks.h"
#include "CSecureMemory.h"
#include "StSaveCurResFile.h"
#include "CString.h"
#include "sharedmem.h"
#include "PGPSharedEncryptDecrypt.h"
#include "StPGPRefs.h"
#include "CPGPException.h"

static const ResType	kAdminConfiguredType	=	'PGPa';
static const SInt16		PGPa_Configured			=	0;

static const StringPtr	kPGPPrefsUpdateCount	=	"\pPGPPrefsUpdateCount";

const ResID	kMinAppearanceVersionAlertResID	= 4758;
const ResID	kSelectPublicKeySFDialogResID 	= 4770;
const ResID	kSelectPrivateKeySFDialogResID 	= 4771;
const ResID	kSelectRandomSeedSFDialogResID 	= 4772;
const ResID	kSelectKeySharesSFDialogResID	= 4773;

const ResID	kFileNameStringResID 			= 4749;

enum
{
	kPGPPrivateKeyFileNameStrIndex	= 1,
	kPGPPublicKeyFileNameStrIndex,
	kPGPRandomSeedFileNameStrIndex,
	kPGPGroupsFileNameStrIndex
};

enum
{
	kShowAllFilesCheckboxItem	= 10
};

const ResID	STRx_UtilStrings						=	8946;
enum {		stringID_SignerKeyImport				=	1,
			stringID_NoPublicKeys,
			stringID_NoPrivateKeys,
			stringID_ImportKeys,
			stringID_AdminPrefsNotFound,
			stringID_BeginSignedMessage,
			stringID_EndSignedMessage,
			stringID_BeginEncryptedAndSignedMessage,
			stringID_EndEncryptedAndSignedMessage,
			stringID_AddKeys,
			stringID_BadSignature,
			stringID_SignatureNotVerified,
			stringID_IncorrectPassphrase
};



static const ResID	icsx_DSARevokedKey		=	3012;
static const ResID	icsx_RSARevokedKey		=	3004;
static const ResID	icsx_DSAExpiredKey		=	3013;
static const ResID	icsx_RSAExpiredKey		=	3005;
static const ResID	icsx_DSADisabledKey		=	3006;
static const ResID	icsx_RSADisabledKey		=	3010;
static const ResID	icsx_DSASplitKeyPair	=	3018;
static const ResID	icsx_RSASplitKeyPair	=	3019;
static const ResID	icsx_DSAKeyPair			=	3008;
static const ResID	icsx_RSAKeyPair			=	3001;
static const ResID	icsx_DSAKey				=	3007;
static const ResID	icsx_RSAKey				=	3000;


static FSSpec		sLibraryFileSpec		= {0, };
static long			sLibraryFileID			= 0;
static short		sLibraryFileRefNum		= -1;
static PGPUInt32	sLibraryFileRefCount	= 0;
static PGPPrefRef	sClientPrefsRef			= kInvalidPGPPrefRef;

#if PGP_BUSINESS_SECURITY
static PGPPrefRef	sAdminPrefsRef			= kInvalidPGPPrefRef;
#endif

	
const ResID		acur_Encode					=	128;
const ResID		acur_Decode					=	129;
const UInt8		kNumFrames					=	18;
const SInt32	kCursorDelay				=	5;


	static void
RegisterPowerPlantClasses(void)
{
	RegisterClass_( LActiveScroller );
	RegisterClass_( LAttachment );
	RegisterClass_( LCaption );
	RegisterClass_( LGAFocusBorder );
	RegisterClass_( LGrafPortView );
	RegisterClass_( LIconPane );
	RegisterClass_( LRadioGroupView );
	RegisterClass_( LTabGroup );
	RegisterClass_( LView );
	
	// Appearance classes:
	
	RegisterClass_( LBevelButton );
	RegisterClass_( LCheckBox );
	RegisterClass_( LEditText );
	RegisterClass_( LIconControl );
	RegisterClass_( LLittleArrows );
	RegisterClass_( LMultiPanelView );
	RegisterClass_( LPopupButton );
	RegisterClass_( LProgressBar );
	RegisterClass_( LPushButton );
	RegisterClass_( LRadioButton );
	RegisterClass_( LStaticText );
	RegisterClass_( LTabsControl );
	RegisterClass_( LTextGroupBox );
	RegisterClass_( LWindowHeader );

	// Register Appearance Manager implementations
	// Note: Do NOT call RegisterAppearanceClient here. That is the
	// client library responsibility.
	
	RegisterClassID_( LAMControlImp, 		LBevelButton::imp_class_ID );
	RegisterClassID_( LAMControlImp, 		LCheckBox::imp_class_ID );
	RegisterClassID_( LAMControlImp, 		LIconControl::imp_class_ID );
	RegisterClassID_( LAMControlImp, 		LProgressBar::imp_class_ID );
	RegisterClassID_( LAMControlImp, 		LTextGroupBox::imp_class_ID );		
	RegisterClassID_( LAMControlImp, 		LWindowHeader::imp_class_ID );
	RegisterClassID_( CPGPAMEditTextImp,	LEditText::imp_class_ID );
	RegisterClassID_( LAMPopupButtonImp,	LPopupButton::imp_class_ID );
	RegisterClassID_( LAMPushButtonImp,		LPushButton::imp_class_ID );
	RegisterClassID_( LAMControlImp, 		LRadioButton::imp_class_ID );
	RegisterClassID_( LAMStaticTextImp,		LStaticText::imp_class_ID );
	RegisterClassID_( LAMTabsControlImp,	LTabsControl::imp_class_ID );
	RegisterClassID_( LAMTrackActionImp, 	LLittleArrows::imp_class_ID );

	RegisterClass_( CPGPActiveScroller );
	RegisterClass_( CPGPModalGrafPortView );
	RegisterClass_( CServerStatusDialog );
	RegisterClass_( CShareholderList );
}

	OSStatus
SetLibraryFSSpec(const FSSpec *fileSpec)
{
	OSStatus	status;
	CInfoPBRec	cpb;
	
	// This routine should be called once ONLY from the library INIT routine.
	pgpAssert( sLibraryFileSpec.name[0] == 0 );
	pgpAssert( sLibraryFileID == 0 );
	
	status = FSpGetCatInfo( fileSpec, &cpb );
	if( IsntErr( status ) )
	{
		HParamBlockRec	pb;
		
		sLibraryFileSpec = *fileSpec;
		
		// Create a file ID so we can track this file later.
		
		pgpClearMemory( &pb, sizeof( pb ) );
		
		pb.fidParam.ioVRefNum	= sLibraryFileSpec.vRefNum;
		pb.fidParam.ioNamePtr	= sLibraryFileSpec.name;
		pb.fidParam.ioSrcDirID	= sLibraryFileSpec.parID;
		
		if( IsntErr( PBCreateFileIDRefSync( &pb ) ) )
		{
			sLibraryFileID = pb.fidParam.ioFileID;
		}
	}
	
	return( status );
}

	OSStatus
PGPGetClientLibFSSpec(FSSpec *fileSpec)
{
	OSStatus	status;
	CInfoPBRec	cpb;
	
	pgpAssertAddrValid( fileSpec, FSSpec );
	AssertSpecIsValid( &sLibraryFileSpec, "PGPGetClientLibFSSpec" );
	
	status = FSpGetCatInfo( &sLibraryFileSpec, &cpb );
	if( IsErr( status ) && sLibraryFileID != 0 )
	{
		// Find the library using the file ID.
		
		HParamBlockRec	pb;
		FSSpec			specCopy;
		
		pgpClearMemory( &pb, sizeof( pb ) );
		
		specCopy = sLibraryFileSpec;
		
		pb.fidParam.ioVRefNum	= specCopy.vRefNum;
		pb.fidParam.ioNamePtr	= specCopy.name;
		pb.fidParam.ioFileID	= sLibraryFileID;
		
		status = PBResolveFileIDRefSync( &pb );
		if( IsntErr( status ) )
		{
			specCopy.parID = pb.fidParam.ioSrcDirID;
			
			status = FSpGetCatInfo( &specCopy, &cpb );
			if( IsntErr( status ) )
			{
				sLibraryFileSpec = specCopy;
			}
		}
	}

	AssertNoErr( status, "GetPGPclientLibFileSpec" );
	
	*fileSpec = sLibraryFileSpec;
	
	return( status );
}

	static void
InitializePowerPlant(void)
{
	static Boolean	firstTime = TRUE;
	
	if( firstTime )
	{
#if PGP_DEBUG
		PP_PowerPlant::UDebugging::gDebugThrow 	= debugAction_LowLevelDebugger;
		PP_PowerPlant::UDebugging::gDebugSignal 	= debugAction_LowLevelDebugger;
#endif

		UEnvironment::InitEnvironment();

		UQDGlobals::SetQDGlobals((QDGlobals*)(*((long*)::LMGetCurrentA5()) - 
				(sizeof(QDGlobals) - sizeof(GrafPtr))));
		
		MacLeaks_Suspend();

		#if USE_MAC_DEBUG_LEAKS
		{ CForceInitLPeriodical	temp; }
		#endif

		RegisterPowerPlantClasses();

		MacLeaks_Resume();
		
		LCommander::SetDefaultCommander( NULL );
		LPane::SetDefaultView( NULL );

		firstTime = FALSE;
	}
}

	PGPError
EnterPGPclientLib(
	PGPContextRef 		context,
	PGPclientLibState 	*state)
{
	CComboError		err;
	
	pgpAssertAddrValid( state, PGPclientLibState );
	
	InitializePowerPlant();

	pgpClearMemory( state, sizeof( *state ) );
	
	GetPort( &state->savedPort );
	
	state->savedResLoad 	= LMGetResLoad();
	state->savedZone		= GetZone();
	state->savedResFile		= CurResFile();
	state->savedView 		= LPane::GetDefaultView();
	state->savedCommander 	= LCommander::GetDefaultCommander();
	state->savedTarget 		= LCommander::GetTarget();
	
	SetResLoad( TRUE );
	LCommander::SetDefaultCommander( NULL );
	LPane::SetDefaultView( NULL );
	LCommander::SwitchTarget( NULL );
	
	if( sLibraryFileRefCount == 0 )
	{
		FSSpec	libraryFileSpec;

		pgpAssert( sLibraryFileRefNum == -1 );
		pgpAssert( ! PGPPrefRefIsValid( sClientPrefsRef ) );
	#if PGP_BUSINESS_SECURITY
		pgpAssert( ! PGPPrefRefIsValid( sAdminPrefsRef ) );
	#endif
	
		err.err	= PGPGetClientLibFSSpec( &libraryFileSpec );
		if( err.IsntError() )
		{
			sLibraryFileRefNum = FSpOpenResFile( &libraryFileSpec, fsRdPerm );
			if( sLibraryFileRefNum > 0 )
			{
				++sLibraryFileRefCount;
				UseResFile( sLibraryFileRefNum );

				if( PGPContextRefIsValid( context ) )
				{
					PGPMemoryMgrRef	memMgr;
					
					memMgr = PGPGetContextMemoryMgr( context );
					pgpAssert( PGPMemoryMgrRefIsValid( memMgr ) );
					
					err.pgpErr = PGPOpenClientPrefs( memMgr, &sClientPrefsRef );

				#if PGP_BUSINESS_SECURITY	// [
					if( err.IsntError() ) 
					{
						err.pgpErr = PGPOpenAdminPrefs( memMgr, &sAdminPrefsRef );
					}
				#endif	// ] PGP_BUSINESS_SECURITY
				}
			}
			else
			{
				err.pgpErr = kPGPError_FileNotFound;
			}
		}
	}
	else
	{
		/* Library already open. Just increment the count */
		
		pgpAssert( sLibraryFileRefNum > 0 );
		
		++sLibraryFileRefCount;
		UseResFile( sLibraryFileRefNum );
	}

	state->clientPrefsRef 	= sClientPrefsRef;
#if PGP_BUSINESS_SECURITY
	state->adminPrefsRef 	= sAdminPrefsRef;
#else
	state->adminPrefsRef 	= kInvalidPGPPrefRef;
#endif

	return( err.ConvertToPGPError() );
}

	void
ExitPGPclientLib(const PGPclientLibState *state)
{
	pgpAssertAddrValid( state, PGPclientLibState );

	--sLibraryFileRefCount;
	if( sLibraryFileRefCount == 0 )
	{
		pgpAssert( sLibraryFileRefNum > 0 );
	
		CloseResFile( sLibraryFileRefNum );
		sLibraryFileRefNum = -1;

		if( PGPPrefRefIsValid( sClientPrefsRef ) )
		{
			(void) PGPClosePrefFile( sClientPrefsRef );
			sClientPrefsRef = kInvalidPGPPrefRef;
		}
			
	#if PGP_BUSINESS_SECURITY
		if( PGPPrefRefIsValid( sAdminPrefsRef ) )
		{
			(void) PGPClosePrefFile( sAdminPrefsRef );
			sAdminPrefsRef = kInvalidPGPPrefRef;
		}
	#endif
	}

	LCommander::SetDefaultCommander( state->savedCommander );
	LPane::SetDefaultView( state->savedView );
	LCommander::SwitchTarget( state->savedTarget );

	SetPort( state->savedPort );
	SetResLoad( state->savedResLoad );
	SetZone( state->savedZone );
	UseResFile( state->savedResFile );
}
	
	PGPError
PGPGetPGPFileName(
	PGPContextRef	context,
	PGPFileSelector whichFile,
	StringPtr 		fileName)
{
	PGPError			err;
	PGPclientLibState	state;
	
	PGPValidatePtr( fileName );
	fileName[0] = 0;
	
	err = EnterPGPclientLib( context, &state );
	if( IsntPGPError( err ) )
	{
		short		strIndex;
		
		switch( whichFile )
		{
			case kPGPFileSelector_PublicKeys:
				strIndex = kPGPPublicKeyFileNameStrIndex;
				break;
				
			case kPGPFileSelector_PrivateKeys:
				strIndex = kPGPPrivateKeyFileNameStrIndex;
				break;
				
			case kPGPFileSelector_RandomSeed:
				strIndex = kPGPRandomSeedFileNameStrIndex;
				break;
			
			case kPGPFileSelector_Groups:
				strIndex = kPGPGroupsFileNameStrIndex;
				break;
				
			default:
				pgpDebugMsg( "PGPGetPGPFileName(): Invalid selector" );
				err = kPGPError_BadParams;
		}

		if( IsntPGPError( err ) )
		{
			GetIndString( fileName, kFileNameStringResID, strIndex );
		}
	}
	
	ExitPGPclientLib( &state );
	
	return( err );
}


	PGPError
PGPGetPGPFileDefaultFSSpec(
	PGPContextRef	context,
	PGPFileSelector whichFile,
	FSSpec			*fileSpec)
{
	CComboError	err;
	Str255		fileName;
	
	PGPValidateParam( IsntNull( fileSpec ) );
	
	err.pgpErr = PGPGetPGPFileName( context, whichFile, fileName );
	if( err.IsntError() )
	{
		short	vRefNum;
		long	dirID;
		
		err.err = FindPGPPreferencesFolder( kOnSystemDisk, &vRefNum, &dirID );
		if( err.IsntError() )
		{
			err.err = FSMakeFSSpec( vRefNum, dirID, fileName, fileSpec );
		}
	}
	
	return( err.ConvertToPGPError() );
}

	PGPError
PGPSetPGPFileFSSpec(
	PGPContextRef	context,
	PGPFileSelector whichFile,
	const FSSpec	*fileSpec)
{
	PGPError			err = kPGPError_NoErr;
	PGPsdkPrefSelector	prefSelector;
	
	PGPValidateParam( PGPContextRefIsValid( context ) );
	PGPValidateParam( IsntNull( fileSpec ) );

	switch( whichFile )
	{
		case kPGPFileSelector_PublicKeys:
			prefSelector = kPGPsdkPref_PublicKeyring;
			break;
			
		case kPGPFileSelector_PrivateKeys:
			prefSelector = kPGPsdkPref_PrivateKeyring;
			break;
			
		case kPGPFileSelector_RandomSeed:
			prefSelector = kPGPsdkPref_RandomSeedFile;
			break;

		case kPGPFileSelector_Groups:
			prefSelector = kPGPsdkPref_GroupsFile;
			break;
		
		default:
			pgpDebugMsg( "PGPSetPGPFileFSSpec(): Invalid selector" );
			err = kPGPError_BadParams;
	}
	
	if( IsntPGPError( err ) )
	{
		PGPFileSpecRef	fileRef;
		
		err = PGPNewFileSpecFromFSSpec(	context, fileSpec, &fileRef );
		if( IsntPGPError( err ) )
		{
			err = PGPsdkPrefSetFileSpec( context, prefSelector, fileRef );
			if( IsntPGPError( err ) )
			{
				err = PGPsdkSavePrefs( context );
			}
			
			(void) PGPFreeFileSpec( fileRef );
		}
	}
	
	return( err );
}

	static pascal Boolean
VisibleItemFileFilterProc(CInfoPBRec *cpb)
{
	Boolean	shouldFilter = FALSE;
	
	pgpAssertAddrValid( cpb, CInfoPBRec);

	if( cpbIsFile( cpb ) )
	{
		if( ( cpb->hFileInfo.ioFlFndrInfo.fdFlags & kIsInvisible ) != 0 )
		{
			shouldFilter = TRUE;
		}
	}
	else
	{
		if( ( cpb->dirInfo.ioDrUsrWds.frFlags & kIsInvisible ) != 0 )
		{
			shouldFilter = TRUE;
		}
	}
	
	return( shouldFilter );
}

typedef struct SFInfo
{
	StandardFileReply	sfReply;
	Boolean				showAllFiles;
	PGPFileSelector		fileSelector;

} SFInfo;

	static pascal Boolean
KeyFileFilterHook(
	CInfoPBRec 		*cpb,
	const SFInfo 	*sfInfo)
{
	Boolean	shouldFilter = FALSE;
	
	shouldFilter = VisibleItemFileFilterProc( cpb );
	if( ! shouldFilter )
	{
		if( cpbIsFile( cpb ) && ! sfInfo->showAllFiles )
		{
			// Show only the files possibly created by PGP.
			switch( cpbFileType( cpb ) )
			{
				case 'PKey':	// ancient MacPGP 2.6.2 filetype
				case kPGPMacFileType_PubRing:
					if( sfInfo->fileSelector != kPGPFileSelector_PublicKeys )
						shouldFilter = TRUE;
					break;
					
				case 'SKey':	// ancient MacPGP 2.6.2 filetype
				case kPGPMacFileType_PrivRing:
					if( sfInfo->fileSelector != kPGPFileSelector_PrivateKeys )
						shouldFilter = TRUE;
					break;
					
				case kPGPMacFileType_RandomSeed:
					if( sfInfo->fileSelector != kPGPFileSelector_RandomSeed )
						shouldFilter = TRUE;
					break;

				case kPGPMacFileType_KeyShares:
					if( sfInfo->fileSelector != kPGPFileSelector_KeyShares )
						shouldFilter = TRUE;
					break;
					
				default:
					shouldFilter = TRUE;
					break;
			}
		}
	}
		
	return( shouldFilter );
}

	static pascal short
KeyFileDialogHook(short item, DialogPtr dialog, SFInfo *sfInfo)
{
	pgpAssertAddrValid( dialog, DialogRecord );
	pgpAssertAddrValid( sfInfo, SFInfo );

	switch( item )
	{
		case kShowAllFilesCheckboxItem:
		{
			sfInfo->showAllFiles = ToggleDialogCheckbox( dialog,
										kShowAllFilesCheckboxItem );
			item = sfHookRebuildList;
			break;
		}
	}
	return item;
}

	PGPError
PGPSelectPGPFile(
	PGPContextRef	context,
	PGPFileSelector whichFile,
	FSSpec			*fileSpec)
{
	PGPError			err;
	PGPclientLibState 	state;
	
	PGPValidateParam( IsntNull( fileSpec ) );
	
	err = EnterPGPclientLib( context, &state );
	if( IsntPGPError( err ) )
	{
		ResID	dialogResID;
		
		switch( whichFile )
		{
			case kPGPFileSelector_PublicKeys:
				dialogResID = kSelectPublicKeySFDialogResID;
				break;
				
			case kPGPFileSelector_PrivateKeys:
				dialogResID = kSelectPrivateKeySFDialogResID;
				break;
				
			case kPGPFileSelector_RandomSeed:
				dialogResID = kSelectRandomSeedSFDialogResID;
				break;
				
			case kPGPFileSelector_KeyShares:
				dialogResID = kSelectKeySharesSFDialogResID;
				break;

			default:
				pgpDebugMsg( "PGPSelectPGPFileFSSpec(): Invalid selector" );
				err = kPGPError_BadParams;
		}
		
		if( IsntPGPError( err ) )
		{
			SFInfo				sfInfo;
			SFTypeList			typeList;
			DlgHookYDUPP		cbHook;
			FileFilterYDUPP		filterUPP;
			static Point		where={0,0};
			
			pgpClearMemory( &sfInfo, sizeof( sfInfo ) );
			
			cbHook 		= NewDlgHookYDProc( KeyFileDialogHook );
			filterUPP	= NewFileFilterYDProc( KeyFileFilterHook );
			typeList[0] = 0;
			
			sfInfo.showAllFiles = FALSE;
			sfInfo.fileSelector	= whichFile;
			
			CustomGetFile( filterUPP, -1, typeList, &sfInfo.sfReply,
					dialogResID, where, cbHook, NULL, NULL, NULL,
					&sfInfo );
								
			DisposeRoutineDescriptor( cbHook );
			DisposeRoutineDescriptor( filterUPP );
			
			if( sfInfo.sfReply.sfGood )
			{
				*fileSpec = sfInfo.sfReply.sfFile;
			}
			else
			{
				err = kPGPError_UserAbort;
			}
		}
	}
	
	ExitPGPclientLib( &state );
	
	return( err );
}

	PGPError
PGPInitClientDecodeEventHandlerData(
	PGPContextRef 					context,
	PGPtlsContextRef				tlsContext,
	PGPKeySetRef 					allKeys,
	PGPClientDecodeEventHandlerData *data)
{
	PGPError	err = kPGPError_NoErr;
	
	PGPValidateParam( PGPContextRefIsValid( context ) );
	PGPValidatePtr( data );
	
	pgpClearMemory( data, sizeof( *data ) );
	
	data->context 		= context;
	data->tlsContext	= tlsContext;
	data->allKeys		= allKeys;
	
	err = PGPNewPassBuffer( PGPGetContextMemoryMgr( context ),
								&data->passBuffer);
	return( err );
}

	void
PGPCleanupClientDecodeEventHandlerData(
	PGPClientDecodeEventHandlerData *data)
{
	if( IsntNull( data ) )
	{
		if( IsntNull( data->outputBuffer ) )
			PGPFreeData( data->outputBuffer );

		if( PGPKeySetRefIsValid( data->recipientKeySet ) )
			(void) PGPFreeKeySet( data->recipientKeySet );

		if( PGPKeySetRefIsValid( data->newKeySet ) )
			(void) PGPFreeKeySet( data->newKeySet );

		if( IsntNull( data->recipientKeyIDList ) )
			PGPFreeData( data->recipientKeyIDList );

		if( PGPPassBufferRefIsValid( data->passBuffer ) )
			PGPFreePassBuffer( data->passBuffer );

		pgpClearMemory( data, sizeof( *data ) );
	}
}

	PGPError
PGPClientDecodeEventHandler(
	PGPContextRef 					context,
	struct PGPEvent 				*event,
	PGPClientDecodeEventHandlerData *data )
{
	PGPError	err = kPGPError_NoErr;
	
	PGPValidatePtr( data );
	PGPValidateParam( data->context == context );
	
	switch( event->type )
	{
		case kPGPEvent_BeginLexEvent:
		{
			pgpAssert( IsNull( data->outputBuffer ) );
			pgpAssert( data->outputBufferSize == 0 );
			
			pgpClearMemory( &data->signatureData, 
						sizeof( data->signatureData ) );
						
			data->signatureDataValid = FALSE;
			data->conventional = FALSE;
			
			break;
		}
		
		case kPGPEvent_EndLexEvent:
		{
			if( IsntNull( data->outputBuffer ) )
				PGPFreeData( data->outputBuffer );
			if( PGPKeySetRefIsValid(data->recipientKeySet) )
				PGPFreeKeySet( data->recipientKeySet );
			if( IsntNull( data->recipientKeyIDList ) )
				PGPFreeData( data->recipientKeyIDList );
			
			data->recipientKeyIDList = NULL;
			data->recipientKeySet = kInvalidPGPKeySetRef;
			data->outputBuffer 		= NULL;
			data->outputBufferSize	= 0;
			
			break;
		}
		
		case kPGPEvent_AnalyzeEvent:
		{
			data->sectionType = event->data.analyzeData.sectionType;
			break;
		}
			
		case kPGPEvent_OutputEvent:
		{
			err = PGPAddJobOptions( event->job,
						PGPOAllocatedOutputBuffer( context, &data->outputBuffer,
								MAX_PGPSize, &data->outputBufferSize ),
						PGPOLastOption( context ) );
			break;
		}
		
		case kPGPEvent_PassphraseEvent:
		{
			char	*passphrase = NULL;
			
			data->split 		= FALSE;
			data->decryptionKey	= kInvalidPGPKeyRef;
			
			/* Clear previous passphrase attempt(s) */
			(void) PGPClearPassBuffer( data->passBuffer );

			if( event->data.passphraseData.fConventional )
			{
				if( data->conventional )
				{
					WarningAlert(kWACautionAlertType, kWAOKStyle, STRx_UtilStrings,
						stringID_IncorrectPassphrase);
				}
				err = PGPConventionalDecryptionPassphraseDialog( context,
							data->prompt[0] != 0 ?
								PGPOUIDialogPrompt( context, data->prompt ) :
								PGPONullOption( context ),
							PGPOUIOutputPassphrase( context, &passphrase ),
							PGPOLastOption( context ) );
			}
			else
			{
				PGPKeySetRef	newKeySet = kInvalidPGPKeySetRef;
				
				err = PGPClientDecryptionPassphraseDialog( context,
							data->tlsContext,
							data->prompt[0] != 0 ? data->prompt : NULL,
							data->recipientKeySet,
							data->recipientKeyIDList, data->recipientKeyIDCount,
							kInvalidPGPOptionListRef, &passphrase,
							&data->decryptionKey, &newKeySet );
				if( IsntPGPError( err ) && PGPKeySetRefIsValid( newKeySet ) )
				{
					if( PGPKeySetRefIsValid( data->newKeySet ) )
					{
						err = PGPAddKeys( data->newKeySet, newKeySet );
						if( IsntPGPError( err ) )
						{
							err = PGPCommitKeyRingChanges( newKeySet );
							
							PGPFreeKeySet( data->newKeySet );
						}
					}
				
					data->newKeySet = newKeySet;
				}
			}
			
			data->conventional = event->data.passphraseData.fConventional;

			if( IsntPGPError( err ) )
			{
				err = PGPPassBufferSetPassphrase( data->passBuffer, passphrase );
				if( IsntPGPError( err ) )
				{
					err = PGPAddJobOptions( event->job,
								PGPOPassphrase( context, passphrase ),
								PGPOLastOption( context ) );
				}
				
				PGPFreeData( passphrase );
			}
			else if( err == kPGPError_KeyUnusableForDecryption )
			{
				PGPByte		*passKey;
				PGPSize		passKeySize;
				
				data->split	= TRUE;
				
				err = PGPReconstitutionDialog( data->decryptionKey,
							data->allKeys, kInvalidPGPtlsContextRef,
							&passKey, &passKeySize);
				
				if( IsntPGPError( err ) )
				{
					pgpAssert( passKey != NULL );
					pgpAssert( passKeySize != 0 );
					
					err = PGPPassBufferSetPassKey( data->passBuffer, passKey,
										passKeySize );
					if( IsntPGPError( err ) )
					{
						err = PGPAddJobOptions( event->job,
									PGPOPasskeyBuffer( context, passKey,
										passKeySize ),
									PGPOLastOption( context ) );
					}
								
					PGPFreeData( passKey );
				}
			}
			
			InitCursor();
			break;
		}
	
		case kPGPEvent_RecipientsEvent:
		{
			pgpAssert( ! PGPKeySetRefIsValid( data->recipientKeySet ) );
			pgpAssert( IsNull( data->recipientKeyIDList ) );
			pgpAssert( data->recipientKeyIDCount == 0 );

			data->recipientKeySet = event->data.recipientsData.recipientSet;
			if( PGPKeySetRefIsValid( data->recipientKeySet ) )
			{
				err = PGPIncKeySetRefCount( data->recipientKeySet );
			}
			
			if( IsntPGPError( err ) &&
				IsntNull( event->data.recipientsData.keyIDArray ) )
			{
				PGPSize	dataSize;
				
				data->recipientKeyIDCount =
									event->data.recipientsData.keyCount;
				dataSize = data->recipientKeyIDCount * sizeof( PGPKeyID );
				
				data->recipientKeyIDList = (PGPKeyID *) PGPNewData(
									PGPGetContextMemoryMgr( data->context ),
									dataSize, 0 );
				if( IsntNull( data->recipientKeyIDList ) )
				{
					pgpCopyMemory( event->data.recipientsData.keyIDArray,
							data->recipientKeyIDList, dataSize );
				}
				else
				{
					err = kPGPError_OutOfMemory;
				}
			}
			
			break;
		}
		
		case kPGPEvent_SignatureEvent:
		{
			PGPBoolean		addedKeySetToJob;
			PGPKeySetRef	newKeySet;

			err = PGPSignerKeyLookup( context, event, kInvalidPGPKeySetRef,
								&addedKeySetToJob, &newKeySet);
			if( IsntPGPError( err ) )
			{
				if( addedKeySetToJob )
				{
					if( PGPKeySetRefIsValid( data->newKeySet ) )
					{
						err = PGPAddKeys( data->newKeySet, newKeySet );
						if( IsntPGPError( err ) )
						{
							err = PGPCommitKeyRingChanges( newKeySet );

							PGPFreeKeySet( data->newKeySet );
						}
					}

					data->newKeySet = newKeySet;
				}
				else
				{
					data->signatureData 		= event->data.signatureData;
					data->signatureDataValid 	= TRUE;
				}
			}

			break;
		}

		case kPGPEvent_KeyFoundEvent:
		{
			if( ! PGPKeySetRefIsValid( data->newKeySet ) )
			{
				err = PGPNewKeySet(context, &data->newKeySet);
			}
			
			if( IsntPGPError( err ) )
			{
				err = PGPAddKeys( event->data.keyFoundData.keySet, data->newKeySet );
			}
			if( IsntPGPError( err ) )
			{
				err = PGPCommitKeyRingChanges( data->newKeySet );
			}

			break;
		}
		
		default:
			break;
	}
	
	return( err );
}



	PGPError
PGPCheckAutoUpdateKeysFromServer(
	PGPMemoryMgrRef	memoryMgr,
	PGPBoolean		launchKeys,
	PGPBoolean *	updateAllKeys,
	PGPBoolean *	updateTrustedIntroducers)
{
#if PGP_BUSINESS_SECURITY
	static const OSType	kPGPkeysType = 'APPL';
	static const OSType	kPGPkeysCreator = 'pgpK';
	PGPError			result;
	PGPUInt32			lastUpdate;
	PGPUInt32			updateInterval;
	UInt32				timeNow;
	PGPPrefRef			clientPrefs;
	PGPPrefRef			adminPrefs;
	PGPBoolean			shouldUpdateAllKeys = false;
	PGPBoolean			shouldUpdateTrustedIntroducers = false;
	FSSpec				libraryFileSpec;
	
	try
	{
		PGPValidateParam(PGPMemoryMgrRefIsValid(memoryMgr));
		
		result = PGPGetClientLibFSSpec(&libraryFileSpec);
		if (IsntPGPError(result))
		{
			StSaveCurResFile	savedFile(libraryFileSpec);
			
			::GetDateTime(&timeNow);
			result = PGPOpenAdminPrefs(	memoryMgr,
										&adminPrefs );
			if (IsntPGPError(result))
			{
				result = PGPOpenClientPrefs(	memoryMgr,
												&clientPrefs );
				if( IsntPGPError( result ) )
				{
					// Check trusted introducers
					result = PGPGetPrefNumber(
								adminPrefs,
								kPGPPrefDaysUpdateTrustedIntroducers,
								&updateInterval);
					if (IsntPGPError(result) && (updateInterval > 0))
					{
						result = PGPGetPrefNumber(
									clientPrefs,
									kPGPPrefLastTrustedIntroducersUpdate,
									&lastUpdate);
						if (IsntPGPError(result) &&
						(((timeNow - lastUpdate) / 86400) >= updateInterval))
						{
							shouldUpdateTrustedIntroducers = true;
						}
					}
					
					// Check all keys
					if (IsntPGPError(result))
					{
						result = PGPGetPrefNumber(	adminPrefs,
													kPGPPrefDaysUpdateAllKeys,
													&updateInterval);
						if (IsntPGPError(result) && (updateInterval > 0))
						{
							result = PGPGetPrefNumber(	clientPrefs,
														kPGPPrefLastAllKeysUpdate,
														&lastUpdate);
							if (IsntPGPError(result) &&
								(((timeNow - lastUpdate) / 86400) >= updateInterval))
							{
								shouldUpdateAllKeys = true;
							}
						}
					}
					PGPClosePrefFile(clientPrefs);
				}
				PGPClosePrefFile(adminPrefs);
				
				// Set booleans
				if (IsntPGPError(result))
				{
					if (updateAllKeys != NULL)
					{
						*updateAllKeys = shouldUpdateAllKeys;
					}
					if (updateTrustedIntroducers != NULL)
					{
						*updateTrustedIntroducers = shouldUpdateTrustedIntroducers;
					}
				}
				
				// Launch keys if necessary
				if (IsntPGPError(result) && launchKeys &&
					(shouldUpdateAllKeys || shouldUpdateTrustedIntroducers))
				{
					result = BringAppToFront(	kPGPkeysType,
												kPGPkeysCreator,
												kFindAppOnAllVolumes,
												true,
												nil,
												nil);
					
					// BringAppToFront returns an OSStatus,
					// so convert to PGPError
					if (result != noErr)
					{
						result = MacErrorToPGPError(result);
					}
				}
			}
		}
	}
	
	catch (...)
	{
		result = kPGPError_UnknownError;
	}
	
	return result;
#else
	(void) memoryMgr;
	(void) launchKeys;
	(void) updateAllKeys;
	(void) updateTrustedIntroducers;

	pgpAssert("PGP_BUSINESS_SECURITY not defined");
	return kPGPError_FeatureNotAvailable;
#endif
}



	PGPError
PGPSignerKeyLookup(
	PGPContextRef	context,
	PGPEvent *		event,
	PGPKeySetRef	defaultKeySet,
	PGPBoolean *	addedKeySetToJob,
	PGPKeySetRef *	newKeySet)
{
	PGPError		result = kPGPError_NoErr;
	PGPPrefRef		prefRef = kInvalidPGPPrefRef;
	PGPKeySetRef	signerKeySet = kInvalidPGPKeySetRef;
	PGPUInt32		count;
	PGPBoolean		sync;

	// Validation
	PGPValidatePtr(event);
	PGPValidateParam(event->type == kPGPEvent_SignatureEvent);
	PGPValidatePtr(addedKeySetToJob);
	*addedKeySetToJob = false;
	if (newKeySet != NULL) {
		PGPValidatePtr(newKeySet);
		*newKeySet = kInvalidPGPKeySetRef;
	}
	
	// Lookup key if necessary
	if (! PGPKeyRefIsValid(event->data.signatureData.signingKey)) {
		result = PGPOpenClientPrefs(	PGPGetContextMemoryMgr(context),
										&prefRef);
		if (IsntPGPError(result)) {
			result = PGPGetPrefBoolean(	prefRef,
										kPGPPrefKeyServerSyncOnVerify,
										&sync);
			PGPClosePrefFile(prefRef);
		}
		
		if (IsntPGPError(result) && sync) {
			result = PGPGetKeyIDFromServer(
						context,
						kInvalidPGPtlsContextRef,
						&event->data.signatureData.signingKeyID,
						&signerKeySet);
			if (IsntPGPError(result) && PGPKeySetRefIsValid(signerKeySet)) {
				result = PGPCountKeys(signerKeySet, &count);
			}
			if (IsntPGPError(result) && (count > 0)) {
				result = PGPAddJobOptions(	event->job,
											PGPOKeySetRef(	context,
															signerKeySet),
											PGPOLastOption(context));
				if (IsntPGPError(result)) {
					*addedKeySetToJob = true;
					
					// Do a selective import
					if (PGPKeySetRefIsValid(defaultKeySet)) {
						try {
							FSSpec	libraryFileSpec;
							
							result = PGPGetClientLibFSSpec(&libraryFileSpec);
							if (IsntPGPError(result)) {
								StSaveCurResFile	savedResFile(
														libraryFileSpec);
								PGPKeySetRef		selectedKeySet;
								
								result = PGPSelectKeysDialog(
											context,
											kPGPSelectKeysImportVariation,							
											CString(CStringFromSTRx(
												STRx_UtilStrings,
												stringID_SignerKeyImport)),							
											signerKeySet,							
											defaultKeySet,
											&selectedKeySet);
								if (IsntPGPError(result)) {
									result = PGPSharedAddKeysToDefaultKeyring(
												selectedKeySet);
									PGPFreeKeySet(selectedKeySet);
								}
							}
						}
						
						catch (...) {
							result = kPGPError_UnknownError;
						}
					}
					
					// Return the keyset
					if (IsntNull(newKeySet)) {
						*newKeySet = signerKeySet;
						signerKeySet = kInvalidPGPKeySetRef;
					}
				}
			}
			if (PGPKeySetRefIsValid(signerKeySet)) {
				PGPFreeKeySet(signerKeySet);
			}
		}
	}
	
	return result;
}
	


	PGPError
PGPGuaranteeMinimumEntropy(
	PGPContextRef		context)
{
	PGPError	err = kPGPError_NoErr;
	
	if( ! PGPGlobalRandomPoolHasMinimumEntropy() )
	{
		PGPUInt32	haveBits;
		PGPUInt32	neededBits;
		
		haveBits 	= PGPGlobalRandomPoolGetEntropy();
		neededBits 	= PGPGlobalRandomPoolGetMinimumEntropy();
		
		pgpAssert( haveBits < neededBits );
		
		if( haveBits < neededBits )
		{
			neededBits = neededBits - haveBits + 100;
		}
		else
		{
			neededBits = 300;	/* Token amount */
		}
		
		err = PGPCollectRandomDataDialog( context, neededBits,
						PGPOLastOption( context ) );
	}
	
	return( err );
}



	PGPError
PGPRSASupported(
	PGPBoolean *	hasRSA)
{
	PGPError					result = kPGPError_NoErr;
	PGPUInt32					numAlgs;
	PGPUInt32					index = 0;
	PGPPublicKeyAlgorithmInfo	info;
	
	PGPValidatePtr(hasRSA);
	*hasRSA = false;
	
	result = PGPCountPublicKeyAlgorithms(&numAlgs);
	while (IsntPGPError(result) && (index < numAlgs)) {
		result = PGPGetIndexedPublicKeyAlgorithmInfo(	index,
														&info);
		if (IsntPGPError(result)) {
			if ((info.algID == kPGPPublicKeyAlgorithm_RSA)
			|| (info.algID == kPGPPublicKeyAlgorithm_RSAEncryptOnly)
			|| (info.algID == kPGPPublicKeyAlgorithm_RSASignOnly)) {
				*hasRSA = true;
				break;
			}
		}
		index++;
	}
	
	return result;
}



	PGPError
PGPIsAdminConfigured(
	PGPBoolean *	isConfigured)
{
	PGPError	result = kPGPError_NoErr;
	FSSpec		libraryFileSpec;
	
	PGPValidatePtr(isConfigured);
	*isConfigured = false;
	
	try {
		result = PGPGetClientLibFSSpec(&libraryFileSpec);
		if (IsntPGPError(result)) {
			StSaveCurResFile	savedFile(libraryFileSpec);
	
			if (ResourceExists(kAdminConfiguredType, 
			PGPa_Configured, true)) {
				*isConfigured = true;
			}
		}
	}

	catch (...) {
		result = kPGPError_UnknownError;
	}
	
	return result;
}



	PGPError
PGPGetKeyIcon(
	PGPKeyRef	inKey,
	PGPIconID *	outIconID,
	Handle *	outSuite)
{
	PGPError	result = kPGPError_NoErr;
	PGPBoolean	expired;
	PGPBoolean	revoked;
	PGPBoolean	disabled;
	PGPBoolean	secret;
	PGPInt32	algorithm;
	PGPIconID	iconID;

	if (outIconID != NULL) {
		PGPValidatePtr(outIconID);
		*outIconID = (PGPIconID) 0;
	}
	if (outSuite != NULL) {
		PGPValidatePtr(outSuite);
		*outSuite = NULL;
	}
	PGPValidateParam(PGPKeyRefIsValid(inKey));
	
	PGPGetKeyBoolean(inKey, kPGPKeyPropIsExpired, &expired);
	PGPGetKeyBoolean(inKey, kPGPKeyPropIsRevoked, &revoked );
	PGPGetKeyBoolean(inKey, kPGPKeyPropIsDisabled, &disabled);
	PGPGetKeyBoolean(inKey, kPGPKeyPropIsSecret, &secret);
	PGPGetKeyNumber(inKey, kPGPKeyPropAlgID, &algorithm);
	if (revoked) {
		iconID = (algorithm == kPGPPublicKeyAlgorithm_DSA) ?	
					kPGPIconID_DSARevokedKey : kPGPIconID_RSARevokedKey;
	} else if (expired) {
		iconID = (algorithm == kPGPPublicKeyAlgorithm_DSA) ?	
					kPGPIconID_DSAExpiredKey : kPGPIconID_RSAExpiredKey;
	} else if (disabled) {
		iconID = (algorithm == kPGPPublicKeyAlgorithm_DSA) ?
					kPGPIconID_DSADisabledKey : kPGPIconID_RSADisabledKey;
	} else if (secret) {
		PGPBoolean	secShared;
		
		PGPGetKeyBoolean(inKey, kPGPKeyPropIsSecretShared,
						&secShared);
		if (secShared) {
			iconID = (algorithm == kPGPPublicKeyAlgorithm_DSA) ?
						kPGPIconID_DSASplitKeyPair : kPGPIconID_RSASplitKeyPair;
		} else {
			iconID = (algorithm == kPGPPublicKeyAlgorithm_DSA) ?
						kPGPIconID_DSAKeyPair : kPGPIconID_RSAKeyPair;
		}
	} else if (algorithm == kPGPPublicKeyAlgorithm_DSA) {
		iconID = kPGPIconID_DSAKey;
	} else {
		iconID = kPGPIconID_RSAKey;
	}
	
	if (outSuite != NULL) {
		result = PGPGetIconIDSuite(PGPGetKeyContext(inKey), iconID, outSuite);
	}
	
	if (IsntPGPError(result) && (outIconID != NULL)) {
		*outIconID = iconID;
	}
	
	return result;
}

	PGPError
PGPGetIconIDSuite(
	PGPContextRef	inContext,
	PGPIconID		inIconID,
	Handle *		outSuite)
{
	CComboError			result;
	Handle				suite;
	PGPclientLibState	state;
	
	PGPValidatePtr(outSuite);
	
	result.pgpErr = EnterPGPclientLib(inContext, &state);
	if (result.IsntError()) {
		result.err = ::GetIconSuite(&suite, inIconID, svAllAvailableData);
		if (result.err == resNotFound) {
			result.pgpErr = kPGPError_BadParams;
		}
		if (result.IsntError()) {
MacLeaks_Suspend();	// This suite will be disposed of in a different session,
					//so suspend leak checking
			result.err = DuplicateIconSuite(suite, outSuite);
MacLeaks_Resume();
			::DisposeIconSuite(suite, true);
		}
	}
	ExitPGPclientLib(&state);
	
	return result.ConvertToPGPError();
}

	void
PGPGetHashStringEven(
	PGPByte		index,
	Str255		str )
{
	pgpAssert( ( index >= 0 ) && ( index <= 255 ) );
	
	CToPString( (char *)hashWordListEven[index], str );
}

	void
PGPGetHashStringOdd(
	PGPByte		index,
	Str255		str )
{
	pgpAssert( ( index >= 0 ) && ( index <= 255 ) );
	
	CToPString( (char *)hashWordListOdd[index], str );
}



struct PGPClientEncodeDecodeData {
							PGPClientEncodeDecodeData(PGPContextRef inContext,
								PGPtlsContextRef inTLSContext);
							~PGPClientEncodeDecodeData();

	PGPContextRef					context;
	PGPMemoryMgrRef					memoryMgr;
	PGPtlsContextRef				tlsContext;
	CSignPassphraseCache *			signingPassphraseCache;
	CPassphraseCache *				decryptPassphraseCache;
	AnimatedCursorRef				encodeCursor;
	AnimatedCursorRef				decodeCursor;
	PGPUInt32						cursorDelay;
	PGPUInt8						frame;
	PGPUInt32						wordWrapWidth;
	PGPCipherAlgorithm *			allowedAlgorithms;
	PGPUInt32						numAlgorithms;
	PGPCipherAlgorithm				preferredAlgorithm;
	CString							commentString;
	PGPClientDecodeEventHandlerData	clientDecodeEventHandlerData;
	PGPByte **						decodeData;
	PGPBoolean						abort;
	PGPBoolean						fyeo;
	PGPBoolean						sigChecked;
	PGPBoolean						sigVerified;
	CString							beginSignedMessage;
	CString							endSignedMessage;
	CString							beginEncryptedAndSignedMessage;
	CString							endEncryptedAndSignedMessage;
};



PGPClientEncodeDecodeData::PGPClientEncodeDecodeData(
	PGPContextRef		inContext,
	PGPtlsContextRef	inTLSContext)
		:	tlsContext(inTLSContext), context(inContext), signingPassphraseCache(0),
			decryptPassphraseCache(0), encodeCursor(0), decodeCursor(0), wordWrapWidth(0),
			allowedAlgorithms(0), frame(0), beginSignedMessage(CStringFromSTRx(STRx_UtilStrings,
			stringID_BeginSignedMessage)), endSignedMessage(CStringFromSTRx(STRx_UtilStrings,
			stringID_EndSignedMessage)), beginEncryptedAndSignedMessage(CStringFromSTRx(
			STRx_UtilStrings, stringID_BeginEncryptedAndSignedMessage)),
			endEncryptedAndSignedMessage(CStringFromSTRx(STRx_UtilStrings,
			stringID_EndEncryptedAndSignedMessage))
{
	OSStatus						err;
	PGPError						pgpErr;
	PGPBoolean						wordWrapEnabled;
	PGPBoolean						cacheEnabled;
	PGPUInt32						cacheDuration;
	StPGPPrefRef					prefRef;
	PGPSize							bufSize;
	PGPCipherAlgorithm *			algorithms;
	PGPUInt32						numPublicKeys;
	PGPUInt32						numPrivateKeys;

	memoryMgr = PGPGetContextMemoryMgr(context);
	signingPassphraseCache = new CSignPassphraseCache(context);
	PGPThrowIfMemFail_(signingPassphraseCache);
	decryptPassphraseCache = new CSignPassphraseCache(context);
	PGPThrowIfMemFail_(decryptPassphraseCache);
	pgpErr = CountKeysInDefaultKeyring(context, &numPublicKeys, &numPrivateKeys);
	PGPThrowIfPGPError_(pgpErr);
	if (numPublicKeys == 0) {
		WarningAlert(kWACautionAlertType, kWAOKStyle, STRx_UtilStrings,
			stringID_NoPublicKeys);
	}
	if (numPrivateKeys == 0) {
		WarningAlert(kWACautionAlertType, kWAOKStyle, STRx_UtilStrings,
			stringID_NoPrivateKeys);
	}
			
#if PGP_BUSINESS_SECURITY
	pgpErr = PGPOpenAdminPrefs(memoryMgr, &prefRef);
	if (IsPGPError(pgpErr)) {
		WarningAlert(	kWAStopAlertType,
						kWAOKStyle,
						STRx_UtilStrings,
						stringID_AdminPrefsNotFound);
		PGPThrowPGPError_(kPGPError_UserAbort);
	}
	PGPGetPrefStringBuffer(prefRef, kPGPPrefComments, commentString.GetMinStorage(), commentString);
	prefRef.Free();
#endif

	pgpErr = PGPOpenClientPrefs(memoryMgr, &prefRef);
	PGPThrowIfPGPError_(pgpErr);
	pgpErr = PGPGetPrefBoolean(prefRef, kPGPPrefWordWrapEnable, &wordWrapEnabled);
	PGPThrowIfPGPError_(pgpErr);
	if (wordWrapEnabled) {
		pgpErr = PGPGetPrefNumber(prefRef, kPGPPrefWordWrapWidth, &wordWrapWidth);
		PGPThrowIfPGPError_(pgpErr);
	}
	if (commentString.GetLength() == 0) {
		PGPGetPrefStringBuffer(prefRef, kPGPPrefComment, commentString.GetMinStorage(), commentString);
	}
	PGPGetPrefData(prefRef, kPGPPrefAllowedAlgorithmsList, &bufSize, &algorithms);
	if (algorithms != 0) {
		allowedAlgorithms = static_cast<PGPCipherAlgorithm *>(PGPNewData(memoryMgr, bufSize,
								kPGPMemoryMgrFlags_None));
		if (allowedAlgorithms != 0) {
			pgpCopyMemory(algorithms, allowedAlgorithms, bufSize);
			numAlgorithms = bufSize / sizeof(PGPCipherAlgorithm);
		}
		PGPDisposePrefData(prefRef, algorithms);
	}
	pgpErr = PGPGetPrefNumber(prefRef, kPGPPrefPreferredAlgorithm, reinterpret_cast<PGPUInt32 *>(
				&preferredAlgorithm));
	if (IsPGPError(pgpErr)) {
		preferredAlgorithm = kPGPCipherAlgorithm_CAST5;
	}
	pgpErr = PGPGetPrefBoolean(prefRef, kPGPPrefSignCacheEnable, &cacheEnabled);
	PGPThrowIfPGPError_(pgpErr);
	if (cacheEnabled) {
		pgpErr = PGPGetPrefNumber(prefRef, kPGPPrefSignCacheSeconds, &cacheDuration);
		PGPThrowIfPGPError_(pgpErr);
	} else {
		cacheDuration = 0;
	}
	signingPassphraseCache->SetCacheSeconds(cacheDuration);
	pgpErr = PGPGetPrefBoolean(prefRef, kPGPPrefDecryptCacheEnable, &cacheEnabled);
	PGPThrowIfPGPError_(pgpErr);
	if (cacheEnabled) {
		pgpErr = PGPGetPrefNumber(prefRef, kPGPPrefDecryptCacheSeconds, &cacheDuration);
		PGPThrowIfPGPError_(pgpErr);
	} else {
		cacheDuration = 0;
	}
	decryptPassphraseCache->SetCacheSeconds(cacheDuration);

	err = Get1AnimatedCursor(acur_Encode, &encodeCursor);
	PGPThrowIfOSError_(err);
	SetAnimatedCursorInitialDelay(encodeCursor, 0);
	SetAnimatedCursorDelay(encodeCursor, 0);
	err = Get1AnimatedCursor(acur_Decode, &decodeCursor);
	PGPThrowIfOSError_(err);
	SetAnimatedCursorInitialDelay(decodeCursor, 0);
	SetAnimatedCursorDelay(decodeCursor, 0);
}



PGPClientEncodeDecodeData::~PGPClientEncodeDecodeData()
{
	if (encodeCursor != 0) {
		DisposeAnimatedCursor(encodeCursor);
	}
	if (decodeCursor != 0) {
		DisposeAnimatedCursor(decodeCursor);
	}
	if (allowedAlgorithms != 0) {
		PGPFreeData(allowedAlgorithms);
	}
	delete signingPassphraseCache;
	delete decryptPassphraseCache;
}



	PGPError
PGPInitClientEncodeDecode(
	PGPContextRef					inContext,
	PGPtlsContextRef				inTLSContext,
	PGPClientEncodeDecodeDataRef *	outRef)
{
	PGPError			result;
	PGPclientLibState	state;
	
	PGPValidatePtr(outRef);
	*outRef = 0;
	PGPValidateParam(PGPtlsContextRefIsValid(inTLSContext));
	PGPValidateParam(PGPContextRefIsValid(inContext));
	
	result = EnterPGPclientLib(inContext, &state);
	if (IsntPGPError(result)) {
		try {
			
			*outRef = new PGPClientEncodeDecodeData(inContext, inTLSContext);
			PGPThrowIfMemFail_(*outRef);
		}
		
		catch (CPGPException & error) {
			result = error.GetPGPError();
		}
		
		catch (...) {
			result = kPGPError_UnknownError;
		}
		
		ExitPGPclientLib(&state);
	}

	return result;
}



	void
PGPCleanupClientEncodeDecode(
	PGPClientEncodeDecodeDataRef	inRef)
{
	if (PGPClientEncodeDecodeDataRefIsValid(inRef)) {
		delete inRef;
	}
}



	PGPError
PGPClearClientEncodeDecodeCaches(
	PGPClientEncodeDecodeDataRef	inRef,
	PGPBoolean						inClearSigningCache,
	PGPBoolean						inClearDecryptCache)
{
	PGPValidateParam(PGPClientEncodeDecodeDataRefIsValid(inRef));
	
	if (inClearSigningCache) {
		inRef->signingPassphraseCache->Forget();
	}
	if (inClearDecryptCache) {
		inRef->decryptPassphraseCache->Forget();
	}
	
	return kPGPError_NoErr;
}


	static PGPError
PGPClientEDEncodeEventHandler(
	PGPContextRef					inContext,
	PGPEvent *						inEvent,
	PGPClientEncodeDecodeDataRef	inRef)
{
	(void) inContext;
	
	switch (inEvent->type) {
		case kPGPEvent_InitialEvent:
		{
			inRef->frame = 0;
			inRef->cursorDelay = ::TickCount();
			ResetAnimatedCursor(inRef->decodeCursor);
		}
		break;
		
		
		case kPGPEvent_NullEvent:
		{
			UInt32	newTicks = ::TickCount();
			SInt32	delay;

			while (inRef->frame < ((inEvent->data.nullData.bytesWritten * kNumFrames)
			/ inEvent->data.nullData.bytesTotal)) {
				AnimateCursor(inRef->encodeCursor);
				inRef->frame++;
				delay = kCursorDelay - (newTicks - inRef->cursorDelay);
				if (delay > 0) {
					UInt32	temp;
					Delay(delay, &temp);
				}
				inRef->cursorDelay = newTicks;
			}
		}
		break;
		
		case kPGPEvent_FinalEvent:
		{
			UInt32 temp;

			while (inRef->frame < kNumFrames) {
				AnimateCursor(inRef->encodeCursor);
				inRef->frame++;
				Delay(5, &temp);
			}
		}
		break;
	}
	
	return kPGPError_NoErr;
}



	PGPError
PGPClientEncode(
	PGPClientEncodeDecodeDataRef	inRef,
	PGPBoolean						inSign,
	PGPGetPassphraseSettings		inSignSettings,
	PGPGetPassphraseOptions			inSignOptions,
	PGPBoolean						inUseSigningCache,
	PGPBoolean						inClearSign,
	PGPBoolean						inDataIsText,
	PGPBoolean						inEncrypt,
	PGPRecipientSettings			inEncryptSettings,
	PGPRecipientOptions				inEncryptOptions,
	PGPUInt32						inNumDefaultRecipients,
	const PGPRecipientSpec *		inDefaultRecipients,
	PGPBoolean						inArmorOutput,
	const PGPByte *					inData,
	PGPSize							inDataSize,
	PGPByte **						outData,
	PGPSize *						outDataSize)
{
	PGPError			result;
	PGPclientLibState	state;
	
	PGPValidatePtr(outData);
	*outData = 0;
	PGPValidatePtr(outDataSize);
	*outDataSize = 0;
	PGPValidatePtr(inData);
	PGPValidateParam(PGPClientEncodeDecodeDataRefIsValid(inRef));
	PGPValidateParam(inDataSize > 0);
	
	result = EnterPGPclientLib(inRef->context, &state);
	if (IsntPGPError(result)) {
		try {
			PGPError		pgpErr;
			StPGPKeySetRef	allKeys;
			StPGPKeySetRef	recipientSet;
			StPGPDataRef	convEncryptPassphrase;
			PGPKeyRef		signingKey;
			CSecureMemory	signingBuffer(inRef->memoryMgr, 256);
			PGPSize			signingPasskeySize;
			Boolean			isPassphrase;
			StPGPDataRef	wrappedData;
			PGPSize			wrappedDataSize;
			
			pgpErr = PGPOpenDefaultKeyRings(inRef->context, kPGPKeyRingOpenFlags_None, &allKeys);
			PGPThrowIfPGPError_(pgpErr);
			
			if (inEncrypt) {
				StPGPKeySetRef	newKeys;
				
				pgpErr = PGPClientRecipientDialog(inRef->context, inRef->tlsContext, allKeys,
							inNumDefaultRecipients, inDefaultRecipients, inEncryptOptions,
							inEncryptSettings, &inEncryptSettings, &recipientSet, &newKeys);
				PGPThrowIfPGPError_(pgpErr);
				
				if (PGPKeySetRefIsValid(newKeys)) {
					StPGPKeySetRef	selectedSet;
					
					pgpErr = PGPSelectKeysDialog(inRef->context, kPGPSelectKeysImportVariation,
								CString(CStringFromSTRx(STRx_UtilStrings,
									stringID_ImportKeys)),
								newKeys, allKeys, &selectedSet);
					if (pgpErr != kPGPError_UserAbort) {
						PGPThrowIfPGPError_(pgpErr);
						pgpErr = PGPSharedAddKeysToDefaultKeyring(selectedSet);
						PGPThrowIfPGPError_(pgpErr);
					}
				}
				
				if ((inEncryptSettings & kPGPRecipientSettingsConvEncrypt) != 0) {
					pgpErr = PGPConventionalEncryptionPassphraseDialog(inRef->context,
								PGPOUIOutputPassphrase(inRef->context, reinterpret_cast<char **>(
									&convEncryptPassphrase)),
								PGPOLastOption(inRef->context));
					PGPThrowIfPGPError_(pgpErr);
				} else {
					PGPUInt32	numKeys;
					
					pgpErr = PGPCountKeys(recipientSet, &numKeys);
					PGPThrowIfPGPError_(pgpErr);
					if (numKeys == 0) {
						if (inSign) {
							inEncrypt = false;
						} else {
							PGPThrowPGPError_(kPGPError_UserAbort);
						}
					}
				}
			}
			
			if (inSign) {
				if (! inUseSigningCache) {
					inRef->signingPassphraseCache->Forget();
				}
				
				if (! inRef->signingPassphraseCache->GetPassphraseOrPasskey(allKeys,
				signingBuffer.mMemory, &isPassphrase, &signingPasskeySize, &signingKey)) {
					pgpErr = PGPClientSigningPassphraseDialog(inRef->context, allKeys,
								0, inSignOptions, inSignSettings, kInvalidPGPKeyRef,
								static_cast<char *>(signingBuffer.mMemory), &inSignSettings, &signingKey);
					if (pgpErr != kPGPError_KeyUnusableForSignature) {
						PGPThrowIfPGPError_(pgpErr);
						inRef->signingPassphraseCache->RememberPassphrase(
							static_cast<char *>(signingBuffer.mMemory), signingKey);
						isPassphrase = true;
					} else {
						StPGPDataRef	passkey;
						
						pgpErr = PGPReconstitutionDialog(signingKey, allKeys,
									inRef->tlsContext, &passkey, &signingPasskeySize);
						PGPThrowIfPGPError_(pgpErr);
						pgpCopyMemory(passkey, signingBuffer.mMemory, signingPasskeySize);
						inRef->signingPassphraseCache->RememberPasskey(passkey, signingPasskeySize,
							signingKey);
						isPassphrase = false;
					}
				}
				
				if ((! inEncrypt) && (inRef->wordWrapWidth > 0)) {
					StPGPIORef		inputIO;
					StPGPIORef		outputIO;
					PGPFileOffset	bufSize;
					
					pgpErr = PGPNewMemoryIOFixedBuffer(inRef->memoryMgr, const_cast<PGPByte *>(inData),
								inDataSize, reinterpret_cast<PGPMemoryIORef *>(&inputIO));
					PGPThrowIfPGPError_(pgpErr);
					pgpErr = PGPNewMemoryIO(inRef->memoryMgr,reinterpret_cast<PGPMemoryIORef *>(&outputIO));
					PGPThrowIfPGPError_(pgpErr);
					pgpErr = pgpWordWrapIO(inputIO, outputIO, inRef->wordWrapWidth, "\r");
					PGPThrowIfPGPError_(pgpErr);
					inputIO.Free();
					pgpErr = PGPIOSetPos(outputIO, 0);
					PGPThrowIfPGPError_(pgpErr);
					pgpErr = PGPIOGetEOF(outputIO, &bufSize);
					PGPThrowIfPGPError_(pgpErr);
					wrappedData = static_cast<PGPByte *>(PGPNewData(inRef->memoryMgr, bufSize,
									kPGPMemoryMgrFlags_None));
					PGPThrowIfMemFail_(wrappedData);
					pgpErr = PGPIORead(outputIO, bufSize, wrappedData, &wrappedDataSize);
					PGPThrowIfPGPError_(pgpErr);
				}	
			}
			
			pgpErr = PGPGuaranteeMinimumEntropy(inRef->context);
			PGPThrowIfPGPError_(pgpErr);
			
			::InitCursor();
			pgpErr = PGPEncode(	inRef->context,
						(wrappedData != 0) ? PGPOInputBuffer(inRef->context, wrappedData,
							wrappedDataSize) : PGPOInputBuffer(inRef->context, inData, inDataSize),
						PGPOAllocatedOutputBuffer(inRef->context, outData, MAX_PGPSize,
							outDataSize),
						PGPOEventHandler(inRef->context, reinterpret_cast<PGPEventHandlerProcPtr>(
							PGPClientEDEncodeEventHandler), inRef),
						PGPOSendNullEvents(inRef->context, 100),
						PGPOForYourEyesOnly(inRef->context, 
							(inEncryptSettings & kPGPRecipientSettingsFYEO) != 0),
						(inRef->commentString.GetLength() != 0) ? PGPOCommentString(inRef->context,
							inRef->commentString) : PGPONullOption(inRef->context),
						PGPOVersionString(inRef->context, pgpVersionHeaderString),
						(inEncrypt && PGPKeySetRefIsValid(recipientSet)) ?
							PGPOEncryptToKeySet(inRef->context, recipientSet) :
							PGPONullOption(inRef->context),
						(inRef->allowedAlgorithms != 0) ? PGPOPreferredAlgorithms(inRef->context,
							inRef->allowedAlgorithms, inRef->numAlgorithms) : PGPONullOption(
							inRef->context),
						(inEncrypt && ((inEncryptSettings & kPGPRecipientSettingsConvEncrypt) != 0)) ?
							PGPOConventionalEncrypt(inRef->context, PGPOPassphrase(inRef->context,
							convEncryptPassphrase), PGPOLastOption(inRef->context)) :
							PGPONullOption(inRef->context),
						(inEncrypt && ((inEncryptSettings & kPGPRecipientSettingsConvEncrypt) != 0)) ?
							PGPOCipherAlgorithm(inRef->context, inRef->preferredAlgorithm) :
							PGPONullOption(inRef->context),
						(inSign) ? PGPOSignWithKey(inRef->context, signingKey, (isPassphrase) ?
							PGPOPassphrase(inRef->context,
							static_cast<const char *>(signingBuffer.mMemory)) :
							PGPOPasskeyBuffer(inRef->context, signingBuffer.mMemory,
							signingPasskeySize), PGPOLastOption(inRef->context)) :
							PGPONullOption(inRef->context),
						PGPOArmorOutput(inRef->context, inArmorOutput),
						PGPODataIsASCII(inRef->context, inDataIsText),
						PGPOClearSign(inRef->context, inClearSign),
						PGPOLastOption(inRef->context));
			PGPThrowIfPGPError_(pgpErr);
		}
		
		catch (CPGPException & error) {
			result = error.GetPGPError();
		}
		
		catch (...) {
			result = kPGPError_UnknownError;
		}
	
		::InitCursor();
		ExitPGPclientLib(&state);
	}
	
	return result;
}



	static PGPError
PGPClientEDDecodeEventHandler(
	PGPContextRef					inContext,
	PGPEvent *						inEvent,
	PGPClientEncodeDecodeDataRef	inRef)
{
	PGPError	result;

	try {
		switch (inEvent->type) {
			case kPGPEvent_InitialEvent:
			{
				inRef->frame = 0;
				inRef->cursorDelay = ::TickCount();
				inRef->fyeo = false;
				inRef->sigChecked = true;
				inRef->sigVerified = true;
				inRef->abort = false;
				ResetAnimatedCursor(inRef->decodeCursor);
			}
			break;
			
			
			case kPGPEvent_NullEvent:
			{
				UInt32	newTicks = ::TickCount();
				SInt32	delay;

				while (inRef->frame < ((inEvent->data.nullData.bytesWritten * kNumFrames)
				/ inEvent->data.nullData.bytesTotal)) {
					AnimateCursor(inRef->decodeCursor);
					inRef->frame++;
					delay = kCursorDelay - (newTicks - inRef->cursorDelay);
					if (delay > 0) {
						UInt32	temp;
						Delay(delay, &temp);
					}
					inRef->cursorDelay = newTicks;
				}
			}
			break;
			
			
			case kPGPEvent_OutputEvent:
			{
				if (inEvent->data.outputData.forYourEyesOnly
				&& (! inRef->fyeo)) {
					PGPByte *	secureData = static_cast<PGPByte *>(PGPNewSecureData(
												inRef->memoryMgr,
												strlen(reinterpret_cast<char *>(
													*inRef->decodeData)) + 1,
												kPGPMemoryMgrFlags_None));
					
					PGPThrowIfMemFail_(secureData);
					strcpy(reinterpret_cast<char *>(secureData),
						reinterpret_cast<char *>(*inRef->decodeData));
					PGPFreeData(*inRef->decodeData);
					*inRef->decodeData = secureData;
					inRef->fyeo = true;
				}
			}
			break;
			
			
			case kPGPEvent_EndLexEvent:
			{
				if (inRef->clientDecodeEventHandlerData.outputBuffer != 0) {
					CString		sigString;
					CString *	prefix = 0;
					CString *	suffix = 0;
					PGPSize		newSize = inRef->clientDecodeEventHandlerData.
										outputBufferSize + strlen(reinterpret_cast<char *>(
										*inRef->decodeData)) + 1;
					
					if (inRef->clientDecodeEventHandlerData.signatureDataValid) {
						SignatureStatusInfo	status;
						
						status.sigData = inRef->
							clientDecodeEventHandlerData.signatureData;
						if (! status.sigData.checked) {
							inRef->sigChecked = false;
						}
						if (! status.sigData.verified) {
							inRef->sigVerified = false;
						}
						result = PGPGetKeyIDString(&status.sigData.signingKeyID,
									kPGPKeyIDString_Abbreviated, status.keyIDString);
						PGPThrowIfPGPError_(result);
						GetSignatureStatusMessage(&status, sigString);
						if (inRef->clientDecodeEventHandlerData.sectionType ==
						kPGPAnalyze_Encrypted) {
							prefix = &inRef->beginEncryptedAndSignedMessage;
							suffix = &inRef->endEncryptedAndSignedMessage;
						} else {
							prefix = &inRef->beginSignedMessage;
							suffix = &inRef->endSignedMessage;
						}
						newSize += sigString.GetLength() + prefix->GetLength()
							+ suffix->GetLength();
					}
					result = PGPReallocData(inRef->memoryMgr, inRef->decodeData,
								newSize, kPGPMemoryMgrFlags_None);
					PGPThrowIfPGPError_(result);
					if (prefix != 0) {
						strcat(reinterpret_cast<char *>(*inRef->decodeData), *prefix);
						strcat(reinterpret_cast<char *>(*inRef->decodeData), sigString);
					}
					strcat(reinterpret_cast<char *>(*inRef->decodeData),
						reinterpret_cast<char *>(inRef->clientDecodeEventHandlerData.
						outputBuffer));
					if (suffix != 0) {
						strcat(reinterpret_cast<char *>(*inRef->decodeData), *suffix);
					}
				}
			}
			break;
			
			
			case kPGPEvent_FinalEvent:
			{
				PGPError	pgpErr;
				char *		passphrase;
				PGPSize		passKeySize;
				
				if (! inRef->clientDecodeEventHandlerData.conventional) {
					switch (PGPGetPassBufferType(inRef->clientDecodeEventHandlerData.passBuffer)) {
						case kPGPPassBufferType_Passphrase:
						{
							pgpErr = PGPPassBufferGetPassphrasePtr(
										inRef->clientDecodeEventHandlerData.passBuffer, &passphrase);
							if (IsntPGPError(pgpErr)) {
								inRef->decryptPassphraseCache->RememberPassphrase(passphrase);
							}
						}
						break;
						
						
						case kPGPPassBufferType_PassKey:
						{
							pgpErr = PGPPassBufferGetPassKeyPtr(
										inRef->clientDecodeEventHandlerData.passBuffer,
										reinterpret_cast<PGPByte **>(&passphrase), &passKeySize);
							if (IsntPGPError(pgpErr)) {
								inRef->decryptPassphraseCache->RememberPasskey(
																reinterpret_cast<PGPByte *>(
																	passphrase),
																passKeySize);
							}
						}
						break;
					}
				}
				
				if (! inRef->abort) {
					UInt32 temp;

					while (inRef->frame < kNumFrames) {
						AnimateCursor(inRef->decodeCursor);
						inRef->frame++;
						Delay(5, &temp);
					}
				}
			}
			break;
		}

		result = PGPClientDecodeEventHandler(inContext, inEvent,
					&inRef->clientDecodeEventHandlerData);
		PGPThrowIfPGPError_(result);
	}
	
	catch (CPGPException & error) {
		result = error.GetPGPError();
	}
	
	catch (...) {
		result = kPGPError_UnknownError;
	}
	
	if (IsPGPError(result)) {
		inRef->abort = true;
	}
	
	return result;
}



	PGPError
PGPClientDecode(
	PGPClientEncodeDecodeDataRef	inRef,
	PGPBoolean						inUseDecryptCache,
	const PGPByte *					inData,
	PGPSize							inDataSize,
	PGPByte **						outData,
	PGPSize *						outDataSize)
{
	PGPError			result;
	PGPBoolean			cleanupClientDecodeEventHandler = false;
	PGPclientLibState	state;
	
	PGPValidatePtr(outData);
	*outData = 0;
	PGPValidatePtr(outDataSize);
	*outDataSize = 0;
	PGPValidatePtr(inData);
	PGPValidateParam(PGPClientEncodeDecodeDataRefIsValid(inRef));
	PGPValidateParam(inDataSize > 0);
	
	result = EnterPGPclientLib(inRef->context, &state);
	if (IsntPGPError(result)) {
		try {
			PGPError		pgpErr;
			StPGPKeySetRef	allKeys;
			StPGPDataRef	decodeData;
			CSecureMemory	decryptBuffer(inRef->memoryMgr, 256);
			PGPSize			decryptPasskeySize;
			PGPBoolean		isPassphrase;
			
			pgpErr = PGPOpenDefaultKeyRings(inRef->context, kPGPKeyRingOpenFlags_None, &allKeys);
			PGPThrowIfPGPError_(pgpErr);
			
			if (! inUseDecryptCache) {
				inRef->decryptPassphraseCache->Forget();
			}
			decodeData = static_cast<PGPByte *>(PGPNewData(inRef->memoryMgr, 1,
												kPGPMemoryMgrFlags_Clear));
			PGPThrowIfMemFail_(decodeData);
			inRef->decodeData = &decodeData;
			pgpErr = PGPInitClientDecodeEventHandlerData(inRef->context, inRef->tlsContext,
						allKeys, &inRef->clientDecodeEventHandlerData);
			PGPThrowIfPGPError_(pgpErr);
			cleanupClientDecodeEventHandler = true;
			pgpErr = PGPDecode(inRef->context, PGPOInputBuffer(inRef->context, inData, inDataSize),
						PGPOEventHandler(inRef->context, reinterpret_cast<PGPEventHandlerProcPtr>(
							PGPClientEDDecodeEventHandler), inRef),
						PGPOSendNullEvents(inRef->context, 100),
						PGPOKeySetRef(inRef->context, allKeys),
						PGPOSendEventIfKeyFound(inRef->context, true),
						PGPOPassThroughIfUnrecognized(inRef->context, true),
						(inRef->decryptPassphraseCache->GetPassphraseOrPasskey(
							decryptBuffer.mMemory, &isPassphrase, &decryptPasskeySize)) ?
							(isPassphrase) ? PGPOPassphrase(inRef->context,
							static_cast<const char *>(decryptBuffer.mMemory)) :
							PGPOPasskeyBuffer(inRef->context, decryptBuffer.mMemory,
							decryptPasskeySize) : PGPONullOption(inRef->context),
						PGPOLastOption(inRef->context));
			PGPThrowIfPGPError_(pgpErr);
			
			if (PGPKeySetRefIsValid(inRef->clientDecodeEventHandlerData.newKeySet)) {
				PGPUInt32		count;
				StPGPKeySetRef	selectedSet;
				
				pgpErr = PGPCountKeys(inRef->clientDecodeEventHandlerData.newKeySet,
							&count);
				PGPThrowIfPGPError_(pgpErr);
				if (count > 0) {
					pgpErr = PGPSelectKeysDialog(inRef->context,
								kPGPSelectKeysImportVariation,
								CString(CStringFromSTRx(STRx_UtilStrings, stringID_AddKeys)),
								inRef->clientDecodeEventHandlerData.newKeySet,
								allKeys, &selectedSet);
					
					if (pgpErr != kPGPError_UserAbort) {
						PGPThrowIfPGPError_(pgpErr);
						pgpErr = PGPSharedAddKeysToDefaultKeyring(selectedSet);
						PGPThrowIfPGPError_(pgpErr);
					}
				}
			}
			
			if ((! inRef->sigChecked) || (! inRef->sigVerified)) {
				WarningAlert(kWAStopAlertType, kWAOKStyle, STRx_UtilStrings,
					(inRef->sigChecked) ? stringID_BadSignature :
					stringID_SignatureNotVerified);
			}
			
			if (inRef->fyeo) {
				PGPForYourEyesOnlyDialog(inRef->context, decodeData);
				PGPThrowPGPError_(kPGPError_UserAbort);
			} else {
				*outData = decodeData;
				*outDataSize = strlen(decodeData);
				decodeData = 0;
			}
		}
		
		catch (CPGPException & error) {
			result = error.GetPGPError();
		}
		
		catch (...) {
			result = kPGPError_UnknownError;
		}
		
		if (cleanupClientDecodeEventHandler) {
			PGPCleanupClientDecodeEventHandlerData(&inRef->clientDecodeEventHandlerData);
		}
	
		::InitCursor();
		ExitPGPclientLib(&state);
	}
	
	return result;
}



	PGPUInt32
PGPGetPGPPrefsUpdateCount()
{
	SInt32		error;
	PGPUInt32 *	sharedMem;

	error = GetNamedBlock(kPGPPrefsUpdateCount, reinterpret_cast<Ptr *>(&sharedMem));
	if (error == err_BlockNotFound) {
		MacLeaks_Suspend();
		error = NewNamedBlock(kPGPPrefsUpdateCount, sizeof(PGPUInt32), reinterpret_cast<Ptr *>(&sharedMem));
		if (error == noErr) {
			*sharedMem = 1;
		}
		MacLeaks_Resume();
	}
	if (error == noErr) {
		return *sharedMem;
	} else {
		return 0;
	}
}



	void
PGPIncPGPPrefsUpdateCount()
{
	SInt32		error;
	PGPUInt32 *	sharedMem;

	error = GetNamedBlock(kPGPPrefsUpdateCount, reinterpret_cast<Ptr *>(&sharedMem));
	if (error == err_BlockNotFound) {
		MacLeaks_Suspend();
		error = NewNamedBlock(kPGPPrefsUpdateCount, sizeof(PGPUInt32), reinterpret_cast<Ptr *>(&sharedMem));
		if (error == noErr) {
			*sharedMem = 1;
		}
		MacLeaks_Resume();
	} else if (error == noErr) {
		*sharedMem += 1;
	}
}

	void
PGPMakeLicenseStr(
	PGPContextRef		context,
	Str255				outLicenseStr )
{
	PGPError			err;
	char				companyStr[128];
	char				nameStr[128];
	char				str[256];
	PGPclientLibState	state;
	
	err = EnterPGPclientLib( context, &state );
	
	companyStr[0] = '\0';
#if PGP_BUSINESS_SECURITY
	err = PGPGetPrefStringBuffer( state.adminPrefsRef,
			kPGPPrefAdminCompanyName, sizeof( companyStr ) - 1, companyStr );
	pgpAssertNoErr( err );
#endif
	if( companyStr[0] == '\0' )
	{
		err = PGPGetPrefStringBuffer( state.clientPrefsRef,
			kPGPPrefCompanyName, sizeof( companyStr ) - 1, companyStr );
		pgpAssertNoErr( err );
	}
	
	err = PGPGetPrefStringBuffer( state.clientPrefsRef, kPGPPrefOwnerName,
								sizeof( nameStr ) - 1, nameStr );
	pgpAssertNoErr( err );
	
	
	strcpy( str, nameStr );
	strcat( str, "\r" );
	strcat( str, companyStr );
	c2pstr( str );
	CopyPString( (uchar *)str, outLicenseStr );
	ExitPGPclientLib( &state );
}

	Boolean
PGPClientVerifyEnvironment(void)
{
	Boolean				canRun = TRUE;
	PGPclientLibState	state;
	
	if( IsntPGPError( EnterPGPclientLib( kInvalidPGPContextRef, &state ) ) )
	{
		if( ! HaveAppearanceMgr() )
		{
			SysBeep( 1 );
			
			StopAlert( kMinAppearanceVersionAlertResID, NULL );
			canRun = FALSE;
		}
	}

	ExitPGPclientLib( &state );
	
	return canRun;
}