/*____________________________________________________________________________
	Copyright (C) 1997 Network Associates Inc. and affiliated companies.
	All rights reserved.

	$Id: PGPKeyServerDialogs.cp,v 1.30 1999/03/10 02:52:16 heller Exp $
____________________________________________________________________________*/

#include <LowMem.h>
#include <OpenTptInternet.h>
#include <string.h>

#include <PP_Messages.h>
#include <UEnvironment.h>

#include "CServerStatusDialog.h"
#include "PGPclientLib.h"
#include "PGPclientLibUtils.h"
#include "PGPclientLibDialogs.h"
#include "WarningAlert.h"
#include "MacStrings.h"

#include "pgpMem.h"
#include "PGPOpenPrefs.h"
#include "pgpClientPrefs.h"
#include "pgpGroups.h"
#include "pgpKeyServer.h"
#include "pgpKeyServerPrefs.h"
#include "pgpErrors.h"

#include "pgpKeys.h"

const ResIDT	kPGPServerStatusDialogID		= 4757;
const PGPInt16	kMaxUserIDLength				= 256;
const ResIDT	kKSStringListID					= 8945;
enum	{	
			kKSAuthenticatePhraseStringID	 	= 9
		};

typedef struct KSCallbackInfo
{
	CServerStatusDialog	*dialog;
	PGPUInt32			lastState;
	ModalFilterUPP		filterUPP;
	Boolean				cancelled; /* KS lib not very good at reporting this */
	Boolean				dialogVisible;
	UInt32				displayTicks;
	DialogRef			dialogRef;
	PGPKeySetRef		allKeys;
	
} KSCallbackInfo;

static PGPError KSCallback(	PGPContextRef	context,
							PGPEvent		*event,
							PGPUserValue	userValue);

	PGPError
KSCallback(
	PGPContextRef	context,
	PGPEvent		*event,
	PGPUserValue	userValue)
{
	PGPError		err = kPGPError_NoErr;
	KSCallbackInfo	*callbackInfo;
	SInt16			saveWindowKind;
	Str255			caption;
	char			cstr[256];
	
	callbackInfo = (KSCallbackInfo *)userValue;
	
	/*
	** If the window kind is not a dialog, set it to dialogKind
	** during the callback
	*/
	
	saveWindowKind = GetWindowKind( callbackInfo->dialogRef );
	if( saveWindowKind != kDialogWindowKind )
		SetWindowKind( callbackInfo->dialogRef, kDialogWindowKind );
		
	if( ! callbackInfo->dialogVisible )
	{
		if( TickCount() >= callbackInfo->displayTicks )
		{
			SetPort( callbackInfo->dialogRef );
			ShowWindow( callbackInfo->dialogRef );
			SelectWindow( callbackInfo->dialogRef );
			UpdateDialog( callbackInfo->dialogRef,
					callbackInfo->dialogRef->visRgn );
			
			callbackInfo->dialogVisible = TRUE;
		}
	}
	
	if( callbackInfo->dialogVisible )
	{
		MessageT	dismissMessage;
		short		itemHit;

		ModalDialog( callbackInfo->filterUPP, &itemHit );
		dismissMessage = callbackInfo->dialog->GetDismissMessage();
		if( dismissMessage == msg_Cancel )
		{
			callbackInfo->cancelled = TRUE;
			err = kPGPError_UserAbort;
		}
	}
	
	if( IsntPGPError( err ) )
		switch( event->type )
		{
			case kPGPEvent_KeyServerIdleEvent:
				break;
			case kPGPEvent_KeyServerEvent:
				if( IsntPGPError( err ) )
				{
					Str255		caption;
					
					if( callbackInfo->lastState !=
							event->data.keyServerData.state )
					{
						GetIndString(caption, kKSStringListID,
										event->data.keyServerData.state);
						callbackInfo->dialog->SetStatusCaption(caption);
						callbackInfo->lastState =
							event->data.keyServerData.state;
					}
				}
				break;
			case kPGPEvent_KeyServerSignEvent:
			{
				PGPKeyRef		authKey = kInvalidPGPKeyRef;
				char			*passphrase = NULL;
				
				GetIndString( caption, kKSStringListID,
								kKSAuthenticatePhraseStringID );
				PToCString( caption, cstr );
				err = PGPSigningPassphraseDialog( context,
							callbackInfo->allKeys,
							&authKey,
							PGPOUIDialogPrompt( context, cstr ),
							PGPOUIOutputPassphrase( context, &passphrase ),
							PGPOLastOption( context ) );
				if( IsntPGPError( err ) && IsntNull( passphrase ) )
				{
					err = PGPAddJobOptions( event->job,
							PGPOSignWithKey( context, authKey,
								PGPOPassphraseBuffer( context, passphrase,
													strlen( passphrase ) ),
								PGPOLastOption( context ) ),
							PGPOClearSign( context, TRUE ),
							PGPOLastOption( context ) );
					PGPFreeData( passphrase );
				}
				break;
			}
			case kPGPEvent_KeyServerTLSEvent:
				// use authentication confirmation dialog+++++
				break;
		}
	SetWindowKind( callbackInfo->dialogRef, kApplicationWindowKind );
	
	return err;
}

	static PGPError
MyPGPQueryKeyServer(
	PGPKeyServerRef 		keyServer, 
	PGPFilterRef 			filter, 
	PGPKeySetRef 			*resultSet )
{
	PGPError	err;
	WindowRef	frontWindow;
	SInt16		saveWindowKind;
	
	/*
	**	OT/PPP has a bug which will cause any dialog in the foreground
	**	to be disposed within OT/PPP if the user cancels the connection.
	**	Our fix is to whack the windowKind of the foreground window if
	**	it is dialogKind.
	*/

	frontWindow = LMGetWindowList();
	if( IsntNull( frontWindow ) )
	{
		saveWindowKind = GetWindowKind( frontWindow );
		if( saveWindowKind == kDialogWindowKind )
			SetWindowKind( frontWindow, kApplicationWindowKind );
	}
	
	err = PGPQueryKeyServer( keyServer, filter,
				resultSet );
				
	if( IsntNull( frontWindow ) )
	{
		SetWindowKind( frontWindow, saveWindowKind );
	}
	
	return( err );
}

	static PGPError
QueryServer(
	PGPContextRef			context,
	PGPtlsContextRef		tlsContext,
	PGPPrefRef				clientPrefsRef,
	const PGPKeyID			*keyID,
	PGPKeyRef				keyRef,
	const char				*domain,
	PGPKeySetRef			*resultSet)
{
	PGPError				err	= kPGPError_NoErr;
	PGPKeyServerEntry		*ksEntries = NULL;
	PGPUInt32				numKSEntries;
	
	PGPValidatePtr( resultSet );
	PGPValidatePtr( context );
	PGPValidatePtr( domain );
	
	*resultSet = kInvalidPGPKeySetRef;
	
	err = PGPCreateKeyServerPath( clientPrefsRef, domain,
									&ksEntries, &numKSEntries);
	if( IsntPGPError( err ) && IsntNull(ksEntries) && numKSEntries > 0)
	{
		PGPKeyServerSpec	*serverList;
		
		serverList = (PGPKeyServerSpec *) PGPNewData(
							PGPGetContextMemoryMgr( context ),
							numKSEntries * sizeof( *serverList ),
							kPGPMemoryMgrFlags_Clear );
		if( IsntNull( serverList ) )
		{
			PGPUInt32			serverIndex;
			
			for( serverIndex = 0; serverIndex < numKSEntries; serverIndex++ )
			{
				err = PGPNewKeyServerFromHostName( context,
										ksEntries[serverIndex].serverDNS,
										ksEntries[serverIndex].serverPort,
										ksEntries[serverIndex].protocol,
										kPGPKeyServerAccessType_Normal,
										kPGPKeyServerKeySpace_Normal,
										&serverList[serverIndex].server );
				if( IsPGPError( err ) )
					break;
				
				serverList[serverIndex].serverName =
						ksEntries[serverIndex].serverDNS;
			}
			
			if( IsntPGPError( err ) )
			{
				PGPOptionListRef	searchObject;

				if( IsntNull( keyID ) )
				{
					pgpAssert( ! PGPKeyRefIsValid( keyRef ) );
					
					searchObject = PGPOUIKeyServerSearchKeyIDList( context,
								1, keyID );
				}
				else
				{
					pgpAssert( IsNull( keyID ) );
					
					searchObject = PGPOUIKeyServerSearchKey( context, keyRef );
				}
				
				err = PGPSearchKeyServerDialog( context, numKSEntries,
							serverList, tlsContext, FALSE, resultSet,
							searchObject, PGPOLastOption( context ) );
			}

			for( serverIndex = 0; serverIndex < numKSEntries; serverIndex++ )
			{
				if( PGPKeyServerRefIsValid( serverList[serverIndex].server ) )
					PGPFreeKeyServer( serverList[serverIndex].server );
			}

			PGPFreeData( serverList );
		}
	}	

	if( IsntNull( ksEntries ) )
		PGPDisposeKeyServerPath( ksEntries );
	
	return err;
}

	static PGPError
pgpGetKeyFromServer(
	PGPContextRef		context,
	PGPtlsContextRef	tlsContext,
	PGPPrefRef			clientPrefsRef,
	PGPKeyID const *	keyID,
	PGPKeyRef			key,
	PGPKeySetRef		*resultSet)
{
	PGPError	err	= kPGPError_NoErr;
	
	if( ! CFM_AddressIsResolved_( PGPKeyServerInit ) )
		return( kPGPError_FeatureNotAvailable );
	
	err = PGPKeyServerInit();
	if( IsntPGPError( err ) )
	{
		{
			StPGPPreserveKeyServerStorage	keyserverStorage;
			char							domainStr[kMaxUserIDLength];

			domainStr[0] = '\0';
			
			if( IsntNull( key ) )
			{
				char	userID[kMaxUserIDLength];
				PGPSize	len;
				
				// If we know the domain, let's see if we have a keyserver
				// for that domain
				
				len = kMaxUserIDLength;
				err = PGPGetPrimaryUserIDNameBuffer(key, sizeof( userID ),
													userID, &len);
				PGPFindEmailDomain( userID, domainStr );
			}
			
			if( IsntPGPError( err ) )
			{
				err = QueryServer( context, tlsContext, clientPrefsRef, keyID,
							key, domainStr, resultSet );
			}
		}
		
		PGPKeyServerCleanup();
	}
	
	return err;
}

	PGPError
PGPGetKeyFromServer(
	PGPContextRef		context,
	PGPtlsContextRef	tlsContext,
	PGPKeyRef			key,		/* if not-NULL, used for domain info */
	PGPKeySetRef		*resultSet)
{
	PGPclientLibState	state;
	PGPError		err;
	
	*resultSet = kInvalidPGPKeySetRef;
	
	err = EnterPGPclientLib( context, &state );
	if( IsntPGPError( err ) )
	{
		err = pgpGetKeyFromServer( context, tlsContext,
					state.clientPrefsRef, NULL, key, resultSet );
	}
	
	ExitPGPclientLib( &state );
	
	return err;
}

	PGPError
PGPGetKeyIDFromServer(
	PGPContextRef		context,
	PGPtlsContextRef	tlsContext,
	PGPKeyID const		*keyID,
	PGPKeySetRef		*resultSet)
{
	PGPclientLibState	state;
	PGPError		err;
	
	*resultSet = kInvalidPGPKeySetRef;
	
	err = EnterPGPclientLib( context, &state );
	if( IsntPGPError( err ) )
	{
		err = pgpGetKeyFromServer( context, tlsContext, state.clientPrefsRef,
						keyID, kInvalidPGPKeyRef, resultSet );
	}
	
	ExitPGPclientLib( &state );
	
	return err;
}

	PGPError
PGPGetGroupFromServerInternal(
	PGPContextRef		context,
	PGPtlsContextRef	tlsContext,
	PGPPrefRef			clientPrefsRef,
	PGPGroupSetRef		groupSet,
	PGPGroupID			groupID,
	const char			*domain,	/* if not-NULL, used for domain info */
	PGPUInt32			dialogDelayTicks,
	PGPKeySetRef		*resultSet)
{
	PGPError			err;
	char				domainStr[kMaxUserIDLength];
	PGPGroupSetRef		flattenedGroupSet;
	PGPGroupID			flattenedGroupID;
	PGPGroupItemIterRef	iterator;
	PGPKeySetRef		foundSet;
	
	if( ! CFM_AddressIsResolved_( PGPKeyServerInit ) )
		return( kPGPError_FeatureNotAvailable );
		
	flattenedGroupSet	= kInvalidPGPGroupSetRef;
	*resultSet 			= kInvalidPGPKeySetRef;
	iterator			= kInvalidPGPGroupItemIterRef;
	foundSet			= kInvalidPGPKeySetRef;
	
	domainStr[0] = '\0';

	if( IsntNull( domain) )
		PGPFindEmailDomain( domain, domainStr );
	
	/* Create a flattened group with unique key ID's and other setup*/
	err = PGPNewGroupSet( PGPGetGroupSetContext( groupSet ),
			&flattenedGroupSet );
	if( IsntPGPError( err ) )
	{
		err = PGPNewFlattenedGroupFromGroup( groupSet, groupID,
					flattenedGroupSet, &flattenedGroupID );
		if( IsntPGPError( err ) )
		{
			err = PGPNewGroupItemIter( flattenedGroupSet,
				flattenedGroupID,
				kPGPGroupIterFlags_Recursive | kPGPGroupIterFlags_Keys,
				&iterator );
			if( IsntPGPError( err ) )
			{
				err = PGPNewKeySet( context, &foundSet );
			}
		}
	}
	
	if( IsntPGPError( err ) )
	{
		CServerStatusDialog		*statusDialog;
		DialogRef				statusDialogRef;
		ModalFilterUPP			filterUPP;
		PGPKeyServerEntry		*ksEntries;
		PGPUInt32				numKSEntries;
		GrafPtr					savePort;
		KSCallbackInfo 			callbackInfo;
		
		GetPort( &savePort );
	
		statusDialogRef = 
			CClientKSGrafPortView::CreateDialog( kPGPServerStatusDialogID );
		pgpAssert( IsntNull( statusDialogRef ) );
		
		statusDialog = (CServerStatusDialog *) GetWRefCon( statusDialogRef );
		pgpAssert( IsntNull( statusDialog ) );
		
		filterUPP = NewModalFilterProc(
						CClientKSGrafPortView::ModalFilterProcHandler );
	
		InitCursor();
		SetPort( statusDialogRef );

		pgpClearMemory( &callbackInfo, sizeof( callbackInfo ) );
		
		callbackInfo.filterUPP		= filterUPP;
		callbackInfo.dialog			= statusDialog;
		callbackInfo.displayTicks	= TickCount() + dialogDelayTicks;
		callbackInfo.dialogRef		= statusDialogRef;

		err = PGPSetKeyServerIdleEventHandler( KSCallback, &callbackInfo );
		pgpAssertNoErr( err );

		err = PGPCreateKeyServerPath( clientPrefsRef, domainStr,
										&ksEntries, &numKSEntries);
		if( IsntPGPError( err ) &&
			IsntNull( ksEntries) &&
			numKSEntries > 0 )
		{
			PGPGroupItem	groupItem;
			
			err = PGPGroupItemIterNext( iterator, &groupItem );
			while( IsntPGPError( err ) )
			{
				PGPFilterRef	searchFilter;
				
				err = PGPNewKeyIDFilter( context, &groupItem.u.key.keyID,
							&searchFilter );
				if( IsntPGPError( err ) )
				{
					Boolean	found = FALSE;
					Str255	caption;

					GetIndString( caption, kKSStringListID,
							kPGPKeyServerState_Opening );
					statusDialog->SetStatusCaption( caption );
			
					for(Int16 serverIndex = 0; serverIndex < numKSEntries;
								serverIndex++)
					{
						PGPKeyServerRef	server;

						err = PGPNewKeyServerFromHostName( context,
										ksEntries[serverIndex].serverDNS,
										ksEntries[serverIndex].serverPort,
										ksEntries[serverIndex].protocol,
										kPGPKeyServerAccessType_Normal,
										kPGPKeyServerKeySpace_Normal,
										&server );
						pgpAssertNoErr( err );
						if( IsntPGPError( err ) )
						{
							PGPKeySetRef		resultKeys =
													kInvalidPGPKeySetRef;
							PGPtlsSessionRef	tls = kInvalidPGPtlsSessionRef;
							char				url[kMaxServerNameLength + 1];
							
							PGPGetKeyServerURL( &ksEntries[serverIndex], url );
							CToPString( url, caption );
							statusDialog->SetServerCaption( caption );

							err = PGPSetKeyServerEventHandler( server,
															KSCallback,
															&callbackInfo );
							pgpAssertNoErr( err );
							if( ksEntries[serverIndex].protocol ==
								kPGPKeyServerType_LDAPS )
							{
								err = PGPNewTLSSession( tlsContext, &tls );
							}
							err = PGPKeyServerOpen( server, tls );
							if( IsntPGPError( err ) )
							{
								err = MyPGPQueryKeyServer(server, searchFilter,
													&resultKeys );
								PGPKeyServerClose( server );
							}
							if( PGPtlsSessionRefIsValid( tls ) )
								PGPFreeTLSSession( tls );
							if( IsntPGPError( err ) &&
								PGPKeySetRefIsValid( resultKeys ) &&
								! callbackInfo.cancelled )
							{
								PGPUInt32	numKeys;

								err = PGPCountKeys(resultKeys, &numKeys);
								if( IsntPGPError( err ) && numKeys >= 1 )
								{
									found = TRUE;
									
									err = PGPAddKeys( resultKeys, foundSet );
									if( IsntPGPError( err ) )
									{
										err = PGPCommitKeyRingChanges(
													foundSet );
									}
								}
							}
							
							if( PGPKeySetRefIsValid( resultKeys ) )
								(void) PGPFreeKeySet( resultKeys );
							
							PGPFreeKeyServer( server );
							
							if( IsntPGPError( err ) &&
								callbackInfo.cancelled )
							{
								err = kPGPError_UserAbort;
							}
						}
					
						if( found || IsPGPError( err ) )
							break;
					}
					
					(void) PGPFreeFilter( searchFilter );
				}
				
				if( IsPGPError( err ) )
					break;
					
				err = PGPGroupItemIterNext( iterator, &groupItem );
			}
			
			if( err == kPGPError_EndOfIteration )
				err = kPGPError_NoErr;
				
			PGPDisposeKeyServerPath( ksEntries );
		}	
		
		DisposeRoutineDescriptor( filterUPP );
		delete statusDialog;
	
		DisposeDialog( statusDialogRef );
		SetPort( savePort );
		PGPSetKeyServerIdleEventHandler( NULL, NULL );
	}
	
	if( IsntPGPError( err ) )
	{
		*resultSet = foundSet;
	}
	else if( PGPKeySetRefIsValid( foundSet ) )
	{
		PGPFreeKeySet( foundSet );
	}
		
	if( PGPGroupItemIterRefIsValid( iterator ) )
		(void) PGPFreeGroupItemIter( iterator );
		
	if( PGPGroupSetRefIsValid( flattenedGroupSet ) )
		(void) PGPFreeGroupSet( flattenedGroupSet );
	
	
	return( err );
}

	PGPError
PGPGroupListOperationOnServerInternal(
	PGPContextRef		context,
	PGPtlsContextRef	tlsContext,
	PGPKeySetRef		allKeys,
	PGPPrefRef			clientPrefsRef,
	PGPBoolean			sendGroups,
	PGPGroupSetRef		*groupSet )
{
	PGPError			err = kPGPError_NoErr;
	PGPKeyRef			authKey = kInvalidPGPKeyRef;
	char				*passphrase = NULL;
		
	if( ! CFM_AddressIsResolved_( PGPKeyServerInit ) )
		return( kPGPError_FeatureNotAvailable );
		
	if( sendGroups )
	{
		Str255				caption;
		char				cstr[256];
		
		GetIndString( caption, kKSStringListID,
						kKSAuthenticatePhraseStringID );
		PToCString( caption, cstr );
		err = PGPSigningPassphraseDialog( context,
					allKeys,
					&authKey,
					PGPOUIDialogPrompt( context, cstr ),
					PGPOUIOutputPassphrase( context, &passphrase ),
					PGPOLastOption( context ) );
	}
	if( IsntPGPError( err ) && ( !sendGroups || IsntNull( passphrase ) ) )
	{
		CServerStatusDialog		*statusDialog;
		DialogRef				statusDialogRef;
		ModalFilterUPP			filterUPP;
		PGPKeyServerEntry		rootServer;
		GrafPtr					savePort;
		KSCallbackInfo 			callbackInfo;
		
		GetPort( &savePort );
	
		statusDialogRef = 
			CClientKSGrafPortView::CreateDialog( kPGPServerStatusDialogID );
		pgpAssert( IsntNull( statusDialogRef ) );
		
		statusDialog = (CServerStatusDialog *) GetWRefCon( statusDialogRef );
		pgpAssert( IsntNull( statusDialog ) );
		
		filterUPP = NewModalFilterProc(
			CClientKSGrafPortView::ModalFilterProcHandler );
	
		InitCursor();
		SetPort( statusDialogRef );

		pgpClearMemory( &callbackInfo, sizeof( callbackInfo ) );
		
		callbackInfo.filterUPP		= filterUPP;
		callbackInfo.dialog			= statusDialog;
		callbackInfo.displayTicks	= TickCount();
		callbackInfo.dialogRef		= statusDialogRef;
		callbackInfo.allKeys		= allKeys;

		err = PGPSetKeyServerIdleEventHandler( KSCallback, &callbackInfo );
		pgpAssertNoErr( err );
		
		err = PGPGetRootKeyServer( clientPrefsRef, &rootServer );
		if( IsntPGPError( err ) )
		{
			PGPKeyServerRef	server;
			Str255			caption;

			GetIndString( caption, kKSStringListID,
					kPGPKeyServerState_Opening );
			statusDialog->SetStatusCaption( caption );

			err = PGPNewKeyServerFromHostName( context,
							rootServer.serverDNS,
							rootServer.serverPort,
							rootServer.protocol,
							kPGPKeyServerAccessType_Normal,
							kPGPKeyServerKeySpace_Normal,
							&server );
			pgpAssertNoErr( err );
			if( IsntPGPError( err ) )
			{
				PGPtlsSessionRef	tls = kInvalidPGPtlsSessionRef;
				char				url[kMaxServerNameLength + 1];
				
				PGPGetKeyServerURL( &rootServer, url );
				CToPString( url, caption );
				statusDialog->SetServerCaption( caption );

				err = PGPSetKeyServerEventHandler( server,
												KSCallback,
												&callbackInfo );
				pgpAssertNoErr( err );
				if( rootServer.protocol == kPGPKeyServerType_LDAPS )
				{
					err = PGPNewTLSSession( tlsContext, &tls );
					if( IsntPGPError( err ) && sendGroups )
					{
						err = PGPtlsSetLocalPrivateKey( tls, authKey,
								kInvalidPGPSigRef, kInvalidPGPKeySetRef,
								PGPOPassphrase( context, passphrase ),
								PGPOLastOption( context ) );
					}
				}
				if( IsntPGPError( err ) )
					err = PGPKeyServerOpen( server, tls );
				if( IsntPGPError( err ) )
				{
					WindowRef	frontWindow;
					SInt16		saveWindowKind;
					
					frontWindow = LMGetWindowList();
					if( IsntNull( frontWindow ) )
					{
						saveWindowKind = GetWindowKind( frontWindow );
						if( saveWindowKind == kDialogWindowKind )
							SetWindowKind( frontWindow, kApplicationWindowKind );
					}
					
					if( sendGroups )
						err = PGPSendGroupsToServer( server, *groupSet );
					else
						err = PGPRetrieveGroupsFromServer( server, groupSet );
					
					if( IsntNull( frontWindow ) )
					{
						SetWindowKind( frontWindow, saveWindowKind );
					}
					PGPKeyServerClose( server );
				}
				PGPFreeKeyServer( server );
				if( PGPtlsSessionRefIsValid( tls ) )
					PGPFreeTLSSession( tls );
				
				if( IsntPGPError( err ) &&
					callbackInfo.cancelled )
				{
					err = kPGPError_UserAbort;
					if( !sendGroups )
					{
						PGPFreeGroupSet( *groupSet );
						*groupSet = kInvalidPGPGroupSetRef;
					}
				}
			}
		}	
		
		DisposeRoutineDescriptor( filterUPP );
		delete statusDialog;
	
		DisposeDialog( statusDialogRef );
		SetPort( savePort );
		PGPSetKeyServerIdleEventHandler( NULL, NULL );
	}
	
	if( IsntNull( passphrase ) )
		PGPFreeData( passphrase );
	
	return( err );
}

	PGPError
PGPGroupListOperationOnServer(
	PGPContextRef		context,
	PGPtlsContextRef	tlsContext,
	PGPKeySetRef		allKeys,
	PGPBoolean			sendGroups,
	PGPGroupSetRef		*groupSet )
{
	PGPclientLibState	state;
	PGPError			err;
	
	PGPValidatePtr( groupSet );
	if( !sendGroups )
		*groupSet = kInvalidPGPGroupSetRef;
	else
		PGPValidatePtr( *groupSet );
	
	err = EnterPGPclientLib( context, &state );
	if( IsntPGPError( err ) )
	{
		err = PGPGroupListOperationOnServerInternal( context, tlsContext,
					allKeys, state.clientPrefsRef, sendGroups, groupSet );
		
	}
	
	ExitPGPclientLib( &state );
	
	return err;
}

	PGPError
PGPGetGroupFromServer(
	PGPContextRef		context,
	PGPtlsContextRef	tlsContext,
	PGPGroupSetRef		groupSet,
	PGPGroupID			groupID,
	const char			*domain,	/* if not-NULL, used for domain info */
	PGPUInt32			dialogDelayTicks,
	PGPKeySetRef		*resultSet)
{
	PGPclientLibState	state;
	PGPError			err;
	
	*resultSet = kInvalidPGPKeySetRef;
	
	err = EnterPGPclientLib( context, &state );
	if( IsntPGPError( err ) )
	{
		err = PGPGetGroupFromServerInternal( context, tlsContext,
					state.clientPrefsRef, groupSet, groupID, domain,
					dialogDelayTicks, resultSet );
		
	}
	
	ExitPGPclientLib( &state );
	
	return err;
}

	static PGPError
PGPSendKeyToServerInternal(
	PGPContextRef		context,
	PGPtlsContextRef	tlsContext,
	PGPPrefRef			clientPrefsRef,
	PGPKeyRef			key,
	PGPKeyServerEntry	*targetServer)
{
	PGPKeySetRef			singleKeySet,
							failedKeys;
	char					userID[kMaxUserIDLength],
							domain[kMaxUserIDLength];
	PGPSize					len;
	PGPKeyServerEntry		*ksEntries;
	PGPUInt32				numKSEntries;
	PGPKeyServerEntry		serverInfo;
	PGPBoolean				foundServer = FALSE;
	PGPError				err = kPGPError_NoErr;

	PGPValidatePtr( key );
	PGPValidatePtr( context );
	
	/* Check for weak link failure */
	
	if( ! CFM_AddressIsResolved_( PGPKeyServerInit ) )
		return( kPGPError_FeatureNotAvailable );
		
	if(IsntNull(targetServer))
	{
		pgpCopyMemory( targetServer, &serverInfo, sizeof( PGPKeyServerEntry ) );
		foundServer = TRUE;
	}
	else
	{
		domain[0] = '\0';
		len = kMaxUserIDLength;
		err = PGPGetPrimaryUserIDNameBuffer(key, sizeof( userID ),
											userID, &len);
		pgpAssertNoErr(err);
		PGPFindEmailDomain(userID, domain);
		err = PGPCreateKeyServerPath(clientPrefsRef, domain,
									&ksEntries, &numKSEntries);
		pgpAssertNoErr(err);
		if( IsntPGPError( err ) )
		{
			if( IsntNull(ksEntries) && numKSEntries > 0 )
			{
				pgpCopyMemory( &ksEntries[0], &serverInfo,
								sizeof( PGPKeyServerEntry ) );
				foundServer = TRUE;
			}
			PGPDisposeKeyServerPath(ksEntries);
		}
	}
	
	if( IsntPGPError(err) && foundServer )
	{
		PGPKeyServerSpec	serverSpec;

		pgpClearMemory( &serverSpec, sizeof( serverSpec ) );
		
		serverSpec.serverName = serverInfo.serverDNS;
		
		err = PGPNewSingletonKeySet( key, &singleKeySet );
		pgpAssertNoErr(err);
		
		err = PGPNewKeyServerFromHostName( context,
					serverInfo.serverDNS,
					serverInfo.serverPort,
					serverInfo.protocol,
					kPGPKeyServerAccessType_Normal,
					kPGPKeyServerKeySpace_Normal,
					&serverSpec.server );
		pgpAssertNoErr(err);
		
		if( IsntPGPError( err ) )
		{
			err = PGPSendToKeyServerDialog(context, &serverSpec,
							tlsContext, singleKeySet, &failedKeys, 
							PGPOLastOption( context ) );
							
			if( PGPKeySetRefIsValid( failedKeys ) )
				PGPFreeKeySet( failedKeys );
				
			PGPFreeKeyServer( serverSpec.server );
		}
		
		PGPFreeKeySet( singleKeySet );
	}
		
	return err;
}

	PGPError
PGPSendKeyToServer(
	PGPContextRef		context,
	PGPtlsContextRef	tlsContext,
	PGPKeyRef			key,
	PGPKeyServerEntry	*targetServer)
{
	PGPclientLibState		state;
	PGPError				err;

	PGPValidatePtr( key );
	PGPValidatePtr( context );
	
	err = EnterPGPclientLib( context, &state );
	if( IsntPGPError( err ) )
	{
		err = PGPSendKeyToServerInternal( context, tlsContext,
						state.clientPrefsRef, key, targetServer );
	}
	
	
	ExitPGPclientLib( &state );

	return err;
}

