/*____________________________________________________________________________
	Copyright (C) 1999 Network Associates, Inc.
	All rights reserved.

	$Id: CHostsPanel.cp,v 1.36.2.2.2.1 1999/06/30 01:16:24 wprice Exp $
____________________________________________________________________________*/
#include <LEditText.h>
#include <LPopupButton.h>
#include <LPopupGroupBox.h>
#include <LPushButton.h>
#include <LStaticText.h>
#include <LTableView.h>
#include <LTextGroupBox.h>
#include <URegistrar.h>
#include <string.h>
#include <OpenTptInternet.h>

#include "CHostsPanel.h"
#include "CStatusPanel.h"
#include "CPGPHierarchyTable.h"
#include "CPGPnet.h"
#include "CPGPStDialogHandler.h"
#include "CString.h"
#include "MacStrings.h"
#include "WarningAlert.h"
#include "pgpClientLib.h"

#include "pgpMem.h"
#include "pgpIKE.h"
#include "pgpUserInterface.h"
#include "PGPnetIPC.h"

namespace
{
	const PaneIDT	button_Edit				=	'bEdi';
	const PaneIDT	button_Remove			=	'bRmv';
	const PaneIDT	button_Add				=	'bAdd';
	const PaneIDT	button_Initiate			=	'bIni';
	
	const PaneIDT	edit_DNS				=	'eDNS';
	const PaneIDT	caption_DNS				=	'cDNS';
	const PaneIDT	button_OK				=	'okay';
	const PaneIDT	chasers_DNS				=	'aCha';

	const MessageT	msg_Edit				=	button_Edit;
	const MessageT	msg_Remove				=	button_Remove;
	const MessageT	msg_Add					=	button_Add;
	const MessageT	msg_Initiate			=	button_Initiate;

	const PaneIDT	table_Hosts				=	'tHos';
		
	const ResIDT	dialog_AddHostExpert	=	3000;
	const ResIDT	dialog_DNSLookup		=	3003;
}

enum {
	STRx_HostsPanel							=	1004,
	kClearPassphraseString					=	1,
	kSetPassphraseString					=	2,
	kBadIPAddressString						=	3,
	kNoHostNameString						=	4,
	kConnectButtonString					=	5,
	kDisconnectButtonString					=	6,
	kDNSLookupFailedString					=	7,
	kIPAddressConflictString				=	8,
	kPGPAuthKeyString						=	9,
	k509AuthKeyString						=	10,
	kUnknownKeyString						=	11,
	kEditString								=	12,
	kAddString								=	13,
	kRemoveString							=	14,
	kConnectKString							=	15,
	kEditKString							=	16,
	kAddKString								=	17,
	kRemoveKString							=	18,
	kDisconnectKString						=	19
};


CHostsPanel::~CHostsPanel()
{
	PGPError	err;
	
	if( IsntNull( mHosts ) )
	{
		err = PGPDisposePrefData( CPGPnet::GetNetPrefs(), mHosts );
		pgpAssertNoErr( err );
	}
}

	void
CHostsPanel::FinishCreateSelf()
{
	RegisterClass_( CAddHostExpertDialog );
	
	mHosts = NULL;
	mNumHosts = 0;
	mCmdButtons = FALSE;

	// Listen to buttons
	( mConnectButton = (LPushButton *) FindPaneByID(button_Initiate))->
		AddListener(this);
	( mEditButton = (LPushButton *) FindPaneByID(button_Edit))->
		AddListener(this);
	( mAddButton = (LPushButton *) FindPaneByID(button_Add))->
		AddListener(this);
	( mRemoveButton = (LPushButton *) FindPaneByID(button_Remove))->
		AddListener(this);
	
	mHostsTable = (CHostsTable *) FindPaneByID(table_Hosts);
	mHostsTable->SetPanel( this );
	mHostsTable->AddListener( this );

	mButtonDisconnect = FALSE;
	Load( NULL );
	CPGPnet::GetApplication()->SetHosts( this );
	SetPanelTarget( mHostsTable );
	StartIdling();
}

	void
CHostsPanel::ListenToMessage(
	MessageT	inMessage,
	void *		/*ioParam*/)
{
	PGPError	err;
	STableCell	cell;
	PGPUInt32	hostIndex;
	UInt32		dataSize = sizeof(PGPUInt32);
	PGPBoolean	selected = FALSE;
	PGPBoolean	child;
	PGPBoolean	hostsChanged = FALSE;
	Str255		pstr;
	
	cell = mHostsTable->GetFirstSelectedCell();
	if( cell.row > 0 )
	{
		selected = TRUE;
		cell.row = mHostsTable->GetWideOpenIndex(cell.row);
		mHostsTable->GetCellData( cell, &hostIndex, dataSize );
		child = ( mHosts[hostIndex].childOf >= 0 );
	}
	switch (inMessage)
	{
		case coltable_SelectChange:
			if( selected )
			{
				((LPushButton *) FindPaneByID(button_Edit))->Enable();
				((LPushButton *) FindPaneByID(button_Remove))->Enable();
				((LPushButton *) FindPaneByID(button_Initiate))->Enable();
				if( mHostsTable->FindHostSA( hostIndex, NULL ) )
				{
					GetIndString( pstr, STRx_HostsPanel, kDisconnectButtonString );
					mConnectButton->SetDescriptor( pstr );
					mButtonDisconnect = TRUE;
				}
				else
				{
					GetIndString( pstr, STRx_HostsPanel, kConnectButtonString );
					mConnectButton->SetDescriptor( pstr );
					mButtonDisconnect = FALSE;
				}
			}
			else
			{
				((LPushButton *) FindPaneByID(button_Edit))->Disable();
				((LPushButton *) FindPaneByID(button_Remove))->Disable();
				((LPushButton *) FindPaneByID(button_Initiate))->Disable();
			}
			break;
		case coltable_DoubleClick:
			AddHostExpert( &cell, FALSE );
			hostsChanged = TRUE;
			break;
		case msg_Edit:
			if( selected )
			{
				AddHostExpert( &cell, FALSE );
				hostsChanged = TRUE;
			}
			break;
		case msg_Remove:
			if( selected )
			{
				TableIndexT		rows,
								cols;
				PGPUInt32		delRows = 0;
				
				do
				{
					if( hostIndex < mNumHosts - 1 )
						pgpCopyMemory( &mHosts[hostIndex + 1],
										&mHosts[hostIndex],
										sizeof(PGPNetPrefHostEntry) *
										( mNumHosts - hostIndex - 1 ) );
					else
						child = TRUE;
					mNumHosts--;
					delRows++;
				} while( !child && ( mHosts[hostIndex].childOf == hostIndex ) );
				err = PGPReallocData( CPGPnet::GetMemMgr(),
							&mHosts, mNumHosts *
							sizeof( PGPNetPrefHostEntry ), 0 );
				pgpAssertNoErr( err );
				mHostsTable->RemoveRows( 1, cell.row, true );
				mHostsTable->GetWideOpenTableSize( rows, cols );
				for( ; cell.row <= rows; cell.row++ )
				{
					mHostsTable->GetCellData( cell, &hostIndex, dataSize );
					hostIndex -= delRows;
					if(  mHosts[hostIndex].childOf != -1 )
						 mHosts[hostIndex].childOf--;
					mHostsTable->SetCellData( cell, &hostIndex, dataSize );
				}
				hostsChanged = TRUE;
			}
			break;
		case msg_Add:
			if( selected && !child &&
				( mHosts[hostIndex].hostType == kPGPnetSecureGateway ) )
			{
				AddHostExpert( &cell, TRUE );
			}
			else
				AddHostExpert( NULL, FALSE );
			hostsChanged = TRUE;
			break;
		case msg_Initiate:
		{
			PGPUInt32	saIndex,
						numSAs;
			PGPikeSA *	saList;
			
			if( selected &&
				!( ( ( mHosts[hostIndex].hostType == kPGPnetInsecureHost ) ||
					 ( mHosts[hostIndex].ipMask != 0xFFFFFFFF ) ) &&
					( mHosts[hostIndex].childOf < 0 ) ) )
			{
				if( mHostsTable->FindHostSA( hostIndex, &saIndex ) )
				{
					do
					{
						CPGPnet::GetApplication()->GetStatus()->
							GetSAList( &saList, &numSAs );
						CPGPnet::PostPNEvent( kPGPnetHLEvent_RemoveSA,
							&saList[saIndex], sizeof( PGPikeSA ) );
						CPGPnet::GetApplication()->GetStatus()->RemoveSA( saIndex );
						saIndex = 0;
					} while( mHostsTable->FindHostSA( hostIndex, &saIndex ) );
				}
				else
					CPGPnet::PostPNEvent( kPGPnetHLEvent_RequestSA, &hostIndex,
								sizeof(PGPUInt32) );
			}
			else
				SysBeep(1);
			break;
		}
	};
	if( hostsChanged )
	{
		err = PGPSetNetHostPrefs( CPGPnet::GetNetPrefs(), mHosts, mNumHosts );
		pgpAssertNoErr(err);
		err = PGPSavePrefFile( CPGPnet::GetNetPrefs() );
		pgpAssertNoErr(err);
		CPGPnet::PostPNEvent( kPGPnetHLEvent_UpdateConfig );
	}
}

	void
CHostsPanel::UpdateStatus()
{
	STableCell		tlCell,
					brCell;
	TableIndexT		rows,
					cols;
	
	mHostsTable->GetTableSize( rows, cols );
	tlCell.row	= 1;		tlCell.col	= 5;
	brCell.row	= rows;		brCell.col	= 5;
	mHostsTable->RefreshCellRange( tlCell, brCell );
	ListenToMessage( coltable_SelectChange, NULL );
}

	PGPBoolean
CHostsPanel::HostExists(
	PGPNetPrefHostEntry *	host,
	TableIndexT				exclude )
{
	PGPUInt32				hostIndex;
	PGPBoolean				exists = FALSE;
	
	for( hostIndex = 0; ( hostIndex < mNumHosts ) && !exists; hostIndex++ )
	{
		if( ( exclude > 0 ) && ( exclude - 1 == hostIndex ) )
			break;
		if( ( host->ipAddress == mHosts[hostIndex].ipAddress ) &&
			( host->ipMask == mHosts[hostIndex].ipMask ) )
			exists = TRUE;
		else if( host->ipMask != 0xFFFFFFFF )
		{
			PGPUInt32	strictMask = ( host->ipMask > mHosts[hostIndex].ipMask ) ?
									host->ipMask : mHosts[hostIndex].ipMask;
			
			if( ( host->ipAddress & strictMask ) ==
				( mHosts[hostIndex].ipAddress & strictMask ) )
				exists = TRUE;
		}
	}
	return exists;
}

	void
CHostsPanel::AddNewHost(
	PGPNetPrefHostEntry *	newHost )
{
	PGPError	err;
	
	mNumHosts++;
	err = PGPReallocData( CPGPnet::GetMemMgr(), &mHosts,
			mNumHosts * sizeof( PGPNetPrefHostEntry ), 0 );
	if( IsntPGPError( err ) )
	{
		PGPUInt32	hostIndex = mNumHosts - 1;
		
		pgpCopyMemory( newHost, &mHosts[hostIndex],
			sizeof(PGPNetPrefHostEntry) );
		mHostsTable->InsertSiblingRows( 1, LArray::index_Last,
				&hostIndex, sizeof(PGPUInt32), FALSE, TRUE );
		err = PGPSetNetHostPrefs( CPGPnet::GetNetPrefs(), mHosts, mNumHosts );
		pgpAssertNoErr(err);
		err = PGPSavePrefFile( CPGPnet::GetNetPrefs() );
		pgpAssertNoErr(err);
	}
	PGPFreeData( newHost );
}

	void
CHostsPanel::AddHostExpert(
	STableCell *			cell,
	PGPBoolean				child )
{
	CPGPnetStDialogHandler addHostDialog( dialog_AddHostExpert,
										CPGPnet::GetApplication() );
	MessageT				dialogMessage;
	PGPUInt32				hostIndex;
	PGPBoolean				valid = FALSE;
	Rect					frame;
	PGPError				err;
	Uint32					dataSize = sizeof(PGPUInt32);
	CAddHostExpertDialog *	ahDialog = 
		(CAddHostExpertDialog *)addHostDialog.GetDialog();
	
	if( IsntNull( cell ) )
	{
		mHostsTable->GetCellData( *cell, &hostIndex, dataSize );
		if( child )
			ahDialog->SetInfo( NULL, child );
		else
			ahDialog->SetInfo( &mHosts[hostIndex], child );
	}
	else
		ahDialog->SetInfo( NULL, child );

tryAgain:
	do
	{
		dialogMessage = addHostDialog.DoDialog();
	} while( dialogMessage != msg_OK && dialogMessage != msg_Cancel );
	if(dialogMessage == msg_OK)
	{
		PGPNetPrefHostEntry	tHost;
		
		pgpClearMemory( &tHost, sizeof( PGPNetPrefHostEntry ) );
		tHost.childOf = -1;
		if( IsntNull( cell ) && !child )
		{
			PGPBoolean	wasGateway =
							( mHosts[hostIndex].hostType == kPGPnetSecureGateway );
			if( mHosts[hostIndex].childOf != -1 )
				tHost.childOf = mHosts[hostIndex].childOf;
			valid = ahDialog->GetInfo( &tHost );
			if( !valid )
				goto tryAgain;
			if( HostExists( &tHost, cell->row ) )
			{
				WarningAlert( kWAStopAlertType, kWAOKStyle, STRx_HostsPanel,
					kIPAddressConflictString );
				goto tryAgain;
			}
			pgpCopyMemory( &tHost, &mHosts[hostIndex],
				sizeof( PGPNetPrefHostEntry ) );
			if( !wasGateway &&
				mHosts[hostIndex].hostType == kPGPnetSecureGateway )
				mHostsTable->MakeCollapsable( cell->row, TRUE );
			else if( wasGateway &&
					( mHosts[hostIndex].hostType != kPGPnetSecureGateway ) )
			{
				UInt32	descendants;
				
				descendants = mHostsTable->CountAllDescendents( cell->row );
				if( descendants )
				{
					mHostsTable->RemoveRows( descendants, cell->row + 1, true );
					if( hostIndex + descendants < mNumHosts - 1 )
					{
						pgpCopyMemory( &mHosts[hostIndex + 1 + descendants],
							&mHosts[hostIndex + 1],
							sizeof( PGPNetPrefHostEntry )
							* ( mNumHosts - 1 - ( hostIndex + descendants ) ) );
					}
					mNumHosts -= descendants;
					err = PGPReallocData( CPGPnet::GetMemMgr(),
								&mHosts, mNumHosts *
								sizeof( PGPNetPrefHostEntry ), 0 );
					pgpAssertNoErr( err );
				}
			}
			else
			{
				mHostsTable->CalcPortFrameRect(frame);
				mHostsTable->InvalPortRect(&frame);
			}
		}
		else
		{
			valid = ahDialog->GetInfo( &tHost );
			if( !valid )
				goto tryAgain;
			if( HostExists( &tHost ) )
			{
				WarningAlert( kWAStopAlertType, kWAOKStyle, STRx_HostsPanel,
					kIPAddressConflictString );
				goto tryAgain;
			}
			err = PGPReallocData( CPGPnet::GetMemMgr(),
						&mHosts, ( mNumHosts + 1 ) *
						sizeof( PGPNetPrefHostEntry ), 0 );
			if( IsntPGPError( err ) )
			{
				if( child )
				{
					STableCell	tcel = *cell;
					TableIndexT	rows,
								cols;
					
					if( hostIndex < mNumHosts - 1 )
						pgpCopyMemory( &mHosts[hostIndex + 1],
									&mHosts[hostIndex + 2],
									sizeof( PGPNetPrefHostEntry ) *
									( mNumHosts - hostIndex - 1 ) );
					tHost.childOf = hostIndex;
					hostIndex++;
					pgpCopyMemory( &tHost, &mHosts[hostIndex],
						sizeof( PGPNetPrefHostEntry ) );
					mHostsTable->InsertChildRows( 1, tcel.row,
							&hostIndex, sizeof(PGPUInt32), FALSE, TRUE );
					mHostsTable->GetWideOpenTableSize( rows, cols );
					for( tcel.row += 2; tcel.row <= rows; tcel.row++ )
					{
						mHostsTable->GetCellData( tcel, &hostIndex, dataSize );
						hostIndex++;
						mHostsTable->SetCellData( tcel, &hostIndex, dataSize );
					}
				}
				else
				{
					PGPUInt32	revIndex = mNumHosts - 1;
					
					pgpCopyMemory( &tHost, &mHosts[mNumHosts],
						sizeof( PGPNetPrefHostEntry ) );
					hostIndex = mNumHosts;
					if( mNumHosts > 0 )
					{
						
						do
						{
							if( mHosts[revIndex].childOf == -1 )
								break;
						} while( revIndex-- > 0 );
					}
					mHostsTable->InsertSiblingRows( 1, revIndex + 1,
							&hostIndex, sizeof(PGPUInt32), FALSE, TRUE );
				}
				mNumHosts ++;
			}
		}
	}
}

	void
CHostsPanel::Load(
	void *			/*inParam*/)
{
	PGPError		err;
	TableIndexT		lastTopRow = 0,
					childRow;
	PGPUInt32		hostIndex = 0;
	PGPBoolean		lastChild = FALSE;
	STableCell		cell(1, 1);
	TableIndexT		rows,
					cols,
					rowIndex;
	
	pgpAssert( IsNull( mHosts ) );
	err = PGPGetNetHostPrefs( CPGPnet::GetNetPrefs(), &mHosts, &mNumHosts );
	pgpAssertNoErr( err );
	
	if( mNumHosts )
	{
		for( hostIndex = 0; hostIndex < mNumHosts; hostIndex++ )
		{
			if( mHosts[hostIndex].childOf >= 0 )
			{
				if( lastChild )
				{
					childRow = mHostsTable->InsertSiblingRows(1,
							childRow, &hostIndex, sizeof(PGPUInt32),
							FALSE, TRUE);
				}
				else
				{
					mHostsTable->InsertChildRows(1, lastTopRow, &hostIndex,
									sizeof(PGPUInt32), FALSE, TRUE);
					childRow = lastTopRow + 1;
				}
				lastChild = TRUE;
			}
			else
			{
				PGPBoolean	collapsable = FALSE;
				
				if( mHosts[hostIndex].hostType == kPGPnetSecureGateway )
					collapsable = TRUE;
				lastTopRow = mHostsTable->InsertSiblingRows(1, lastTopRow,
						&hostIndex, sizeof(PGPUInt32), collapsable, TRUE);
				lastChild = FALSE;
			}
		}
	}
	mHostsTable->GetWideOpenTableSize(rows, cols);
	for(rowIndex = rows;rowIndex > 0;rowIndex--)
	{
		cell.row = mHostsTable->GetExposedIndex(rowIndex);
		if( mHostsTable->IsCollapsable(rowIndex) )
			mHostsTable->DeepCollapseRow(rowIndex);
	}
	cell.row = cell.col = 1;
	mHostsTable->ScrollCellIntoFrame(cell);
}

	void
CHostsPanel::Save(
	void * )
{
}

	void
CHostsPanel::SpendTime(
	const EventRecord &inMacEvent )
{
	Str255	pstr;
	
	if( IsVisible() )
	{
		if( inMacEvent.what == nullEvent )
			if( inMacEvent.modifiers & cmdKey )
			{
				if( !mCmdButtons )
				{
					if( IsActive() )
					{
						if( mButtonDisconnect )
							GetIndString( pstr, STRx_HostsPanel , kDisconnectKString );
						else
							GetIndString( pstr, STRx_HostsPanel , kConnectKString );
						mConnectButton->SetDescriptor( pstr );
						GetIndString( pstr, STRx_HostsPanel , kEditKString );
						mEditButton->SetDescriptor( pstr );
						GetIndString( pstr, STRx_HostsPanel , kAddKString );
						mAddButton->SetDescriptor( pstr );
						GetIndString( pstr, STRx_HostsPanel , kRemoveKString );
						mRemoveButton->SetDescriptor( pstr );
						mCmdButtons = TRUE;
					}
				}
			}
			else
			{
				if( mCmdButtons )
				{
					if( mButtonDisconnect )
						GetIndString( pstr, STRx_HostsPanel , kDisconnectButtonString );
					else
						GetIndString( pstr, STRx_HostsPanel , kConnectButtonString );
					mConnectButton->SetDescriptor( pstr );
					GetIndString( pstr, STRx_HostsPanel , kEditString );
					mEditButton->SetDescriptor( pstr );
					GetIndString( pstr, STRx_HostsPanel , kAddString );
					mAddButton->SetDescriptor( pstr );
					GetIndString( pstr, STRx_HostsPanel , kRemoveString );
					mRemoveButton->SetDescriptor( pstr );
					mCmdButtons = FALSE;
				}
			}
	}
}

	Boolean
CHostsPanel::HandleKeyPress(const EventRecord&	inKeyEvent)
{
	Boolean		keyHandled	= true;
	char		theKey		= inKeyEvent.message & charCodeMask;

	if( (inKeyEvent.modifiers & cmdKey) )
	{
		switch( toupper(theKey) )
		{
			case 'S':
				ListenToMessage( msg_Initiate, NULL );
				break;
			case 'E':
				ListenToMessage( msg_Edit, NULL );
				break;
			case 'D':
				ListenToMessage( msg_Add, NULL );
				break;
			case 'R':
				ListenToMessage( msg_Remove, NULL );
				break;
			default:
				keyHandled = false;
				break;
		}
		
	}
	else
		keyHandled = false;
	if( !keyHandled )
		keyHandled = LCommander::HandleKeyPress(inKeyEvent);
	return keyHandled;
}

	void
CHostsTable::FinishCreateSelf()
{
	CColumnTable::FinishCreateSelf();
}

	Boolean
CHostsTable::DrawCellCustom(
	const STableCell		&inCell,
	const Rect				&inLocalRect,
	Boolean					 )
{
	if( inCell.col == 4 )
	{
		PGPUInt32			hostIndex;
		UInt32				dataSize = sizeof(PGPUInt32);
		PGPNetPrefHostEntry *hosts = mPanel->mHosts;
		STableCell			cell = inCell;
		
		cell.col = 1;
		GetCellData( cell, &hostIndex, dataSize );
		if( hosts[hostIndex].hostType != kPGPnetInsecureHost )
		{
			if( hosts[hostIndex].sharedSecret[0] )
				DrawIcon( inCell, icon_SharedSecret, inLocalRect, 0, kTransformNone );
			if( CPGPnet::HasPGPAuth() )
				DrawIcon( inCell, icon_PGPKey, inLocalRect, 1, kTransformNone );
			if( CPGPnet::HasX509Auth() )
				DrawIcon( inCell, icon_X509Cert, inLocalRect, 2, kTransformNone );
		}
	}
	return FALSE;
}

	Boolean
CHostsTable::GetCellDrawData(
	STableCell			inCell,
	ResIDT				&iconID,
	Int16				&indentLevel,
	Str255				data,
	StyleParameter		& )
{
	PGPUInt32			hostIndex;
	UInt32				dataSize = sizeof(PGPUInt32);
	PGPNetPrefHostEntry *hosts = mPanel->mHosts;
	char				cstr[32];
	TableIndexT			col;
	Boolean				custom = FALSE;
	
	col = inCell.col;
	inCell.col = 1;
	GetCellData( inCell, &hostIndex, dataSize );
	switch( col )
	{
		case 1:
			if( hosts[hostIndex].childOf >= 0 )
				indentLevel = 1;
			CToPString( hosts[hostIndex].hostName, data );
			switch( hosts[hostIndex].hostType )
			{
				case kPGPnetInsecureHost:
					if( hosts[hostIndex].ipMask < 0xFFFFFFFF )
						iconID = icon_InsecSubnet;
					else
						iconID = icon_InsecHost;
					break;
				case kPGPnetSecureHost:
					if( hosts[hostIndex].ipMask < 0xFFFFFFFF )
						iconID = icon_SecSubnet;
					else
						iconID = icon_SecHost;
					break;
				case kPGPnetSecureGateway:
					iconID = icon_Gateway;
					break;
			}
			break;
		case 2:
			OTInetHostToString( hosts[hostIndex].ipAddress, cstr );
			CToPString( cstr, data );
			break;
		case 3:
			if( hosts[hostIndex].ipMask < 0xFFFFFFFF )
			{
				OTInetHostToString( hosts[hostIndex].ipMask, cstr );
				CToPString( cstr, data );
			}
			break;
		case 4:
			custom = TRUE;
			break;
		case 5:
			if( FindHostSA( hostIndex, NULL ) )
				iconID = icon_GreenDot;
			break;
	}
	return custom;
}

	PGPBoolean
CHostsTable::FindHostSA(
	PGPUInt32			hostIndex,
	PGPUInt32 *			saIndexP )
{
	PGPUInt32			saIndex,
						numSAs;
	PGPikeSA *			saList;
	PGPNetPrefHostEntry *hosts = mPanel->mHosts;

	CPGPnet::GetApplication()->GetStatus()->GetSAList( &saList, &numSAs );
	for( saIndex = 0; saIndex < numSAs; saIndex++ )
	{
		if( ( saList[saIndex].ipAddress == hosts[hostIndex].ipAddress ) ||
			( ( saList[saIndex].ipAddress & hosts[hostIndex].ipMask ) ==
				( hosts[hostIndex].ipAddress & hosts[hostIndex].ipMask ) ) ||
			( ( hosts[hostIndex].childOf != -1 ) && 
				( hosts[hostIndex].ipAddress == saList[saIndex].ipAddrStart ) &&
					( hosts[hostIndex].ipMask == saList[saIndex].ipMaskEnd ) )
			 )
		{
			*saIndexP = saIndex;
			return TRUE;
		}
	}
	return FALSE;
}

	void
CAddHostExpertDialog::FinishCreateSelf()
{
	LGADialog::FinishCreateSelf();
	
	mPGPRadio = (LRadioButton *)FindPaneByID( radio_PGP );
	m509Radio = (LRadioButton *)FindPaneByID( radio_X509 );
	mAnyRadio = (LRadioButton *)FindPaneByID( radio_AnyKey );
	
	mKeyCaption = (LStaticText *)FindPaneByID( caption_Key );
	mKeyIcon	= (CIconPane *)FindPaneByID( icon_Key );
	
	mPGPAuthKey 	= kInvalidPGPKeyRef;
	m509AuthKey 	= kInvalidPGPKeyRef;
	m509AuthCert	= kInvalidPGPSigRef;
}

	void
CAddHostExpertDialog::SetInfo(
	PGPNetPrefHostEntry *		entry,
	PGPBoolean					child )
{
	Str255						pstr;
	char						cstr[256];
	LPopupGroupBox *			typePopup;
	PGPBoolean					subnet = FALSE,
								secret = FALSE;
	PGPUInt32					idType;
	MenuHandle					typeMenu;
	
	if( IsntNull( entry ) )
		pgpCopyMemory( entry, &mEntry, sizeof( PGPNetPrefHostEntry ) );
	else
	{
		InetInterfaceInfo info;

		pgpClearMemory( &mEntry, sizeof( PGPNetPrefHostEntry ) );
		mEntry.hostType			= kPGPnetSecureHost;
		mEntry.childOf			= -1;
		mEntry.authKeyAlg		= kPGPPublicKeyAlgorithm_Invalid;
		mEntry.ipMask			= 0xFFFFFFFF;
		mEntry.identityType		= kPGPike_ID_IPV4_Addr;
		if( !OTInetGetInterfaceInfo( &info, 0 ) )
			OTInetHostToString( info.fAddress, (char *)mEntry.identity );
		else
			strcpy( (char *)mEntry.identity, "0.0.0.0" );
		mEntry.identityLen		= strlen( (char *)mEntry.identity );
	}
	typePopup = (LPopupGroupBox *)FindPaneByID( popup_HostType );
	( (LPushButton *)FindPaneByID( button_DNSLookup ) )->AddListener(this);
	((LPushButton *)FindPaneByID( button_SecretPhrase ))->AddListener(this);

	CToPString( mEntry.hostName, pstr );
	( (LEditText *)FindPaneByID( edit_HostName ) )->SetDescriptor( pstr );
	OTInetHostToString( mEntry.ipAddress, cstr );
	CToPString( cstr, pstr );
	( (LEditText *)FindPaneByID( edit_IPAddress ) )->SetDescriptor( pstr );
	OTInetHostToString( mEntry.ipMask, cstr );
	CToPString( cstr, pstr );
	( (LEditText *)FindPaneByID( edit_Mask ) )->SetDescriptor( pstr );
	if( mEntry.sharedSecret[0] )
	{
		GetIndString( pstr, STRx_HostsPanel, kClearPassphraseString );
		( (LPushButton *)FindPaneByID( button_SecretPhrase ) )->
			SetDescriptor( pstr );
	}
	if( mEntry.identityType )
	{
		LEditText *		editID = (LEditText *)FindPaneByID( edit_Identity );
		LPopupButton *	popupID = (LPopupButton *)FindPaneByID( popup_IdentityType );
		
		switch( mEntry.identityType )
		{
			case kPGPike_ID_IPV4_Addr:
			case kPGPike_ID_FQDN:
			case kPGPike_ID_UserFDQN:
				idType = mEntry.identityType;
				break;
			case kPGPike_ID_DER_ASN1_DN:
				idType = 4;
				break;
			default:
				idType = 1;
				break;
		}
		popupID->SetValue( idType );
		popupID->AddListener(this);
		CToPString( (char *)mEntry.identity, pstr );
		editID->SetDescriptor( pstr );
		if( mEntry.identityType == kPGPike_ID_IPV4_Addr )
			editID->Disable();
	}
	switch( mEntry.hostType )
	{
		case kPGPnetInsecureHost:
			if( mEntry.ipMask < 0xFFFFFFFF )
				subnet = TRUE;
			break;
		case kPGPnetSecureHost:
			if( mEntry.ipMask < 0xFFFFFFFF )
				subnet = TRUE;
			secret = TRUE;
			break;
		case kPGPnetSecureGateway:
			secret = TRUE;
			break;
	}
	if( subnet )
	{
		( (LStaticText *)FindPaneByID( caption_Mask ) )->Show();
		( (LEditText *)FindPaneByID( edit_Mask ) )->Show();
	}
	if( secret )
	{
		( (LTextGroupBox *)FindPaneByID( group_SharedSecret ) )->Show();
		( (LTextGroupBox *)FindPaneByID( group_RemoteAuth ) )->Show();
	}
	typePopup->SetValue( mEntry.hostType + ( subnet ? 3 : 0 ) + 1 );
	typePopup->AddListener(this);
	typeMenu = typePopup->GetMacMenuH();
	if( ( mEntry.childOf != -1 ) || child )
	{
		DisableItem( typeMenu, 2 );
		DisableItem( typeMenu, 3 );
		DisableItem( typeMenu, 5 );
		if( child && ( mEntry.ipMask == 0xFFFFFFFF ) )
			typePopup->SetValue( 1 );
	}
#if PGP_FREEWARE
	DisableItem( typeMenu, 3 );
	m509Radio->Disable();
#endif
	
	if( mEntry.authKeyAlg != kPGPPublicKeyAlgorithm_Invalid )
	{
		PGPError	err;
		PGPKeyID	keyID;
		
		err = PGPImportKeyID( &mEntry.authKeyExpKeyID, &keyID );
		if( IsPGPError( err ) )
			goto done;
		if( mEntry.authCertIASNLength )
		{
			err = PGPX509CertFromExport( CPGPnet::GetPGPContext(),
						(PGPUInt32)mEntry.authKeyAlg,
						(PGPByte *)&mEntry.authKeyExpKeyID,
						mEntry.authCertIASN, mEntry.authCertIASNLength,
						CPGPnet::GetDefaultKeyrings(),
						&m509AuthKey, &m509AuthCert );
			if( IsntPGPError( err ) )
			{
				Str255		pstr;
				char		cstr[256];
				PGPSize		strSize = sizeof(cstr) - 1;
				
				PGPGetSigPropertyBuffer( m509AuthCert,
						kPGPSigPropX509LongName, strSize,
						(char *)&cstr, &strSize );
				cstr[strSize] = '\0';
				CToPString( cstr, pstr );
				mKeyCaption->SetDescriptor( pstr );
				mKeyIcon->SetIconID( icon_X509Cert );
				mKeyIcon->Show();
				m509Radio->SetValue(1);
			}
		}
		else
		{
			err = PGPGetKeyByKeyID( CPGPnet::GetDefaultKeyrings(),
				&keyID, mEntry.authKeyAlg,
				&mPGPAuthKey );
			if( PGPKeyRefIsValid( mPGPAuthKey ) )
			{
				CString			userID;
				PGPSize			userIDSize = userID.GetMinStorage();
				
				err = PGPGetPrimaryUserIDNameBuffer(	mPGPAuthKey, 
														userIDSize,
														userID,
														&userIDSize);
				if( IsntPGPError( err ) )
				{
					mKeyCaption->SetDescriptor( userID );
					
					Handle						iconSuite;
					
					err = PGPGetKeyIcon( mPGPAuthKey, NULL, &iconSuite );
					mKeyIcon->SetIconSuite( iconSuite, true );
					mKeyIcon->Show();
					mPGPRadio->SetValue(1);
				}
			}
		}
		if( IsPGPError( err ) )
		{
			char	keyIDString[kPGPMaxKeyIDStringSize];
			
			err = PGPGetKeyIDString( &keyID, kPGPKeyIDString_Abbreviated,
					keyIDString );
			if( IsntPGPError( err ) )
			{
				Str255	pstr,
						pstr2;
				
				if( mEntry.authCertIASNLength > 0 )
					m509Radio->SetValue(1);
				else
					mPGPRadio->SetValue(1);
				CToPString( keyIDString, pstr2 );
				GetIndString( pstr, STRx_HostsPanel, kUnknownKeyString );
				AppendPString( pstr2, pstr );
				mKeyCaption->SetDescriptor( pstr );
			}
		}
	}
done:
	mPGPRadio->AddListener(this);
	m509Radio->AddListener(this);
	mAnyRadio->AddListener(this);
	Show();
}

	PGPBoolean
CAddHostExpertDialog::GetInfo(
	PGPNetPrefHostEntry *	entry )
{
	PGPBoolean				valid = FALSE;
	PGPNetPrefHostEntry		host;
	PGPUInt32				alg = kPGPPublicKeyAlgorithm_DSA;
	PGPKeyID				keyID;
	PGPSize					keyIDSize = 0;
	Str255					pstr;
	char					cstr[256];
	PGPUInt32				idType,
							hostType,
							mask,
							invalidReason = 0;
	OSStatus				err = 0;
	
	pgpCopyMemory( entry, &host, sizeof( PGPNetPrefHostEntry ) );
	
	( (LEditText *)FindPaneByID( edit_HostName ) )->GetDescriptor( pstr );
	if( !pstr[0] )
	{
		invalidReason = 1;
		goto done;
	}
	PToCString( pstr, host.hostName );
	( (LEditText *)FindPaneByID( edit_IPAddress ) )->GetDescriptor( pstr );
	PToCString( pstr, cstr );
	if( OTInetStringToHost( cstr, (Uint32 *)&host.ipAddress ) ||
		!host.ipAddress )
		goto done;
	( (LEditText *)FindPaneByID( edit_Mask ) )->GetDescriptor( pstr );
	PToCString( pstr, cstr );
	if( cstr[0] )
		err = OTInetStringToHost( cstr, (Uint32 *)&mask );
	hostType = ( (LPopupButton *)FindPaneByID( popup_HostType ) )->GetValue();
	switch( hostType )
	{
		case 1:
		case 2:
		case 3:
			host.ipMask = 0xFFFFFFFF;
			hostType--;
			break;
		case 4:
			if( err )
				goto done;
			host.ipMask = mask;
			hostType = kPGPnetInsecureHost;
			break;
		case 5:
			if( err )
				goto done;
			host.ipMask = mask;
			hostType = kPGPnetSecureHost;
			break;
	}
	host.hostType = (PGPNetHostType)hostType;
	idType = ((LPopupButton *)FindPaneByID( popup_IdentityType ) )->GetValue();
	switch( idType )
	{
		default:
		case 1:
		case 2:
		case 3:
			host.identityType = (PGPipsecIdentity)idType;
			break;
		case 4:
			host.identityType = kPGPike_ID_DER_ASN1_DN;
			break;
	}
	( (LEditText *)FindPaneByID( edit_Identity ) )->GetDescriptor( pstr );
	PToCString( pstr, (char *)host.identity );
	if( hostType != kPGPnetInsecureHost )
	{
		if( host.identityType == kPGPike_ID_IPV4_Addr )
		{
			err = OTInetStringToHost( (char *)host.identity, (Uint32 *)&mask );
			if( err )		// This makes sure the user entered right
			{
				goto done;
			}
		}
		else if( host.identityType == kPGPike_ID_DER_ASN1_DN )
		{
			// *****
			goto done;
		}
	}
	strcpy( host.sharedSecret, mEntry.sharedSecret );
	if( PGPKeyRefIsValid( mPGPAuthKey ) )
	{
		if( IsntPGPError( PGPGetKeyIDFromKey( mPGPAuthKey, &keyID ) ) )
		{
			PGPGetKeyNumber( mPGPAuthKey, kPGPKeyPropAlgID, (PGPInt32 *)&alg );
			host.authKeyAlg = (PGPPublicKeyAlgorithm)alg;
			PGPExportKeyID( &keyID, host.authKeyExpKeyID, &keyIDSize );
		}
	}
	else if( PGPKeyRefIsValid( m509AuthKey ) )
	{
		PGPUInt32				iasnSize = 0;
		PGPByte *				iasn;
		
		PGPGetKeyNumber( m509AuthKey, kPGPKeyPropAlgID, (PGPInt32 *)&alg );
		if( IsntPGPError( PGPX509CertToExport( CPGPnet::GetPGPContext(),
								m509AuthKey, m509AuthCert,
								&alg, (PGPByte *)&keyID,
								&iasn, &iasnSize ) ) )
		{
			host.authKeyAlg = (PGPPublicKeyAlgorithm)alg;
			pgpCopyMemory( &keyID, host.authKeyExpKeyID, sizeof(PGPKeyID) );
			pgpCopyMemory( iasn, host.authCertIASN, iasnSize );
			host.authCertIASNLength = iasnSize;
			PGPFreeData( iasn );
		}
	}
	else
	{
		if( !m509Radio->GetValue() && !mPGPRadio->GetValue() )
		{
			host.authKeyAlg			= kPGPPublicKeyAlgorithm_Invalid;
			host.authCertIASNLength	= 0;
		}
		else
		{
			host.authKeyAlg			= mEntry.authKeyAlg;
			host.authCertIASNLength	= mEntry.authCertIASNLength;
			pgpCopyMemory( mEntry.authKeyExpKeyID, host.authKeyExpKeyID,
				kMaxNetHostKeyIDSize );
			pgpCopyMemory( mEntry.authCertIASN, host.authCertIASN,
				kMaxNetHostIASNLength );
		}
	}
	pgpCopyMemory( &host, entry, sizeof( PGPNetPrefHostEntry ) );
	valid = TRUE;
	
done:
	if( !valid )
	{
		WarningAlert( kWAStopAlertType, kWAOKStyle, STRx_HostsPanel,
			( invalidReason == 1 ) ? kNoHostNameString : kBadIPAddressString );
	}
	return valid;
}	

static PGPBoolean	gOTDNSLookupComplete;
static OTResult		gOTDNSLookupResult;

static pascal void OTNotifyProc(void */*context*/, OTEventCode code,
									OTResult result, void */*cookie*/)
{
	switch( code )
	{
		case T_DNRSTRINGTOADDRCOMPLETE:
			gOTDNSLookupComplete = TRUE;
			gOTDNSLookupResult = result;
			break;
	}
}

	void
CAddHostExpertDialog::ListenToMessage(
	MessageT		inMessage,
	void *			ioParam)
{
	Str255			pstr;
	PGPUInt32		val;
	PGPError		err;
	PGPKeySetRef	tempKeySet = kInvalidPGPKeySetRef;
	
	switch (inMessage)
	{
		case button_DNSLookup:
		{
			CPGPnetStDialogHandler	theHandler(dialog_DNSLookup, this);
			LWindow*				theDialog = theHandler.GetDialog();
			Str255					dnsStr;
			char					dnsCStr[256];
			
			LEditText*		theField = dynamic_cast<LEditText*>
								(theDialog->FindPaneByID(edit_DNS));
			while(true)
			{
				MessageT	hitMessage = theHandler.DoDialog();
				
				if (hitMessage == msg_Cancel)
					break;
				else if (hitMessage == msg_OK)
				{
					theField->GetDescriptor( dnsStr );
					theField->Disable();
					(theDialog->FindPaneByID(button_OK))->Disable();
					(theDialog->FindPaneByID(caption_DNS))->Disable();
					(theDialog->FindPaneByID(chasers_DNS))->Show();
					if( dnsStr[0] )
					{
						InetSvcRef		inetSvcRef;
						InetHostInfo	ihinfo;
						OSStatus		status;
						PGPBoolean		found = FALSE;
						
						if( ( ( inetSvcRef = OTOpenInternetServices(
									kDefaultInternetServicesPath,
									NULL, &status) ) != NULL ) && !status )
						{
							PToCString( dnsStr, dnsCStr );
							gOTDNSLookupComplete = FALSE;
							gOTDNSLookupResult = 1;
							OTSetAsynchronous( inetSvcRef );
							OTInstallNotifier( inetSvcRef,
								(OTNotifyProcPtr)&OTNotifyProc, NULL );
							status = OTInetStringToAddress( inetSvcRef,
										dnsCStr, &ihinfo);
							do
							{
								hitMessage = theHandler.DoDialog();
							} while( !gOTDNSLookupComplete &&
									( hitMessage != msg_Cancel ) );
							if( !status && !gOTDNSLookupResult )
							{
								PGPByte *p = (PGPByte *)&ihinfo.addrs[0];
								sprintf( dnsCStr, "%ld.%ld.%ld.%ld",
										(long)p[0],
										(long)p[1],
										(long)p[2],
										(long)p[3]);
								CToPString( dnsCStr, dnsStr );
								( (LEditText *)FindPaneByID( edit_IPAddress ) )->
									SetDescriptor( dnsStr );
								found = TRUE;
							}
							OTCloseProvider( inetSvcRef );
						}
						if( !found )
						{
							WarningAlert( kWAStopAlertType, kWAOKStyle,
								STRx_HostsPanel,
								kDNSLookupFailedString );
						}
					}
					break;
				}
			}
			break;
		}
		case popup_HostType:
		{
			PGPBoolean	subnet = FALSE,
						secret = FALSE;
			
			val = *(PGPUInt32 *)ioParam;
			switch( val )
			{
				case 1:
					break;
				case 2:
					secret = TRUE;
					break;
				case 3:
					secret = TRUE;
					break;
				case 4:
					subnet = TRUE;
					break;
				case 5:
					subnet = secret = TRUE;
					break;
			}
			pgpClearMemory( mEntry.sharedSecret,
							sizeof( mEntry.sharedSecret ) );
			GetIndString( pstr, STRx_HostsPanel, kSetPassphraseString );
			( (LPushButton *)FindPaneByID( button_SecretPhrase ) )->
				SetDescriptor( pstr );
			if( subnet )
			{
				( (LStaticText *)FindPaneByID( caption_Mask ) )->Show();
				( (LEditText *)FindPaneByID( edit_Mask ) )->Show();
			}
			else
			{
				( (LStaticText *)FindPaneByID( caption_Mask ) )->Hide();
				( (LEditText *)FindPaneByID( edit_Mask ) )->Hide();
				( (LEditText *)FindPaneByID( edit_Mask ) )->
					SetDescriptor( "\p255.255.255.255" );
			}
			if( secret )
			{
				( (LTextGroupBox *)FindPaneByID( group_RemoteAuth ) )->Show();
				( (LTextGroupBox *)FindPaneByID( group_SharedSecret ) )->Show();
			}
			else
			{
				( (LTextGroupBox *)FindPaneByID( group_SharedSecret ) )->Hide();
				( (LTextGroupBox *)FindPaneByID( group_RemoteAuth ) )->Hide();
			}
			break;
		}
		case button_SecretPhrase:
			if( mEntry.sharedSecret[0] )
			{
				pgpClearMemory( mEntry.sharedSecret,
								sizeof( mEntry.sharedSecret ) );
				GetIndString( pstr, STRx_HostsPanel, kSetPassphraseString );
			}
			else
			{
				char			*phrase = NULL;
				
				err = PGPConfirmationPassphraseDialog( CPGPnet::GetPGPContext(),
					//PGPOUIDialogPrompt( CPGPnet::GetPGPContext(), cstr ),
					PGPOUIOutputPassphrase( CPGPnet::GetPGPContext(), &phrase ),
					PGPOLastOption( CPGPnet::GetPGPContext() ) );
				if( IsntPGPError( err ) )
				{
					strcpy( mEntry.sharedSecret, phrase );
					PGPFreeData( phrase );
					GetIndString( pstr, STRx_HostsPanel, kClearPassphraseString );
				}
				else
					GetIndString( pstr, STRx_HostsPanel, kSetPassphraseString );
			}
			( (LPushButton *)FindPaneByID( button_SecretPhrase ) )->
				SetDescriptor( pstr );
			break;
		case popup_IdentityType:
		{
			LEditText *	editID = (LEditText *)FindPaneByID( edit_Identity );
			
			val = *(PGPUInt32 *)ioParam;
			if( val == 1 )
			{
				InetInterfaceInfo	info;
				char				cstr[128];
				Str255				pstr;
				
				editID->Disable();
				if( !OTInetGetInterfaceInfo(&info, 0) )
				{
					OTInetHostToString( info.fAddress, cstr );
					CToPString( cstr, pstr );
					editID->SetDescriptor( pstr );
				}
			}
			else
			{
				editID->Enable();
				editID->SetDescriptor( "\p" );
				SwitchTarget( editID );
			}
			break;
		}
		case radio_PGP:
			if( *(PGPUInt32 *)ioParam != 1 )
				break;
			err = PGPSelectKeysDialog(
					CPGPnet::GetPGPContext(),
					kPGPSelectKeysSingleKey,
					CString(CStringFromSTRx(STRx_HostsPanel, kPGPAuthKeyString)),
						CPGPnet::GetDefaultKeyrings(), NULL,
						&tempKeySet);
			if( IsntPGPError( err ) )
			{
				StPGPKeyListRef	keyList;
				StPGPKeyIterRef	keyIter;
				CString			userID;
				PGPSize			userIDSize = userID.GetMinStorage();
				
				err = PGPOrderKeySet( tempKeySet, kPGPAnyOrdering, &keyList );
				if( IsPGPError( err ) )
					goto error;
				err = PGPNewKeyIter( keyList, &keyIter );
				if( IsPGPError( err ) )
					goto error;
				err = PGPKeyIterNext( keyIter, &mPGPAuthKey );
				if( IsPGPError( err ) )
					goto error;
				err = PGPGetPrimaryUserIDNameBuffer(	mPGPAuthKey,
														userIDSize,
														userID,
														&userIDSize);
				if( IsPGPError( err ) )
					goto error;
				FindPaneByID( caption_Key )->SetDescriptor( userID );
				
				Handle						iconSuite;
				
				err = PGPGetKeyIcon( mPGPAuthKey, NULL, &iconSuite );
				if( IsPGPError( err ) )
					goto error;
				mKeyIcon->Hide();
				mKeyIcon->SetIconSuite( iconSuite, true );
				mKeyIcon->Show();
				m509AuthKey		= kInvalidPGPKeyRef;
				if( PGPKeySetRefIsValid( m509KeySet ) )
					m509KeySet.Free();
				if( PGPKeySetRefIsValid( mPGPKeySet ) )
					mPGPKeySet.Free();
				mPGPKeySet = tempKeySet;
			}
			else
			{
			error:	
				mAnyRadio->SetValue( 1 );
			}
			break;
		case radio_X509:
		{
			PGPKeyRef	x509Key;
			PGPSigRef	x509Cert;
			
			if( *(PGPUInt32 *)ioParam != 1 )
				break;
			err = PGPSelect509Dialog(
					CPGPnet::GetPGPContext(), 0,
					CString(CStringFromSTRx(STRx_HostsPanel, k509AuthKeyString)),
					CPGPnet::GetDefaultKeyrings(),
					&tempKeySet, &x509Key, &x509Cert );
			if( IsntPGPError( err ) )
			{
				Str255		pstr;
				char		cstr[256];
				PGPSize		strSize = sizeof(cstr) - 1;
				
				err = PGPGetSigPropertyBuffer( x509Cert,
							kPGPSigPropX509LongName, strSize,
							(char *)&cstr, &strSize );
				if( IsPGPError( err ) )
					goto error;
				cstr[strSize] = '\0';
				CToPString( cstr, pstr );
				mKeyCaption->SetDescriptor( pstr );
				mKeyIcon->SetIconSuite( NULL, false );
				mKeyIcon->SetIconID( icon_X509Cert );
				mKeyIcon->Show();
				
				if( PGPKeySetRefIsValid( mPGPKeySet ) )
					mPGPKeySet.Free();
				if( PGPKeySetRefIsValid( m509KeySet ) )
					m509KeySet.Free();
				mPGPAuthKey		= kInvalidPGPKeyRef;
				m509KeySet		= tempKeySet;
				m509AuthKey		= x509Key;
				m509AuthCert	= x509Cert;
			}
			else
				goto error;
			break;
		}
		case radio_AnyKey:
			mKeyCaption->SetDescriptor( "\p" );
			mKeyIcon->Hide();
			mPGPAuthKey		= kInvalidPGPKeyRef;
			m509AuthKey		= kInvalidPGPKeyRef;
			if( PGPKeySetRefIsValid( m509KeySet ) )
				m509KeySet.Free();
			if( PGPKeySetRefIsValid( mPGPKeySet ) )
				mPGPKeySet.Free();
			break;
		default:
			LGADialog::ListenToMessage(inMessage, ioParam);
			break;
	};
}

