/*
 * The author of this software is William Dorsey.
 * Copyright (c) 1993, 1994, 1995 by William Dorsey.  All rights reserved.
 *
 * THIS SOFTWARE IS BEING PROVIDED "AS IS", WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTY.  IN PARTICULAR, THE AUTHOR DOES NOT MAKE ANY CLAIM OR
 * WARRANTY OF ANY KIND CONCERNING THE MERCHANTABILITY OF THIS SOFTWARE OR
 * ITS FITNESS FOR ANY PARTICULAR PURPOSE.
 */

/* cli.c
 *
 * SCCS ID:  @(#)cli.c 1.29 96/05/27
 *
 * REVISION HISTORY
 *
 * DATE      RESPONSIBLE PARTY  DESCRIPTION
 * -------------------------------------------------------------------------
 * 93/12/08  W. Dorsey          Module created by breakup of nautilus.c
 * 95/03/11  W. Dorsey          Added basic crypto stuff
 * 95/09/17  P. Kronenwetter    Added socket support from S. Parekh's 
 *                              patches.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>

#include "nautilus.h"

/* external variables */
extern struct param_t params;	/* operating parameters */
extern struct coder_t coders[];	/* coder table */
extern struct cipher_t ciphers[];    /* cipher table */
extern struct negotiate_t negotiate; /* capability negotiation parameters */

extern char    *optarg;
extern int      optind;

void 
main(int argc, char *argv[])
{
    int             c;
    int             vlogon = TRUE;
    char           *fname;
    char           *pw;
    char            pw1[MAX_SKEY_LEN+1];
    char            pw2[MAX_SKEY_LEN+1];
    char            tbuf[128];
    UINT8           audio_buf[2048];
    void            debug_puts(char *);

    /* initialize internal variables */
    init();

    /* display title message */
    title();

    /* OS specific system initialization (interrupts, etc.) */
    if (SysInit() == FAIL) {
	fprintf(stderr, "System initialization failed.\n");
	exit(1);
    }

    /* read configuration file */
    if ((fname = getenv("NAUTILUS_CONFIG_FILE")) != NULL) {
	if (ReadConfigFile(fname) == FAIL) {
	    exit(1);
	}
    }

    /* parse arguments */
#ifndef unix
    while ((c = getopt(argc, argv, "aAhoxc:e:k:l:p:s:v")) != -1)
#else
    while ((c = getopt(argc, argv, "aAhoixc:e:k:l:n:p:s:v")) != -1)
#endif
	switch (c) {
	case 'a':
	    params.mode = AUTO_ANSWER;
	    break;
	case 'A':
	    params.mode = ANSWER;
	    break;
	case 'h':
	    help();
	    exit(0);
	case 'o':
	    params.mode = ORIGINATE;
	    break;
	case 'x':
	    vlogon = FALSE;
	    break;
#ifdef unix
	case 'i':
	    params.net_flag = TRUE;
	    /* Setup/override some defaults */
	    params.net.portnum = 12370;
	    params.rp_timeout = 30;
	    params.sp_timeout = 30;
	    break;
	case 'n':
	    params.net.portnum = atoi(optarg);
	    break;
#endif /* #ifdef unix */
	case 'c':
	    if ((params.coder.index = FindCoder(optarg)) == -1) {
		fprintf(stderr, "%s: invalid coder\n", optarg);
		ListCoders();
		exit(1);
	    }
	    negotiate.coder = params.coder.index;
	    break;
	case 'e':
	    if ((params.crypto.key1.type = FindCipher(optarg)) == -1) {
		fprintf(stderr, "%s: invalid cipher\n", optarg);
		ListCiphers();
		exit(1);
	    }
	    params.crypto.key2.type = params.crypto.key1.type;
	    negotiate.encrypt_type = params.crypto.key1.type;
	    break;
 	case 'k':
 	    if ((params.crypto.keyexch_type = FindKeyExch(optarg)) == -1) {
 		fprintf(stderr, "%s: invalid key exchange protocol\n", optarg);
#ifndef ENABLE_DH_ENORMOUS
 		fprintf (stderr,
  "valid protocols are: pp (passphrase), dh (768), dh512, dh768, dh1024");
#else
 		fprintf (stderr,
  "valid protocols are: pp (passphrase), dh (768), dh512, dh768, dh1024, dh2048");
#endif
 		exit(1);
 	    }
 	    negotiate.keyexch_type = (UINT8) params.crypto.keyexch_type;
 	    break;
	case 'l':
	    if (!Strncasecmp("coders", optarg, strlen(optarg))) {
		ListCoders();
	    }
	    else if (!Strncasecmp("ciphers", optarg, strlen(optarg))) {
		ListCiphers();
	    }
	    else {
		fprintf(stderr, "'%s' is an invalid parameter to -l argument\n",
		        optarg);
		fprintf(stderr, "\n");
		usage();
		exit(1);
	    }
	    exit(0);
	case 'p':
	    if (strcpy(params.port.name, GetPort(optarg)) == NULL) {
		fprintf(stderr, "%s: invalid serial port\n", optarg);
		exit(1);
	    }
	    break;
	case 's':
	    params.port.speed = atoi(optarg);
	    break;
	case 'v':
	    params.verbose = TRUE;
	    break;
	case '?':
	    fprintf(stderr, "Invalid argument(s).\n");
	    fprintf(stderr, "\n");
	    usage();
	    exit(1);
	}

    /* Finish processing arguments */
    if (params.mode == ORIGINATE) {
	if (optind + 1 == argc) {
#ifdef unix
	  if (params.net_flag == TRUE)
	    strcpy(params.hostname, argv[optind]);
	  else
#endif /* unix */
	    strcpy(params.telno, argv[optind]);
	}
	else if (optind != argc) {
	    fprintf(stderr, "Extraneous argument(s).\n");
	    fprintf(stderr, "\n");
	    usage();
	    exit(1);
	}
	else {
	    fprintf(stderr, "-o option requires argument.\n");
	    fprintf(stderr, "\n");
	    usage();
	    exit(1);
	}
    }
    else if ((params.mode == ANSWER) || (params.mode == AUTO_ANSWER)) {
	if (optind != argc) {
	    fprintf(stderr, "Extraneous argument(s).\n");
	    fprintf(stderr, "\n");
	    usage();
	    exit(1);
	}
    }
    else {
	credits();
	fputs("No mode specified\n", stderr);
	usage();
	exit(1);
    }

    /*
     * Test the speed of all the speech coders to determine which are
     * fast enough to be used.  Test each of them for at least 200
     * ms.
     */
    CoderSpeedTest(200, params.verbose);

    /*
     * Initialize communications channel (network or serial)
     */
    if (params.net_flag) {
        params.session = nsp_create(debug_puts, "udp");
        if (params.session == NULL) {
            error(MSG_FATAL, "Could not initialize socket");
        }
        sprintf(tbuf, "%d", params.net.portnum);
        nsp_ioctl(params.session, "udp_port", tbuf);
    }
    else {
        params.session = nsp_create(debug_puts, "modem");
        if (params.session == NULL) {
            error(MSG_FATAL, "Could not initialize serial port");
	}
        sprintf(tbuf, "%s,%d", params.port.name, params.port.speed);
        if (nsp_ioctl(params.session, "modem_open", tbuf) == -1) {
            error(MSG_FATAL, "Could not open serial port");
        }
    }

    /*
     * Initialize audio device for possible playing of startup sound,
     * and for recording of sound samples that will be used to
     * initialize the pseudo-random number generator.
     */
    if (InitAudio(8000, 128) == FAIL) {
        error(MSG_FATAL, "Could not initialize audio device");
        /* NOTREACHED */
    }

    /* Print startup information. */
    if (params.net_flag == TRUE) {
        printf("Selected Port   : %5d\t\t", params.net.portnum);
        printf("Network Protocol: %s\n", "UDP");
    }
    else {
        printf("Selected Port : %s\t\t", params.port.name);
        printf("DTE Speed     : %u\n", params.port.speed);
    }

    /* Play logon file unless supressed. */
    if (vlogon) {
	VoiceLogon(LOGON_FILE);
    }

    if ((params.crypto.keyexch_type == PASSPHRASE) &&
        (params.crypto.key1.type != NONE)) {
	/*
	 * Get the key either from the configuration file, or from the
	 * keyboard.
	 */
	if ((pw = getenv("NAUTILUS_PASSPHRASE")) != NULL) {
	    strncpy(pw1, pw, MAX_SKEY_LEN);
	}
	else {
	    for (;;) {
		if (GetPassPhrase(pw1, MAX_SKEY_LEN, "\nEnter passphrase: ") < 0) {
		    error(MSG_FATAL, "could not get passphrase");
		    /* NOTREACHED */
		}
		if (GetPassPhrase(pw2, MAX_SKEY_LEN, "Enter it again  : ") < 0) {
		    error(MSG_FATAL, "could not get passphrase");
		    /* NOTREACHED */
		}
		if (strcmp(pw1, pw2) == 0)
		    break;
		fprintf(stderr, "\nPassphrases did not match, try again.\n");
	    }
	}

	/*
	 * Initialize key structure with the passphrase.
	 */
	if (strlen(pw1) == 0) {
	    /*
	     * This used to set the passphrase to the version string if
	     * the user didn't type in any passphrase at all.  However,
	     * that causes incompatibility between versions, so I now
	     * hardcoded it to a specific string that should remain
	     * fixed between versions that are compatible with each
	     * other.
	     */
	    strcpy(pw1, "UNODIR, ESBAM");
	}
	if (keyinit(&params.crypto.key1, pw1, strlen(pw1), params.mode) < 0) {
	    error(MSG_FATAL, "key initialization failed.");
	    /* NOTREACHED */
	}
	if (keyinit(&params.crypto.key2, pw1, strlen(pw1), !params.mode) < 0) {
	    error(MSG_FATAL, "key initialization failed.");
	    /* NOTREACHED */
	}
	memset(pw1, '\0', MAX_SKEY_LEN);
	memset(pw2, '\0', MAX_SKEY_LEN);
    }
 
    if (params.crypto.key1.type != NONE) {
	/*
	 * Sample audio data to use as "random" bits for initializing
	 * the pseudo-random number generator.
	 */
	memset(audio_buf, '\0', 2048);
        for (;;) {
            float entropy;

            AudioFlow(TRANSMIT);
            if (ReadAudio(audio_buf, 2048/128) < 0)
                error(MSG_FATAL, "ReadAudio() failure.");
	    AudioFlow(RECEIVE);

	    entropy = ComputeEntropy(audio_buf, 2048);
            if (entropy > 10.0) {
                printf("\nAudio Entropy is %.1f (satisfactory).\n", entropy);
                break;
	    }
	    else {
	        printf("\nWARNING:  Audio Entropy is %.1f (low).\n", entropy);
	        printf("\nPlease make sure the microphone is on.  Blow into the microphone for\n");
	        printf("about 1 second, and hit the <Return> key while blowing.  If you want\n");
	        printf("to override the entropy check, type 'q' followed by <Return>: ");
		fflush(stdout);
		gets(tbuf);
		if (toupper(tbuf[0]) == 'Q')
		    break;
	    }
        }
	random_init(audio_buf, 2048);
    }

    /* close the audio device */
    CloseAudio();

    /* begin communicating */
    if (params.net_flag) {
        switch (params.mode) {
        case ANSWER:
        case AUTO_ANSWER:
            printf("Waiting for incoming Nautilus connection...\n");
            if (nsp_open(params.session, NULL, -1) == -1) {
                error(MSG_FATAL, "Failed to connect to remote socket.");
                exit(1);
            }
            break;
	case ORIGINATE:
            if (nsp_open(params.session, params.hostname, -1) == -1) {
                error(MSG_FATAL, "Failed to connect to remote socket.");
                exit(1);
            }
            break;
        }
    }
    else {
        printf("\n");
        switch (params.mode) {
        case ANSWER:
            if (nsp_open(params.session, NULL, 60) == -1) {
                error(MSG_FATAL, "Failed to connect to remote modem.");
                exit(1);
            }
            break;
        case AUTO_ANSWER:
            if (nsp_open(params.session, NULL, 0) == -1) {
                error(MSG_FATAL, "Failed to connect to remote modem.");
                exit(1);
            }
            break;
        case ORIGINATE:
            if (nsp_open(params.session, params.telno, 60) == -1) {
                error(MSG_FATAL, "Failed to connect to remote modem.");
                exit(1);
            }
            break;
        }
    }

    /* get modem connect speed */
    if (!params.net_flag)
        params.modem.speed = nsp_ioctl(params.session, "connect_speed", NULL);

    /* exchange operating parameters with other side */
    if (XChange(params.mode) == FAIL) {
	fprintf(stderr, "Failed to startup.\n");
	exit(2);
    }

    /* Print coder & cipher information. */
    printf("\nSelected Coder: %s\n", coders[params.coder.index].name);
    printf("Encryption    : %s\n", ciphers[params.crypto.key1.type].name);

    /* Print DCE speed if we're using the modem to connect */
    if (!params.net_flag) {
        if (params.modem.speed)
            printf("Modem Speed   : %u\n", params.modem.speed);
        else
            printf("Modem Speed   : UNKNOWN\n");
    }

    /*
     * Create turnaround beep at the correct sampling rate (i.e., the
     * sampling rate used by the selected speech coder.
     */
    InitBeep(coders[params.coder.index].sample_rate);

    /* Initialize audio device for the selected speech coder. */
    if (InitAudio(coders[params.coder.index].sample_rate,
		  coders[params.coder.index].frame_size) == FAIL) {
	fprintf(stderr, "Could not initialize audio device.\n");
	exit(2);
    }

    /* begin talking */
    Talk(params.mode == ORIGINATE ? TRANSMIT : RECEIVE);

    if (params.crypto.key1.type != NONE) {
	keydestroy(&params.crypto.key1);	/* wipe key schedule */
    }

    nsp_close(params.session);		/* close communications channel */
    CloseAudio();			/* close audio device */

    exit(0);
}


void
title(void)
{
    fprintf(stderr, "\nNautilus Digital Voice Communicator -- %s\n", VERSION_STRING);
    fprintf(stderr, "Copyright (c) 1993-1996 William W. Dorsey, All Rights Reserved\n");
    fprintf(stderr, "\n");
}


void
credits(void)
{
    fprintf(stderr, "Nautilus was originally developed by William Dorsey, Pat Mullarky, and\n");
    fprintf(stderr, "Paul Rubin.  Andy Fingerhut, Paul Kronenwetter, and William Soley have\n");
    fprintf(stderr, "made major contributions to subsequent releases.\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "The developers may be reached by sending email to <nautilus@lila.com>.\n");
    fprintf(stderr, "For up-to-date information about the current status of Nautilus, see the\n");
    fprintf(stderr, "Nautilus web page at <http://www.lila.com/nautilus/>.\n");
    fprintf(stderr, "\n");
}


void
help(void)
{
    char *cipher = ciphers[params.crypto.key1.type].name;

    fprintf(stderr, "Here's a quick summary of Nautilus commands...\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "-h produces this summary\n");
    fprintf(stderr, "-o selects originate mode\n");
    fprintf(stderr, "-a/-A selects auto/manual answer mode\n");
    fprintf(stderr, "-p <port> selects serial port modem is connected to\n");
    fprintf(stderr, "-s <speed> selects serial DTE (default=%d)\n", params.port.speed);
    fprintf(stderr, "-c <coder> overrides automatic coder selection\n");
    fprintf(stderr, "-e <cipher> selects encryption cipher (default=%s)\n", cipher);
    fprintf(stderr, "-k <protocol> selects keyexch protocol (default=768-bit DH)\n");
    fprintf(stderr, "-l <coders|ciphers> lists available coders or ciphers\n");
    fprintf(stderr, "-x suppresses start up sound\n");
    fprintf(stderr, "-v verbose option.  Currently only gives detailed coder speed testing data.\n");
#ifdef unix
    fprintf(stderr, "-i selects the use of sockets rather than modems\n");
    fprintf(stderr, "   -n <port> selects socket number to use. (>1024)\n");
#endif /* unix */
    fprintf(stderr, "\n");
    fprintf(stderr, "To originate a call, type:\n");
    fprintf(stderr, "\tnautilus -o -p COM1 <phone number>\n");
    fprintf(stderr, "To accept an incoming call, type:\n");
    fprintf(stderr, "\tnautilus -a -p COM1\n");
    fprintf(stderr, "\n");
#if defined(unix)
    fprintf(stderr, "To connect to another unix host via a network (on port #12345):\n");
    fprintf(stderr, "\tnautilus -o -i -n 12345 <hostname>\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "To accept an incoming call via a network (on port #12345):\n");
    fprintf(stderr, "\tnautilus -a -i -n 12345\n");
    fprintf(stderr, "\n");
#endif /* unix */
    fprintf(stderr, "For more information, refer to the documentation.\n");
}


void
usage(void)
{
    fprintf(stderr, "For a usage summary, type:  nautilus -h\n");
}


/*
 * display list of available coders
 */

void
ListCoders(void)
{
    int             i;

    printf("List of available coders:\n");
    for (i = 0; i < NCODERS; i++) {
	printf("%s\t%s\n", coders[i].name, coders[i].desc);
    }
    printf("\n");
}


/*
 * display list of available ciphers
 */

void
ListCiphers(void)
{
    int             i;

    printf("List of available ciphers:\n");
    for (i = 0; i < NCIPHERS; i++) {
	printf("%10s  %s\n", ciphers[i].name, ciphers[i].desc);
    }
    printf("\n");
}


/*
 * show user what mode we're in (transmit or receive)
 */

void
ShowMode(enum flow mode)
{
    if (mode == RECEIVE) {
	printf("\rListening...");
    }
    else {
	printf("\rGo ahead... ");
    }
    printf("           (q=quit, <CR>=Talk/Listen)");
    fflush(stdout);
}


/*
 * error message handler
 */

void
error(enum err_type type, char *fmt, ...)
{
    va_list args;

    va_start (args, fmt);

    fprintf (stderr, "\n");
    vfprintf (stderr, fmt, args);
    fprintf (stderr, "\n");
    fflush(stderr);
    if (type == MSG_FATAL)
	exit(2);
}


/*
 * print a string of debug information to stderr
 */

void
debug_puts(char *s)
{
    fputs(s, stderr);
    fputc('\n', stderr);
    fflush(stderr);
}


/*
 * print a string of debug information to stderr without a newline
 */

void
debug_puts_nr(char *s)
{
    fputs(s, stderr);
    fflush(stderr);
}


/*
 * print a character of debug information to stderr
 */

void
debug_putc(char c)
{
    putc(c, stderr);
    fflush(stderr);
}
