/* sexpconv.c
 *
 * Reads a sexp in given form from, and writes it in given form.
 *
 * $Id$ */

/* lsh, an implementation of the ssh protocol
 *
 * Copyright (C) 1999 Balazs Scheidler
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "lsh.h"
#include "sexp.h"
#include "io.h"
#include "xalloc.h"
#include "blocking_write.h"
#include "werror.h"

#include <string.h>
#include <getopt.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>

#include "sexpconv.c.x"

struct sexp_format
{
  char *name;
  UINT32 id;
};

static UINT32 lookup_sexp_format(const char *name)
{
  struct sexp_format sexp_formats[] = {
    { "transport", SEXP_TRANSPORT },
    { "canonical", SEXP_CANONICAL },
    { "advanced", SEXP_ADVANCED },
    { "international", SEXP_INTERNATIONAL },
    { NULL, 0 }
  };
  int i;

  for (i = 0; sexp_formats[i].name; i++)
    {
      if (strcasecmp(sexp_formats[i].name, name) == 0)
	return sexp_formats[i].id;
    }
  return -1;
}

/* GABA:
   (class
     (name convert_sexp)
     (super sexp_handler)
     (vars
       (output_format . UINT32)
       (output_fd . int)))
*/

static int do_convert_key(struct sexp_handler *c, struct sexp *e)
{
  CAST(convert_sexp, self, c);

  if (LSH_CLOSEDP(A_WRITE(make_blocking_write(self->output_fd, 0), sexp_format(e, self->output_format, 2))))
    {
      werror("Writing key failed\n");
      return LSH_FAIL | LSH_DIE;
    }
  return LSH_OK | LSH_DIE;
}

int main(int argc, char *argv[])
{
  int option;
  int input_format = SEXP_TRANSPORT;
  int output_format = SEXP_ADVANCED;

  NEW(io_backend, backend);
  NEW(convert_sexp, handler);

  while((option = getopt(argc, argv, "i:o:")) != -1)
    switch(option)
      {
      case 'i':
	/* specify input format */
	input_format = lookup_sexp_format(optarg);
	break;
      case 'o':
	/* specify output format */
	output_format = lookup_sexp_format(optarg);
	break;
      case -1:
	break;
      }

  init_backend(backend);

  handler->super.handler = do_convert_key;
  handler->output_format = output_format;
  handler->output_fd = STDOUT_FILENO;

  io_read(make_io_fd(backend, STDIN_FILENO),
	  make_read_sexp(&handler->super, 8192, input_format, 0),
	  NULL);

  io_run(backend);

  return 0;
}
