/*
 * This file is part of LibEuFin.
 * Copyright (C) 2024 Taler Systems S.A.

 * LibEuFin is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation; either version 3, or
 * (at your option) any later version.

 * LibEuFin is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General
 * Public License for more details.

 * You should have received a copy of the GNU Affero General Public
 * License along with LibEuFin; see the file COPYING.  If not, see
 * <http://www.gnu.org/licenses/>
 */

package tech.libeufin.common.crypto

import tech.libeufin.common.encodeBase64
import tech.libeufin.common.rand
import java.security.SecureRandom

data class PasswordHashCheck(
    val match: Boolean,
    val outdated: Boolean
)

/** Cryptographic operations for secure password storage and verification */
object PwCrypto {
    private val SECURE_RNG = SecureRandom()

    /** Hash [pw] using the strongest supported hashing method */
    fun hashpw(pw: String): String {
        val salt = ByteArray(16).rand(SECURE_RNG)
        /* TODO Argon2id
        val pwh = CryptoUtil.hashArgon2id(pw, salt)
        return "argon2id\$${salt.encodeBase64()}\$${pwh.encodeBase64()}"
        */
        val saltEncoded = salt.encodeBase64()
        val pwh = CryptoUtil.hashStringSHA256("$saltEncoded|$pw").encodeBase64()
        return "sha256-salted\$$saltEncoded\$$pwh"
    }

    /** Check whether [pw] match hashed [storedPwHash] and if it should be rehashed */
    fun checkpw(pw: String, storedPwHash: String): PasswordHashCheck {
        val components = storedPwHash.split('$', limit = 4)
        return when (val algo = components[0]) {
            "sha256" -> {
                require(components.size == 2) { "bad password hash format" }
                val hash = components[1]
                val pwh = CryptoUtil.hashStringSHA256(pw).encodeBase64()
                PasswordHashCheck(pwh == hash, true)
            }
            "sha256-salted" -> {
                require(components.size == 3) { "bad password hash format" }
                val salt = components[1]
                val hash = components[2]
                val pwh = CryptoUtil.hashStringSHA256("$salt|$pw").encodeBase64()
                PasswordHashCheck(pwh == hash, false)
            }
            /* TODO Argon2id 
            "argon2id" -> {
                require(components.size == 3) { "bad password hash format" }
                val salt = components[1].decodeBase64()
                val hash = components[2]
                val pwh = CryptoUtil.hashArgon2id(pw, salt).encodeBase64()
                PasswordHashCheck(pwh == hash, false)
            } */
            else -> throw Exception("unsupported hash algo: '$algo'")
        }
    }
}
