/* 
 * arf2arf.c - Specialized Archive reading/copying utility.
 */

/*
   Copyright (C) 2003-2004 James H. Lowe, Jr.
   All rights reserved.
  
   COPYING TERMS AND CONDITIONS
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 */

#include "swuser_config.h"
#include "swprog_versions.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <string.h>
#include <sys/stat.h>
#include <time.h>
#include <unistd.h>
#include "usgetopt.h"
#include "swparse.h"
#include "swlex_supp.h"
#include "swevents_array.h"
#include "swevents.h"
#include "swparser_global.h"
#include "swlib.h"
#include "swi.h"
#include "xformat.h"
#include "swutilname.h"

YYSTYPE yylval;
static int verboseG;

static int
usage(FILE * file, char * name)
{

fprintf(file, "%s", 
"arf2arf : A signature and digest checking tool of the swbis project.\n"
"Usage arf2arf [options] [archive_file]\n"
"Translates an archive to stdout with specified filtering.\n"
"  -H, --format FORMAT   Output format  ustar, gnutar, newc, crc, odc\n"
"  --verbose-level=LEVEL\n"
"  -v, --verbose\n"
"  -n, --sig-number=N  select Nth signature for verification; first is N=1,\n"
"                      last is N=0.\n"
"  -m, --numeric-owner\n"
"       --checksig-mode  fifoname  used by swverify --checksig.\n"
"       --sleep TIME  A configurable delay in seconds.\n"
"       --gpg-prog gpgexecname     used by --checksig-mode.\n"
"       --noop        legal option with no effect.\n"
"       --sha1        applies to -Sd only, show the sha1 digest.\n"
"       --decode      applies to -C and -S and --audit only.\n"
"                     Re-encodes the header with the swbis personality.\n"
"  --audit  Pass the input thru to the output unchanged and audit.\n"
"  -p, --pass-thru   Pass the input thru to the output unchanged.\n"
"  -C, --catalog-sign-file  Write the catalog signature input file to stdout.\n"
"  -S, --storage-digest-file  Write the storage digest input file to stdout.\n"
"  -s, --adjunct-digest-file  Write the adjunct digest input file to stdout.\n"
"  -d, --show-auth-files    Writes the relevent security file to stderr.\n"
"              i.e  .../catalog/*/md5sum, or, adjunct_md5sum, or signature.\n"
"                   is copied directly from the archive to stderr.\n"
"    --test1  Write digest archive file on stdout and adjunct on stderr.\n"
"  -G --get-sig-if FILE Checks md5sum and if valid writes sigfile to FILE.\n"
"                   and the signed file to stdout.\n"
"\n"
" Note: The adjunct_md5sum attribute does not cover SYMLNK files in the\n"
"        archive.  It is a concession to an inability to preserve\n"
"        the modification time in the installed directory form of a package.\n"
" Example:  Check the md5sum and the (lesser) adjunct_md5sum:\n"
"      arf2arf -d -S <serial_archive_package | /usr/bin/md5sum\n"
"      arf2arf -d -s <serial_archive_package | /usr/bin/md5sum\n"
"         Note: The pairs of digests should be identical.\n"
" Example:  Check the package signature:\n"
"       arf2arf -G /dev/tty packageName.tgz | gpg --verify /dev/tty -\n"
"       (Copy and paste the sigfile, then hit Ctrl-D.)\n"
);
	fprintf(file, "\n");
	fprintf(file, "Date: " SWBIS_DATE "\n");
	fprintf(file, "Copyright (C) 2003 Jim Lowe\n");
	fprintf(file,
"This program is distributed under the GNU GPL Software License.\n");
	return 0;
}

static
void
enforce_tar_format(int format) 
{
	if (format != USTAR_FILEFORMAT) {
		fprintf(stderr, 
		"operation not supported for this archive format[%d]\n",
		format);
		exit(1);
	}
}

static
int
run_checksig(char * sigfilename, char * thisprog,
		char * filearg, char * gpg_prog, int uverbose,
		char * which_sig_arg)
{
	int ret1;
	int ret2;
	char * absname;
	SHCMD * cmd[3];
	pid_t pid1[1];
	pid_t pid2[1];
	int status1[1];
	int status2[1];
	int u_verbose = uverbose;
	STROB * tmp = strob_open(30);

	cmd[0] = shcmd_open();
	cmd[1] = shcmd_open();
	cmd[2] = NULL;

	shcmd_add_arg(cmd[0], thisprog);  
	strob_sprintf(tmp, 0, "--util-name=%s", swlib_utilname_get());
	shcmd_add_arg(cmd[0], strob_str(tmp));
	shcmd_add_arg(cmd[0], "-G");
	shcmd_add_arg(cmd[0], sigfilename); /* FIFO */
	shcmd_add_arg(cmd[0], "-n");
	shcmd_add_arg(cmd[0], which_sig_arg);
	while (u_verbose > 1) {
		shcmd_add_arg(cmd[0], "-v");
		u_verbose--;
	}
	shcmd_add_arg(cmd[0], "--sleep");
	shcmd_add_arg(cmd[0], "1");
	shcmd_add_arg(cmd[0], filearg);

	absname = shcmd_find_in_path(getenv("PATH"), gpg_prog);
	if (!absname) {
		fprintf(stderr,
			"swbis: %s: Not found in current path\n", gpg_prog);
		return 1;
	}
	if (uverbose >= 2) {	
		fprintf(stderr, 
			"%s: using GNU Privacy Guard : %s\n", swlib_utilname_get(), absname);
	}
	shcmd_add_arg(cmd[1], absname);		/* GPG command */
	shcmd_add_arg(cmd[1], "--verify");
	shcmd_add_arg(cmd[1], sigfilename); 	/* Other end of FIFO */
	shcmd_add_arg(cmd[1], "-");
	if (uverbose < 2) {
		shcmd_set_errfile(cmd[1], "/dev/null");
	}
	shcmd_cmdvec_exec(cmd);

	pid1[0] = shcmd_get_pid(cmd[0]);  /* arf2arf pid */
	pid2[0] = shcmd_get_pid(cmd[1]);  /* gpg pid */

	/*
	* Wait on the arf2arf process.
	*/
	swlib_wait_on_all_pids(pid1, 1, status1, 0, verboseG);

	/*
	* analyze the status of the arf2arf process.
	*/

	if (WIFEXITED(status1[0]) == 0 || 
			((ret1=WEXITSTATUS(status1[0])) && ret1 != 63)) {
		/*
		* If there is a bad outcome of arf2arf process, then
		* Kill gpg.
		*/
		if (uverbose > 3) 
			fprintf(stderr, 
	"arf2arf: run_checksig error : child: exited=%d exitstatus=%d\n",
			WIFEXITED(status1[0]), WEXITSTATUS(status1[0]));
		kill(pid2[0], SIGTERM);
		return 22;
	}

	/*
	* 63 is the gzip warning indicator, don't react to it.
	*/
	
	/*
	* Wait on the gpg process.
	*/
	swlib_wait_on_all_pids(pid2, 1, status2, 0, verboseG);

	/*
	* analyze the status of the arf2arf process.
	*/
	if (WIFEXITED(status1[0]) == 0) {
		ret1 = 7;
	} else {
		ret1 = WEXITSTATUS(status1[0]);
	}

	/*
	* analyze the status of the GPG process.
	*/
	if (WIFEXITED(status2[0]) == 0) {
		ret2 = 10;
	} else {
		ret2 = WEXITSTATUS(status2[0]);
	}

	shcmd_close(cmd[0]);
	shcmd_close(cmd[1]);
	if (ret2) 
		return ret2;
	else if (ret1) 
		return ret1;
	else
		return 0;
}

int
main (int argc, char *argv[])
{
	int fd;
	int c;
	int debugmode = 0;
	int format=arf_ustar;
	int detected_format;
	int numeric_uids = 0;
	int do_decode_pass_thru = 0;
	int ret = 0;
	int ifd;
	int flags;
	int cat_sign = 0;
	int store_sign = 0;
	int opt_gnutar = 0;
	int do_checksig = 0;
	int adjunct_store_sign = 0;
	int do_test1 = 0;
	int verbose_stderr = 0;
	int pass_thru_mode = 0;
	int do_audit = 0;
	int get_sign_if = 0;
	int uverbose = 1;
	int digest_type = 0; /* 0 Means get the md5, 3 means get the sha1 */
	int iret;
	int which_sig = 0; /* 0 means last */
	unsigned long int statbytes;
	char * gpg_prog = NULL;
	char * filearg;
	char * sleeptime = NULL;
	XFORMAT * xformat;
	char * sigfilename = NULL;
	char * which_sig_arg;

	flags = UINFILE_DETECT_FORCEUXFIOFD | 
			UINFILE_DETECT_IEEE | 
				UINFILE_UXFIO_BUFTYPE_MEM;
	verboseG = uverbose;
	which_sig_arg = strdup("0");

	/*
	* This line is required by the parser.*/	
	yylval.strb = strob_open (8);

         while (1)
           {
             int option_index = 0;
             static struct option long_options[] =
             {
               {"format", 1, 0, 'H'},
               {"get-sig-if", 1, 0, 'G'},
               {"verbose", 0, 0, 'v'},
               {"numeric-owner", 0, 0, 'x'},
               {"decode", 0, 0, 151},
               {"debug-mode", 0, 0, 'D'},
               {"catalog-sign-file", 0, 0, 'C'},
               {"storage-sign-file", 0, 0, 'S'},
               {"adjunct-storage-sign-file", 0, 0, 's'},
               {"show-auth-files", 0, 0, 'd'},
               {"sig-number", 1, 0, 'n'},
               {"pass-thru", 0, 0, 'p'},
               {"test1", 0, 0, 152},
               {"help", 0, 0, 150},
               {"noop", 0, 0, 158},
               {"checksig-mode", 1, 0, 159},
               {"sha1", 0, 0, 160},
               {"gpg-prog", 1, 0, 161},
               {"sleep", 1, 0, 162},
               {"audit", 0, 0, 163},
               {"verbose-level", 1, 0, 164},
               {"util-name", 1, 0, 165},
               {0, 0, 0, 0}
             };

             c = ugetopt_long (argc, argv, "n:pdsSCxDH:G:v",
                        long_options, &option_index);
             if (c == -1) break;

             switch (c)
               {
		case 'x':
			numeric_uids = 1;
			break;
		case 'p':
			pass_thru_mode = 1;
			break;
		case 'v':
			uverbose++;
			verboseG++;
			break;
		case 'd':
			verbose_stderr = 1;
			break;
		case 'D':
			debugmode = 1;
			break;
		case 'n':
			which_sig = atoi(optarg);
			which_sig_arg = strdup(optarg);
			break;	
		case 'G':
			get_sign_if = 1;
			sigfilename=strdup(optarg);	
			break;	
		case 'H':
			if (!strcmp(optarg,"ustar")) {
  				format=arf_ustar;
			} else if (!strcmp(optarg,"gnutar")) {
				opt_gnutar = 1;
  				format=arf_ustar;
			} else if (!strcmp(optarg,"newc")) {
  				format=arf_newascii;
			} else if (!strcmp(optarg,"crc")) {
  				format=arf_crcascii;
			} else if (!strcmp(optarg,"odc")) {
  				format=arf_oldascii;
			} else {
				fprintf (stderr,
					"unrecognized format: %s\n",
						optarg);
				exit(2);
			}
			break;
		case 'C':
			cat_sign = 1;
			break;
		case 'S':
			store_sign = 1;
			break;
		case 's':
			adjunct_store_sign = 1;
			break;
		case 160:
			digest_type = 3;
			break;
		case 158:
			break;
		case 150:
			usage(stdout, argv[0]);
			exit(0);
			break;
		case 151:
			do_decode_pass_thru = 1;
			break;
		case 152:
			do_test1 = 1;
			break;
		case 159:
			do_checksig = 1;
			sigfilename=strdup(optarg);	
			break;
		case 161:
			gpg_prog=strdup(optarg);	
			break;
		case 162:
			sleeptime=strdup(optarg);	
			break;
		case 163:
			do_audit = 1;
			flags = UINFILE_DETECT_FORCEUXFIOFD | 
					UINFILE_DETECT_IEEE | 
					UINFILE_DETECT_OTARALLOW | 
					UINFILE_UXFIO_BUFTYPE_DYNAMIC_MEM;
			break;
		case 164:
			uverbose = atoi(optarg);	
			verboseG = atoi(optarg);	
			break;
		case 165:
			swlib_utilname_set(optarg);
			break;
		default:
			usage(stderr, argv[0]);
			exit (2);
		break;
               }
	}

	if (do_decode_pass_thru && verbose_stderr) {
		fprintf(stderr, "option combination not supported.\n");
		exit(1);
	}

	if (sleeptime && atoi(sleeptime)) {
		sleep(atoi(sleeptime));
	}

	if (optind < argc) {
		if (strcmp(argv[optind], "-")) {
			fd = open(argv[optind], O_RDONLY, 0);
			if (fd <  0) {
				fprintf(stderr,
					"%s : %s\n",
					argv[optind], strerror(errno)); 
				exit(1);
			}
			filearg = strdup(argv[optind]);
		} else {
			filearg = strdup("-");
			fd = STDIN_FILENO;
		}
	} else {
		filearg = strdup("-");
		fd = STDIN_FILENO;
	}
	
	if (do_checksig) {
		ret = run_checksig(sigfilename, 
				SWBISLIBEXECDIR "/swbis/arf2arf", 
					filearg, gpg_prog, uverbose, which_sig_arg);
		if (uverbose>3) fprintf(stderr, 
				"do_checksig returned %d\n", ret);
		exit(ret);
	}

	xformat = xformat_open(STDIN_FILENO, STDOUT_FILENO, format);
	if (!xformat) exit(1);

	if (xformat_open_archive_by_fd(xformat, fd, flags, 0)) {
		if (get_sign_if) {
			int sigfd = open(sigfilename, 
					O_RDWR|O_TRUNC|O_CREAT, 0600);
			if (sigfd < 0) {
				fprintf(stderr,
					"open failed ( %s ) : %s\n",
						sigfilename, strerror(errno));
			} else {
				close(sigfd);
			}
		}
		close(STDOUT_FILENO);
		exit(1);
	}

	detected_format = xformat_get_format(xformat);	
	xformat_set_numeric_uids(xformat, numeric_uids);

	if (opt_gnutar) {
		xformat_set_tarheader_flag(xformat, 
					TARU_TAR_GNU_OLDGNUTAR, 1 /* ON */);
		xformat_set_tarheader_flag(xformat, 
					TARU_TAR_GNU_LONG_LINKS, 1 /* ON */);
	}

	if (store_sign) {
		/*
		* Write the storage section.
		*/

		/*
		* Set taru flags not to tolerate any hiccups.
		*/
		enforce_tar_format(detected_format);
		xformat_set_tarheader_flag(xformat, 
					TARU_TAR_FRAGILE_FORMAT, 1 /*on*/ );

		if (do_decode_pass_thru) {
			/*
			* decodes and re-encodes the archive therefore 
			* possibly changing it.
			*/
			ret = swlib_write_signing_files(xformat, 
						STDOUT_FILENO, 1, 0);
		} else {
			/*
			* Decodes, but writes output that is identical
			* to the input.
			*/
			ret = taruib_write_storage_stream(
					(void *)xformat,
					STDOUT_FILENO,
					1 		/*version*/, 
					-1		/*adjunct_ofd*/ , 
					verbose_stderr 	/*verbose*/, 
					digest_type);
		}
	} else if (get_sign_if) {
		enforce_tar_format(detected_format);
		xformat_set_tarheader_flag(xformat,
					TARU_TAR_FRAGILE_FORMAT, 1 /*on*/ );
		ret = taruib_write_signedfile_if(
				(void*)xformat, 
				STDOUT_FILENO, 
				sigfilename, 
				uverbose,
				which_sig /*last signature*/); 
		if (uverbose >= 3) {
			fprintf(stderr,
				"taruib_write_signedfile_if returned %d\n",
							ret);
		}
	} else if (adjunct_store_sign) {
		enforce_tar_format(detected_format);
		if (do_decode_pass_thru) {
			swlib_write_signing_files(xformat,
					STDOUT_FILENO, 1, 1);
		} else {
			int nullfd;
			nullfd = open("/dev/null", O_RDWR, 0);
			if (nullfd < 0) {
				fprintf(stderr, "error opening /dev/null \n");
				exit(2);
			}
			taruib_write_storage_stream(
					(void*)xformat, 
					nullfd, 
					1 /* version */, 
					STDOUT_FILENO, 
					verbose_stderr, 
					digest_type);
		}
	} else if (cat_sign) {
		enforce_tar_format(detected_format);
		if (do_decode_pass_thru) {
			ret = swlib_write_signing_files(
				xformat, 
				STDOUT_FILENO, 
				0, 
				0);
		} else {
			ret = taruib_write_catalog_stream(
				(void*)xformat, 
				STDOUT_FILENO, 
				1 /* version */, 
				verbose_stderr);
		}
	} else if (do_test1) {
		xformat_set_tarheader_flag(xformat,
				TARU_TAR_FRAGILE_FORMAT, 1 /*on*/ );
		ret = taruib_write_storage_stream(
			xformat, 
			STDOUT_FILENO, 
			1 /*version*/, 
			STDERR_FILENO, 
			0 /*verbose*/, 
			digest_type);	

	} else if (pass_thru_mode) {
		ret = taruib_write_pass_files(
			(void*)xformat, 
			STDOUT_FILENO, 
			-1);
	} else if (do_audit) {
		/*
		* Test use of the swlib_arfcopy() and swi_<*> routines.
		*/
		ret = swlib_audit_distribution(xformat, 
					do_decode_pass_thru, 
					STDOUT_FILENO, 
					&statbytes, 
					(int*)(NULL),
					(void (*)(int))(NULL));
	} else {
		xformat_set_output_format(xformat, (int)format);
		xformat_set_false_inodes(xformat, XFORMAT_OFF);
	
		ifd = xformat_get_ifd(xformat);

		while ((ret = xformat_read_header(xformat)) > 0) {
			if (xformat_is_end_of_archive(xformat)){
				break;
			}
			xformat_write_header(xformat);
			xformat_copy_pass(xformat, STDOUT_FILENO, ifd);
		}
		
		if (ret >= 0) {
			xformat_write_trailer(xformat);
			if (xformat_get_pass_fd(xformat)) {
				xformat_clear_pass_buffer(xformat);
			}
		}
	}


	iret = xformat_close(xformat);
	if (fd != STDIN_FILENO) close(fd);

	if (uverbose >= 3) {
		fprintf(stderr, "%s : ret=%d\n", argv[0], ret);
		fprintf(stderr, "%s : iret=%d\n", argv[0], iret);
	}
	if (ret == 0) ret = iret;

	if (uverbose >= 3) {
		fprintf(stderr, "%s : exiting with %d\n", argv[0], ret);
	}
	close(STDERR_FILENO);
	if (ret < 0) ret = 1;
	exit(ret);
}
