/*
 * dip		A program for handling dialup IP connections.
 *
 *		UNIX terminal I/O support functions.  This file takes
 *		care of opening, setting up and maintaining the line,
 *		and it takes care of allocating the buffers and init-
 *		ializing their control structures.
 *
 *
 * Version:	@(#)tty.c	1.02	1.July.1995
 *
 *
 * New Author:	Joachim Bartz, <injb@sun.rz.tu-clausthal.de>
 *
 *		There have been so many changes to get  dip work under
 *		BSD...   So I believe, you can call me an author, too.
 *		Please read the  ReadMe  file and the man page to know
 *		what, when, why.
 *
 *
 * Org.Author:	Fred N. van Kempen, <waltje@uWalt.NL.Mugnet.ORG>
 *		Copyright 1988-1993 MicroWalt Corporation
 *
 *
 *		This program is free software; you can redistribute it
 *		and/or  modify it under  the terms of  the GNU General
 *		Public  License as  published  by  the  Free  Software
 *		Foundation;  either  version 2 of the License, or  (at
 *		your option) any later version.
 *
 *		This program is  distributed in the  hope that it will
 *		be useful, but WITHOUT ANY WARRANTY;  without even the
 *		implied warranty of MERCHANTABILITY  or  FITNESS FOR A
 *		PARTICULAR  PURPOSE.    See  the  GNU  General  Public
 *		License for more details.
 *
 *		You  should have received  a copy  of the  GNU General
 *		Public License along with this program;  if not, write
 *		to the
 *			Free Software Foundation, Inc.
 *			675 Mass Ave
 *			Cambridge, MA 02139, USA
 */


#include "dip.h"

#include <sys/stat.h>
#include <sys/fcntl.h>
#include <sys/ioctl.h>

#include <ctype.h>
#include <paths.h>



#define TTY_BUFSIZE	256	/* size of one serial buffer	*/


static struct
{
  char	*speed;
  int	code;
} tty_speeds [] =	/* table of usable baud rates	*/
{
  { "50",	B50	},
  { "75",	B75  	},
  { "110",	B110	},
  { "134",	B134	},
  { "150",	B150	},
  { "200",	B200	},
  { "300",	B300	},
  { "600",	B600	},
  { "1200",	B1200	},
  { "1800",	B1800	},
  { "2400",	B2400	},
  { "4800",	B4800	},
#ifdef B7200
  { "7200",	B7200	},
#endif
  { "9600",	B9600	},
#ifdef B14400
  { "14400",	B14400	},
#endif
  { "19200",	B19200	},
#ifdef B28800
  { "28800",	B28800	},
#endif
  { "38400",	B38400	},
#ifdef B57600
  { "57600",	B57600	},
#endif
#ifdef B76800
  { "76800",	B76800	},
#endif
#ifdef B115200
  { "115200",	B115200	},
#endif
#ifdef B230400
  { "230400",	B230400	},
#endif
  { NULL,	0	}
};


static struct termios	tty_saved;	/* saved TTY device state	*/
static struct termios	tty_current;	/* current TTY device state	*/

static char		*in_buff,	/* line input/output buffers	*/
			*out_buff,
			*in_ptr,
			*out_ptr;

static int		in_size,	/* buffer sizes and counters	*/
			out_size,
			in_cnt,
			out_cnt,
			tty_sdisc,	/* saved TTY line discipline	*/
			tty_ldisc,	/* current TTY line discipline	*/
			tty_fd	  = -1;	/* TTY file descriptor		*/

static	char		*saved_path = NULL;
static	short		saved_lock  = 0;



/*
 * Disable any messages to the input channel of this process.
 */
int  tty_nomesg ( void )
{
  return fchmod ( STDIN_FILENO, S_IRUSR | S_IWUSR );
}



/*
 * Lock or unlock a terminal line.
 */
int  tty_lock ( char *path, int mode )
{

  struct passwd	 *pw;

  int		 fd;


  if ( mode )					/* locking on ?		*/
  {				
    if ( path  ==  NULL )
    {
      return 0;					/* standard input	*/
    }

    if ( saved_path != NULL )
    {
      free ( saved_path );
      saved_path = NULL;
    }

    saved_path = (char*) malloc ( sizeof ( PATH_LOCKD ) +
							strlen ( path ) + 1 );

    (void) strcpy ( saved_path, PATH_LOCKD );
    (void) strcat ( saved_path, path );


    if ( ( fd = creat ( saved_path, 0644 ) )  <  0)
    {
      if ( errno  !=  EEXIST )
      {
	(void) fprintf ( stderr, "DIP: tty: lock: (%s): %s\n",
					saved_path, strerror ( errno ) );
      }

      return -1;
    }

    (void) close ( fd );


    /*
     *  Make sure UUCP owns the lockfile.  Required by some packages.
     */
    if ( ( pw = getpwnam ( UID_UUCP ) )  ==  NULL )
    {
      (void) fprintf ( stderr, "DIP: tty: lock: UUCP user  %s  unknown!\n",
								UID_UUCP );
      return 0;		/* keep the lock anyway */
    }

    (void) chown ( saved_path, pw->pw_uid, pw->pw_gid );

    saved_lock = 1;
  }
  else	/* unlock */
  {

    if ( ( ! saved_lock ) || ( saved_path == NULL ) )
    {
      return 0;
    }

    if ( unlink ( saved_path )  <  0)
    {
      (void) fprintf ( stderr, "DIP: tty: unlock: (%s): %s\n", saved_path,
							strerror ( errno ) );
      return -1;
    }

    saved_lock = 0;
  }


  return 0;
}



/*
 *  Find a serial speed code in the table.
 */
static  int  tty_find_speed ( char *speed )
{

  int i = 0;


  while ( tty_speeds[i].speed  !=  NULL )
  {
    if ( ! strcmp ( tty_speeds[i].speed, speed ) )
      return tty_speeds[i].code;

    i++;
  }


  return -EINVAL;
}



/*
 *  Set the number of stop bits.
 */
static  int  tty_set_stopbits ( struct termios *tty, char *stopbits )
{

  if ( opt_verb )
  {
    (void) printf ( "DIP: tty: set_stopbits: %c\n", *stopbits );
  }


  switch ( *stopbits )
  {
    case '1':	tty->c_cflag &= ~CSTOPB;
		break;

    case '2':	tty->c_cflag |= CSTOPB;
		break;

    default:	return -EINVAL;
  }


  return 0;
}



/*
 *  Set the number of data bits.
 */
static  int  tty_set_databits ( struct termios *tty, char *databits )
{

  if ( opt_verb )
  {
    (void) printf ( "DIP: tty: set_databits: %c\n", *databits );
  }


  tty->c_cflag &= ~CSIZE;


  switch ( *databits )
  {
    case '5':	tty->c_cflag |= CS5;
		break;

    case '6':	tty->c_cflag |= CS6;
		break;

    case '7':	tty->c_cflag |= CS7;
		break;

    case '8':	tty->c_cflag |= CS8;
		break;

    default:	return -EINVAL;
  }


  return 0;
}



/*
 * Set the type of parity encoding.
 */
static int  tty_set_parity ( struct termios *tty, char *parity )
{

  if ( opt_verb )
  {
    printf("DIP: tty: set_parity: %c\n", *parity );
  }


  switch ( toupper ( *parity ) )
  {
    case 'N':	tty->c_cflag &= ~(PARENB | PARODD);
		break;  

    case 'O':	tty->c_cflag &= ~(PARENB | PARODD);
		tty->c_cflag |= (PARENB | PARODD);
		break;

    case 'E':	tty->c_cflag &= ~(PARENB | PARODD);
		tty->c_cflag |= PARENB;
		break;

    default:	return -EINVAL;
  }


  return 0;
}



/*
 *  Set the line speed of a terminal line.
 */
static  int  tty_set_speed ( struct termios *tty, char *speed )
{

  int code;


  if ( opt_verb )
  {
    printf ( "DIP: tty: set_speed: %s\n", speed );
  }


  if ( ( code = tty_find_speed ( speed ) )  <  0 )
  {
    return -1;
  }


  tty->c_ispeed = tty->c_ospeed = code;


  return 0;
}



/*
 *  Put a terminal line in a transparent state.
 */
static  int  tty_set_raw ( struct termios *tty )
{

  int	i;


  for ( i = 0; i < NCCS; tty->c_cc[i++] = '\0' ); /* no spec chr	*/

  tty->c_cc [ VMIN  ] = 1;			/* wait for 1 char	*/
  /* tty->c_cc [ VTIME ] = 0; */		/* wait forever		*/
  tty->c_iflag = IGNBRK | IGNPAR;
  tty->c_cflag = CLOCAL | CRTSCTS | /* HUPCL | */ CREAD;
  tty->c_oflag =
  tty->c_lflag = 0;


  return 0;
}



/*
 *  Fetch the state of a terminal.
 */
static  int  tty_get_state ( struct termios *tty )
{

  if ( tcgetattr ( tty_fd, tty )  <  0 )
  {
    (void) fprintf ( stderr, "DIP: tty: get_state: %s\n",
							strerror ( errno ) );
    return -errno;
  }


  return 0;
}



/*
 *  Set the state of a terminal.
 */
static int  tty_set_state ( struct termios *tty )
{

  if ( tcsetattr ( tty_fd, TCSANOW, tty )  <  0 )
  {
    (void) fprintf ( stderr, "DIP: tty: set_state: %s\n",
							strerror ( errno ) );
    return -errno;
  }


  return 0;
}



/*
 *  Get the line discipline of a terminal line.
 */
int  tty_get_disc ( int *disc )
{

  if ( ioctl ( tty_fd, TIOCGETD, disc )  <  0 )
  {
    (void) fprintf ( stderr, "DIP: tty: get_disc: %s\n",
							strerror ( errno ) );
    return -errno;
  }


  return 0;
}



/*
 *  Set the line discipline of a terminal line.
 */
int  tty_set_disc ( int disc )

{
  if ( disc  ==  -1 )
    disc = tty_sdisc;	/* set saved tty discipline */


  if ( ioctl ( tty_fd, TIOCSCTTY, 0 )  <  0 )
  {
    syslog ( LOG_ERR, "DIP: TIOCSCTTY: %m" );
    return -errno;
  }


  if ( tcsetpgrp ( tty_fd, getpid ( ) )  <  0 )
  {
    syslog ( LOG_ERR, "DIP: TIOCGPGRP(%d): %m",	getpgrp() );
    return -errno;
  }


  if ( ioctl ( tty_fd, TIOCSETD, &disc )  <  0 )
  {
    syslog ( LOG_ERR, "DIP: TIOCSETD(%d): %m", disc );
    return -errno;
  }


  if ( ( disc == SLIPDISC ) )
  {

    int	 tmp_unit = -1;


    if ( ioctl ( tty_fd, SLIOCGUNIT, (caddr_t) &tmp_unit )  <  0 )
    {
      syslog ( LOG_ERR, "DIP: SLIOCGUNIT: %m" );
      return -errno;
    }


    if ( tmp_unit < 0 )
    {
      syslog ( LOG_ERR, "DIP: SLIP-IF = %d ? : %m", tmp_unit );
      return -errno;
    }


    (void) sprintf ( mydip.ifname, "sl%d", tmp_unit );

/*
 *  Note: Do not set this preprocessor macro in any circumstances.
 *
 *  May be, in future I excactly know, how to set the SLIP MTU
 *  _directly_ at the socket port (if it is possible at all by
 *  the system).
 *
 *  But at the moment, this code doesn't work as it should.
 *
 *  All possible settings are done via the external script
 *  (parameter -s) or the ifconfig(8) and route(8) commands.
 */

#undef USE_SOCKET_DIRECTLY

#ifdef USE_SOCKET_DIRECTLY

    if ( ( mydip.protonr ) && ( mydip.mtu ) )
    {

      struct ifreq	ifr;
      int		s;


      /*
       *  Try to get a free socket.
       */
      if ( ( s = socket ( AF_INET, SOCK_DGRAM, 0 ) )  <  0 )
      {
	syslog ( LOG_ERR, "DIP: get(SOCK_DGRAM): %m" );
	return -errno;
      }


      /*
       *  Copy the desired socket name to the interface-request structure.
       */
      (void) strncpy ( ifr.ifr_name, mydip.ifname, IFNAMSIZ );


      /*
       *  Get the flags for the interface
       */
      if ( ioctl ( s, SIOCGIFFLAGS, (caddr_t) &ifr )  <  0 )
      {
	syslog ( LOG_ERR, "DIP: SIOCGIFFLAGS: %m" );
	return -errno;
      }


      ifr.ifr_flags &= ~( IFF_LINK0 | IFF_LINK1 | IFF_LINK2  );

      if ( mydip.protonr  ==  1 )
	;				/* SLIP, disable VJ compression	*/
      else
	if ( mydip.protonr  ==  2 )
	  ifr.ifr_flags |= IFF_LINK0;	/* CSLIP, enable VJ compression	*/
/*	else
	  ;
*/
      ifr.ifr_flags |= IFF_LINK2;	/* add AUTO-Mode 		*/


      if ( ioctl ( s, SIOCSIFFLAGS, (caddr_t) &ifr )  <  0 )
      {
	syslog ( LOG_ERR, "DIP: SIOCSIFFLAGS: %m" );
	return -errno;
      }


      /*
       *  Now get and set the MTU directly at the socket.
       *  Seems, that there is no other way than going this path.
       */
      if ( ioctl ( s, SIOCGIFMTU, (caddr_t) &ifr )  <  0 )
      {
	syslog ( LOG_ERR, "DIP: SIOCGIFMTU: %m" );
	return -errno;
      }
    
      ifr.ifr_metric = mydip.mtu;


      /*
       *  Set the MTU directly at the socket.
       */
      if ( ioctl ( s, SIOCGIFMTU, (caddr_t) &ifr )  <  0 )
      {
	syslog ( LOG_ERR, "DIP: SIOCSIFMTU: %m" );
	return -errno;
      }

      (void) close ( s );
    }

#endif

  }

  return 0;
}



/*
 *  Read one character (byte) from the TTY link.
 */
int  tty_getc ( short waitfor )
{
  /*
   *  Ist der interne Read-Buffer schon/noch leer?
   *  Dann alle anstehenden Zeichen abholen.
   */
  if ( in_cnt  <=  0 )
  {
    /*
     *  Soll auf ankommende Zeichen gewartet werden?
     *  Abbruch durch Signals oder eben ankommende Zeichen.
     */
    if ( waitfor )
    {

      fd_set  mask;

      FD_ZERO ( &mask );
      FD_SET  ( tty_fd, &mask );

      if ( select ( 32, &mask, (fd_set*) NULL, (fd_set*) NULL,
					(struct timeval*) NULL )  >  0 )
      {
	if ( FD_ISSET ( tty_fd, &mask ) )
	  in_cnt = read ( tty_fd, in_ptr = in_buff, in_size );
      }
    }
    else
      in_cnt = read ( tty_fd, in_ptr = in_buff, in_size );
  }


  if ( in_cnt <= 0 )
    return -1;


  if ( opt_debg  ==  2 )
  {
    int   ic  = in_cnt;
    char  *cp = in_ptr;
    char  c;

    (void) printf ( "in_cnt = %d, inptr = \"", ic );

    while ( ic--  >  0 )
    {
      switch ( c = *cp++ )
      {
	case '\r' : c = 'r'; break;
	case '\n' : c = 'n';
      }

      (void) printf ( "%c", c );
    }
    (void) printf ( "\"\n" );
  }

  in_cnt--;

  return (int) *in_ptr++;
}



/*
 * Write one character (byte) to the TTY link.
 */
int  tty_putc ( int c )
{

  int s;


  if ( ( out_cnt == out_size ) || ( c == -1 ) )
  {
    s = write ( tty_fd, out_buff, out_cnt );

    out_cnt = 0;
    out_ptr = out_buff;

    if ( s < 0 )
      return -1;
  }


  if ( c != -1 )
  {
    *out_ptr = (char) c;
    out_ptr++;
    out_cnt++;
  }


  return 0;
}



/*
 *  Output a string of characters to the TTY link.
 */
void  tty_puts ( char *s )
{

  while ( *s != '\0' )
  {
    tty_putc ( (int) *s++ );
  }


  tty_putc ( -1 );	/* flush */
}



/*
 * Return the TTY link's file descriptor.
 */
int  tty_askfd ( void )
{
  return tty_fd;
}



/*
 * Set the number of databits a terminal line.
 */
int  tty_databits ( char *bits )
{

  if ( tty_set_databits ( &tty_current, bits )  <  0)
  {
    return -1;
  }


  return tty_set_state ( &tty_current );
}



/*
 *  Set the number of stopbits of a terminal line.
 */
int  tty_stopbits ( char *bits )
{

  if ( tty_set_stopbits ( &tty_current, bits )  <  0)
  {
    return -1;
  }


  return tty_set_state ( &tty_current );
}



/*
 *  Set the type of parity of a terminal line.
 */
int  tty_parity ( char *type )
{

  if ( tty_set_parity ( &tty_current, type )  <  0)
  {
    return -1;
  }


  return tty_set_state ( &tty_current );
}



/*
 *  Set the line speed of a terminal line.
 */
int  tty_speed ( char *speed )
{

  if ( tty_set_speed ( &tty_current, speed )  <  0 )
  {
    return -1;
  }


  return tty_set_state ( &tty_current );
}


/*
 *  Drop DTR of a terminal line.
 */
int  tty_dropdtr ( void )
{

  if ( ioctl ( tty_fd, TIOCCDTR )  <  0 )
  {
    (void) fprintf ( stderr, "DIP: tty: drop_dtr: %s\n",
							strerror ( errno ) );
    return -errno;
  }


  return 0;
}


/*
 *  Raise DTR of a terminal line.
 */
static  int  tty_raisedtr ( void )
{

  if ( ioctl ( tty_fd, TIOCSDTR )  <  0 )
  {
    (void) fprintf ( stderr, "DIP: tty: raise_dtr: %s\n",
							strerror ( errno ) );
    return -errno;
  }


  return 0;
}



/*
 *  Hangup the line.
 *
static  int  hanguptty ( void )
{

  (void) tty_dropdtr ( );

  (void) save_sleep ( 1 );

  (void) tty_raisedtr ( );


  return 0;
}
*/


/*
 *  Flush input on the terminal.
 */
int  tty_flush ( void )
{

  char	buf [ 32 ];


  while ( read ( tty_fd, &buf, 32 ) > 0) ;


  return 0;
}



/*
 *  Now we have to watch for the presence of the carrier.
 *  If there is a lost carrier, the tty sends us a SIGHUP.
 */
void  tty_set_online ( void )
{

  struct termios  tty;

  /*
   * Fetch the current state of the terminal.  Errors are ignored!
   */
  (void) tty_get_state ( &tty );

  tty.c_cflag &= ~CLOCAL;  /* local mode off    */
  tty.c_cflag |= HUPCL;    /* watch for carrier */

  /*
   *  Now set the new state of the terminal.  Errors are ignored, too.
   */
  (void) tty_set_state ( &tty );
}



/*
 *  
 */
void  tty_set_offline ( void )
{

  struct termios  tty;

  /*
   * Fetch the current state of the terminal.  Errors are ignored!
   */
  (void) tty_get_state ( &tty );

  tty.c_cflag &= ~HUPCL;  /* ignore carrier state */
  tty.c_cflag |= CLOCAL;  /* local mode           */

  /*
   *  Now set the new state of the terminal.  Errors are ignored, too.
   */
  (void) tty_set_state ( &tty );

  (void) tty_raisedtr ( );
}



/*
 * Close down a terminal line.
 */
int  tty_close ( void )
{

  (void) tty_dropdtr ( );


  if ( mydip.isattached )  {
    (void) attach ( DO_DETACH );
    mydip.isattached = 0;
  }


  if ( mydip.setsdisc )  {
    (void) tty_set_disc ( tty_sdisc );  /* reset tty to saved discipline */
    mydip.setsdisc = 0;
  }


  (void) tty_lock ( NULL, 0 );


  return 0;
}



/*
 *  Open and initialize a terminal line.
 */
int  tty_open ( char *name )
{

  int	fd;
  char	*sp;
  char	path [ MAXPATHLEN ];


  /*
   *  Try opening the TTY device.
   */
  if ( name  !=  NULL )
  {
    (void) strcpy ( path, name );

    if ( strncmp ( _PATH_DEV, name, sizeof( _PATH_DEV ) - 1 ) )
    {
      (void) strcpy ( path, _PATH_DEV );
      (void) strcat ( path, name );
    }


    if ( ( sp = strrchr ( path, '/') ) != (char*) NULL)
      sp++;
    else
      sp = name;


    if ( ( fd = open ( path, O_RDWR | O_NONBLOCK ) )  <  0 )
    {
      (void) fprintf ( stderr, "DIP: tty: open(%s, RW): %s\n",
						path, strerror ( errno ) );
      return -errno;
    }


    tty_fd = fd;


    if ( opt_verb )
    {
      printf ( "DIP: tty: open: %s (%d)\n", path, fd );
    }
  }
  else
  {
    tty_fd = 0;

    sp = (char*) NULL;
  }


  /*
   * Size and allocate the I/O buffers.
   */
  in_size  = TTY_BUFSIZE;
  in_buff  = (char*) malloc ( in_size  );

  out_size = in_size;
  out_buff = (char*) malloc ( out_size );

  if ( ( in_buff  == (char*) NULL  )
    || ( out_buff == (char*) NULL) )
  {
    (void) fprintf ( stderr, "DIP: tty: open: cannot alloc(%d, %d) buffers "
					"(%d)\n", in_size, out_size, errno );
    return -ENOMEM;
  }

  in_cnt = 0;
  in_ptr = in_buff;

  out_cnt = 0;
  out_ptr = out_buff;
  out_size -= 4;	/* safety */

  if ( opt_verb )
  {
    (void) printf ( "DIP: tty: open: IBUF=%d OBUF=%d\n", in_size, out_size );
  }


  /*
   * Fetch the current state of the terminal.
   */
  if ( tty_get_state ( &tty_saved )  <  0)
  {
    (void) fprintf ( stderr, "DIP: tty: open: cannot get current state!\n" );

    return -errno;
  }

  tty_current = tty_saved;


  /*
   * Fetch the current line discipline of this terminal.
   */
  if ( tty_get_disc ( &tty_sdisc )  <  0)
  {
    (void) fprintf ( stderr, "DIP: tty: open: cannot get current "
							"line disc!\n" );
    return -errno;
  } 


  tty_ldisc = tty_sdisc;


  /*
   * Put this terminal line in a 8-bit transparent mode.
   */
  if ( tty_set_raw ( &tty_current )  <  0)
  {
    (void) fprintf ( stderr, "DIP: tty: open: cannot set RAW mode!\n" );
    return -errno;
  }


  /*
   * If we are running in MASTER mode, set the default speed.
   */
  if ( ( path  !=  NULL )
    && ( tty_set_speed ( &tty_current, "9600" )  !=  0 ) )
  {
    (void) fprintf ( stderr, "DIP: tty: open: cannot set 9600 bps!\n" );

    return -errno;
  }


  /*
   * Set up a completely 8-bit clean line.
   */
  if ( tty_set_databits ( &tty_current, "8" )
    || tty_set_stopbits ( &tty_current, "1" )
    || tty_set_parity   ( &tty_current, "N" ) )
  {
    (void) fprintf ( stderr, "DIP: tty: open: cannot set 8N1 mode!\n" );

    return -errno;
  }


  /*
   * Set the new line mode.
   */
  if ( ( fd = tty_set_state ( &tty_current ) )  <  0)
  {
    return ( fd );
  }


  /*
   * And raise DTR.
   */
  if ( tty_raisedtr ( )  <  0)
  {
    return 0 ;
  }


  /*
   * OK, all done.  Lock this terminal line.
   */
  return tty_lock ( sp, 1 );
}
