/*
 * dip		A program for handling dialup IP connections.
 *		This program handles the connections needed for dialup
 *		IP  links,  like  SLIP  or  PPP.  It  can  handle both
 *		incoming  and  outgoing  connections,  using  password
 *		security for incoming connections.
 *
 *		This module contains the  main, initialization and the
 *		usage functions.
 *
 *
 * Usage:	See help display below.
 *
 *
 * Version:	@(#)dip.c	1.02	1.July.1995
 *
 *
 * New Author:	Joachim Bartz, <injb@sun.rz.tu-clausthal.de>
 *
 *		There have been so many changes to get  dip work under
 *		BSD...   So I believe, you can call me an author, too.
 *		Please read the  ReadMe  file and the man page to know
 *		what, when, why.
 *
 *
 * Org.Author:	Fred N. van Kempen, <waltje@uWalt.NL.Mugnet.ORG>
 *		Copyright 1988-1993 MicroWalt Corporation
 *
 *
 *		This program is free software; you can redistribute it
 *		and/or  modify it under  the terms of  the GNU General
 *		Public  License as  published  by  the  Free  Software
 *		Foundation;  either  version 2 of the License, or  (at
 *		your option) any later version.
 *
 *		This program is  distributed in the  hope that it will
 *		be useful, but WITHOUT ANY WARRANTY;  without even the
 *		implied warranty of MERCHANTABILITY  or  FITNESS FOR A
 *		PARTICULAR  PURPOSE.    See  the  GNU  General  Public
 *		License for more details.
 *
 *		You  should have received  a copy  of the  GNU General
 *		Public License along with this program;  if not, write
 *		to the
 *			Free Software Foundation, Inc.
 *			675 Mass Ave
 *			Cambridge, MA 02139, USA
 */

#include "dip.h"

#include <netdb.h>
extern  int h_errno;

struct dip  mydip;		/* Hauptstruktur		*/


short	opt_debg;		/* Debugger-Modi 0, 1, 2	*/
short	opt_verb;		/* Verbose-Modus 0, 1		*/
short   opt_i;                  /* DialIn-Modus  0, 1           */


#define VERSION	"  v1.02  (1. July 1995)"



/*
 *
 */
static  void  pre_catcher1 ( int sig )
{
  switch ( sig )
  {
    case SIGHUP  : exit ( 2 );
    case SIGINT  : exit ( 3 );
    case SIGTERM : exit ( 4 );
  }

  exit ( 100 + sig );
}



/*
 *  Grundsaetzliche Initialisierung.
 *
 *  Ausfuellen der dip-Struktur mit den System-Werten.
 *  Im ersten Teil werden nur die Userdaten eingetragen. Die anderen
 *  Daten (z.B. ueber das Netzwerk) werden erst dann eingestellt und
 *  geholt, wenn es wirklich notwendig wird.
 */
static int  initialize_dip1 ( void )
{

  struct passwd   *pw;


  /*
   *  Ermitteln der allgemeinen User-Daten des aufrufenden Users.
   */
  if ( ( pw = getpwuid ( getuid ( ) ) )  ==  (struct passwd*) NULL )
  {
    /*
     *  Dieser User existiert in diesem System nicht.
     *  Melden und Programm gleich beenden!
     */
    (void) fprintf ( stderr, "DIP: You do not exist. Go away.\n" );

    return -1;
  }


  /*
   *  Merken des User-Namen
   */
  (void) strncpy ( mydip.username, pw->pw_name, MAXUSERNMLEN-1 );


  /*
   *  Merken des User-Password
   */
  (void) strncpy ( mydip.userpasswd, pw->pw_passwd, MAXPASSWDLEN-1 );


  /*
   *  Merken des home-Pfades
   */
  (void) strncpy ( mydip.userhome, pw->pw_dir, MAXPATHLEN-1 );


  /*
   *  Merken, wie die Datei mit der  dip-pid  heisst, damit  dip -k
   *  auch wirklich nur den richtigen Job killt
   */
  (void) strcpy ( mydip.pidfile, PATH_DIP_PID EXT_DIP_PID );


  return 0;
}



/*
 *  Grundsaetzliche Initialisierung.
 *
 *  Ausfuellen der dip-Struktur mit den System-Werten.
 *  In diesem zweiten Init-Teil werden die Netzwerk-relevanten Daten
 *  ermittelt.
 *  Diese Trennung ist notwendig, damit dip nicht haengt, wenn eine Anfrage
 *  bei einem Nameserver aufgrund 'Network down' scheitert.
 *
 *  Kann z.B bei einem  dip -k  sehr unangenehm werden, wenn der dip-Daemon
 *  den Carrierverlust (durch Zufall) nicht mitbekommt.
 */
static int  initialize_dip2 ( void )
{

  struct hostent  *hp;


  /*
   *  Ermitteln dieses Hostnamen
   */
  if ( gethostname ( mydip.local, MAXHOSTNAMELEN-1 )  <  0 )
  {
    /*
     *  Wenn der Hostname nicht verfuegbar ist, bekommen wir hier einen
     *  eine -1 zurueck und einen Wert in errno
     */
    (void) fprintf ( stderr, "DIP: gethostname: %s\n", strerror ( errno ) );

    return -1;
  }


  /*
   *  Eintraege aus der Network-Tabelle holen
   */
  if ( ( hp = gethostbyname ( mydip.local ) )  ==  (struct hostent*) NULL )
  {
    /*
     *  Wenn keine Eintraege in der Network-Tabelle  verfuegbar sind,
     *  den ermittelten h_errno-Wert auswerten. Programm wird beendet.
     */
    herror ( mydip.local );

    return -1;
  }


  /*
   *  Aus der eben ermittelten Hostentry-Struktur jetzt den offiziellen
   *  Hostnamen kopieren und merken
   */
  (void) strncpy ( mydip.local, hp->h_name, MAXHOSTNAMELEN-1 );


  /*
   *  Die Adressenliste auch noch kopieren
   */
  (void) memcpy ( (char*) &mydip.ip_loc, (char*) hp->h_addr_list[0],
								hp->h_length );

  /*
   *  SLIP als Default-Protokoll-Zeichenkette eintragen, falls nicht schon
   *  per command-line geschehen.
   */
  if ( ! mydip.protonr )
  {
    (void) strncpy ( mydip.protostr, DEF_PROTOCOL, MAXPROTSTRLEN-1 );

    /*
     *  Die zum Default-Protokoll gehoerige Protokoll-Nummer holen und
     *  merken fuer spaetere Zugriffe
     */
    mydip.protonr = get_protocol ( mydip.protostr );
  }


  /*
   *  Default-Netmask (String) einstellen
   */
  (void) strcpy ( mydip.netmask, DEF_NETMASK );


  /*
   *  Default-MTU (Zahlenwert) einstellen
   */
  if ( ! mydip.mtu )
  {
    mydip.mtu = DEF_MTU;
  }


  return 0;
}



/*
 *  Kill eines laufenden Protokoll-Servers anhand der dip.pid-Datei
 */
static int  dip_kill ( void )
{

  FILE	*fp;
  int	pid;
  char	buffer [ 16 ];


  /*
   *  Oeffnen der angegebenen .pid-Datei, in der eine PID eines dip-Jobs
   *  abgespeichert wurde.
   */
  if ( ( fp = fopen ( mydip.pidfile, "r" ) )  ==  (FILE*) NULL )
  {
    /*
     *  Wenn die Datei nicht existiert, kann auch kein Job entfernt werden.
     *  Dann muss der User per Hand den gemeinten Job killen.
     */
    (void) fprintf ( stderr, "DIP: Cannot open %s (%s)\n",
					mydip.pidfile, strerror ( errno ) );
    return -1;
  }


  /*
   *  Die ersten (max.) 16 Bytes sollten eine PID enthalten.
   */
  (void) fgets ( buffer, 16, fp );


  /*
   *  Datei schliessen.
   */
  (void) fclose ( fp );


  /*
   *  SIGTERM an den ermittelten Job schicken.
   */
  if ( kill ( pid = atoi ( buffer ), SIGTERM ) )
  {
    /*
     *  Das ging leider nicht. Dann Programm beenden, aber sicherheitshalber
     *  die pid-Datei nicht entfernen.
     */
    (void) fprintf ( stderr, "DIP: Cannot kill process %d (%s)\n",
						pid, strerror ( errno ) );
    return -1;
  }

  (void) printf ( "DIP: Process %d killed\n", pid );


  /*
   *  Kill war erfolgreich, also pid-Datei entfernen.
   */
  (void) unlink ( mydip.pidfile );


  return 0;
}



/*
 *  Usage... alle Programm-Optionen in Kurzform ausgeben
 */
static  void  usage ( void )
{
  printf ( "\nUsage: dip -i [-v]   or   diplogin [-v]\n"
	   "       dip -k [-v] [-f file]\n"
	   "       dip -t [-v] [-c count] [-f file]\n"
	   "       dip script.dip [-v] [-d] [-c count] [-f file] [-m mtu]\n"
	   "           [-p proto] [-s /setup]\n\n"
	   "Flags: -c count   Set startup time limit to count.\n"
	   "       -d         Debug mode. Includes verbose mode!\n"
	   "       -f file    File containing the pid for later -kill.\n"
	   "       -h         Show these hints.\n"
	   "       -i         Use DIP in dial-in server mode.\n"
	   "       -k         Kill the specified DIP protocol server.\n"
	   "       -m mtu     Mtu to use as default.\n"
	   "       -p proto   Protcol to use as default.\n"
	   "       -s /setup  Shellscript to complete protocol server setup.\n"
	   "       -t         Test mode. Give commands by stdin.\n"
	   "       -v         Verbose mode. See debug mode also!\n\n" );
}



/* 
 *  Hauptroutine.
 *
 *  - Zuerst wird die Systemumgebung untersucht und die wichtigen Daten
 *    in der  dip-Struktur festgehalten.
 *  - Anschliessend werden die uebermittelten Parameter ausgewertet.
 */
int main ( int argc, char *argv[] )
{
  int	 s;

  char	 *sp;
  FILE   *fp;
  char   cmdfile [ 256+8 ];

  short  opt_kill = 0;
  short  opt_test = 0;  


  /*
   *  Ausnullen der dip-Struktur, damit keine falschen Daten im
   *  Speicher bleiben.
   */
  (void) memset ( (char*) &mydip, 0, sizeof ( struct dip ) );


  /*
   *  Kurze Beschreibung, worum es bei diesem Programm eigentlich geht.
   */
  (void) printf ( "\nDIP: Dialup IP Protocol Driver\n"
		  "Original idea & source:  Fred N. van Kempen, "
		  "MicroWalt Corporation\n"
		  "NetBSD implementation :  Joachim Bartz  " VERSION "\n\n" );


  /*
   *  Grundinitialisierung (Teil 1) der dip-Datenstruktur, dh. holen der
   *  ganzen bekannten Daten aus /etc/ und sonstiger Konfiguration.
   */
  if ( initialize_dip1 ( ) )
  {
    return -1;
  }


  /*
   *  Feststellen, ob das Programm als  diplogin laeuft.
   *  Wenn ja, ist es als  dip -i  zu behandeln.
   *
   *  Erstmal den Programmnamen extrahieren.
   */
  if ( ( sp = strrchr ( argv[0], '/' ) )  !=  NULL )
  {
    sp++;
  }
  else 
  {
    sp = argv [ 0 ];
  }

  if ( ( *sp == '-' )  ||  ( ! strcmp ( sp, "diplogin" ) ) )
  {
    opt_i = 1;
  }


  /*
   *  Kommandozeilen-Parameter analysieren
   */
  opterr = 0;

  while ( ( s = getopt ( argc, argv, "c:df:hikm:p:s:tv" ) )  !=  EOF )
  {
    switch ( s )
    {
      case 'c' : /*
		  *  Setzen der Timeout-Zeit
		  */
		 if ( ( mydip.timeout = atoi ( optarg ) )  <  0 )
		 {
		   usage ( );
		   return -1;
		 }
		 break;

      case 'd' : /*
		  *  Debugging Ausgaben einschalten.
		  *  Enthaelt -verbose
		  */
		 if ( opt_debg )
		   opt_debg = 2;
		 else
		   opt_debg = opt_verb = 1;
		 break;

      case 'f' : /*
		  *  Explizite dip.pid-Dateiangabe zB. fuer kill-Aufruf
		  *  Es darf nur eine Datei angegeben werden, der Pfad liegt
		  *  fest auf /etc/, die Endung lautet immer .pid
		  */
		 (void) strcpy ( mydip.pidfile, PATH_DIP_PID "." );
		 (void) strcat ( mydip.pidfile, optarg		 );
		 (void) strcat ( mydip.pidfile, EXT_DIP_PID	 );
		 break;

      case 'h' : /*
		  *  Ausgabe der Programm-Optionen
		  */
		 usage ( );
		 return -1;

      case 'i' : /*
		  *  Aktivierung des Dial-In-Modus
		  */
		 opt_i = 1;
		 break;

      case 'k' : /*
		  *  Kill des angegebenen Protokoll-Servers
		  */
		 opt_kill = 1;
		 break;

      case 'm' : /*
		  *  MTU setzen
		  */
		 if ( ( mydip.mtu = atoi ( optarg ) )  <=  2 )
		 {
		   usage ( );
		   return -1;
		 }
		 break;

      case 'p' : /*
		  *  Protokoll setzen
		  */
		 if ( ( mydip.protonr = get_protocol ( strncpy (
			   mydip.protostr, optarg, MAXPROTSTRLEN-1 ) ) ) == 0 )
		 {
		   usage ( );
		   return -1;
		 }
		 break;

      case 's' : /*
		  *  Shell Skript zur Vervollstaendigung des Protokoll-Setup
		  *  statt Aufruf von ifconfig und route
		  */
		 (void) strncpy ( mydip.shellfile, optarg, 256-1 );
		 break;

      case 't' : /*
		  *  Test-Modus aktivieren
		  */
		 opt_test = 1;
		 break;

      case 'v' : /*
		  *  Verbose einschalten
		  */
		 opt_verb = 1;
		 break;

      default  : usage ( );
		 return -1;
    }
  }


  /*
   *  Wurde der -k Parameter gesetzt? Dann angegebenen Server killen.
   */
  if ( opt_kill )
  {
    return  dip_kill ( );
  }


  /*
   *  Installieren der Signal-Handler
   */
  for ( s = 1;  s < 32;  (void) signal ( s++, SIG_IGN ) ) ;

  (void) signal ( SIGHUP,  pre_catcher1 );
  (void) signal ( SIGINT,  pre_catcher1 );
  (void) signal ( SIGTERM, pre_catcher1 );


  /*
   *  Zweiter Teil der Initialisierung.  Netzwerk-Daten besorgen.
   */
  if ( initialize_dip2 ( ) )
  {
    return -1;
  }


  /*
   *  Wurde -i oder diplogin angegeben?
   */
  if ( opt_i )
  {
    sp  =  ( optind == argc )  ?  mydip.username : argv [ optind ];

    (void) do_login ( sp, &argv[0] );

    /*
     *  Hier kommen wir nur hin, wenn der login abgewiesen wird.
     */
    return -1;
  }


  /*
   *  Wurde -t angegeben? Dann wechseln in den Testmodus.
   *  Eingaben erfolgen hier durch Tastatur (stdin).
   */
  if ( opt_test )
  {
    if ( optind  !=  argc )
    {
      usage ( );
      return -1;
    }

    /*
     *  Ausfuehren der direkt eingegebenen Befehle (stdin).
     *  In  s  den Rueckgabewert nur kurz merken.
     */
    s = do_command ( stdin );

    (void) printf ( "Leaving DIP command mode\n\n" );

    /*
     *  Nur wenn exit eingegeben wird, kommen wir hier wieder hin.
     */
    return s;
  }


  /*
   *  Nun bleibt nur noch der Dial-Out-Modus.
   *  Der letzte (noch nicht geparste Parameter) ist die Befehlsdatei.
   */
  if ( optind  !=  ( argc - 1 ) )
  {
    usage ( );
    return -1;
  }

  /*
   *  Den Namen der Befehlsdatei extrahieren
   */
  (void) strncpy ( cmdfile, argv [ optind ], 256-1 );

  if ( ( sp = strrchr ( cmdfile, '/' ) )  !=  (char*) NULL )
  {
    sp++;
  }
  else
  {
    sp = cmdfile;
  }

  /*
   *  Falls nicht so angegeben, das obligatorische .dip anhaengen.
   */
  if ( strstr ( sp, EXT_DIP_DIP )  ==  (char*) NULL )
  {
    (void) strcat ( cmdfile, EXT_DIP_DIP );
  }

  /*
   *  Die Command-Datei nur zum Lesen oeffnen.
   */
  if ( ( fp = fopen ( cmdfile, "r" ) )  ==  (FILE*) NULL )
  {
    /*
     *  Ohne Command-Datei geht es nicht.
     */
    (void) fprintf ( stderr, "DIP: %s (%s)\n", cmdfile, strerror ( errno ) );

    return -1;
  }

  (void) setbuf ( fp, (char*) NULL );

  /*
   *  Ausfuehren der Befehle direkt aus der Datei.
   */
  s = do_command ( fp );

  (void) fclose ( fp );

  (void) printf ( "Leaving DIP command mode\n\n" );


  return s;
}

