/*
 * dip		A program for handling dialup IP connections.
 *
 *		This module contains the command processor.
 *
 *
 * Version:	@(#)command.c	1.02	1.July.1995
 *
 *
 * New Author:	Joachim Bartz, <injb@sun.rz.tu-clausthal.de>
 *
 *		There have been so many changes to get  dip work under
 *		BSD...   So I believe, you can call me an author, too.
 *		Please read the  ReadMe  file and the man page to know
 *		what, when, why.
 *
 *
 * Org.Author:	Fred N. van Kempen, <waltje@uWalt.NL.Mugnet.ORG>
 *		Copyright 1988-1993 MicroWalt Corporation
 *
 *
 *		This program is free software; you can redistribute it
 *		and/or  modify it under  the terms of  the GNU General
 *		Public  License as  published  by  the  Free  Software
 *		Foundation;  either  version 2 of the License, or  (at
 *		your option) any later version.
 *
 *		This program is  distributed in the  hope that it will
 *		be useful, but WITHOUT ANY WARRANTY;  without even the
 *		implied warranty of MERCHANTABILITY  or  FITNESS FOR A
 *		PARTICULAR  PURPOSE.    See  the  GNU  General  Public
 *		License for more details.
 *
 *		You  should have received  a copy  of the  GNU General
 *		Public License along with this program;  if not, write
 *		to the
 *			Free Software Foundation, Inc.
 *			675 Mass Ave
 *			Cambridge, MA 02139, USA
 */

#include "dip.h"

#include <ctype.h>


struct commands
{
  char	*name;
  int	(*func) ( int, char ** );
};


static  FILE	*scriptfp  = (FILE*) NULL;  /* input script pointer	*/

static  int	errlevel;		/* script command return code	*/
static  short	timeout;		/* script "wait" timed out	*/
static  short	var_echo   = 0;		/* enable line data echoing	*/
static  char	*var_modem = NULL;	/* name of modem we are using	*/
static  char	*var_speed = NULL;	/* terminal line speed		*/
static  char	*var_port  = NULL;	/* terminal line used		*/



/*
 *  Catch any signals.
 */
static  void  pre_catcher2 ( int sig )
{
  /*
   *  Wenn ein Signal kommt, erstmal  tty  schliessen.
   */
  (void) tty_close ( );


  if ( sig  ==  SIGALRM )
  {
    (void) fprintf ( stderr, "\nDIP: Catched ALARM-Signal "
			"(timeout=%d sec). Immediate exit!\n", mydip.timeout );
    exit ( 5 );
  }


  if ( opt_verb )
    (void) fprintf ( stderr, "\nDIP: Catched Signal (%d). "
						"Immediate exit!\n", sig );

  switch ( sig )
  {
    case SIGHUP  : exit ( 2 );
    case SIGINT  : exit ( 3 );
    case SIGTERM : exit ( 4 );
  }

  exit ( 100 + sig );
}



static  void  TimeOut ( int sig )
{
  (void) sig;

  timeout = 1;
}



/*
 *  Convert a C-style backslash sequence to ASCII.
 */
static char  cvt_char ( char c )
{

  if ( c  ==  '\0' )
  {
    return c;
  }


  switch ( c )
  {
    case 'a' : return ( '\a' );
    case 'b' : return ( '\b' );
    case 'f' : return ( '\f' );
    case 'n' : return ( '\n' );
    case 'r' : return ( '\r' );
    case 's' : return ( ' '  );
    case 't' : return ( '\t' );
    case 'v' : return ( '\v' );
    case '\\': return ( '\\' );
    case '\'': return ( '\'' );
    case '"' : return ( '\"' );
    case '?' : return ( '\?' );
/*  default  : return ( '?'  ); */
  }


  /*
   *  Wenn wir hier hin kommen, ist was kaputt...
   */
  return '?';
}



/*
 *  Split the input string into multiple fields.
 */
static int  getargs ( char *string, char *arguments[] )
{

  char  *sp;

  int   argc;
  int   i;


  arguments [ i = 0 ] = sp = string;


  while ( *sp && ( i < 32 ) )
  {
    while ( *sp && ( ( *sp == ' ' ) || ( *sp == '\t' ) ) )
      sp++;

    arguments[ i++ ] = sp;

    while ( *sp && ( *sp != ' ' ) && ( *sp != '\t' ) )
      sp++;

    if ( *sp != '\0' )
      *sp++ = '\0';
  }


  argc = i;


  while ( i < 32 )
    arguments[ i++ ] = (char*) NULL;


  return  argc;
}



/************************************************************************
 *									*
 *		Internal Scripting Commands				*
 *									*
 ************************************************************************/

/*
 *  Enable/Disable echoing of data from terminal line.
 */
static  int  do_echo ( int argc, char *argv[] )
{

  if ( argc  !=  2 )
  {
    (void) fprintf ( stderr, "Usage: echo on|off\n" );
    return -1;
  }
  

  if ( strcmp ( argv[1], "on" )  ==  0 )
  {
    var_echo = 1;
  }
  else
  {
    if ( strcmp ( argv[1], "off" )  ==  0 )
      var_echo = 0;
    else
    {
      (void) fprintf ( stderr, "Usage: echo on|off\n" );
      return -1;
    }
  }
	

  if ( opt_verb )
  {
    (void) printf ( "Display modem output: %sabled\n",
					( var_echo != 1 ) ? "dis" : "en" );
  }


  return 0;
}
 


/*
 *  Go to some label in the script.
 */
static  int  do_goto ( int argc, char *argv[] )
{

  char   buff [ 1024 ];

  off_t	 oldpos;

  char	 *label,
	 *xp,
	 *sp;


  if ( argc  !=  2 )
  {
    (void) fprintf ( stderr, "Usage: goto label\n" );
    return -1;
  }


  if ( scriptfp  ==  stdin )
  {
    (void) fprintf ( stderr, "DIP: GOTO not possible in TEST mode!\n" );
    return -1;
  }


  label  = argv[1];


  oldpos = ftell ( scriptfp );

  (void) rewind ( scriptfp );

  (void) fflush ( scriptfp );


  do
  {
    if ( fgets ( buff, 1024, scriptfp ) == (char*) NULL )
      break;

    if ( ( sp = strchr ( buff, '\n') ) != (char*) NULL )
      *sp = '\0';

    sp = buff;
    while ( ( *sp == ' ' ) || ( *sp == '\t' ) )
      sp++;

    if ( ( *sp == '#' ) || ( *sp == '\0' ) )
      continue;

    if ( ( xp = strchr ( sp, ':' ) )  ==  (char*) NULL )
      continue;

    *xp = '\0';

    if ( ! strcmp ( label, sp ) )
    {
      oldpos = ftell ( scriptfp );
      (void) fseek ( scriptfp, oldpos, SEEK_SET );
      return 0;
    }
  } while ( 1 );


  (void) fseek ( scriptfp, oldpos, SEEK_SET );

  (void) fflush ( scriptfp );


  return -1;
}



/*
 *  Check some error (result) code.
 */
static int  do_if ( int argc, char *argv[] )
{

  char	*cmd [ 3 ];

  long	val;
  long	var;
  int	ret = -1;


  if ( argc  !=  6 )
  {
    (void) fprintf ( stderr, "Usage: if expr goto label\n" );
    return -1;
  }


  if ( strcmp ( argv[4], "goto" ) )
  {
    (void) fprintf ( stderr, "Warning: keyword not \"goto\" !\n" );

    argv[4] = "goto";
  }


  if ( ! strcmp ( argv[1], "$errlvl"  ) )
    var = (long) errlevel;
  else
  if ( ! strcmp ( argv[1], "$counter" ) )
    var = (long) mydip.counter;
  else
  if ( ! strcmp ( argv[1], "$locip"   ) )
    var = (long) mydip.ip_loc.s_addr;
  else
  if ( ! strcmp ( argv[1], "$rmtip"   ) )
    var = (long) mydip.ip_rmt.s_addr;
  else
  {
    (void) fprintf ( stderr, "Invalid variable \"%s\" !\n", argv[1] );

    return -1;
  }


  val = (long) atol ( argv[3] );


  if ( ! strcmp ( argv[2], "==" ) )
    ret = ( var == val );
  else
  if ( ! strcmp ( argv[2], "!=" ) )
    ret = ( var != val );
  else
  if ( ! strcmp ( argv[2], "<"  ) )
    ret = ( var <  val );
  else
  if ( ! strcmp ( argv[2], ">"  ) )
    ret = ( var >  val );
  else
  if ( ! strcmp ( argv[2], "<=" ) )
    ret = ( var <= val );
  else
  if ( ! strcmp ( argv[2], ">=" ) )
    ret = ( var >= val );
  else
  {
    (void) fprintf ( stderr, "Syntax error: \"%s\" is not a relop!\n",
								     argv[2] );
    return -1;
  }


  if ( ret )
  {
    cmd[1] = argv[4];
    cmd[1] = argv[5];
    cmd[2] = (char*) NULL;

    return do_goto ( 2, cmd );
  }


  return errlevel;
}



/*
 *  Increase the counter variable.
 */
static int  do_inc ( int argc, char *argv[] )
{

  int  i = 1;


  if ( ( ( argc < 2 )  ||  ( argc > 3 ) )
    || ( strcmp ( argv[1], "$counter" ) )  )
  {
    (void) fprintf ( stderr, "Usage: inc $counter [value]\n" );

    return -1;
  }
  else
    if ( argc  ==  3 )
      i = atol ( argv [ 2 ] );


  mydip.counter += i;


  return mydip.counter;
}



/*
 *  Decrease the counter variable.
 */
static int  do_dec ( int argc, char *argv[] )
{

  int  i = 1;


  if ( ( ( argc < 2 )  ||  ( argc > 3 ) )
    || ( strcmp ( argv[1], "$counter" ) )  )
  {
    (void) fprintf ( stderr, "Usage: dec $counter [value]\n" );

    return -1;
  }
  else
    if ( argc  ==  3 )
      i = atol ( argv [ 2 ] );


  mydip.counter -= i;


  return mydip.counter;
}



/*
 *  Print the contents of some variable.
 */
static int  do_print ( int argc, char *argv[] )
{

  char  *sp;
  int   i = 0;


  if ( argc  ==  1 )
  {
    (void) printf ( "\n" );

    return 0 ;
  }


  while ( argv[ ++i ] != (char*) NULL )
  {
    sp = argv[i];

    if ( i  !=  1 )
      (void) printf(" ");

    if ( *sp  ==  '$' )
    {
	if ( ! strcmp ( ++sp, "errlvl" ) )
	  (void) printf ( "%d", errlevel );
	else
	if ( ! strcmp ( sp, "counter" ) )
	  (void) printf ( "%d", mydip.counter );
	else
	if ( ! strcmp ( sp, "mtu" ) )
	  (void) printf ( "%d", mydip.mtu );
	else
	if ( ! strcmp ( sp, "locip" ) )
	  (void) printf("%s", inet_ntoa ( mydip.ip_loc ) );
	else
	if ( ! strcmp ( sp, "rmtip" ) )
	  (void) printf("%s", inet_ntoa ( mydip.ip_rmt ) );
	else
	if ( ! strcmp ( sp, "local" ) )
	  (void) printf ( "%s", mydip.local );
	else
	if ( ! strcmp ( sp, "remote" ) )
	  (void) printf ( "%s", mydip.remote );
	else
	if ( ! strcmp ( sp, "modem" ) )
	  (void) printf ( "%s", var_modem );
	else
	if ( ! strcmp ( sp, "port" ) )
	  (void) printf ( "%s", var_port );
	else
	if ( ! strcmp ( sp, "speed" ) )
	  (void) printf ( "%s", var_speed );
	else
	if ( ! strcmp ( sp, "netmask" ) )
	  (void) printf ( "%s", mydip.netmask );
	else
	  (void) fprintf ( stderr, "Unknown variable %s\n", sp );
    }
    else
      (void) printf ( "%s", sp );
  }


  (void) printf ( "\n" );


  return 0;
}



/*
 *  Send a string to the serial driver.
 */
static  int  do_send ( int argc, char *argv[] )
{
  int   i;
  char  *sp;
  char	c;
  
  if ( argc  <  2 )
  {
    (void) fprintf ( stderr, "Usage: send text...\n" );

    return -1;
  }


  for ( i = 1;  i < argc;  i++ )
  {
    sp = argv[i];

    while ( *sp  !=  '\0' )
    {
      switch ( *sp )
      {
	case '~' : tty_putc ( '\r' );
		   tty_putc ( '\n' );
		   break;

	case '\\': sp++;
		   c = cvt_char ( *sp );
		   tty_putc ( (int) c );
		   break;

	default  : c = *sp;
		   tty_putc ( (int) c );
      }

      sp++;
    }

    if ( i  <  (argc - 1) )
      tty_putc ( ' ' );
  }


  /*
   *  Flush the connection
   */
  tty_putc ( -1 );


  return 0;
}



/*
 *  Wait some time.
 */
static  int  do_sleep ( int argc, char *argv[] )
{

  if ( argc  !=  2 )
  {
    (void) fprintf ( stderr, "Usage: sleep time_in_secs\n" );

    return -1;
  }


  /*
   * save_sleep() in save_sleep.c
   */
  (void) save_sleep ( atoi ( argv[1] ) );


  return 0;
}



/*
 *  Flush tty input
 */
static  int  do_flush ( int argc, char *argv[] )
{
  if ( argc  !=  1 )
  {
    (void) fprintf ( stderr, "Usage: flush\n" );

    return -1;
  }


  tty_flush ( );


  return 0;
}



/*
 *  Wait for some string to arrive.
 */
static  int  do_wait ( int argc, char *argv[] )
{

  void	(*oldsig) (int) = SIG_IGN;

  int	i;
  int   leave;
  int 	howlong;
  int	oldalarm;

  char	*p [ 10 ];
  char	c;
  char	c2;


  if ( ( argc < 3 ) || ( argc > 12 ) )
  {
    (void) fprintf ( stderr, "Usage: wait timeout text1 "
						"[text2 ... [text10]]\n" );

    return -1;
  }


  for ( i = 0; i < 10; i++ )
  {
    p[i] = ( i <= argc-3 )  ?  argv[i+2] : NULL;
  }


  leave = timeout = howlong = oldalarm = 0;

  if ( ! mydip.timeout )
  {
    /*
     *  argv[1] <= -1 : Wait forever (until one of the strings arrives)
     *          ==  0 : Test the buffer for the string and then return.
     *          >=  1 : Normal function.
     */
    if ( ( howlong = atoi ( argv[1] ) )  >=  0 )
      oldalarm = alarm ( howlong );
    else    
      oldalarm = alarm ( 0 );

    oldsig = signal ( SIGALRM, TimeOut );
  }


  while ( ( ! timeout )  &&  ( ! leave ) )
  {
    if ( ( i = tty_getc ( howlong!=0 ) )  ==  -1 )	/* chars in buffer? */
    {
      if ( howlong  ==  0 )
      {
	timeout = 2;
	break;
      }
      continue;
    }

    c = ( (char) i ) & 0x7F;

    if ( var_echo )
    {
      (void) fputc ( c, stdout );
      (void) fflush ( stdout );
    }

    for ( i = 0; i < 10; i++ )
    {
      if ( p[i] != NULL )
      {
	if ( *p[i] == '\\' )
	  c2 = cvt_char ( *++p[i] );
	else
	  c2 = *p[i];

	if ( c2 != c )
	  p[i] = argv[i+2];
	else
	  p[i]++;

	if ( *p[i] == '\0' )
	{
	  leave = i+1;
	  break;
	}
      }
    } /* for */
  } /* while */


  if ( ! mydip.timeout )
  {
    (void) alarm ( oldalarm );
    (void) signal ( SIGALRM, oldsig );
  }

  /*
   *  Return codes: -2 : Buffer is empty, no specified string reveived
   *		    -1 : Timeout was given, but no spec. string came in
   *		   > 1 : String # recognized
   */
  return  timeout  ?  -timeout  :  leave ;
}



/*
 * Show some help.
 */
static  int  do_help ( int argc, char *argv[] )
{

  extern  struct commands  commands [ ];

  int	i = 0;
  int	j = 0;


  (void) printf ( "DIP knows about the following commands:\n\n" );


  while ( commands[i].name != (char*) NULL)
  {
    if ( j++  ==  0 )
      (void) printf ( "\t" );

    (void) printf ( "%-8.8s ", commands[i].name );

    if ( j  ==  5 )
    {
      (void) printf ( "\n" );

      j = 0;
    }

    i++;
  }


  if ( j )
    (void) printf ( "\n\n" );
  else
    (void)printf ( "\n" );


  return 0;
}



/************************************************************************
 *									*
 *		Modem Handling and Dialing Commands			*
 *									*
 ************************************************************************/


/*
 *  Set the name of the terminal port to use.
 */
static  int  do_port ( int argc, char *argv[] )
{

  int	i;


  if ( argc  !=  2 )
  {
    (void) fprintf ( stderr, "Usage: port tty_name\n" );
    return -1;
  }


  if ( var_port  !=  NULL )
  {
    (void) fprintf ( stderr, "PORT: terminal port already set to \"%s\".\n",
								var_port );
    return -1;
  }


  /*
   *  Initialize the terminal line.
   */
  if ( tty_open ( argv[1] )  <  0 )
  {
    var_port = NULL;

    return -1;
  }


  var_port = (char*) malloc ( strlen ( argv[1] ) + 1 );

  (void) strcpy ( var_port, argv[1] );


  if ( opt_verb )
  {
    (void) printf ( "PORT: terminal port set to \"%s\".\n", var_port );
  }


  for ( i = 1; i < 32; (void) signal ( i++, SIG_IGN ) );

  (void) signal ( SIGHUP,  pre_catcher2 );
  (void) signal ( SIGINT,  pre_catcher2 );
  (void) signal ( SIGTERM, pre_catcher2 ); 

  if ( mydip.timeout )
    (void) signal ( SIGALRM, pre_catcher2 );

  (void) alarm ( mydip.timeout );


  if ( opt_verb )
  {
    (void) printf ( "PORT: signal processing, timeout set to %d.\n",
							mydip.timeout );
  }


  return 0;
}



/*
 *  Setzen DATABITS.
 */
static  int  do_databits ( int argc, char *argv[] )
{

  if ( argc != 2 )
  {
    (void) fprintf ( stderr, "Usage: databits bits\n" );

    return -1;
  }


  /*
   *  Ist schon ein Port eingestellt worden?
   */
  if ( var_port  ==  NULL )
  { 
    (void) fprintf ( stderr, "Please set PORT first.\n" );

    return -1;
  }  


  return tty_databits ( argv[1] );
}



/*
 *  Define a modem "INIT" string.
 */
static  int  do_init ( int argc, char *argv[] )
{

  if ( argc  !=  2 )
  {
    (void) fprintf ( stderr, "Usage: init init_string\n" );

    return -1;
  }


  return mdm_init ( argv[1] );
}



/*
 *  Befehl: Waehlen.
 */
static  int  do_dial ( int argc, char *argv[] )
{

  if ( argc  !=  2 )
  {
    (void) fprintf ( stderr, "Usage: dial telno\n" );

    return -1;
  }


  /*
   *  Ist schon ein Modem-Typ eingestellt worden?
   */
  if ( var_modem  ==  NULL )
  {
    fprintf ( stderr, "Please set MODEM first.\n" );

    return -1;
  }


  /*
   *  Schon Port eingestellt?
   */
  if ( var_port  ==  NULL )
  {
    (void) fprintf ( stderr, "Please set PORT first.\n" );

    return -1;
  }


  return mdm_dial ( argv[1] );
}



/*
 *  Welches Modem haetten's denn gern?
 */
static  int  do_modem ( int argc, char *argv[] )
{

  if ( argc  !=  2 )
  {
    (void) fprintf ( stderr, "Usage: modem modem_name\n" );

    return -1;
  }


  /*
   *  Ist schon ein Modem-Typ eingestellt worden?
   */
  if ( var_modem  !=  NULL )
  {
    (void) fprintf ( stderr, "MODEM: modem already set to \"%s\".\n",
								var_modem );
    return -1;
  }


  /*
   *  Initialize this modem.
   */
  if ( ( var_modem = mdm_modem ( argv[1] ) )  ==  NULL )
  {
    return -1;
  }


  /*
   *  Zur Sicherheit ausgeben, was nun wirklich eingestellt ist.
   */
  if ( opt_verb )
  {
    (void) printf ( "MODEM: modem set to \"%s\".\n", var_modem );
  }


  return 0;
}



/*
 *  Modem Reset.
 */
static  int  do_reset ( int argc, char *argv[] )
{

  if ( argc  !=  1 )
  {
    (void) fprintf ( stderr, "Usage: reset\n" );

    return -1;
  }


  /*
   *  Schon Modem-Typ eingestellt?
   */
  if ( var_modem  ==  NULL )
  {
    (void) fprintf ( stderr, "Please set MODEM first.\n" );

    return -1;
  }


  /*
   *  Schon Port eingestellt?
   */
  if ( var_port  ==  NULL )
  {
    (void) fprintf ( stderr, "Please set PORT first.\n" );

    return -1;
  }


  /*
   *  Modem Reset.
   */
  return  mdm_reset ( );
}



/*
 *  Setzen PARITY
 */
static  int  do_parity ( int argc, char *argv[] )
{

  if ( argc  !=  2 )
  {
    (void) fprintf ( stderr, "Usage: parity E/O/N\n" );

    return -1;
  }


  /*
   *  Schon Port eingestellt?
   */
  if ( var_port  ==  NULL )
  {
    (void) fprintf ( stderr, "Please set PORT first.\n" );

    return -1;
  }


  return tty_parity ( argv[1] );
}



/*
 *  Setzen der SPEED.
 */
static  int  do_speed ( int argc, char *argv[] )
{

  if ( argc  !=  2 )
  {
    (void) fprintf ( stderr, "Usage: speed baudrate\n" );

    return -1;
  }


  /*
   *  Schon Port eingestellt?
   */
  if ( var_port  ==  NULL )
  {
    (void) fprintf ( stderr, "Please set PORT first.\n" );

    return -1;
  }


  return tty_speed ( argv[1] );
}



/*
 *  Setzen der STOPBITS.
 */
static  int  do_stopbits ( int argc, char *argv[] )
{

  if ( argc  !=  2 )
  {
    (void) fprintf ( stderr, "Usage: stopbits bits\n" );

    return -1;
  }


  /*
   *  Schon Port eingestellt?
   */
  if ( var_port  ==  NULL )
  {
    (void) fprintf ( stderr, "Please set PORT first.\n" );

    return -1;
  }


  return tty_stopbits ( argv[1] );
} 



/*
 *  TERMINAL Modus einschalten.
 */
static  int  do_term ( int argc, char *argv[] )
{

  if ( argc  !=  1 )
  {
    (void) fprintf ( stderr, "Usage: term\n" );

    return -1;
  }


  /*
   *  Schon Port eingestellt?
   */
  if ( var_port  ==  NULL )
  {
    (void) fprintf ( stderr, "Please set PORT first.\n" );

    return -1;
  }


  /*
   *  Terminal-Modus. Direkt-Eingaben an das Modem moeglich.
   */
  do_terminal ( );


  return 0;
}



/************************************************************************
 *									*
 *		Connection Setup Commands				*
 *									*
 ************************************************************************/

/*
 *  Get or ask for the value of a variable.
 *  Get local IP#, Remote IP#, Netmask and MTU
 */

#define VAR_IP		1
#define VAR_NUM		2
#define VAR_STR		3

static  int  do_get ( int argc, char *argv[] )
{

  char		  str [ MAXHOSTNAMELEN ];
  struct hostent  *hp;
  int		  var = 0;
  void		  *v1 = NULL;
  void		  *v2 = NULL;
  void	(*oldsig) ( int ) = 0;


  str[0] = '\0';
   

  if ( ( argc  <=  2 ) || ( argc  >  4 ) || ( argv[1][0]  !=  '$' ) )
  {
    (void) fprintf ( stderr,
		"Usage: get $variable [value|ask|remote [timeout_value]]\n" );
    return -1;
  }

   
  if ( ( !strcmp ( argv[1], "$locip" ) )
    || ( !strcmp ( argv[1], "$local" ) ) )
  {
    v1  = (void*) mydip.local;
    v2  = (void*) &mydip.ip_loc;
    var = VAR_IP;
  }
  else
  {
    if ( ( !strcmp ( argv[1], "$rmtip"  ) )
      || ( !strcmp ( argv[1], "$remote" ) ) )
    {
      v1  = (void*) mydip.remote;
      v2  = (void*) &mydip.ip_rmt;
      var = VAR_IP;
    }
    else
    {
      if ( !strcmp ( argv[1], "$netmask" ) )
      {
	v1  = (void*) &mydip.netmask;
	var = VAR_STR;
      }
      else
      {
	if ( !strcmp ( argv[1], "$mtu" ) )
	{
	  v1  = (void*) &mydip.mtu;
	  var = VAR_NUM;
	}
	else
	{
	  if ( !strcmp ( argv[1], "$counter" ) )
	  {
	    v1  = (void*) &mydip.counter;
	    var = VAR_NUM;
	  }
	  else
	  {
	    (void) fprintf ( stderr, "Variables must be: $locip|$rmtip|"
				   "$local|$remote|$netmask|$mtu|$counter\n" );
	    return -1;
	  }
	}
      }
    }
  }
 

  if ( argc > 2 )
  {
    if ( ! strcmp ( argv[2], "ask" ) )
    {
      /*
       * Ask the stdinput for the value of the variable.
       */
      register char  *sp;

      (void) fprintf ( stderr, "Enter the value for %s: ", argv[1] );

      if ( fgets ( str, MAXHOSTNAMELEN-1, stdin )  ==  (char*) NULL)
	return 1;

      if ( ( sp = strchr ( str, '\n' ) )  !=  (char*) NULL)
	*sp = '\0';
    }
    else
      if ( ! strcmp ( argv[2], "remote" ) )
      {
	/*
	 *  Get the variable string from the "remote" line
	 */
	char	c,
		*p;

	int  state;

	int  howlong = 0;

	if ( ! mydip.timeout )
	{

	  if ( argc  ==  4 )
	    howlong = atoi ( argv[3] );

	  /*
	   *  argv[3] <= -1 : Wait forever (until one of the strings arrives)
	   *          ==  0 : Test the buffer for the string and then return.
	   *          >=  1 : Normal function.
	   */
	  oldsig = signal ( SIGALRM, TimeOut );

	  if ( howlong  >  0 )
	    (void) alarm ( howlong );
	  else    
	    (void) alarm ( 0 );
	}

	p = str;

	timeout = 0;

	if ( ( var == VAR_IP )  ||  ( var == VAR_NUM ) )
	  state = 1;
	else
	  state = 0;

	while ( ( ! timeout )  &&  ( state >= 0 ) )
	{
	  int i;

	  if ( ( i = tty_getc ( howlong!=0 ) )  ==  -1 ) /* chars in buffer? */
	  {
	    if ( howlong  ==  0 )
	    {
	      timeout = 2;
	      break;
	    }
	    continue;
	  }

	  c = ( (char) i ) & 0x7F;

	  if ( timeout )
	    break;

	  switch ( state )
	  {
	    case  0: /* throw away "white space" */

		     if ( isspace ( c ) )
			break;

		     if ( var == VAR_STR )
		     {
			*(p++) = c;
			state = 10;
			break;
		     }
		     state = 1;

	    case  1: /* Read first part xxx. of an IP # or name */

		     if ( isdigit ( c ) )
		     {
			*(p++) = c;
			if ( var == VAR_IP )
			  state = 2;
			else
			  state = 5;
		     }
		     break; 

	    case  2: /* Read 2. part of an IP */
	    case  3: /* Read 3. part of an IP */
	    case  4: /* Read 4. part of an IP */

		     if ( isdigit ( c )  ||  ( c == '.' ) )
		     {
			*(p++) = c;

			if (c == '.')
			  state++;

			break;
		     } 
		     p = str;
		     state = 0;
		     break;

	    case  5: /* Read last part of an IP */

		     if ( isdigit ( c ) )
			*(p++) = c;
		     else
			state = -1;
		     break;

	    case 10: /* stop at trailing white spaces */

		     if ( isspace ( c ) )
			state = -1;
		     else
			*(p++) = c;

	    default: break;
	  } 
	}

	*p = '\0';

	if ( ! mydip.timeout )
	{
	  (void) alarm ( 0 );
	  (void) signal ( SIGALRM, oldsig );
	}

	if ( timeout )
	  return  -timeout;
      }
      else
      {
	/*
	 *  The third argv is the value for the variable.
	 */
	(void) strncpy ( str, argv[2], MAXHOSTNAMELEN-1 );
      }
  }


  if ( opt_verb )
  {
    (void) printf ( "About to set variable %s to %s\n", argv[1], str );
  }


  if ( *str )
  {
    switch ( var )
    {
      case VAR_IP : if ( ( hp=gethostbyname(str) ) == (struct hostent*) NULL)
		    {
		      herror ( str );
		      return -1;
		    }

		    (void) strncpy ( (char*) v1, hp->h_name,
							MAXHOSTNAMELEN-1 );
		    (void) memcpy ( (char*) v2, (char*) hp->h_addr_list[0],
								hp->h_length );
		    break;

      case VAR_NUM: *(int *) v1 = atoi ( str );
		    break;

      case VAR_STR: (void) strcpy ( (char*) v1, str );
    }
  }

  return 0;
}



/*
 *  Enter a specific protocol.
 */
static int  do_mode ( int argc, char *argv[] )
{

  int  i;


  if ( ( argc  !=  2 ) || ( ( i = get_protocol ( argv[1] ) )  ==  0 ) )
  {
    (void) fprintf ( stderr, "Usage: mode protocol_name\n" );

    return -1;
  }


  if ( mydip.ip_rmt.s_addr  ==  INADDR_ANY )
  {
    (void) fprintf ( stderr, "Please set REMOTE first.\n" );

    return -1;
  }


  mydip.protonr = i;

  (void) strncpy ( mydip.protostr, argv[1], MAXPROTSTRLEN-1 );


  /*
   *  Enter background mode here!
   *
   *  In "normal" circumstances we will never reach the end of
   *  dip_daemon(), because the daemon is killed and has no chance
   *  to terminate itself.
   *
   *  If a -1 is returned, there was an error creating the pid-file.
   */
  if ( dip_daemon ( )  ==  -1 )
    return -1;

  /*
   *  Otherwise: A 0 (zero) is returned, meaning "HUPCL" received.
   *  Via errorlevel we give the dip script the chance to do all the
   *  necassary actions to redial the remote system.
   */
  for ( i = 1; i < 32; (void) signal ( i++, SIG_IGN ) );

  (void) signal ( SIGHUP,  pre_catcher2 );
  (void) signal ( SIGINT,  pre_catcher2 );
  (void) signal ( SIGTERM, pre_catcher2 ); 

  if ( mydip.timeout )
    (void) signal ( SIGALRM, pre_catcher2 );

  return 1;
}



/*
 *  Set routing as default
 */
static int  do_default ( int argc, char *argv[] )
{

  if ( argc  !=  1 )
  {
    (void) fprintf ( stderr, "Usage: default\n" );

    return -1;
  }


  if ( opt_verb )
  {
    (void) printf ( "Destination net/address set to 'default'\n" );
  }


  mydip.rtdefault = 1;


  return 0;
}



struct commands commands[] =
{
  { "databits",         do_databits     },
  { "dec",		do_dec		},
  { "default",		do_default	},
  { "dial",		do_dial		},
  { "echo",		do_echo		},
  { "exit",		NULL		},
  { "flush",		do_flush	},
  { "get",		do_get   	},
  { "goto",		do_goto		},
  { "help",		do_help		},
  { "if",		do_if		},
  { "inc",		do_inc		},
  { "init",		do_init		},
  { "mode",		do_mode		},
  { "modem",		do_modem	},
  { "parity",           do_parity       },
  { "print",		do_print	},
  { "port",		do_port		},
  { "reset",		do_reset	},
  { "send",		do_send		},
  { "sleep",		do_sleep	},
  { "speed",		do_speed	},
  { "stopbits",         do_stopbits     },
  { "term",		do_term		},
  { "wait",		do_wait		},
  { (char*) NULL,	NULL		}
};



/*
 *  Analyse der eingegebenen Befehle. Entweder aus der angegbenen Datei
 *  oder von stdin.
 */
int  do_command ( FILE *fp )
{

  char	cline [ 1024 ];	/* Eingabezeile					  */

  char	*argv [ 32 ];	/* Zeiger auf die einzelnen Parameter der Eingabe */
  int	argc;		/* Anzahl der tatsaechlichen Parameter der Eing.  */

  int	i;
  char  *sp;


  /*
   *  Initialize the command level module.
   */
  var_speed = DEF_SPEED;
  scriptfp  = fp;
  timeout   = 0;
  errlevel  = 0;


  /*
   *  - Einstellen, welcher Modem-Typ gemeint ist. Momentan ist nur HAYES
   *    bekannt und deshalb aus fest im Programm einkodiert.
   *  - Ein anderes Problem ist: Wie soll sichergestellt werden, dass immer
   *    MODEM xxx als erster Befehl gegeben wird?
   */
  var_modem = mdm_modem ( DEF_MODEM );


  while ( 1 )
  {
    /*
     *  Wenn von Tastatur eingegeben wird, vorher Prompt ausgeben.
     */
    if ( scriptfp  ==  stdin )
    {
      if ( opt_verb )
	(void) printf ( "DIP [%-4d]> ", errlevel );
      else
	(void) printf ( "DIP> " );

      (void) fflush ( stdout );
    }


    /*
     *  Eine Textzeile holen.
     *  Bei Fehlern & EOF zurueckspringen, dann Ende des Programms.
     */
    if ( fgets ( cline, 1024, scriptfp )  ==  (char*) NULL )
    {
      return  feof ( scriptfp )  ?  0 : errno;
    }


    /*
     *  Letztes NewLine entfernen.
     */
    if ( ( sp = strchr ( cline, '\n' ) )  !=  (char*) NULL )
      *sp = '\0';


    /*
     *  Fuehrende White Spaces entfernen.
     */
    sp = cline;
    while ( ( *sp == ' ' ) || ( *sp == '\t' ) )
      sp++;


    /*
     *  Kommentare ueberlesen
     */
    if ( ( *sp == '#' ) || ( *sp == '\0' ) )
      continue;


    /*
     *  Ausgeben, was wirklich in der Eingabezeile steht.
     */
    if ( opt_verb )
      (void) fprintf ( stderr, ">> %s\n", sp );


    /*
     *  Eingabezeile in einzelne Worte zerlegen.
     *  Wenn keine Worte vorhanden sind, Zeile ueberlesen.
     */
    if ( ( argc = getargs ( sp, argv ) )  ==  0 )
      continue;


    /*
     *  Sprungmarken/Label ueberlesen.
     */
    if ( strchr ( argv[0], ':' )  !=  (char*) NULL )
      continue;


    /*
     * So, welches Kommando haben wir denn?
     */
    if ( ! strcmp ( argv[0], "exit" ) )
    {
      /*
       *  Falls noch eine PID-Datei existieren sollte: Weg damit!
       */
      (void) unlink ( mydip.pidfile );

      /*
       *  Exit heisst Ende...
       */
      return  ( argv[1] != NULL )  ?  atoi ( argv[1] ) : 0;
    }

    /*
     *  Zugehoerige Funktion suchen.
     */
    i = 0;
    while ( commands[i].name  !=  (char*) NULL )
    {
      if ( ! strcmp ( commands[i].name, argv[0] ) )
	break;

      i++;
    }

    if ( commands[i].name  !=  (char*) NULL )
      errlevel = (*commands[i].func) ( argc, argv );
    else
      (void) printf ( "? Unknown command (%s).\n", argv[0] );

  } /* while(1) */

}
