/* points.c */

/*
 * Mesa 3-D graphics library
 * Version:  1.2
 * Copyright (C) 1995  Brian Paul  (brianp@ssec.wisc.edu)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


/*
$Id: points.c,v 1.14 1995/06/20 16:20:50 brianp Exp $

$Log: points.c,v $
 * Revision 1.14  1995/06/20  16:20:50  brianp
 * do float-to-int depth scaling here instead of in draw.c
 *
 * Revision 1.13  1995/06/05  20:27:26  brianp
 * better clipping of points with size > 1
 *
 * Revision 1.12  1995/05/22  21:02:41  brianp
 * Release 1.2
 *
 * Revision 1.11  1995/05/12  16:57:22  brianp
 * replaced CC.Mode!=0 with INSIDE_BEGIN_END
 *
 * Revision 1.10  1995/04/18  15:48:23  brianp
 * fixed assignment of NULL to function pointers to prevent warnings on Suns
 *
 * Revision 1.9  1995/04/12  15:36:15  brianp
 * updated to use DD.draw_* function pointers
 *
 * Revision 1.8  1995/03/24  15:33:32  brianp
 * introduced VB
 *
 * Revision 1.7  1995/03/07  14:20:55  brianp
 * updated for new XSetForeground/GC scheme
 *
 * Revision 1.6  1995/03/04  19:29:44  brianp
 * 1.1 beta revision
 *
 * Revision 1.5  1995/03/04  19:16:47  brianp
 * added size clamp
 *
 * Revision 1.4  1995/03/02  19:18:34  brianp
 * new RasterMask logic
 *
 * Revision 1.3  1995/02/27  22:48:59  brianp
 * modified for PB
 *
 * Revision 1.2  1995/02/27  15:08:12  brianp
 * added Vcolor/Vindex scheme
 *
 * Revision 1.1  1995/02/24  14:26:49  brianp
 * Initial revision
 *
 */


#include "context.h"
#include "dd.h"
#include "feedback.h"
#include "list.h"
#include "macros.h"
#include "pb.h"
#include "span.h"
#include "vb.h"




void glPointSize( GLfloat size )
{
   if (CC.CompileFlag) {
      /* TODO: gl_save_pointsize( size ); */
   }
   if (CC.ExecuteFlag) {
      if (size<=0.0) {
	 gl_error( GL_INVALID_VALUE, "glPointSize" );
	 return;
      }
      if (INSIDE_BEGIN_END) {
	 gl_error( GL_INVALID_OPERATION, "glPointSize" );
	 return;
      }

      CC.PointFunc = ( void (*)() ) 0;
      CC.Point.Size = size;

      gl_update_rasterflags();
   }
}



/**********************************************************************/
/*****                    Rasterization                           *****/
/**********************************************************************/


/*
 * There are 3 pairs (RGBA, CI) of point drawing functions:
 *   1. simple:  size=1 and no special rasterization functions (fastest)
 *   2. size1:  size=1 and any rasterization functions
 *   3. general:  any size and rasterization functions (slowest)
 */



/*
 * Put point in feedback buffer.
 */
static void feedback_point( GLuint p )
{
   GLfloat x, y, z, w;
   GLfloat tex[4];  /* texture coord */

   x = VB.Win[p][0];
   y = VB.Win[p][1];
   z = VB.Win[p][2];
   w = VB.Clip[p][3];

   APPEND_TOKEN( (GLfloat) GL_POINT_TOKEN );
   gl_feedback_vertex( x, y, z, w, VB.Color[p], VB.Index[p], tex );
}



/*
 * Put point in selection buffer.
 */
static void select_point( GLuint p )
{
   GLfloat z = VB.Win[p][2];
 
   CC.HitFlag = GL_TRUE;
   if (z < CC.HitMinZ) {
      CC.HitMinZ = z;
   }
   if (z < CC.HitMinZ) {
      CC.HitMaxZ = z;
   }
}



/*
 * Simple CI point.
 */
static void simple_ci_point( GLuint p )
{
   dd_index( (GLuint) VB.Index[p] );
   (*DD.draw_pixel)( (GLint) VB.Win[p][0], (GLint) VB.Win[p][1] );
}


/*
 * Simple RGBA point.
 */
static void simple_rgba_point( GLuint p )
{
   dd_color( VB.Color[p] );
   (*DD.draw_pixel)( (GLint) VB.Win[p][0], (GLint) VB.Win[p][1] );
}


/*
 * CI point with size == 1.0
 */
static void size1_ci_point( GLuint p )
{
   GLint x, y, z;

   x = (GLint) VB.Win[p][0];
   y = (GLint) VB.Win[p][1];
   z = (GLint) (VB.Win[p][2] * DEPTH_SCALE);

   PB_WRITE_CI_PIXEL( x, y, z, (GLuint) VB.Index[p] );
   PB_CHECK_FLUSH
}


/*
 * RGBA point with size == 1.0
 */
static void size1_rgba_point( GLuint p )
{
   GLint x, y, z;
   GLint red, green, blue, alpha;

   x = (GLint) VB.Win[p][0];
   y = (GLint) VB.Win[p][1];
   z = (GLint) (VB.Win[p][2] * DEPTH_SCALE);

   red   = (GLint) (VB.Color[p][0] * CC.RedScale);
   green = (GLint) (VB.Color[p][1] * CC.GreenScale);
   blue  = (GLint) (VB.Color[p][2] * CC.BlueScale);
   alpha = (GLint) (VB.Color[p][3] * CC.AlphaScale);

   PB_WRITE_RGBA_PIXEL( x, y, z, red, green, blue, alpha );
   PB_CHECK_FLUSH
}



/*
 * General CI point.
 */
static void general_ci_point( GLuint p )
{
   GLint x, y, z;
   GLint x0, x1, y0, y1;
   GLint ix, iy;
   GLint isize;

   x = (GLint) VB.Win[p][0];
   y = (GLint) VB.Win[p][1];
   z = (GLint) (VB.Win[p][2] * DEPTH_SCALE);

   isize = (GLint) (CLAMP(CC.Point.Size,MIN_POINT_SIZE,MAX_POINT_SIZE) + 0.5F);
   if (isize<1) {
      isize = 1;
   }

   if (isize&1) {
      /* odd size */
      x0 = x - isize/2;
      x1 = x + isize/2;
      y0 = y - isize/2;
      y1 = y + isize/2;
   }
   else {
      /* even size */
      x0 = (GLint) (x + 0.5F) - isize/2;
      x1 = x0 + isize-1;
      y0 = (GLint) (y + 0.5F) - isize/2;
      y1 = y0 + isize-1;
   }

   PB_SET_INDEX( (GLuint) VB.Index[p] );

   for (iy=y0;iy<=y1;iy++) {
      for (ix=x0;ix<=x1;ix++) {
	 PB_WRITE_PIXEL( ix, iy, z );
      }
   }
   PB_CHECK_FLUSH
}


/*
 * General RGBA point.
 */
static void general_rgba_point( GLuint p )
{
   GLint x, y, z;
   GLint x0, x1, y0, y1;
   GLint ix, iy;
   GLint isize;

   x = (GLint) VB.Win[p][0];
   y = (GLint) VB.Win[p][1];
   z = (GLint) (VB.Win[p][2] * DEPTH_SCALE);

   isize = (GLint) (CLAMP(CC.Point.Size,MIN_POINT_SIZE,MAX_POINT_SIZE) + 0.5F);
   if (isize<1) {
      isize = 1;
   }

   if (isize&1) {
      /* odd size */
      x0 = x - isize/2;
      x1 = x + isize/2;
      y0 = y - isize/2;
      y1 = y + isize/2;
   }
   else {
      /* even size */
      x0 = (GLint) (x + 0.5F) - isize/2;
      x1 = x0 + isize-1;
      y0 = (GLint) (y + 0.5F) - isize/2;
      y1 = y0 + isize-1;
   }

   PB_SET_COLOR( VB.Color[p] );

   for (iy=y0;iy<=y1;iy++) {
      for (ix=x0;ix<=x1;ix++) {
	 PB_WRITE_PIXEL( ix, iy, z );
      }
   }
   PB_CHECK_FLUSH
}




/*
 * Examine the current context to determine which point drawing function
 * should be used.
 */
void gl_set_point_function( void )
{
   /* TODO: texture mapping */
   /* TODO: antialiased points */

   if (CC.RenderMode==GL_RENDER) {
      if (CC.Point.Size==1.0) {
	 if (CC.RasterMask & RASTER_BITS) {
	    /* size=1, any raster ops */
	    CC.PointFunc = CC.RGBAflag ? size1_rgba_point : size1_ci_point;
	 }
	 else {
	    /* size=1, no raster ops */
	    CC.PointFunc = CC.RGBAflag ? simple_rgba_point : simple_ci_point;
	 }
      }
      else {
	 /* every other kind of point rendering */
	 CC.PointFunc = CC.RGBAflag ? general_rgba_point : general_ci_point;
      }
   }
   else if (CC.RenderMode==GL_FEEDBACK) {
      CC.PointFunc = feedback_point;
   }
   else {
      /* GL_SELECT mode */
      CC.PointFunc = select_point;
   }

}


