
/*  @(#)optimise.c 1.3 93/06/01
 *
 *  popi parse tree machine independant optimizations.
 *
 *  1991 Jeremy Fitzhardinge jeremy@softway.sw.oz.au
 *
 *  Popi was originally written by Gerard J. Holzmann - AT&T Bell Labs.
 *  This version is based on the code in his Prentice Hall book,
 *  "Beyond Photography - the digital darkroom," ISBN 0-13-074410-7,
 *  which is copyright (c) 1988 by Bell Telephone Laboratories, Inc.
 *
 *  Permission is given to distribute these extensions, as long as these
 *  introductory messages are not removed, and no monies are exchanged.
 *
 *  No responsibility is taken for any errors or inaccuracies inherent
 *  either to the comments or the code of this program, but if reported
 *  (see README file) then an attempt will be made to fix them.
 */

#include <math.h>
#include "popi.h"
#include "expr.h"

static int cexp         P((Tree *)) ;
static int depend       P((Tree *)) ;
static int depends      P((Tree *)) ;
static int rmsub        P((Tree *)) ;

static void depvar      P((Tree *)) ;
static void miscopt     P((Tree *)) ;
static void optvar      P((Tree *)) ;

static Tree *addcare    P((Tree *)) ;
static Tree *copytree   P((Tree *)) ;
static Tree *docse      P((Tree *)) ;
static Tree *mvloopinv  P((Tree *, Tree **)) ;
static Tree *narnop     P((Tree *, enum tree)) ;
static Tree *widenbinop P((Tree *, enum tree)) ;
static Tree *stredmult  P((Tree *, Tree **, Tree **, int)) ;

#define  depY       1
#define  depX       2
#define  depV       (depX | depY)
#define  depCplx    4
#define  depNoMv    8

/*  Entry point:  This module takes a tree and applies various optimizations
 *  to it, returning another expression tree that the code generator can
 *  generate faster-executing code for (one hopes).
 *
 *  The main optimizations performed are:
 *      Constant folding
 *         expressions that evaluate to a constant value are replaced
 *         with that constant, rather than being recalculated.
 *      Loop invariant moving
 *         expressions that will not change value during the execution
 *         of the inner loop are moved out of the loop, and are
 *         replaced with a variable.
 *      Constant Subexpression Elimination
 *         An expression used in more than one place is calculated once and
 *         has it's value propagated about the expression.
 *      Strength reduction
 *         Multiplies which have x+C or y+C as an argument (where C is a
 *         constant) are replaced with adds at the end of the loop where
 *         they are used, eliminating the need for a multiply in these
 *         cases.  Mod and Div could be treated similarly.
 *      Miscellaneous optimizations
 *         A+-B is replaced with A-B (mainly because this form is
 *                 deliberately introduced during optimization)
 *         *, -, % are strength reduced if one of the
 *                 arguments is a constant of the form 2^C.
 *
 *      A "Top" node is added which has 5 children.  This is a special case
 *      of the basic blocks used in the generated code.  These 5 trees are
 *      inserted into the code:
 *           Before the outer Y loop
 *           Before the inner X loop
 *           In the inner X loop
 *           Just before the end of the X loop
 *           Just before the end of the Y loop
 */

Tree *
opttree(t)
Tree *t ;
{
  Tree *nt ;

  dtree("Before:\n", t, 0) ;

/* Do it here, before tree gets complex */

  if (careful) t = addcare(t) ;

/* If there are any user-specified variables, work out dependencies */

  depvar(t) ;

/* Turn x-y into x+ -y ;  this makes things easier later */

  rmsub(t) ;

/* Iterate over expression tree, removing all constant expressions. */

  do
    {
      t = widenbinop(t, T_Mul) ;
      t = widenbinop(t, T_Add) ;
      t = widenbinop(t, T_And) ;
      t = widenbinop(t, T_Or) ;
      t = widenbinop(t, T_Xor) ;
    }
  while (cexp(t)) ;

/* Convert n-ary operators back into binary ones, so cgen can cope. */

  t = narnop(t, T_Mul) ;
  t = narnop(t, T_Add) ;
  t = narnop(t, T_And) ;
  t = narnop(t, T_Or) ;
  t = narnop(t, T_Xor) ;

/* Convert easy /%* before complex strength reduction */

  miscopt(t) ;    

/* CSE pass */

  t = docse(t) ;

/* Move expressions around; move loop-invariants outside the loop, */

  nt = mkempty(T_Nop) ;
  t = mkTop(nt, nt, t, nt, nt) ;
  t->kids[2] = mvloopinv(t->kids[2], &t->kids[1]) ;

/* Strength reduction on multiplies - get rid of multiples by x or y */

  t->kids[2] = stredmult(t->kids[2], &t->kids[1], &t->kids[3], depX) ;
  t->kids[1] = stredmult(t->kids[1], &t->kids[0], &t->kids[4], depY) ;

/* This is tidyup, after all the gunge put in above */

  miscopt(t) ;

/* Optimize variables - just remove unnecessary intermediaries */

  optvar(t) ;
  dtree("After\n", t, 0) ;
  return(t) ;
}


/* Replace a node to another */

static Tree *
replnode(t, from, to, inf)
Tree *t, *to ;
int (*from)P((Tree *, int)) ;
int inf ;
{
  int i ;

  for (i = 0; i < t->siz; i++)
    t->kids[i] = replnode(t->kids[i], from, to, inf) ;

  if ((*from)(t, inf))
    {
      if (t->t == T_Var)  devalVar(t->i) ;
      if (to->t == T_Var) valVar(to->i) ;
      return(to) ;
    }
  return(t) ;
}


/* Return true if var is changed by an expression */

static int
varchng(t, var)
Tree *t ;
int var ;
{
  int i ;

  if (t->t == T_Assign && t->i == var) return(1) ;

  for (i = 0; i < t->siz; i++)
    if (varchng(t->kids[i], var)) return(1) ;
  return(0) ;
}


static int
findvar(t, inf)
Tree *t ;
int inf ;
{
  return(t->t == T_Var && t->i == inf) ;
}


static int
findtype(t, inf)
Tree *t ;
int inf ;
{
  enum tree type = inf ;

  return(t->t == type) ;
}


static int
countnode(t, match, inf)
Tree *t ;
int (*match)P((Tree *, int)) ;
int inf ;
{
  int i, sum = 0 ;

  for (i = 0; i < t->siz; i++)
    sum += countnode(t->kids[i], match, inf) ;

  if ((*match)(t, inf)) sum++ ;

  return(sum) ;
}


/* Eliminate unused intermediate vars. */

static void
optinter(t, top, fix, notif)
Tree *t, **top, **fix, **notif ;
{
  int i, from ;

  for (i = 0; i < t->siz; i++)
    optinter(t->kids[i], top, fix, notif) ;

  if (t->t == T_Assign && t->kids[0]->t == T_Var && !varchng(*notif, t->i))
    {
      Tree *to = mkempty(T_Var) ;

      t->t = T_Nop ;
      t->siz = 0 ;
      from = t->i ;
      to->i = t->kids[0]->i ;
      *top = replnode(*top, findvar, to, from) ;
      *fix = replnode(*fix, findvar, to, from) ;
      *notif = replnode(*notif, findvar, to, from) ;
    }
}


static Tree *
optunused(t, top)
Tree *t, *top ;
{
  int i ;

  if (t->t == T_Comma) t->kids[0] = optunused(t->kids[0], top) ;

  if (t->t == T_Assign)
    {
      int num = countnode(top, findvar, t->i) ;

      if (num <= 1)
        {
          t->t = T_Nop ;
          t->siz = 0 ;
          replnode(top, findvar, t->kids[0], t->i) ;
        }
    }
  return(t) ;
}


static void
optvar(t)
Tree *t ;
{
#if 0         /* A little too bodgy - this is not best done in a tree */
  optinter(t->kids[1], &t->kids[1], &t->kids[2], &t->kids[3]) ;
  t->kids[2] = optunused(t->kids[2], t->kids[2]) ;
#endif
}

struct chain {
  Tree *node ;
  struct chain *nxt ;
} ;


linkhash(t, inf)
Tree *t ;
int inf ;
{
  struct chain **chain = (struct chain **) inf ;
  struct chain *ch = alloc(sizeof(struct chain)) ;

  if (cost(t) > 1 && !(depends(t) & depNoMv))
    {
      ch->nxt = *chain ;
      *chain = ch ;
      ch->node = t ;
    }
  return(0) ;
}


/* Generate a hash code for each node, depending on itself and its kids. */

#define  HASH(h, v)  ((h << 2) ^ (v))

static int
genhash(t)
Tree *t ;
{
  int hash = 7, i ;

  for (i = 0; i < t->siz; i++)
    hash = HASH(hash, genhash(t->kids[i])) ;

  hash = HASH(hash, t->t) ;
  hash = HASH(hash, t->siz) ;
  switch (t->t)
    {
      case T_Inum   :
      case T_Var    :
      case T_Assign :
      case T_Coord  : hash = HASH(hash, t->i) ;
    }
  t->hash = hash ;
  return(hash) ;
}


/* Compare two trees */

static int
cmptree(t1, t2)
Tree *t1, *t2 ;
{
  int i ;

  if (t1->t != t2->t || t1->siz != t2->siz) return(0) ;

  switch (t1->t)
    {
      case T_Inum   :
      case T_Var    :
      case T_Assign :
      case T_Coord  : if (t1->i != t2->i) return(0) ;
    }

  for (i = 0; i < t1->siz; i++)
    if (!cmptree(t1->kids[i], t2->kids[i]))
      return(0) ;
}


static int
sameexpr(t1, inf)
Tree *t1 ;
int inf ;
{
  Tree *t2 = (Tree *) inf ;
  int i ;

  if (t1->t != t2->t || t1->siz != t2->siz || t1->hash != t2->hash)
    return(0) ;

  switch (t1->t)
    {
      case T_Inum   :
      case T_Var    :
      case T_Assign :
      case T_Coord  : if (t1->i != t2->i) return(0) ;
                       break ;
    }
  for (i = 0; i < t1->siz; i++)
    if (!sameexpr(t1->kids[i], (int)(t2->kids[i])))
      return(0) ;

  return(1) ;
}


static Tree *
replcse(t, cs)
Tree *t, *cs ;
{
  int last, inkids = 0, i ;

  if (cost(cs) <= 2) return(t) ;

  for (i = 0; i < t->siz; i++)
    if (countnode(t->kids[i], sameexpr, (int) cs) != 0)
      {
        last = i ;
        inkids++ ;
      }

  if (inkids == 0) return(t) ;

  if (inkids == 1)
    t->kids[last] = replcse(t->kids[last], cs) ;
  else
    {
      char *var = tmpvar() ;
      Tree *to = mkVar(var) ;
      Tree *nt ;

      setvardep(to->i, depends(t)) ;
      t = mk(T_Comma, mkAssign(var, cs), t) ;
      t->kids[1] = replnode(t->kids[1], sameexpr, to, (int) cs) ;
    }

  return(t) ;
}


static Tree *
docse(t)
Tree *t ;
{
  int i ;
  struct chain *chain = 0, *ch ;

/*  Attach a hash code to each node in the tree, which is dependant on the
 *  node and it's children.
 */

  genhash(t) ;

/*  Use countnode to walk the tree, and link all the nodes into the hash
 *  buckets.
 */

  countnode(t, linkhash, (int) &chain) ;

  ch = chain ;
  while (ch)
    {
      t = replcse(t, ch->node) ;
      ch = ch->nxt ;
    }
  return(t) ;
}


static int
depends(t)
Tree *t ;
{
  int dep = 0, i ;

  for (i = 0; i < t->siz; i++)
    dep |= depends(t->kids[i]) ;

  switch (t->t)
    {
      case T_Ycoord   : dep |= depY ;
                        break ;
      case T_Xcoord   : dep |= depX ;
                        break ;
      case T_Var      : dep = getvardep(t->i) ;
                        break ;
      case T_Rand     : dep |= depNoMv ;
                        break ;
      case T_Pang     :
      case T_Prad     : dep |= depX | depY ;
                        break ;
      case T_Mapcoord : dep |= depNoMv ;
      case T_Coord    : if (t->siz == 0)
                          dep |= depX | depY ;
    }

  switch (t->t)
    {
      case T_Add    :
      case T_Sub    :
      case T_Inum   :
      case T_Var    :
      case T_Xcoord :
      case T_Ycoord :
      case T_Assign :
      case T_Comma  : break ;
      default       : dep |= depCplx ;
    }
  return(dep) ;
}


static void
depvar(t)
Tree *t ;
{
  int i ;

  if (t->t == T_Assign)
    setvardep(t->i, getvardep(t->i) | depends(t->kids[0])) ;

  for (i = 0; i < t->siz; i++)
    depvar(t->kids[i]) ;
}


static void
makeinit(t, val, type)
Tree *t ;
int val ;
enum tree type ;
{
  int i ;

  if (t->t == type)
    {
      t->t = T_Inum ;
      t->i = val ;
      t->siz = 0 ;
    }

  for (i = 0; i < t->siz; i++)
    makeinit(t->kids[i], val, type) ;
}


static Tree *
copytree(t)
Tree *t ;
{
  int i ;
  Tree *nt ;

  nt = treealloc(t->siz) ;
  *nt = *t ;
  
  for (i = 0 ; i < nt->siz ; i++)
    nt->kids[i] = copytree(t->kids[i]) ;

  return(nt) ;
}


/* Get rid of multiplies by x and/or y */

static Tree *
stredmult(t, init, post, dep)
Tree *t, **init, **post ;
int dep ;
{
  int deps[2], i, red, exp ;
  Tree *tp, *vn ;
  char *var ;

  for (i = 0; i < t->siz; i++)
    t->kids[i] = stredmult(t->kids[i], init, post, dep) ;

  if (t->t != T_Mul) return(t) ;

  deps[0] = depends(t->kids[0]) ;
  deps[1] = depends(t->kids[1]) ;

  if (deps[0] & deps[1] & depCplx) return(t) ;
  if ((deps[0] & depV) == dep &&
     !(deps[0] & depCplx) &&
     !(deps[1] & dep))
    {
      red = 0 ;
      exp = 1 ;
    }
  else if ((deps[1] & depV) == dep &&
          !(deps[1] & depCplx) &&
          !(deps[0] & dep))
    {
      red = 1 ; 
      exp = 0 ;
    }
  else return(t) ;      /* Both sides have too complex expressions */
  
/*  Put the initialization in - this is the reductable side
 *  (x or y only) with all the Xcoord or Ycoord nodes replaced with
 *  their initial values.
 */

  var = tmpvar() ;
  *init = mk(T_Comma, *init, vn = mkAssign(var, tp = copytree(t))) ;
  setvardep(vn->i, deps[exp]) ;
  makeinit(tp, dep == depX ? Xsize : Ysize,
               dep == depX ? T_Xcoord : T_Ycoord) ;
  cexp(tp) ;

/*  Now do the post side, where the tmp var is decremented by the
 *  non-reducable argument.
 */

  *post = mk(T_Comma, *post,
             mkAssign(var, mk(T_Sub, mkVar(var), t->kids[exp]))) ;

/* Now replace expression with var. */

  t->t = T_Var ;
  t->i = allocName(var) ;
  t->siz = 0 ;

  return(t) ;
}


/* Move expressions dependant on "y" only out of the X inner loop. */

static Tree *
mvloopinv(t, to)
Tree *t, **to ;
{
  int i ;
  int dep ;
  char *var ;

  dep = depends(t) ;
  if (!(dep & (depX | depNoMv)) && cost(t) > 2)
    {
      Tree *vn ;

      var = tmpvar() ;
      *to = mk(T_Comma, *to, vn = mkAssign(var, t)) ;
      setvardep(vn->i, dep) ;
      t = mkVar(var) ;
    }

  for (i = 0; i < t->siz; i++)
    t->kids[i] = mvloopinv(t->kids[i], to) ;

  return(t) ;
}


/* Remove "Sub" nodes */

static int
rmsub(t)
Tree *t ;
{
  int chng = 0 ;
  int i ;

  if (t->t == T_Sub)
    {
      t->kids[1] = mkun(T_Neg, t->kids[1]) ;
      t->t = T_Add ;
      chng = 1 ;
    }

  for (i = 0; i < t->siz; i++)
    chng |= rmsub(t->kids[i]) ;

  return(chng) ;
}


static int
kidcomp(tp1, tp2)
Tree **tp1, **tp2 ;
{
  int ret = 0 ;
  Tree *t1 = *tp1 ;
  Tree *t2 = *tp2 ;

  if (!t1) return(-1) ;
  if (!t2) return(1) ;

  ret = t1->t - t2->t ;

  if (t1->t == T_Ycoord) ret = 1 ;
  if (t1->t == T_Xcoord) ret = -1 ;
  if (t2->t == T_Ycoord) ret = -1 ;
  if (t2->t == T_Xcoord) ret = 1 ;
  return(ret) ;
}


/* Convert n-ary operators back to binary. */

static Tree *
narnop(t, type)
Tree *t ;
enum tree type ;
{
  int i ;

  if (!t) return(t) ;

/*  Make sure Xcoord and Ycoord terms are as separated as possible
 *  to help later loop invarient code.
 */

  if (t->t == type && t->siz > 1)
    qsort((char *) &t->kids[0], t->siz, sizeof(Tree *), kidcomp) ;
  for (i = 0; i < t->siz; i++)
    t->kids[i] = narnop(t->kids[i], type) ;

  if (t->t == type && t->siz > 2)
    {
      Tree *nn, **tn, *ln ;

      tn = &nn ;
      for (i = 0; i < t->siz - 1; i++)
        {
          *tn = ln = treealloc(2) ;
          ln->t = type ;
          ln->kids[1] = t->kids[i] ;
          tn = &ln->kids[0] ;
        }
      *tn = t->kids[i] ;
      t = nn ;
    }
  return(t) ;
}


/* Convert binary operators into n-ary ones. */

static Tree *
widenbinop(t, type)
Tree *t ;
enum tree type ;
{
  int sum ;
  Tree *nt ;
  int i, j, k ;

  if (!t) return(t) ;

  for (i = 0; i < t->siz; i++)
    t->kids[i] = widenbinop(t->kids[i], type) ;

  if (t->t != type) return(t) ;

/* Find new number of kids. */

  sum = t->siz ;
  for (i = 0; i < t->siz; i++)
    {
      if (t->kids[i]->t == type)
        sum += t->kids[i]->siz - 1 ;  /* Ignore parent */
    }
  if (sum == t->siz) return(t) ;

  nt = treealloc(sum) ;
  nt->t = type ;
  for (k = i = 0; i < t->siz; i++)
    {
      if (t->kids[i]->t != type)
        nt->kids[k++] = t->kids[i] ;
      else
        for (j = 0; j < t->kids[i]->siz; j++)
          nt->kids[k++] = t->kids[i]->kids[j] ;
    }
  return(nt) ;
}


/* Turn constant expressions into literal constants. */

static int
cexp(t)
Tree *t ;
{
  int chng = 0 ;
  int ret ;
  int val = 0 ;
  int sum ;
  int i, j ;
  Tree *nt, *nc ;

  if (!t) return ;

  ret = 1 ;
  for (i = 0; i < t->siz; i++)
    {
      chng |= cexp(t->kids[i]) ;
      if (t->kids[i]->t != T_Inum) ret = 0 ;
    }
  
  switch (t->t)
    {
      case T_Inum : ret = 0 ;
                    break ;

/* Unary operators */

      case T_Bang : if (t->kids[0]->t == T_Inum)
                      t->i = val = !t->kids[0]->i ;
                    break ;

      case T_Not  : if (t->kids[0]->t == T_Inum)
                      val = ~t->kids[0]->i ;
                    break ;

      case T_Neg  : if (t->kids[0]->t == T_Inum)
                      val = -t->kids[0]->i ;
                    break ;

      case T_Sin  :
      case T_Cos  :
      case T_Tan  : if (t->kids[0]->t == T_Inum)
                      {
                        val = (t->t == T_Sin ? sin_cache :
                               t->t == T_Cos ? cos_cache :
                               tan_cache)[t->kids[0]->i % MAX_ANG] ;
                      }
                    break ;

      case T_Abs  : if (t->kids[0]->t == T_Inum) 
                      val = t->kids[0]->i < 0 ? -t->kids[0]->i
                                              :  t->kids[0]->i ;
                    break ;

      case T_Sqr  : if (t->kids[0]->t == T_Inum) 
                      val = t->kids[0]->i * t->kids[0]->i ;
                    break ;
  
      case T_Sqrt : if (t->kids[0]->t == T_Inum)
                      val = (int) sqrt(t->kids[0]->i * 1.0) ;
                    break ;

      case T_Mul  :
      case T_Add  :
      case T_And  :
      case T_Or   :
      case T_Xor  : switch (t->t)
                      {
                        case T_And : val = ~0 ;
                                     break ;
                        case T_Add :
                        case T_Or  :
                        case T_Xor : val = 0 ;
                                     break ;
                        case T_Mul : val = 1 ;
                      }
                    j = val ;
                    for (sum = t->siz, i = 0; i < t->siz; i++)
                      {
                        nt = t->kids[i] ;
                        if (nt->t == T_Inum)
                          {
                            switch (t->t)
                              {
                                case T_Mul : val *= nt->i ;
                                             break ;
                                case T_Add : val += nt->i ;
                                             break ;
                                case T_Or  : val |= nt->i ;
                                             break ;
                                case T_And : val &= nt->i ;
                                             break ;
                                case T_Xor : val ^= nt->i ;
                              }
                            sum-- ;
                          }
                        else nc = nt ;
                      }

/* No variables, or multiply by zero: replace with T_Inum */

                    if (sum == 0 ||
                        (val == 0 && (t->t == T_And || t->t == T_Mul)))
                      {
                        ret = 1 ;
                        break ;
                      }

/* Only make space for the constant if there are any */

                    if (val != j) sum++ ;

                    if (sum == 1)
                      {
                        t = nc ;
                        for (i = 0; i < nc->siz; i++)
                          t->kids[i] = nc->kids[i] ;
                        break ;
                      }
                    if (sum == t->siz)
                      {
                        ret = 0 ;
                        break ;
                      }
                    ret = 2 ;  /* We changed the tree, but not into a const. */
                    nt = treealloc(sum) ;
                    nt->t = t->t ;
                    if (val != j)
                      {
                        nt->kids[0] = mkInum(val) ;
                        sum = 1 ;
                      }
                    else sum = 0 ;
                    for (i = 0; i < t->siz; i++)
                      {
                        if (t->kids[i]->t != T_Inum)
                          nt->kids[sum++] = t->kids[i] ;
                      }
                    for (i = 0; i < nt->siz; i++)
                      t->kids[i] = nt->kids[i] ;
                    t->siz = nt->siz ;
                    break ;

      case T_Sub  : PRINTF("Warning: cexp: unexpected Sub node\n") ;
                    if (t->kids[0]->t == T_Inum && t->kids[1]->t == T_Inum)
                      val = t->kids[0]->i - t->kids[1]->i ;
                    break ;

      case T_Div  : if (t->kids[0]->t == T_Inum && t->kids[1]->t == T_Inum)
                      {
                        if (t->kids[1]->i == 0) run_error(ERR_DIV) ;
                        val = t->kids[0]->i / t->kids[1]->i ;
                      }
                    break ;

      case T_Mod  : if (t->kids[0]->t == T_Inum && t->kids[1]->t == T_Inum)
                      {
                        if (t->kids[1]->i == 0) run_error(ERR_MOD) ;
                        val = t->kids[0]->i % t->kids[1]->i ;
                      }
                    break ;

      case T_Eq   : if (t->kids[0]->t == T_Inum && t->kids[1]->t == T_Inum)
                      val = t->kids[0]->i == t->kids[1]->i ;
                    break ;

      case T_Ne   : if (t->kids[0]->t == T_Inum && t->kids[1]->t == T_Inum)
                      val = t->kids[0]->i != t->kids[1]->i ;
                    break ;

      default     : ret = 0 ;
    }

  if (ret == 1)
    {
      t->t = T_Inum ;
      t->i = val ;
      t->siz = 0 ;
    }
  return(ret) ;
}


/*  Put range checking in Coordpair nodes
 *  The cse pass wull make this neater later on
 */

static Tree *
addcare(t)
Tree *t ;
{
  int i ;
  int co = t->t == T_Mapcoord ? 1 : 0 ;
  int sz = t->t == T_Mapcoord ? 2 : 1 ;

  for (i = 0; i < t->siz; i++)
    t->kids[i] = addcare(t->kids[i]) ;

  if ((t->t == T_Mapcoord || t->t == T_Coord) &&
       t->siz == sz && t->kids[co]->t != T_Polar)
    {
      char *tmp1, *tmp2 ;
      Tree *xarg = t->kids[co]->kids[0] ;
      Tree *yarg = t->kids[co]->kids[1]->kids[0] ;
      Tree *zero = mkInum(0) ;

      t = mkCond(mk(T_Land, mk(T_Lt, xarg, mkInum(Xsize)),
                 mk(T_Land, mk(T_Ge, xarg, zero),
                 mk(T_Land, mk(T_Lt, yarg, mkInum(Ysize)),
                            mk(T_Ge, yarg, zero)))), t, zero) ;
    }
  return(t) ;
}


static void
miscopt(t)
Tree *t;
{
  int num ;
  int shift, i ;

  switch (t->t)
    {

/* Accumulate shifts */

      case T_Lshift :
      case T_Rshift : {
                        enum tree kt = t->kids[0]->t ;
                        int kv, ka ;

                        if (t->kids[1]->t == T_Inum &&
                           (kt == T_Lshift || kt == T_Rshift) &&
                            t->kids[0]->kids[1]->t == T_Inum)
                          {
                            int kv = t->kids[0]->kids[1]->i ;
  
                            ka = t->kids[1]->i ;
                            ka += kt == t->t ? kv : -kv ;
                            if (ka < 0)
                              {
                                ka = -ka ;
                                t->t = t->t == T_Lshift ? T_Rshift : T_Lshift ;
                              }
                            t->kids[1]->i = ka ;
                            t->kids[0] = t->kids[0]->kids[0] ;
                          }
                      }

/* Undo the de-Subbing done above, so the code-gen can use sub */

      case T_Add    : if (t->kids[0]->t == T_Neg)
                        {
                          Tree *tmp = t->kids[1] ;
                          t->kids[1] = t->kids[0] ;
                          t->kids[0] = tmp ;
                        }
                      if (t->kids[1]->t == T_Neg)
                        {
                          t->t = T_Sub ;
                          t->kids[1] = t->kids[1]->kids[0] ;
                        }
                      break ;

/* Replace *,/,% with logical ops if possible */

      case T_Div    :
      case T_Mul    : if (t->kids[0]->t == T_Inum || t->kids[1]->t == T_Inum)
                        {
                          if (t->kids[0]->t == T_Inum)
                            {
                              Tree *tmp = t->kids[0] ;
                              if (t->t == T_Div) break ;
                              t->kids[0] = t->kids[1] ;
                              t->kids[1] = tmp ;
                            }
                          num = t->kids[1]->i ;
                          if ((num & (num - 1)) == 0)
                            {
                              for (shift = 0 ; num != 0 ; shift++, num >>= 1)
                                continue ;
                              if (shift)
                                {
                                  t->t = t->t == T_Div ? T_Rshift : T_Lshift ;
                                  t->kids[1]->i = --shift ;
                                }
                              else
                                {
                                  if (t->t == T_Mul)
                                    {
                                      t->t = T_Inum ;
                                      t->i = 0 ;
                                      t->siz = 0 ;
                                    }
                                  else run_error(ERR_DIV) ;
                                }
                            }
                        }
                      break ;
      case T_Mod    : if (t->kids[1]->t == T_Inum)
                        {
                          num = t->kids[1]->i ;
                          if (num & (num - 1) != 0) break ;
                          t->t = T_And ;
                          t->kids[1]->i = num-1 ;
                        }
    }

/*  Move constants to the right (well, kids[1]), 'cause the code
 *  generator seems to want them there.
 */

  switch (t->t)
    {
      case T_Add :
      case T_And :
      case T_Or  :
      case T_Xor :
      case T_Mul : if (t->kids[0]->t == T_Inum && t->kids[1]->t != T_Inum)
                     {
                       Tree *tmp = t->kids[1] ;
                       t->kids[1] = t->kids[0] ;
                       t->kids[0] = tmp ;
                     }
                   break ;
    }

  for (i = 0; i < t->siz; i++)
    if (t->kids[i])
      miscopt(t->kids[i]) ;
}
