/*LINTLIBRARY*/

/*  @(#)next.m 1.3 92/04/16
 *
 *  Popi device driver for a NeXT machine
 *  Written by Joe Freeman.
 *  Modified by Stephan Wasserroth <wasserroth@fokus.berlin.gmd.dbp.de>
 *
 *  Popi was originally written by Gerard J. Holzmann - AT&T Bell Labs.
 *  This version is based on the code in his Prentice Hall book,
 *  "Beyond Photography - the digital darkroom," ISBN 0-13-074410-7,
 *  which is copyright (c) 1988 by Bell Telephone Laboratories, Inc. 
 *
 *  Permission is given to distribute these extensions, as long as these
 *  introductory messages are not removed, and no monies are exchanged.
 *
 *  No responsibility is taken for any errors or inaccuracies inherent
 *  either to the comments or the code of this program, but if reported
 *  (see README file) then an attempt will be made to fix them.
 */

#include "popi.h"
#import "appkit/appkit.h"	/* lazy tsk tsk */

/*  These are the exportable routines used by the popi program.
 *
 *  These are:
 *
 *  disp_init(argc, argv)    - called from main at the start.
 *  disp_finish()            - called from main prior to exit.
 *  disp_imgstart(w,h,c,l)   - called prior to drawing an image.
 *  disp_imgend()            - called after drawing an image.
 *  disp_putline(line,y,w,c) - to draw an image scanline.
 *  disp_getchar()           - to get the next character typed.
 *  disp_prompt()            - display popi prompt and clear input buffer.
 *  disp_error(errtype,pos)  - display error message.
 *  disp_percentdone(n)      - display percentage value of conversion.
 *  disp_resize(w,h)         - resize popi-window
 *  disp_colormap(n,r,g,b)   - create new colormap
 */

id	NXApp;
id	drawWindow,percentWindow;
extern int tcache;	/* turn on trig cache */
int	lastX, lastY;	/* size of the last displayed image */
int	curXsize;	/* length of an allocated scanline */

#define	PERCENTHEIGHT	50
#define	PERCENTWIDTH	120
#define	LOGHEIGHT	250
#define	LOGWIDTH	200
#define	DRAWBASEX		250
#define	DRAWBASEY 	250

/*ARGSUSED*/
void
disp_init(argc,argv)           /* called from main at the atart. */
int argc;
char *argv[];
{
	NXRect	bound;
	NXApp = [Application new];
	bound.origin.x = DRAWBASEX;
	bound.origin.y = DRAWBASEY;
	bound.size.width = lastX = Xsize;
	bound.size.height = lastY = Ysize;
	drawWindow = [Window newContent:&bound
						style:NX_TITLEDSTYLE
						backing: NX_RETAINED
						buttonMask: NX_MINIATURIZEBUTTONMASK
						defer : NO];
	[[drawWindow contentView] setFlipped:YES];	/* start at top */
	[drawWindow orderFront:nil];
	[drawWindow setTitle:"popi image window"];
	[drawWindow display];
	bound.origin.x = DRAWBASEX - PERCENTWIDTH;
	bound.origin.y = DRAWBASEY + Ysize - PERCENTHEIGHT;
	bound.size.width = PERCENTWIDTH; bound.size.height = PERCENTHEIGHT;
	percentWindow = [Window newContent:&bound
						style:NX_TITLEDSTYLE
						backing: NX_RETAINED
						buttonMask: 0
						defer: NO];
	[percentWindow setTitle:"percent done"];
	bound.origin.x = 40.0; bound.origin.y = 350;
	bound.size.width = LOGWIDTH; bound.size.height = LOGHEIGHT;
	[percentWindow display];
	bound.origin.x = 0.0; bound.origin.y = DRAWBASEY;
	bound.size.width = DRAWBASEX ;  bound.size.height = Ysize - PERCENTHEIGHT;
	NXPing();
	scr_depth = 32;   /* !!!! assume NeXT-Dimension !!!! */
	tcache = 1;	/* set trig-cache on unconditionally */
	Verbose = 1;	/* set Verbose-flag unconditionally */
}


void
disp_finish()                  /* called from main prior to exit. */
{
	[drawWindow free];
	[percentWindow free];
	[NXApp free];
}


void
disp_imgstart(width, height, ncol, len) /* called prior to drawing an image. */
int width, height, ncol, len;
{
    NXRect		theRect ;
    NXRect		theFrame ;

    curXsize = len ;
    if (Verbose)
    {
	PRINTF("Displaying %s image in %s mode ... ",
		(ncol == 3) ? "color" : "gray",
		(dtype == IS_MONO) ? "mono" :
		(dtype == IS_GRAY) ? "gray" : "color") ;
	FFLUSH(stdout);
	if (dtype == IS_MONO)
		PRINTF("\nMONOchrome display NOT SUPPORTED\n") ;
    }
    if ((width != lastX) || (height != lastY))	/* size changed -> clear */
    {
/* clear the window and draw a border line around image -- alternative 1 */
/*OUT
	[[drawWindow contentView] lockFocus];
	PSgsave();
	theRect.origin.x = 0.0;
	theRect.origin.y = 0.0;
	theRect.size.width = (float)Xsize;
	theRect.size.height = (float)Ysize;
	NXSetColor(NX_COLORDKGRAY);
	NXRectFill(&theRect);
	NXSetColor(NX_COLORBLACK);
	PSmoveto(0.0, (float)height);
	PSlineto((float)width, (float)height);
	PSlineto((float)width, 0.0);
	PSstroke();
 	PSgrestore();
	[[drawWindow contentView] unlockFocus];
OUT*/
/* resize the window to the actual image size -- alternative 2 */
	[drawWindow getFrame:&theFrame];
	theFrame.origin.y += lastY - height;
	theFrame.size.height += height - lastY;
	theFrame.size.width += width - lastX;
	[drawWindow placeWindow:&theFrame];
	lastX = width ;
	lastY = height ;
    }
}


void
disp_imgend()                  /* called after drawing an image. */
{
	if (dtype == IS_MONO) return ;
	[drawWindow flushWindow];
	NXPing();
	PRINTF("done\n");
}


void
disp_putline(lines, y, width, ncolors)	/* called to draw image scanline y. */
pixel_t **lines;
int y, width, ncolors;
{
	unsigned char	*line ;
	int		mask ;
	NXRect		theRect ;
	if (dtype == IS_MONO) return ; 	/* MONOchrome NOT SUPPORTED */
	if ((dtype == IS_GRAY) && ncolors == 3)
		line = ntsc_luma(lines, y, width) ;
	else
		line = &lines[0][y*curXsize] ;
	if ((dtype == IS_COLOR) && ncolors == 3)
		mask = NX_COLORMASK ;
	else
		mask = NX_MONOTONICMASK ;
	[[drawWindow contentView] lockFocus];
	PSgsave();
	theRect.origin.x = 0.0;
	theRect.origin.y = (float)y;
	theRect.size.width = (float)width;
	theRect.size.height = (float)1.0;
	NXImageBitmap(&theRect, width, 1.0, 
					8 , (dtype == IS_COLOR) ? ncolors : 1,
					NX_PLANAR, mask, 
					line, &lines[1][y*curXsize],
					&lines[2][y*curXsize], NULL, NULL);
 	PSgrestore();
	[[drawWindow contentView] unlockFocus];
}


disp_getchar()                 /* get next user typed character. */
{
    return(getchar());
}


disp_prompt()                  /* display popi prompt. */
{
    char	*prompt = "-> ";

    PRINTF(prompt);
    FFLUSH(stdout);
    return sizeof prompt - 1;
}


void
disp_error(errtype, pos)            /* display error message. */
int	errtype,
	pos;
{
    extern int  errno;
    extern char *sys_errlist[];

    if (errtype & ERR_PARSE)
    {
        int     i;
 
        for (i=1; i < pos; ++i)
            PUTC('-', stderr);
        PUTC('^', stderr);
        PUTC('\n', stderr);
    }
 
    FPRINTF(stderr, "%s\n", ErrBuf);
    /* we assume errno hasn't been reset by the preceding output */
    if (errtype & ERR_SYS)
        FPRINTF(stderr, "\t(%s)\n", sys_errlist[errno]);
    FFLUSH(stdout);
}

void
disp_percentdone(percent)
int	percent;
{
    static	int	lastpercent = 100;
    static	NXRect	bound;

    if (!Verbose)
	return;
    if (percent == 100)
    {
    	[percentWindow orderOut:nil];
	return;
    }
    if ((percent == 0) && (lastpercent != 0))
    {
    	[[percentWindow  contentView] lockFocus];
    	[[percentWindow  contentView] getBounds:&bound];
	NXDrawGrayBezel( &bound, &bound);
    	[[percentWindow contentView] unlockFocus];
    	[percentWindow orderFront:nil];
	NXPing();
	bound.origin.x = bound.origin.y = 10.0;
	bound.size.height = PERCENTHEIGHT - (2.0 * 10.0);
	lastpercent = percent;
    }
    if (percent != lastpercent && percent % 5 == 0)
    {
	[[percentWindow contentView] lockFocus];
	bound.size.width = (float)percent;
	NXDrawButton(&bound,&bound);
	[[percentWindow contentView] unlockFocus];
	NXPing();
	lastpercent = percent;
    }
}

void
disp_colormap (a, r, g, b)
int a;
unsigned char *r, *g, *b;
{
}

void
disp_resize(width, height)              /* Resize popi image window. */
int width, height ;
{
	NXRect	theFrame, theRect;
	
	PRINTF ("resizing window to X= %i, Y= %i\n", width, height) ;
	[drawWindow getFrame:&theFrame];
	theFrame.origin.y += lastY - height;
	theFrame.size.height += height - lastY;
	theFrame.size.width += width - lastX;
	[drawWindow placeWindow:&theFrame];
	[[drawWindow contentView] lockFocus];
	PSgsave();
	theRect.origin.x = 0.0;
	theRect.origin.y = 0.0;
	theRect.size.width = (float)width;
	theRect.size.height = (float)height;
	NXSetColor(NX_COLORLTGRAY);
	NXRectFill(&theRect);
 	PSgrestore();
	[[drawWindow contentView] unlockFocus];
	lastX = width;
	lastY = height;
	NXPing();
}
