/* Copyright (C) 1992 Aladdin Enterprises.  All rights reserved.
   Distributed by Free Software Foundation, Inc.

This file is part of Ghostscript.

Ghostscript is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY.  No author or distributor accepts responsibility
to anyone for the consequences of using it or for whether it serves any
particular purpose or works at all, unless he says so in writing.  Refer
to the Ghostscript General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
Ghostscript, but only under the conditions described in the Ghostscript
General Public License.  A copy of this license is supposed to have been
given to you along with Ghostscript so you can know your rights and
responsibilities.  It should be in a file named COPYING.  Among other
things, the copyright notice and this notice must be preserved on all
copies.  */

/* gxrefct.h */
/* Definitions for reference-counted data structures */

/* ------ Allocate/free ------ */

#define rc_alloc_n(vp, stype, mprocs, errstat, cname, rcinit)\
   {	if ( (vp = (stype *)(*(mprocs)->alloc)(1, sizeof(stype), cname)) == 0 )\
	   errstat;\
	vp->ref_count = rcinit;\
   }
#define rc_alloc0(vp, stype, mprocs, errstat, cname)\
  rc_alloc_n(vp, stype, mprocs, errstat, cname, 0)
#define rc_alloc1(vp, stype, mprocs, errstat, cname)\
  rc_alloc_n(vp, stype, mprocs, errstat, cname, 1)

#define rc_free(vp, mprocs, cname)\
  (*(mprocs)->free)((char *)vp, 1, sizeof(*vp), cname)

/* ------ Reference counting ------ */

/* Increment a reference count. */
#define rc_increment(vp)\
  if ( vp != 0 ) vp->ref_count++

/* Increment a reference count, allocating the structure if necessary. */
#define rc_allocate(vp, stype, mprocs, errstat, cname)\
  if ( vp != 0 )\
	vp->ref_count++;\
  else\
   {	rc_alloc1(vp, stype, mprocs, errstat, cname);\
   }

/* Guarantee that a structure is not shared. */
#define rc_unshare(vp, stype, mprocs, errstat, cname)\
  if ( vp == 0 || vp->ref_count > 1 )\
   {	stype *new;\
	rc_alloc1(new, stype, mprocs, errstat, cname);\
	if ( vp ) vp->ref_count--;\
	vp = new;\
   }

/* Decrement a reference count, freeing the structure if necessary. */
#define rc_decrement(vp, mprocs, cname)\
  if ( vp != 0 && !--(vp->ref_count) )\
   {	rc_free(vp, mprocs, cname);\
	vp = 0;\
   }

/* Adjust a reference count either up or down. */
#define rc_adjust(vp, delta, mprocs, cname)\
  if ( vp != 0 && !(vp->ref_count += delta) )\
   {	rc_free(vp, mprocs, cname);\
	vp = 0;\
   }

/* Assign a pointer, adjusting reference counts. */
#define rc_assign(vpto, vpfrom, mprocs, cname)\
  if ( vpto != vpfrom )\
   {	rc_decrement(vpto, mprocs, cname);\
	vpto = vpfrom;\
	rc_increment(vpto);\
   }
