/* Copyright (C) 1989, 1990, 1991 Aladdin Enterprises.  All rights reserved.
   Distributed by Free Software Foundation, Inc.

This file is part of Ghostscript.

Ghostscript is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY.  No author or distributor accepts responsibility
to anyone for the consequences of using it or for whether it serves any
particular purpose or works at all, unless he says so in writing.  Refer
to the Ghostscript General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
Ghostscript, but only under the conditions described in the Ghostscript
General Public License.  A copy of this license is supposed to have been
given to you along with Ghostscript so you can know your rights and
responsibilities.  It should be in a file named COPYING.  Among other
things, the copyright notice and this notice must be preserved on all
copies.  */

/* iname.c */
/* Name lookup for Ghostscript interpreter */
#include "memory_.h"
#include "ghost.h"
#include "alloc.h"
#include "errors.h"
#include "name.h"
#include "store.h"

/* Definitions and structure for the name table. */
/* The first entry is left unused. */
/* 1-character names are the next nt_1char_size entries. */
#define nt_log2_sub_size 7
#define nt_sub_size (1 << nt_log2_sub_size)
#define nt_hash_size 256		/* must be a power of 2 */
#define nt_1char_size 256		/* must cover a full byte */
typedef name name_sub_table[nt_sub_size];
typedef struct {
	ushort hash[nt_hash_size];
	name *table[1 << (16 - nt_log2_sub_size)];	/* name_sub_table */
	ref count;			/* t_integer */
#define nt_count(nt) (uint)((nt)->count.value.intval)
} name_table;
#define name_index_ptr(nt, index)\
  ((nt)->table[(index) >> nt_log2_sub_size] + ((index) & (nt_sub_size - 1)))

/* The one and only name table (for now). */
private name_table *the_nt;

/* Forward references */
private int name_alloc_sub(P1(name_table *));

/* Make a t_name ref out of a name * */
#define make_name(pref, pnm) make_tv(pref, t_name, pname, pnm)

/* Initialize the name table */
void
name_init()
{	uint i;
	the_nt = (name_table *)alloc(1, sizeof(name_table), "name_init");
	memset(the_nt, 0, sizeof(name_table));
	make_int(&the_nt->count, 1);
	for ( i = 0; i <= nt_1char_size; i += nt_sub_size )
	   {	the_nt->count.value.intval = i + 1;
		name_alloc_sub(the_nt);
	   }
}

/* Look up or enter a name in the table. */
/* Return 0 or an error code. */
/* The return may overlap the characters of the string! */
/* See name.h for the meaning of enterflag. */
int
name_ref(const byte *ptr, uint isize, ref *pref, int enterflag)
{	register name *pname;
	const byte *cptr;
	ushort size = (ushort)isize;	/* see name.h */
	if ( size == 1 )
	   {	pname = name_index_ptr(the_nt, *ptr + 1);
		if ( pname->string_size != 0 )
		   {	make_name(pref, pname);
			return 0;
		   }
		if ( enterflag < 0 ) return e_undefined;
		pname->index = *ptr + 1;
		pname->next_index = 0;
	   }
	else
	   {	ushort *phash =
		  the_nt->hash +
		    ((ushort)string_hash(ptr, size) & (nt_hash_size - 1));
		uint nidx = *phash;
		while ( nidx != 0 )
		   {	pname = name_index_ptr(the_nt, nidx);
			if ( pname->string_size == size &&
			     !memcmp(ptr, pname->string_bytes, size)
			   )
			   {	make_name(pref, pname);
				return 0;
			   }
			nidx = pname->next_index;
		   }
		/* Not in table, allocate a new entry. */
		if ( enterflag < 0 ) return e_undefined;
		if ( !(nt_count(the_nt) & (nt_sub_size - 1)) )
		   {	int code = name_alloc_sub(the_nt);
			if ( code < 0 ) return code;
		   }
		nidx = nt_count(the_nt);
		pname = name_index_ptr(the_nt, nidx);
		pname->index = nidx;
		pname->next_index = *phash;
		*phash = nidx;
		ref_save(&the_nt->count, "name_ref(count)");
		the_nt->count.value.intval = nidx + 1;
	   }
	/* Name was not in the table.  Make a new entry. */
	if ( enterflag )
	   {	cptr = (const byte *)alloc(size, 1, "name_ref(string)");
		if ( cptr == 0 ) return e_VMerror;
		memcpy((byte *)cptr, ptr, size);
	   }
	else
		cptr = ptr;
	pname->string_size = size;
	pname->string_bytes = cptr;
	pname->pvalue = pv_no_defn;
	make_name(pref, pname);
	return 0;
}

/* Get the string for a name. */
void
name_string_ref(const ref *pnref /* t_name */,
  ref *psref /* result, t_string */)
{	name *pname = pnref->value.pname;
	make_tasv(psref, t_string, a_read+a_execute, pname->string_size,
		  bytes, (byte *)pname->string_bytes);
}

/* Convert a t_string object to a name. */
/* Copy the executable attribute. */
int
name_from_string(const ref *psref, ref *pnref)
{	int exec = r_has_attr(psref, a_executable);
	int code = name_ref(psref->value.bytes, r_size(psref), pnref, 1);
	if ( code < 0 ) return code;
	if ( exec ) r_set_attrs(pnref, a_executable);
	return code;
}

/* Enter a name during initialization. */
/* Fatal error if the entry fails. */
void
name_enter(const char *str, ref *pref)
{	if ( name_ref((const byte *)str, strlen(str), pref, 0) )
		lprintf1("name_enter failed - %s", str),
		gs_exit(1);
}

/* Get the name with a given index. */
void
name_index_ref(uint index, ref *pnref)
{	make_name(pnref, name_index_ptr(the_nt, index));
}

/* Get the current name count. */
ushort
name_next_index()
{	return nt_count(the_nt);
}

/* Clean up the name table before a restore. */
/* The count will be reset, and added subtables will be freed. */
/* All we have to do is remove initial entries from the hash chains, */
/* since we know they are linked in decreasing index order. */
/* (There will be some spurious non-zero entries in the subtable table, */
/* but this doesn't matter since they will never be accessed.) */
void
name_restore(uint old_count)
{	register ushort *phash = &the_nt->hash[0];
	register uint i;
	for ( i = 0; i < nt_hash_size; phash++, i++ )
	   {	while ( *phash >= old_count )
		  *phash = name_index_ptr(the_nt, *phash)->next_index;
	   }
}

/* ------ Internal procedures ------ */

/* Allocate the next sub-table. */
private int
name_alloc_sub(name_table *nt)
{	name *sub = (name *)alloc(1, sizeof(name_sub_table), "name_alloc_sub");
	if ( sub == 0 ) return e_VMerror;
	memset(sub, 0, sizeof(name_sub_table));
	nt->table[nt_count(nt) >> nt_log2_sub_size] = sub;
	return 0;
}
