/* Copyright (C) 1989, 1992 Aladdin Enterprises.  All rights reserved.
   Distributed by Free Software Foundation, Inc.

This file is part of Ghostscript.

Ghostscript is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY.  No author or distributor accepts responsibility
to anyone for the consequences of using it or for whether it serves any
particular purpose or works at all, unless he says so in writing.  Refer
to the Ghostscript General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
Ghostscript, but only under the conditions described in the Ghostscript
General Public License.  A copy of this license is supposed to have been
given to you along with Ghostscript so you can know your rights and
responsibilities.  It should be in a file named COPYING.  Among other
things, the copyright notice and this notice must be preserved on all
copies.  */

/* dict.h */
/* Interfaces for Ghostscript dictionary package */

/********************************
 * NOTE: on MS-DOS systems, the dict stack is stored in the data segment.
 * This leads to large performance gains, at the expense of having to swap
 * the stack explicitly when switching contexts or handling segment under-
 * or overflow (none of which are implemented yet!).
 ********************************/

/*
 * Contrary to our usual practice, we expose the (first-level)
 * representation of a dictionary in the interface file,
 * because it is so important that access checking go fast.
 * The access attributes for the dictionary are stored in
 * the contents ref.
 */
struct dict_s {
	ref count;		/* t_integer, # of occupied entries */
	ref keys;		/* t_shortarray or t_array, keys */
	ref values;		/* t_array, values */
};

/* Define the maximum size of a dictionary */
extern const uint dict_max_size;

/* Create a dictionary */
extern int dict_create(P2(uint maxlength, ref *dict));

/* Return a pointer to a ref that holds the access attributes */
/* for a dictionary. */
#define dict_access_ref(pdref) (&(pdref)->value.pdict->values)
#define check_dict_read(dict) check_read(*dict_access_ref(&dict))
#define check_dict_write(dict) check_write(*dict_access_ref(&dict))

/* Look up in a stack of dictionaries.  Store a pointer to the value slot */
/* where found, or to the (value) slot for inserting. */
/* Return 1 if found, 0 if not and there is room for a new entry, */
/* or e_dictfull if the dictionary is full and the key is missing. */
/* The caller is responsible for ensuring key is not a null. */
/* Note that pdbot <= pdtop, and the search starts at pdtop. */
extern int dict_lookup(P4(const ref *pdbot, const ref *pdtop, const ref *key, ref **ppvalue));
/* Look up in just one dictionary. */
#define dict_find(dict,key,ppvalue) dict_lookup(dict,dict,key,ppvalue)

/* Enter a key-value pair in a dictionary. */
/* Return 0, e_dictfull, or e_VMerror if the key was a string */
/* and a VMerror occurred when converting it to a name. */
extern int dict_put(P3(ref *dict, const ref *key, const ref *pvalue));

/* Remove a key-value pair from a dictionary. */
/* Return 0 or e_undefined. */
extern int dict_undef(P2(ref *dict, const ref *key));

/* Return the number of elements in a dictionary. */
extern uint dict_length(P1(const ref *dict));

/* Return the capacity of a dictionary. */
extern uint dict_maxlength(P1(const ref *dict));

/* Copy one dictionary into another. */
/* Return 0 or e_dictfull. */
extern int dict_copy(P2(const ref *dfrom, ref *dto));

/* Grow or shrink a dictionary. */
/* Return 0 or e_dictfull. */
extern int dict_resize(P2(ref *dict, uint newmaxlength));

/* Prepare to enumerate a dictionary. */
/* Return an integer suitable for the first call to dict_next. */
extern int dict_first(P1(const ref *dict));

/* Enumerate the next element of a dictionary. */
/* index is initially the result of a call on dict_first. */
/* Either store a key and value at eltp[0] and eltp[1] */
/* and return an updated index, or return -1 */
/* to signal that there are no more elements in the dictionary. */
extern int dict_next(P3(const ref *dict, int index, ref *eltp));

/****** The remaining definitions take explicit note of ******/
/****** the existence of the dictionary stack. ******/

/* Define the dictionary stack and the standard dictionaries. */
extern ref dstack[];
#define systemdict (dstack[0])
#define userdict (dstack[1])

/* Define the dictionary stack pointers. */
typedef ref _ds *ds_ptr;
extern ds_ptr dsp, dstop;

/* Define a special fast entry for name lookup in the interpreter. */
/* The key is known to be a name; search the entire dict stack. */
/* Return the pointer to the value slot. */
/* If the name isn't found, just return 0. */
extern ref *dict_find_name(P1(ref *pname));
