// Copyright (C) 2005 Open Source Telecom Corp.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include <cc++/process.h>
#include <cc++/slog.h>
#include "server.h"

namespace server {
using namespace ost;
using namespace std;

BuildThread::BuildThread(ScriptInterp *interp, BayonneAudio *au, Audio::Info *inf, const char **list) :
ScriptThread(interp, 0)
{
	memcpy(&to, inf, sizeof(to));
	in = au;
	buffer = NULL;
	lbuffer = NULL;
	paths = list;
	completed = false;
}

BuildThread::~BuildThread()
{
	terminate();
	if(in)
		in->cleanup();

	strcpy(in->var_position, "00:00:00.000");
	if(out.isOpen())
	{
		if(!completed)
			remove(pathname);
		else
			out.getPosition(in->var_position, 12);	
		out.close();
	}

	if(buffer)
		delete[] buffer;

	if(lbuffer)
		delete[] lbuffer;

	in = NULL;
	buffer = NULL;
	lbuffer = NULL;
}

void BuildThread::run(void)
{
	const char *fn = *(paths++);

	fn = in->getFilename(fn, true);
	if(!fn)
		exit("unable-to-access");

	setString(pathname, sizeof(pathname), fn);
	fn = pathname;

	remove(fn);
	out.create(fn, &to, false, to.framing);
	if(!out.isOpen())
		exit("unable-to-create");

	Thread::yield();

	out.getInfo(&to);

	in->play(paths, modeReadAny);
	if(!in->isOpen())
		exit("unable-to-read");

	in->getInfo(&from);
	Thread::yield();

	if(to.encoding == from.encoding)
		copyDirect();
	else
	{
		if(!in->isStreamable() || !out.isStreamable())
		{
			remove(pathname);
			exit("unable-to-convert");
		}
		copyConvert();
	}
	exit(NULL);
}	

void BuildThread::copyDirect(void)
{
	unsigned samples, bufsize;
	bool endian = false;
	Error status = errSuccess;

	if(to.order && from.order && to.order != from.order)
		endian = true;

	samples = from.framecount;
	bufsize = from.framesize;
	buffer = new unsigned char[bufsize];

	while(status == errSuccess)
	{
		Thread::yield();
		status = in->getSamples(buffer, samples);
		if(status == errReadIncomplete)
			break;

		if(endian)
		{
                        swapEndian(from, buffer, samples);
                        swapEndian(to, buffer, samples);
		}

		Thread::yield();
		status = out.putSamples(buffer, samples);
	}
	completed = true;
}

void BuildThread::copyConvert(void)
{
	unsigned long samples;
	unsigned pages, npages;
	bool mono = true;

	if(isStereo(from.encoding) || isStereo(to.encoding))
		mono = false;

	samples = in->getCount();

	if(mono)
		lbuffer = new Sample[samples];
	else
		lbuffer = new Sample[samples * 2];

	for(;;)
	{
		Thread::yield();
		if(mono)
			pages = in->getMono(lbuffer, 1);
		else
			pages = in->getStereo(lbuffer, 1);

		if(!pages)
			break;

		Thread::yield();

		if(mono)
			npages = out.bufMono(lbuffer, samples);
		else
			npages = out.bufStereo(lbuffer, samples);
	}
	completed = true;
}

CopyThread::CopyThread(ScriptInterp *interp, const char *from, const char *to) :
ScriptThread(interp, 0)
{
	src = from;
	dest = to;
	in = NULL;
	out = NULL;
}

CopyThread::~CopyThread()
{
	terminate();
	if(in)
		fclose(in);
	if(out)
	{
		fclose(out);
		remove(dest);
	}
	in = out = NULL;
}

void CopyThread::run(void)
{
	unsigned len;

	in = fopen(src, "rb");
	if(!in)
		exit("source-invalid");
	
	remove(dest);
	out = fopen(dest, "wb");
	if(!out)
	{
		fclose(in);
		in = NULL;
		exit("target-invalid");
	}

	while(!feof(in))
	{
		len = fread(buf, 1, sizeof(buf), in);
		if(len < 1)
			break;
		fwrite(buf, len, 1, out);
		Thread::yield();
	}

	fclose(in);
	fclose(out);
	in = out = NULL;
	exit(NULL);
}	

} // end namespace
