/****************************** includes *************************************/
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>
/****************************************************************************/
typedef int BOOL;
#define TRUE 1;
#define FALSE 0;

BOOL PrepareRtf(char *filename, FILE **f); /* creates file and writes */
BOOL PreparePbm(char *filename, FILE **f);  /* opens file for reading */
BOOL ClosePbm(FILE **f);
BOOL CloseRtf(FILE **f);
void warning(char * text);
void error(char * text);
void IgnoreTo(char cEnd);
void ConvertImage(int p, int q);
int WriteImageHeader(int linelength, int linenumber); /* return number of bytes */
long int DimenToTwips(float size, char *unit);

/********************************* global variables *************************/
FILE *fRtf = NULL;		/* file pointer to RTF output file */
FILE *fStd = stdout;		/* file pointer to LOG file */
FILE *fPgm = NULL;		/* file pointer to input PBM file */
char Version[20]="pbm2pict, 0.1";
char *progname;			/* name of the executable file */
extern int getopt(int, char **, char *);
extern char *optarg;
extern int optind, opterr;
void OutputError(char *text);
char *output = NULL;
char *input = NULL;
char pnmtype=' ';
int NbytesPerLine=0;

/****************************************************************************/
int main(int argc, char **argv)
/****************************************************************************
 ****************************************************************************/
{
  int c,errflag=0;
  char errormessage[256]="";
  char letterP=' ';
  int linelength=0;
  int linenumber=0;

  progname=argv[0];

  printf("%s:\n %s", progname, Version);


  while((c = getopt(argc, argv, "o:")) != EOF)
  {
    switch(c)
    {
      case 'o':
        output = optarg;
        break;
      default:
        errflag = 1;
    }
  };
  sprintf(errormessage,"Usage: %s [-o outfile] inputfile",
      progname);
  if(argc > optind + 1 || errflag)
  {
    OutputError(errormessage);      
    return(1);
  }
  if(argc == optind + 1)
  {
    input = argv[optind];
  };

  if (output == NULL)
  {
    OutputError(errormessage);      
    return(1);
  }

/* opening input file */

  PreparePbm(input,&fPgm);
  PrepareRtf(output,&fRtf);
  fprintf(fStd,"\nInput file  %s",input);
  fprintf(fStd,"\nOutput file %s",output);

/* reading file header */
/*  fscanf(fPgm,"%c%c",letterP,pnmtype); */
  fread(&letterP,1,1,fPgm);
  fread(&pnmtype,1,1,fPgm);
  fprintf(fStd,"\nFile header=\'%c%c\'",letterP,pnmtype);
  if (letterP != 'P') error("input file is not a PNM file");
  if (pnmtype != '5') error("only PBM one byte/pixel format accepted");

  fscanf(fPgm,"%d%d",&linelength,&linenumber);
  fprintf(fStd,"\nLine length=%d, \tline number=%d",linelength,linenumber);

  IgnoreTo('\n');

  NbytesPerLine=WriteImageHeader(linelength,linenumber);
  ConvertImage(linelength,linenumber);
  CloseRtf(&fRtf);
}
/****************************************************************************/
void OutputError(char *text)
/****************************************************************************
purpose: writes error or diagnostic message on both stderr and stdout
 ****************************************************************************/
{
  fprintf(fStd,"\n%s",text);  	
}

/****************************************************************************/
void error(char * text)
/****************************************************************************
purpose: writes error message
globals: reads progname;
 ****************************************************************************/
{
  char errormessage[256] = "";
  sprintf(errormessage,"%s: ERROR: %s",progname,text);
  OutputError(errormessage);  	
  OutputError("Program aborted\n");
  exit(-1);
}

/****************************************************************************/
BOOL PrepareRtf(char *filename, FILE **f)  /* creates file */
/****************************************************************************
purpose: creates output file.
params: filename - name of outputfile
        f - pointer to filepointer to store file ID
 ****************************************************************************/
{ char Error_Msg_Text[256]="";
  if(filename != NULL)
  {
      if ((*f = fopen(filename,"wb")) == NULL)	 /* open file */
      {
      	sprintf(Error_Msg_Text,"Error opening RTF-file %s",filename);
        error(Error_Msg_Text);
        exit(1);
      }
  };
  return TRUE;
 }


/****************************************************************************/
BOOL PreparePbm(char *filename, FILE **f)  /* opens file for reading */
/****************************************************************************
purpose: opens input file.
params: filename - name of inputfile
        f - pointer to filepointer to store file ID
 ****************************************************************************/
 
{ char errormessage[256];

  if(filename != NULL)
  {
      if ((*f = fopen(filename,"r")) == NULL)	/* open file */
      { 
      	sprintf(errormessage,"Error opening image file %s",filename);
        error(errormessage);
/*	error("Error opening LATEX-file"); */
        exit(1);
      }
  }
  return TRUE;
}


/****************************************************************************/
BOOL ClosePbm(FILE **f)
/****************************************************************************
purpose: closes input file.
params: f - pointer to filepointer to invalidate
 ****************************************************************************/
{
  if(*f != stdin)
      fclose(*f);
  *f = NULL;
  return TRUE;
}


/****************************************************************************/
BOOL CloseRtf(FILE **f)
/****************************************************************************
purpose: closes output file.
params: f - pointer to filepointer to invalidate
 ****************************************************************************/
{
  fprintf(*f,"}\n");
  if(*f != stdout)
  {
      if(fclose(*f) == EOF)
      {
        warning("error closing RTF-File");
      }
  }
  *f = NULL;
  return TRUE;
}

/****************************************************************************/
void warning(char * text)
/****************************************************************************
purpose: writes diagnostic message
 ****************************************************************************/
{
  char errormessage[256] = "";
  sprintf(errormessage,"WARNING: %s",text);
  OutputError(errormessage);  	
}


/****************************************************************************/
void IgnoreTo(char cEnd)
/****************************************************************************
purpose: ignores anything from inputfile till character cEnd
params:  character to stop ignoring
globals: changes inputfile fPgm
 ****************************************************************************/
{
  char c;
  while (fread(&c, 1,1,fPgm) >= 1)
  {
    if (c == cEnd)  return;
  }
  error("End of input file found before image text");
}


/****************************************************************************/
int WriteImageHeader(int linelength, int linenumber)
/****************************************************************************
purpose: builds the {\\pict and associated characteristics
params:  linelength in pixels, linenumber
globals: changes inputfile fPgm
 ****************************************************************************/
{
  long int Width=DimenToTwips((linelength*1.0)/300.0,"in");
  long int Height=DimenToTwips((linenumber*1.0)/300.0,"in");
  int nbytes=((linelength+15)/16)*2;
  int Scale=100;
  fprintf(fRtf,"{\\pict\\wbitmap0\\picw%d\\pich%d",linelength,linenumber);
  fprintf(fRtf,"\\wbmbitspixel1\\wbmplanes1\\wbmwidthbytes%d",nbytes);
  fprintf(fRtf,"\n\\picwgoal%d\\pichgoal%d",Width,Height);
  fprintf(fRtf,"\n\\picscalex%d\\picscaley%d",Scale);
  fprintf(fRtf,"\n");
  return nbytes;
}	
/****************************************************************************/
void ConvertImage(int linelength, int linenumber)
/****************************************************************************
purpose: converts binary image into hexa
params:  linelength in pixels, linenumber
globals: changes inputfile fPgm
 ****************************************************************************/
{
  long int bytenumber=0;	
  long int bitnumber=0;	
  long int maxnumber=0;	
  char c, cCompl;
  char hexa[16]={'\0'};

  maxnumber=linelength*linenumber;
  fprintf(fStd,"\nNumber of bytes expected: %d",maxnumber);
  while (fread(&c, 1,1,fPgm) >= 1)
  {
    bytenumber +=8;
/*    cCompl=~c; */
    sprintf(hexa,"%8x",cCompl);
    if(hexa[7] == ' ') hexa[7]='0';
    if(hexa[6] == ' ') hexa[6]='0'; 
/*    fprintf(fStd,"\n|%c%c%c%c%c%c%c%c|%s|"
    ,hexa[0],hexa[1],hexa[2],hexa[3],hexa[4],hexa[5],hexa[6],hexa[7],hexa); */
    fprintf(fRtf,"%c%c",hexa[6],hexa[7]);
/*    fprintf(fStd,"\n%8X",c); */
    if (bitnumber % 256 == 0) fprintf(fRtf,"\n");
    if (bitnumber >= maxnumber) break;
  };
  if(bitnumber < maxnumber)  warning("End of input file found");
  fprintf(fRtf,"\n");
  fprintf(fStd,"\n Number of bits found %d",bitnumber);
}

/******************************************************************************/
long int DimenToTwips(float size, char *unit)
/******************************************************************************
  purpose : stores the requested vspace value
 ******************************************************************************/
{
  char errormessage[256];
  int SizeTwips=0;

  if (strcmp(unit,"pt") == 0)
  { SizeTwips = 20*size;
  }
  else if (strcmp(unit,"pc") == 0)
  { SizeTwips = 12*20*size;
  }
  else if (strcmp(unit,"in") == 0)
  { SizeTwips = 72.27*20*size;
  }
  else if (strcmp(unit,"bp") == 0)
  { SizeTwips = (72.27*20*size)/72.0;
  }
  else if (strcmp(unit,"cm") == 0)
  { SizeTwips = (72.27*20*size)/2.54;
  }
  else if (strcmp(unit,"mm") == 0)
  { SizeTwips = (72.27*20*size)/25.4;
  }
  else if (strcmp(unit,"dd") == 0)
  { SizeTwips = (1238.0*20*size)/1157.0;
  }
  else if (strcmp(unit,"cc") == 0)
  { SizeTwips = 12*(1238.0*20*size)/1157.0;
  }
  else if (strcmp(unit,"sp") == 0)
  { SizeTwips = (20*size)/65536.;
  }
  else
  {
    sprintf(errormessage,"Illegal dimension unit %s.");
    error(errormessage);	
  };
  
  return SizeTwips;
}

