#include <math.h>

typedef struct { float re ; float im ; } complex ;

extern complex one ;
extern complex zero ;

#define CABS(x) hypot( (x).re, (x).im )

complex cadd(), csub(), cmult(), scmult(), cdiv(), conjg(), csqrt() ;

/*
 * given N samples of digital waveform x, lpa computes M+1 coefficients
 * by maximum entropy (autocorrelation) method for spectral estimation--
 * these are returned in b[] (b[0] is always set to 1). lpa itself returns
 * the a0 (residual energy) coefficient.
 */
float lpa( x, N, b, M ) float x[], b[] ; int N, M ; {
 int i, j ;
 float s, at ;
 float a0, *rx, *rc ;
 char *malloc() ;
    rx = (float *) malloc( (M+2)*sizeof(float) ) ;
    rc = (float *) malloc( (M+2)*sizeof(float) ) ;
    for ( i = 0 ; i <= M + 1 ; i++ )
	for ( rx[i] = j = 0 ; j < N - i ; j++ )
	    rx[i] += x[j]*x[i + j] ;
    b[0] = 1. ;
    b[1] = rc[0] = -rx[1]/rx[0] ;
    a0 = rx[0] + rx[1]*rc[0] ;
    for ( i = 1 ; i < M ; i++ ) {
	for ( s = j = 0 ; j <= i ; j++ )
	    s += rx[i - j + 1]*b[j] ;
	rc[i] = -s/a0 ;
	for ( j = 1 ; j <= (i + 1)>>1 ; j++ ) {
	    at = b[j] + rc[i]*b[i-j+1] ;
	    b[i-j+1] += rc[i]*b[j] ;
	    b[j] = at ;
	}
	b[i+1] = rc[i] ;
	a0 += rc[i]*s ;
	if ( a0 <= 0. ) {
	    errmess( "lpa: matrix singularity" ) ;
	    return( a0 ) ;
	}
    }
    free( rx ) ;
    free( rc ) ;
    return( a0 ) ;
}
float lpamp( omega, a0, coef, M ) float omega, a0, *coef ; int M ; {
 register float wpr, wpi, wr, wi, re, im, temp ;
 register int i ;
    if ( a0 == 0. )
	return( 0. ) ;
    wpr = cos( omega ) ;
    wpi = sin( omega ) ;
    re = wr = 1. ;
    im = wi = 0. ;
    for ( coef++, i = 1 ; i <= M ; i++ ) {
	wr = (temp = wr)*wpr - wi*wpi ;
	wi = wi*wpr + temp*wpi ;
	re += *coef*wr ;
	im += (*coef++)*wi ;
    }
    if ( re == 0. && im == 0. )
	return( HUGE ) ;
    else
	return( a0/sqrt( (re*re + im*im) ) ) ;
}
lpresp( y, N, coef, M ) float y[], coef[] ; int N, M ; {
 int i, j ;
    for ( y[0] = sqrt( (double) coef[0] ), i = 1 ; i < N ; i++ )
	for ( y[i] = 0., j = 1 ; j <= M ; j++ )
	    if ( j <= i )
		y[i] -= coef[j]*y[i-j] ;
}
findpoles( coef, pole, M ) float coef[] ; complex pole[] ; int M ; {
 int i ;
 complex *a ;
 char *malloc() ;

    a = (complex *) malloc( (M+1)*sizeof(complex) ) ;
    for ( i = 0 ; i <= M ; i++ ) {
	a[M-i].re = coef[i] ;
	a[i].im = 0. ;
    }
    findroots( a, pole, M ) ;
/*
    for ( i = 0 ; i < M ; i++ )
	pole[i] = cdiv( one, conjg( pole[i] ) ) ;
*/
    free( a ) ;
}
/*
 * a[] contains M+1 complex polynomial coefficients in the order
 *         a[0] + a[1]*x + a[2]*x^2 + ... + a[M]*x^M
 * find and return its M roots in r[0] through r[M-1]
 */
findroots( a, r, M ) complex a[], r[] ; int M ; {
 complex x, b, c, laguerre() ;
 float eps = 1.e-6 ;
 int i, j, jj ;
 static complex *ad ;
 static int LM ;
    if ( M != LM ) {
	if ( ad )
	    free( ad ) ;
	ad = (complex *) malloc( (M+1)*sizeof(complex) ) ;
	LM = M ;
    }
/*
 * make temp copy of polynomial coefficients
 */
    for ( i = 0 ; i <= M ; i++ )
	ad[i] = a[i] ;
/*
 * use Laguerre's method to estimate each root
 */
    for ( j = M ; j > 0 ; j-- ) {
	x = zero ;
	x = laguerre( ad, j, x, eps, 0 ) ;
	if ( fabs( x.im ) <= pow( 2.*eps, 2.*fabs( x.re ) ) )
	    x.im = 0. ;
	r[j-1] = x ;
/*
 * factor each root as it is found out of the polynomial
 * using synthetic division
 */
	b = ad[j] ;
	for ( jj = j - 1 ; jj >= 0 ; jj-- ) {
	    c = ad[jj] ;
	    ad[jj] = b ;
	    b = cadd( cmult( x, b ), c ) ;
	}
    }
/*
 * polish each root, (i.e., improve its accuracy)
 * also by using Laguerre's method
    for ( j = 0 ; j < M ; j++ )
	r[j] = laguerre( a, M, r[j], eps, 1 ) ;
 */
/*
 * sort roots by their real parts
 */
    for ( i = 0 ; i < M-1 ; i++ ) {
	for ( j = i + 1 ; j < M ; j++ ) {
	    if ( r[j].re < r[i].re ) {
		x = r[i] ;
		r[i] = r[j] ;
		r[j] = x ;
	    }
	}
    }
}
/*
 * polynomial is in a[] in the form
 *         a[0] + a[1]*x + a[2]*x^2 + ... + a[M]*x^M
 * if P is 0, laguerre attempts to return a root to
 * within eps of its value, given an initial guess x;
 * if P is nonzero, eps is ignored and laguerre attempts
 * to improve the guess x to within the achievable
 * roundoff limit, specified as "tiny"
 */
complex laguerre( a, M, x, eps, P )
 complex a[], x ; float eps ; int M, P ;
{
 complex dx, x1, b, d, f, g, h, mh, sq, gp, gm, g2, q ;
 int i, j, npol ;
 float dxold, cdx, tiny = 1.e-15 ;
    if ( P ) {
	dxold = CABS( x ) ;
	npol = 0 ;
    }
/*
 * iterate up to 100 times
 */
    for ( i = 0 ; i < 100 ; i++ ) {
	b = a[M] ;
	d = zero ;
	f = zero ;
/*
 * compute polynomial and its first two derivatives
 */
	for ( j = M-1 ; j >= 0 ; j-- ) {
	    f = cadd( cmult( x, f ), d ) ;
	    d = cadd( cmult( x, d ), b ) ;
	    b = cadd( cmult( x, b ), a[j] ) ;
	}
	if ( CABS( b ) <= tiny )      /* are we on the root? */
	    dx = zero ;
	else if ( CABS( d ) <= tiny && CABS( f ) <= tiny ) {
	    q = cdiv( b, a[M] ) ;  /* this is a special case */
	    dx.re = pow( CABS( q ), 1./M ) ;
	    dx.im = 0. ;
	} else {          /* general case: Laguerre's method */
	    g = cdiv( d, b ) ;
	    g2 = cmult( g, g ) ;
	    h = csub( g2, scmult( 2., cdiv( f, b ) ) ) ;
	    sq = csqrt( 
		scmult( (float) M-1,
		    csub( scmult( (float) M, h ), g2 )
		)
	    ) ;
	    gp = cadd( g, sq ) ;
	    gm = csub( g, sq ) ;
	    if ( CABS( gp ) < CABS( gm ) )
		gp = gm ;
	    q.re = M ; q.im = 0. ;
	    dx = cdiv( q, gp ) ;
	}
	x1 = csub( x, dx ) ;
	if ( x1.re == x.re && x1.im == x.im )
	    return( x ) ;                  /* converged */
	x = x1 ;
	if ( P ) {
	    npol++ ;
	    cdx = CABS( dx ) ;
	    if ( npol > 9 && cdx >= dxold )
		return( x ) ; /* reached roundoff limit */
	    dxold = cdx ;
	} else 
	    if ( CABS( dx ) <= eps*CABS( x ) )
		return( x ) ;              /* converged */
    }
    errmess( "root (pole) convergence failure" ) ;
    return( x ) ;                          /* best we could do */
}
/*
 * complex arithmetic routines
 */
complex cadd( x, y ) complex x, y ; { /* return x + y */
 static complex z ;
    z.re = x.re + y.re ;
    z.im = x.im + y.im ;
    return( z ) ;
}
complex csub( x, y ) complex x, y ; { /* return x - y */
 static complex z ;
    z.re = x.re - y.re ;
    z.im = x.im - y.im ;
    return( z ) ;
}
complex cmult( x, y ) complex x, y ; { /* return x*y */
 static complex z ;
    z.re = x.re*y.re - x.im*y.im ;
    z.im = x.re*y.im + x.im*y.re ;
    return( z ) ;
}
complex scmult( s, x ) float s ; complex x ; { /* return s*x */
 static complex z ;
    z.re = s*x.re ;
    z.im = s*x.im ;
    return( z ) ;
}
complex cdiv( x, y ) complex x, y ; { /* return x/y */
 static complex z ;
 float mag, ang ; /* polar arithmetic more robust here */
    mag = CABS( x )/CABS( y ) ;
    if ( x.re != 0. && y.re != 0. )
	ang = atan2( x.im, x.re) - atan2( y.im, y.re) ;
    else
	ang = 0. ;
    z.re = mag*cos( ang ) ;
    z.im = mag*sin( ang ) ;
    return( z ) ;
}
complex conjg( x ) complex x ; { /* return x* */
 static complex y ;
    y.re = x.re ;
    y.im = -x.im ;
    return( y ) ;
}
complex csqrt( x ) complex x ; { /* return sqrt(x) */
 static complex z ;
 float mag, ang ;
    mag = sqrt( CABS( x ) ) ;
    if ( x.re != 0. )
	ang = atan2( x.im, x.re)/2. ;
    else
	ang = 0. ;
    z.re = mag*cos( ang ) ;
    z.im = mag*sin( ang ) ;
    return( z ) ;
}
