/*
** Roland mpu-401 interface driver
** Original by Gareth Loy, UCSD
** Hacked by psl 5/86
*/

#include "mpu.h"
#if NMPU > 0

#include "../h/param.h"
#include "../h/systm.h"
#include "../h/ioctl.h"
#include "../h/tty.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/proc.h"
#include "../h/buf.h"
#include "../h/file.h"
#include "../h/map.h"
#include "../h/uio.h"

#ifdef sun2
#include "../sun2/psl.h"
#include "../sun2/mmu.h"
#include "../sun2/pte.h"
#else sun2
#include "../sun3/psl.h"
#include "../sun3/mmu.h"
#include "../sun3/pte.h"
#endif sun2
#include "../sundev/mbvar.h"

#include "../sundev/mpuvar.h"
#include "../sundev/mpureg.h"

#define dprintf  if (MPUdebug) printf
int	MPUdebug	= 0;
int	MPUdebugreset	= 0;	/**** only used on sun3s ****/
int	MPUblox		= 0;	/**** number of blocks we're holding ****/
int	MPUhwm		= 0;	/**** high-water mark of mpublox ****/
int	MPUlost		= 0;	/**** blocks lost (and reported) ****/
int	mpu_ndelay	= 1000;	/**** only used on sun3s ****/

# define MPUBUFSIZ	(4*CBSIZE)
# define MPU_NLOOPS	1000

# define MPUDELAY(n)	while(mpur->mpur_stat&MPU_STAT_DRR)DELAY(n);DELAY(9*(n))

# define MPU_UNIT(dev)	(minor(dev))

# define MPU_PRI_HI	(PZERO)		/* not interruptable	*/
# define MPU_PRI_LOW	(PZERO+1)	/* interruptable	*/

/* states of mpu */
# define MPUD_CLOSE	0		/* turn off all bits	*/
# define MPUD_OPEN	bit(0)		/* open			*/
# define MPUD_ACK	bit(1)		/* ack'd a command	*/
# define MPUD_WACK	bit(2)		/* sleeping on ack	*/
# define MPUD_MPU_DATA	bit(3)		/* command w/mpu data	*/
# define MPUD_MIDI_DATA	bit(4)		/* command w/midi data	*/
# define MPUD_TT	bit(5)		/* time tag data	*/
# define MPUD_SMESS	bit(6)		/* system message	*/
# define MPUD_ERROR	bit(7)		/* data format error	*/
# define MPUD_SELWAIT	bit(8)		/* select waiting	*/

extern unsigned char	mpugetbyte();
extern struct mpu_l	*mpugetl();

extern int		mpuprobe();
extern int		mpuattach();
extern int		mpupoll();

struct mb_device	*mpudinfo[NMPU];

struct mb_driver	mpudriver = {
	mpuprobe, 0, 0, 0, 0,
	mpupoll,
	sizeof(struct mpu_reg), "mpu", mpudinfo,
	0, 0, 0, 0
};

/*
** The following struct is used for data from the user that is to be sent
** to the mpu.
** Play and conductor data is arranged as linked lists (1 per track) of
** system buffers.  Each time the user issues a write(), the data is copied
** into a system buffer and that buffer is linked onto the track list.
** Since the mpu issues an interrupt each time it needs more play data,
** the play data is asynchronous.
** Mpu command data is synchronous; the user process blocks until the mpu
** command data has all been sent to the mpu.
** It's been a while since I wrote this driver and at the moment I'm wondering
** why mpu command data couldn't also be asynchronous.
*/
struct mpu_l {
	struct buf	mpul_head;	/* head of buffer list		*/
	int		mpul_tcount;	/* number of bytes in track	*/
	int		mpul_bcount;	/* number of bytes in buffer	*/
	int		mpul_contin;	/* number of bytes in continuation */
	unsigned char	*mpul_cp;	/* byte pointer into buffer	*/
};

/*
** data and state information for the mpu.
*/
struct mpu_device {
	struct mpu_l	mpud_trackl[MPU_TTR_MAX];
					/* data, com, & cond track lists */
	struct clist	mpud_iq;	/* incoming data		*/
	struct proc	*mpud_sel;	/* select pending		*/
	int		mpud_state;	/* state of mpu			*/
	int		mpud_cnt;	/* # of bytes for mpuintr	*/
	int		mpud_contin;	/* continuation bytes (mpuintr)	*/
	char		mpud_track;	/* current active track		*/
	char		mpud_busy;	/* non-zero if device busy	*/
} mpu_device[NMPU];

int	*MPUstataddr	= &mpu_device[0].mpud_state;	/**** laziness ****/

/*
** Mpuprobe determines whether the mpu really exists.
** It is possible that the interface card is plugged into the computer
** but that no mpu is connected to the card, so we send a reset command
** to the mpu and see if we get an ack back.
*/
/*ARGSUSED*/
mpuprobe(reg, unit)
caddr_t	reg;
int	unit;
{
	register struct mpu_reg	 *mpur;
	register int ch, i;

	mpur = (struct mpu_reg *) reg;
#ifdef sun2
	if (peekc((char *) &mpur->mpur_stat) == -1) {
#else sun2
	if (peek((char *) &mpur->mpur_stat) == -1) {
#endif sun2
		printf("mpu%d: no stat reg @ 0x%x\n", unit, &mpur->mpur_stat);
		return(0);
	}
	for (i = MPU_NLOOPS; --i > 0; ) {		/* wait for DRR */
#ifdef sun2
		if ((UCH(peekc((char *) &mpur->mpur_stat)) & MPU_STAT_DRR) == 0)
#else sun2
		if ((UCH(peek((char *) &mpur->mpur_stat)) & MPU_STAT_DRR) == 0)
#endif sun2
			break;
		DELAY(100);
	}
	if (i == 0) {
		printf("mpu%d: no DRR in stat @0x%x\n", unit, &mpur->mpur_stat);
		return(0);
	}
#ifdef sun2
	if (pokec((char *) &mpur->mpur_com, MPU_COM_RESET)) {	/* reset mpu */
#else sun2
	if (poke((char *) &mpur->mpur_com, MPU_COM_RESET)) {	/* reset mpu */
#endif sun2
		printf("mpu%d: no com reg @0x%x?\n", unit, &mpur->mpur_com);
		return(0);
	}
	for (i = MPU_NLOOPS; --i > 0; ) {		/* wait for DSR */
#ifdef sun2
		if ((UCH(peekc((char *) &mpur->mpur_stat)) & MPU_STAT_DSR) == 0)
#else sun2
		if ((UCH(peek((char *) &mpur->mpur_stat)) & MPU_STAT_DSR) == 0)
#endif sun2
			break;
		DELAY(100);
	}
	if (i == 0) {
		printf("mpu%d: no DSR in stat @0x%x\n", unit, &mpur->mpur_stat);
		return(0);
	}
#ifdef sun2
	if ((ch = peekc((char *) &mpur->mpur_data)) == -1) {	/* read data */
#else sun2
	if ((ch = peek((char *) &mpur->mpur_data)) == -1) {	/* read data */
#endif sun2
		printf("mpu%d: no data reg @0x%x?\n", unit, &mpur->mpur_data);
		return(0);
	}
	if (UCH(ch) != MPU_MESS_ACK) {
		printf("mpu%d: reset didn't get ack (0x%x) got 0x%x\n",
		 unit, MPU_MESS_ACK, UCH(ch));
		return(0);
	}
	return(0x100); /* not sizeof(struct mpu_reg) due to interface design */
}

/*ARGSUSED*/
mpuopen(dev, flag)
dev_t	dev;
int	flag;
{
	register struct mpu_device *mpud;
	register struct mb_device *md;
	register struct buf *bp;
	register int unit, i;

dprintf("mpuopen(0x%x, %d)\n", dev, flag);
	if ((unit = MPU_UNIT(dev)) >= NMPU	/* check minor device number */
	 || (md = mpudinfo[unit]) == 0
	 || md->md_alive == 0)
		return(ENXIO);
	mpud = &mpu_device[unit];
	if ((mpud->mpud_state & MPUD_OPEN) == MPUD_OPEN && u.u_uid != 0)
		return(EEXIST);			/* exclusive use */
	mpud->mpud_state = MPUD_OPEN;		/* mark it open */
	mpud->mpud_track = MPU_CTR_COM;		/* command track by default */
	mpud->mpud_cnt = 0;			/* nothing sent yet */
	for (i = MPU_TTR_MAX; --i >= 0; ) {	/* init track buffer lists */
		bp = &mpud->mpud_trackl[i].mpul_head;
		bp->av_forw = bp;
		bp->av_back = bp;
	}
	return(0);
}

/*ARGSUSED*/
mpuclose(dev, flag)
dev_t	dev;
int	flag;
{
	register struct mpu_device *mpud;
	register int unit, i;
	int err, opri;

dprintf("mpuclose(0x%x, %d)\n", dev, flag);
	unit = MPU_UNIT(dev);
	mpud = &mpu_device[unit];
	opri = splx(pritospl(mpudinfo[unit]->md_intpri));
	for (;;) {		/* check if any tracks are still running */
	    for (i = MPU_TTR_MAX; --i >= 0; )
		if (mpud->mpud_trackl[i].mpul_tcount != 0)
		    break;
	    if (i < 0)
		break;
	    dprintf("mpuclose: need to wait for track %d\n", i);
	    if (mpusleep((caddr_t) &mpud->mpud_track, MPU_PRI_LOW)) {
		mpupurge(mpud);			/* ^C will get us here */
		break;				/* we hope */
	    }
	}
	splx(opri);
	if (!mpureset(unit))			/* send a reset command */
		printf("mpu%d: can't reset mpu\n", unit);
	ndflush(&mpud->mpud_iq, mpud->mpud_iq.c_cc);	/* flush input clist */
	err = mpud->mpud_state & MPUD_ERROR;
	mpud->mpud_state = MPUD_CLOSE;
	if (MPUblox > MPUlost) {
		printf("mpuclose: lost block count went from %d to %d\n",
		 MPUlost, MPUblox);
		MPUlost = MPUblox;
	}
	return(err? EINVAL : 0);
}

/*
** Record data is copied into a clist.  Clists aren't very big; I've wondered
** if it is possible for a verbose synth to use up all of the system's clists.
*/
mpuread(dev, uio)
dev_t	dev;
register struct uio *uio;
{
	register struct mpu_device *mpud;
	register struct mb_device *md;
	char mpurdbuf[MPUBUFSIZ];
	int unit, err, opri, mincnt, cc;

dprintf("mpuread(0x%x, 0x%x)\n", dev, uio);
	if ((unit = MPU_UNIT(dev)) >= NMPU)
		return(ENXIO);
	md = mpudinfo[unit];
	mpud = &mpu_device[unit];
	err = 0;
	/*
	** while the user needs more data and there haven't been any errors
	** get the data and send it to user.
	*/
	while ((uio->uio_resid > 0) && (err == 0)) {
		opri = splx(pritospl(md->md_intpri));
		while (mpud->mpud_iq.c_cc <= 0)
			sleep((caddr_t) &mpud->mpud_iq, MPU_PRI_LOW);
		splx(opri);
		mincnt = MIN(uio->uio_resid, MPUBUFSIZ);
		/*
		** q_to_b() transfers bytes from a clist (a "q") to a byte
		** array (a "b"uffer) in kernel memory.  We then use uiomove()
		** to transfer the data from kernel space to user space.
		*/
		if ((cc = q_to_b(&mpud->mpud_iq, mpurdbuf, mincnt)) == 0)
			break;
		err = uiomove(mpurdbuf, cc, UIO_READ, uio);
	}
	return(err);
}

mpuwrite(dev, uio)
dev_t	dev;
register struct uio *uio;
{
	register struct mpu_device *mpud;
	register struct mpu_l *mpul;
	register struct buf *bp, *hp;
	register int unit;
	int err, cnt, opri, bsiz;
	caddr_t addr;
	extern int mpustrategy();
	/* extern unsigned int minphys(); */
	void minphys();
	extern struct buf *geteblk();

dprintf("mpuwrite(0x%x, 0x%x)\n", dev, uio);
	if ((unit = MPU_UNIT(dev)) >= NMPU)
		return(ENXIO);
	mpud = &mpu_device[unit];
	mpul = mpugetl(mpud);
	hp = &mpul->mpul_head;
	/*
	** Mpu commands are handled differently from play data.
	** Commands are immediately sent to the mpu with no delay or queueing.
	*/
	if (mpul == &mpud->mpud_trackl[MPU_CTR_COM])
		return(physio(mpustrategy, hp, dev, B_WRITE, minphys, uio));
	/*
	** Play data is handled as a linked list of buffers.  First we see if
	** this request will fit in the last buffer, if not we get an empty
	** system buffer; then we copy the data from user space to kernel
	** space; finally, we link the buffer onto the track list if it's new.
	*/
	cnt = uio->uio_iov->iov_len;
	opri = splx(pritospl(mpudinfo[unit]->md_intpri));
	if ((bp = hp->av_back) == hp || bp->b_bufsize - bp->b_bcount < cnt) {
dprintf("mpuwrite: new buffer for %d byte(s)\n", cnt);
		bsiz = ((uio->uio_iov->iov_len + 0x03FF) & 0xFC00);
		bp = geteblk(bsiz);			/* mult of 1024 */
/****/MPUblox++; if (MPUblox > MPUhwm) MPUhwm = MPUblox;
		addr = bp->b_un.b_addr;
		if (!(err = uiomove(bp->b_un.b_addr, cnt, UIO_WRITE, uio))) {
			bp->b_bcount = cnt;
			mpul->mpul_tcount += cnt;
			if (hp->av_back == hp) {	/* the only buffer */
				mpul->mpul_cp = (unsigned char *) addr;
				mpul->mpul_bcount = bp->b_bcount;
			}
			hp->av_back->av_forw = bp;
			bp->av_back = hp->av_back;
			hp->av_back = bp;
			bp->av_forw = hp;
		}
	} else {
dprintf("mpuwrite: %d byte(s) added to old buffer\n", cnt);
		addr = bp->b_un.b_addr + bp->b_bcount;
		if (!(err = uiomove(addr, cnt, UIO_WRITE, uio))) {
			bp->b_bcount += cnt;
			mpul->mpul_tcount += cnt;
			if (bp == hp->av_forw)	/* current buffer */
				mpul->mpul_bcount += cnt;
		}
	}
	splx(opri);
	return(err);
}

/*ARGSUSED*/
mpuioctl(dev, cmd, data, flag)
dev_t	dev;
int	cmd, flag;
caddr_t	data;
{
        register struct mpu_device *mpud;
	register struct mb_device *md;
	register int track, opri, i;

dprintf("mpuioctl(0x%x, 0x%x, 0x%x, %d)\n", dev, cmd, data, flag);
	md = mpudinfo[MPU_UNIT(dev)];
	mpud = &mpu_device[MPU_UNIT(dev)];
	if (cmd == MPU_IOC_TRACK) {		/* set current track */
		if (0 <= (track = *(int *) data) && track < MPU_TTR_MAX)
			mpud->mpud_track = track;
		else
			return(EINVAL);
	} else if (cmd == MPU_IOC_PURGE) {	/* purge all track buffers */
		mpupurge(mpud);
	} else if (cmd == MPU_IOC_RESID) {	/* get resid count of tracks */
		for (i = MPU_DTR_NUM; --i >= 0; )
			((mpu_resid *) data)->mr_tresid[i]
			 = mpud->mpud_trackl[i].mpul_bcount;
		((mpu_resid *) data)->mr_cresid
		 = mpud->mpud_trackl[MPU_CTR_CND].mpul_bcount;
	} else if (cmd == FIONREAD) {		/* return # bytes unread */
		opri = splx(pritospl(md->md_intpri));
		*(int *) data = mpud->mpud_iq.c_cc;
		splx(opri);
	} else
		return(ENOTTY);
	return(0);
}

/* STILL NO WORKEE */
mpuselect(dev, flag)
dev_t	dev;
int	flag;
{
        register struct mpu_device *mpud;
	register struct mb_device *md;
	int opri;

dprintf("mpuselect(0x%x, %d)\n", dev, flag);
	mpud = &mpu_device[MPU_UNIT(dev)];
	md = mpudinfo[MPU_UNIT(dev)];
	if (flag == FREAD) {
		opri = splx(pritospl(md->md_intpri));
		if (mpud->mpud_iq.c_cc > 0) {
			splx(opri);
			return(1);
		}
		if ((mpud->mpud_sel != 0)
		 && (mpud->mpud_sel->p_wchan == (caddr_t) &selwait))
			mpud->mpud_state |= MPUD_SELWAIT;
		else
			mpud->mpud_sel = u.u_procp;
		splx(opri);
		return(0);
	} else if (flag == FWRITE) {
		return(1);
	} else
		return(0);
}

/*ARGSUSED*/
mpummap(dev, off, prot)
dev_t	dev;
off_t	off;
int	prot;
{
dprintf("mpummap(0x%x, %d, %d)\n", dev, off, prot);
	if (off)
		return(-1);
	return(getkpgmap(mpudinfo[MPU_UNIT(dev)]->md_addr) & PG_PFNUM);
}

mpupoll()
{
	register struct mb_device *md;
	register struct mpu_reg	 *mpur;
	register struct mpu_device *mpud;
	register int unit;
	unsigned char data;
	int didint;

	didint = 0;
	for (unit = 0; unit < NMPU; unit++) {
		if ((md = mpudinfo[unit]) == 0 || md->md_alive == 0)
			continue;
		mpud = &mpu_device[unit];
		mpur = (struct mpu_reg *) mpudinfo[unit]->md_addr;
		while ((mpur->mpur_stat & MPU_STAT_DSR) == 0) {
			didint = 1;
			if (mpud->mpud_state & MPUD_OPEN)	/* real */
				mpuintr(unit);
			else {					/* spurious */
				data = mpur->mpur_data;		/* suck crud */
				printf("mpupoll: spurious, unit %d\n", unit);
			}
		}
	}
	return(didint);
}

mpuintr(unit)
register int unit;
{
	register struct mpu_reg	 *mpur;
	register struct mpu_device *mpud;
	unsigned char data;

dprintf ("mpuintr(%d)\t", unit);
	mpud = &mpu_device[unit];
	mpur = (struct mpu_reg *) mpudinfo[unit]->md_addr;
	data = mpur->mpur_data;
	if (mpud->mpud_state & MPUD_TT) {	/* If prev byte was time tag */
		if (data == UCH(0xf8)		/* determine how many follow */
		 || data == UCH(0xf9)
		 || data == UCH(0xfc))
			mpud->mpud_cnt = 1;
		else {
			/*
			** mpud_contin is used to keep track of how many bytes
			** to expect with running status.
			*/
			if (data < UCH(0x80))		/* running status */
				mpud->mpud_cnt = mpud->mpud_contin - 1;
			else {				/* explicit status */
				mpud->mpud_cnt = mpumidicnt(data);
				mpud->mpud_contin = mpud->mpud_cnt;
			}
		}
		mpud->mpud_state &= ~MPUD_TT;
	}
	if (mpud->mpud_state & MPUD_SMESS) {	/* If prev byte was Sys Mess */
		mpud->mpud_cnt = mpumidicnt(data);   /* find how many follow */
		mpud->mpud_state &= ~MPUD_SMESS;
	}
	/*
	** By now mpud_cnt has been set up.
	**  >0 ==> normal MIDI data
	**  -1 ==> sys excl, pass bytes until 0xF7 found
	**   0 ==> between commands, use mpuisubr to decode new one
	*/
	if (mpud->mpud_cnt > 0) {		/* normal midi data */
		if (putc(data, &mpud->mpud_iq) == -1)
			printf("mpu: putc(0x%x) failed\n", data);
		mpud->mpud_cnt--;
	} else if (mpud->mpud_cnt == -1) {	/* system exclusive midi data */
		if (putc(data, &mpud->mpud_iq) == -1)
			printf("mpu: putc(0x%x) failed\n", data);
		if (data == UCH(0xf7))
			mpud->mpud_cnt = 0;
	} else {				/* starting a new command */
		if (mpuisubr(mpud, data, unit))
			if (putc(data, &mpud->mpud_iq) == -1)
				printf("mpu: putc(0x%x) failed\n", data);
	}
	wakeup((caddr_t) &mpud->mpud_iq);
	if (mpud->mpud_sel) {
		selwakeup(mpud->mpud_sel, mpud->mpud_state & MPUD_SELWAIT);
		mpud->mpud_state &= ~MPUD_SELWAIT;
		mpud->mpud_sel = 0;
	}
}

/*
** mpuisubr, called by mpuintr to decode a MIDI command byte.
*/
mpuisubr(mpud, data, unit)
register struct mpu_device *mpud;
unsigned char data;
int	unit;
{
	int touser = 0;

dprintf("mpuisubr(0x%x, 0x%x, %d)\n", mpud, data, unit);
	/*
	** If the upper nibble is 0xF, it's an mpu message or a data request.
	** Otherwise it's a time tag.  Some things get sent back to the user,
	** some don't; we use touser to keep track of what gets sent back.
	*/
	if ((data & UCH(0xf0)) == UCH(0xf0)) {
	    if (data & bit(3)) {
		switch (data) {
		case MPU_MESS_CONDREQ:	/* 0xf9, conductor request */
		    mpusendcond(mpud, unit);
		    break;
		case MPU_MESS_CASSREQ:	/* 0xfa, cassette data req */
		    printf("mpuisubr: 0xfa\n");
		    break;
		case MPU_MESS_ACK:	/* 0xfe, command ack */
		    if (mpud->mpud_state & MPUD_WACK) {
			mpud->mpud_state |= MPUD_ACK;
			wakeup((caddr_t) &mpud->mpud_trackl[MPU_CTR_COM]);
		    }
		    break;
		default:
		/*
		** timing overflow		(0xf8)
		** cassette data received	(0xfb)
		** all end			(0xfc)
		** clock to host		(0xfd)
		** system message		(0xff)
		*/
		    if (data == UCH(0xff))
			mpud->mpud_state |= MPUD_SMESS;
		    touser = 1;
		    break;
		}
	    } else			/* track data request */
		mpusendtrack((int) (data & UCH(0x7)), unit);
	} else {				/* time tag */
	    mpud->mpud_state |= MPUD_TT;
	    touser = 1;
	}
	return(touser);
}

mpustrategy(bp)		/* only used for the command track */
register struct buf *bp;
{
	register struct mb_device *md;
	register struct mpu_device *mpud;
	register struct mpu_l *mpul;
	int opri;

dprintf("mpustrategy(0x%x)\n", bp);
	mpud = &mpu_device[MPU_UNIT(bp->b_dev)];
	md = mpudinfo[MPU_UNIT(bp->b_dev)];
	opri = splx(pritospl(md->md_intpri));
	while (mpud->mpud_busy)
		sleep((caddr_t) &mpud->mpud_busy, MPU_PRI_HI);
	mpud->mpud_busy = 1;
	mpul = &mpud->mpud_trackl[MPU_CTR_COM];
	mpul->mpul_cp = (unsigned char *) bp->b_un.b_addr;
	mpul->mpul_bcount = bp->b_bcount;
	mpusendcom(MPU_UNIT(bp->b_dev));
	mpud->mpud_busy = 0;
	wakeup((caddr_t) &mpud->mpud_busy);
	splx(opri);
}

/*
** mpusendcom, used to send the next command data byte to the mpu.
*/
mpusendcom(unit)
{
	register struct mpu_device *mpud;
	register struct mpu_l *mpul;
	register struct mb_device *md;
	register struct mpu_reg	 *mpur;
	unsigned char byte;
	int opri;

	md = mpudinfo[unit];
	mpud = &mpu_device[unit];
	mpur = (struct mpu_reg *) mpudinfo[unit]->md_addr;
	mpul = &mpud->mpud_trackl[MPU_CTR_COM];
	while (mpul->mpul_bcount > 0) {
		MPUDELAY(100);
		byte = *mpul->mpul_cp;
		/*
		** If we're sending midi data we use mpul_contin to keep track
		** of how many bytes to send.  We want to avoid what Roland
		** calls "looping problem", in which we send midi data out as
		** an mpu command, instead of playing it as track data.
		*/
		if (mpud->mpud_state & MPUD_MIDI_DATA) {
			if ((mpul->mpul_contin = mpumidicnt(byte)) == 0) {
				printf("mpusendcom: mpumidicnt=0\n");
				mpud->mpud_state |= MPUD_ERROR;
				mpul->mpul_bcount = 0;
				break;
			}
			mpud->mpud_state &= ~MPUD_MIDI_DATA;
		}
		if (mpud->mpud_state & MPUD_MPU_DATA) {	/* data for mpu cmd */
dprintf("mpusendcom: mpu data 0x%x\n", byte);
			mpur->mpur_data = byte;
			mpud->mpud_state &= ~MPUD_MPU_DATA;
		} else if (mpul->mpul_contin > 0) {	/* MIDI data */
dprintf("mpusendcom: MIDI data 0x%x\n", byte);
			mpur->mpur_data = byte;
			mpul->mpul_contin--;
		} else if (mpul->mpul_contin == -1) {	/* sys excl data */
dprintf("mpusendcom: sys excl data 0x%x\n", byte);
			mpur->mpur_data = byte;
			if (byte == UCH(0xf7))
				mpul->mpul_contin = 0;
		} else {				/* start of mpu cmd */
dprintf("mpusendcom: mpu cmd 0x%x\n", byte);
			mpur->mpur_com = byte;
			/* wait for ack from the command */
			opri = splx(pritospl(md->md_intpri));
			mpud->mpud_state &= ~MPUD_ACK;
			while ((mpud->mpud_state & MPUD_ACK) == 0) {
				mpud->mpud_state |= MPUD_WACK;
				sleep((caddr_t) &mpud->mpud_trackl[MPU_CTR_COM],
				 MPU_PRI_HI);
			}
			mpud->mpud_state &= ~MPUD_WACK;
			splx(opri);
			if ((byte & UCH(0xf0)) == UCH(0xe0)) /* need mpu data */
				mpud->mpud_state |= MPUD_MPU_DATA;
			else if ((byte & UCH(0xf0)) == UCH(0xd0))/* midi data */
				mpud->mpud_state |= MPUD_MIDI_DATA;
		}
		mpul->mpul_bcount--;
		mpul->mpul_cp++;
	}
	if (mpul->mpul_bcount == 0)	/* all command i/o done for now */
		biodone(&mpul->mpul_head);
}

/*
** mpusendtrack is used to send the next byte
** from a track of play data.
*/
mpusendtrack(track, unit)
{
	register struct mpu_reg	 *mpur;
	register struct mpu_device *mpud;
	register struct mpu_l *mpul;
	register int nbytes, i;
	unsigned char byte;

	if ((track < 0) || (track > 7))
		panic("mpusendtrack");
	mpud = &mpu_device[unit];
	mpur = (struct mpu_reg *) mpudinfo[unit]->md_addr;
	mpul = &mpud->mpud_trackl[track];
	if (mpul->mpul_tcount <= 0) {			/* nothing to send */
		MPUDELAY(100);
		mpur->mpur_data = 0;			/* timing byte */
dprintf("mpusendtrack: sending 0xfc\n");
		MPUDELAY(100);
		mpur->mpur_data = UCH(0xfc);		/* send end of track */
		wakeup((caddr_t) &mpud->mpud_track);	/* for mpuclose() */
		return;
	}
	if ((byte = mpugetbyte(mpul, 0)) == UCH(0xf8)) {
dprintf("mpusendtrack: sending 0xf8\n");
		MPUDELAY(100);			/* send 0xF8s (TCIP) as is */
		mpur->mpur_data = mpugetbyte(mpul, 1);
		return;
	}
dprintf("mpusendtrack: sending 0x%x\n", UCH(mpugetbyte(mpul, 0)));
	MPUDELAY(100);
	mpur->mpur_data = mpugetbyte(mpul, 1);		/* time tag for mpu */
	if (mpul->mpul_tcount <= 0) {
dprintf("mpusendtrack: sending 0xfc\n");
		MPUDELAY(100);			/* time tag with no MIDI? */
		mpur->mpur_data = UCH(0xfc);		/* send end of track */
		wakeup((caddr_t) &mpud->mpud_track);	/* for mpuclose() */
		return;
	}
	/*
	** Figure out how many bytes to send by examing the current byte in
	** the track.  Here again, we use mpul_contin to keep track of how
	** many bytes to send when the mpu uses running status.
	*/
	byte = mpugetbyte(mpul, 0);
	if (byte & bit(7)) {
		if (byte == UCH(0xf8)			/* no op */
		 || byte == UCH(0xf9)			/* measure end mark */
		 || byte == UCH(0xfc))			/* data end mark */
			nbytes = 1;
		else {					/* midi command */
			nbytes = mpumidicnt(byte);
			mpul->mpul_contin = nbytes - 1;
		}
	} else						/* running status */
		nbytes = mpul->mpul_contin;
	/*
	** Now send the data
	*/
	if (nbytes == -1) {				/* system exclusive */
		/*
		** If the user doesn't terminate the sys excl with 0xf7, we
		** we could get stuck in this loop.  To guard against this,
		** we check mpul_tcount and if the track is empty we bail out.
		*/
		while (mpul->mpul_tcount > 0) {
			byte = mpugetbyte(mpul, 1);
dprintf("mpusendtrack: sending 0x%x\n", UCH(byte));
			MPUDELAY(100);
			mpur->mpur_data = byte;
			/*
			** Anything with bit 7 on can terminate sys excl, but
			** if it is a real-time message we stay in sys excl.
			*/
			if ((byte & bit(7)) && byte < UCH(0xf8))
				break;
		}
	} else {
		/* We guard against expecting more bytes than there are. */
		for (i = 0; i < nbytes && mpul->mpul_tcount > 0; i++) {
dprintf("mpusendtrack: sending 0x%x\n", UCH(mpugetbyte(mpul, 0)));
			MPUDELAY(100);
			mpur->mpur_data = mpugetbyte(mpul, 1);
		}
	}
	if (mpul->mpul_tcount <= 0)
		wakeup((caddr_t) &mpud->mpud_track);	/* for mpuclose() */
}

/*
** mpusendcond, used to send conductor data.
** It's similar to, but not as complicated as, mpusendtrack.
*/
mpusendcond(mpud, unit)
register struct mpu_device *mpud;
{
	register struct mpu_l *mpul;
	register struct mpu_reg *mpur;
	register int nbytes, i;
	unsigned char byte;

dprintf("mpusendcond(0x%x, %d)\n", mpud, unit);
	mpur = (struct mpu_reg *) mpudinfo[unit]->md_addr;
	mpul = &mpud->mpud_trackl[MPU_CTR_CND];
	if (mpul->mpul_tcount <= 0) {		/* nothing to send */
		MPUDELAY(100);
		mpur->mpur_data = 0;		/* send a timing byte */
		MPUDELAY(100);
		mpur->mpur_data = UCH(0xfc);	/* data end */
		wakeup((caddr_t) &mpud->mpud_track);	/* for mpuclose() */
		return;
	}
	byte = mpugetbyte(mpul, 1);		/* time tag */
	MPUDELAY(100);
	mpur->mpur_data = byte;
	if (byte == UCH(0xf8))
		return;
	if (mpul->mpul_tcount <= 0) {		/* just in case */
		MPUDELAY(100);
		mpur->mpur_data = UCH(0xfc);	/* data end */
		wakeup((caddr_t) &mpud->mpud_track);	/* for mpuclose() */
		return;
	}
	byte = mpugetbyte(mpul, 0);
	nbytes = ((UCH(0xe0) <= byte) && (byte <= UCH(0xef)))? 2 : 1;
	for (i = nbytes; --i >= 0 && mpul->mpul_tcount > 0; ) {
		MPUDELAY(100);
		mpur->mpur_data = mpugetbyte(mpul, 1);
	}
}

/*
** mpugetbyte returns the next byte for a track.
** Track data is arranged as a linked list of system buffers.
** The update arg specifies whether to advance to the next byte.
*/
unsigned char
mpugetbyte(mpul, update)
register struct mpu_l *mpul;
{
	register struct buf *bp;
	unsigned char byte;

	if ((bp = mpul->mpul_head.av_forw) == &mpul->mpul_head
	 || mpul->mpul_bcount <= 0)
	    return(UCH(0xfc));		/* no more buffs */
	byte = *mpul->mpul_cp;
	if (update) {
	    --mpul->mpul_tcount;
	    mpul->mpul_cp++;
	    if (--mpul->mpul_bcount <= 0) {		/* setup next buf */
		bp->av_back->av_forw = bp->av_forw;
		bp->av_forw->av_back = bp->av_back;
		brelse(bp);
/****/if (--MPUblox < 0) printf("MPUblox is %d???\n", MPUblox);
		if ((bp = mpul->mpul_head.av_forw) != &mpul->mpul_head) {
		    mpul->mpul_cp = (unsigned char *) bp->b_un.b_addr;
		    mpul->mpul_bcount = bp->b_bcount;
		}
	    }
	}
	return(byte);
}

/*
** mpumidicnt examines the midi command "byte" and
** returns the number of bytes that should follow it.
*/
mpumidicnt(byte)
unsigned char byte;
{
	register int status;

	if (byte < UCH(0x80))
		printf("mpumidicnt: byte=%x\n", byte);	/* "can't happen" */
	status = (byte & UCH(0xF0));
	if (status <= UCH(0xB0)  /* note off|on, after touch, control change */
	 || status == UCH(0xe0))	/* pitch wheel */
		return(3);
	if (status == UCH(0xc0)		/* program change */
	 || status == UCH(0xd0))	/* change after touch */
		return(2);
	if (status == UCH(0xf0)) {	/* system messages */
		if ((byte & UCH(0x0f)) == 0)	/* system exclusive	*/
			return(-1);		/* (watch for eox)	*/
		else if ((byte & UCH(0x0f)) == 2) /* song position	*/
			return(3);
		else if ((byte & UCH(0x0f)) == 3) /* song select	*/
			return(2);
		else if ((byte & UCH(0x0f)) == 6) /* tune request	*/
			return(1);
	}
	/* "can't happen" */
	return(0);
}

struct mpu_l *
mpugetl(mpud)		/* return a pointer to a track buffer */
struct mpu_device *mpud;
{
	if (0 <= mpud->mpud_track && mpud->mpud_track < MPU_TTR_MAX)
		return(&mpud->mpud_trackl[mpud->mpud_track]);
	printf("mpugetl: track=%d\n", mpud->mpud_track);
	panic("mpugetl");
	return(0);
}

mpupurge(mp)			/* purge all track buffers */
register struct mpu_device *mp;
{
	register struct mpu_l *tlp;

dprintf("mpupurge(0x%x)\n", mp);
	for (tlp = &mp->mpud_trackl[MPU_TTR_MAX]; --tlp >= mp->mpud_trackl; ) {
		mpufreeall(&tlp->mpul_head);
		tlp->mpul_tcount = 0;
		tlp->mpul_bcount = 0;
		tlp->mpul_contin = 0;
		tlp->mpul_cp = 0;
	}
	mp->mpud_contin = 0;
}

/*
** return all buffers of track dp to
** the system.
*/
mpufreeall(dp)
register struct buf *dp;
{
	register struct buf *bp;

	for (bp = dp->av_forw; bp != dp; bp = dp->av_forw) {
		bp->av_back->av_forw = bp->av_forw;
		bp->av_forw->av_back = bp->av_back;
		brelse(bp);
/****/if (--MPUblox < 0) printf("MPUblox is %d???\n", MPUblox);
	}
}

/*
** Like tsleep for the mpu driver (stolen from archive driver).
** Mpusleep will return 1 if it was interrupted by the user typing ^C
** otherwise it returns 0 if it was awakened with wakeup().
*/
mpusleep(cp, pri)
caddr_t	cp;
int	pri;
{
	label_t q;

	q = u.u_qsave;
	if (setjmp(&u.u_qsave))
		return(1);	/* interrupted  DOES THIS NEED u.u_qsave = q? */
	sleep(cp, pri);
	u.u_qsave = q;
	return(0);
}

/*
** Mpureset sends a reset command to the mpu.
** Return 1 on success, 0 on failure.
*/
mpureset(unit)
{
	register struct mb_device *md;
	register struct mpu_reg *mpur;
#ifdef sun2
	int opri, reset;
#else sun2
	int opri, reset, retry = 1;
#endif sun2

	md = mpudinfo[unit];
	mpur = (struct mpu_reg *) md->md_addr;
	opri = splx(pritospl(md->md_intpri));
#ifdef sun2
  	MPUDELAY(100);
  	mpur->mpur_com = MPU_COM_RESET;
  	MPUDELAY(100);
  	reset = (mpur->mpur_data == MPU_MESS_ACK);
#else sun2
	do {
		MPUDELAY(mpu_ndelay);
		mpur->mpur_com = MPU_COM_RESET;
		MPUDELAY(mpu_ndelay);
		reset = mpur->mpur_data & 0xff;
		if (MPUdebugreset) printf("mpureset: got %x\n", reset);
	} while (reset != MPU_MESS_ACK && retry--);
#endif sun2
	splx(opri);
#ifdef sun2
	return(reset);
#else sun2
	return(reset == MPU_MESS_ACK);
#endif sun2
}

# ifdef NOTUSED
mpudelay(n, unit)
register int n;
{
	register struct mpu_reg *mpur;
	register int i;

	mpur = (struct mpu_reg *) mpudinfo[unit]->md_addr;
	for (i = 0; i < MPU_NLOOPS; i++) {
		if ((mpur->mpur_stat & MPU_STAT_DRR) == 0)
			break;
		DELAY(n);
	}
	if (i == n)
		printf("mpu%d: drr timeout\n", unit);
}
# endif NOTUSED
# endif NMPU


