#include <stdio.h>
#include <ctype.h>
#include <math.h>
#include <carl/Player.h>

double Scale[] = {
	261.625565,	/* c4 */
	277.182631,	/* cs4 */
	293.664768,	/* d4 */
	311.126984,	/* ds4 */
	329.627557,	/* e4 */
	349.228231,	/* f4 */
	369.994423,	/* fs4 */
	391.995436,	/* g4 */
	415.304698,	/* gs4 */
	440.000000,	/* a4 */
	466.163762,	/* as4 */
	493.883301	/* b4 */
};

/*
 * pc_oct -	compute pitch class and octave from cyclic list element
 * takes:	
 *	str - string in Pitch format
 * 	oct - pointer to float to return octave
 * 	ac_list - state structure for keysig
 * returns:	
 *	pitch
 *	-1 on errors
 *	-2 if key signature field is scanned.
 * side effects:
 *	Sets Player_stat if expr ==  STOPSTR or RESTR.
 * 	Sets prevailing octave by assignment into argument oct.
 *	Sets key signature and accidentals via ac_list.
 * notes:	
 *	Limited to 12-tone pitch classes.
 */

pc_oct(str, oct, ac_list) 
	register char *str;
	register float *oct;
	struct keysig *ac_list;
{
	register int pitchclass, acoff;
	register char pc, ac = '\0';

	Player_stat &= ~(P_STOP_SET | P_REST_SET | P_EXPR_ERR);
	if (!strcmp(str, STOPSTR)) {
		Player_stat |= P_STOP_SET;
		return(-1);
	}
	if (!strcmp(str, RESTR)) {
		Player_stat |= P_REST_SET;
		return(-1);
	}
	if (*str == KEYSIG) {
		set_keysig(str+1, ac_list);
		return(-2);
	}
	if (*str >= 'a' && *str <= 'g') 	/* get pitch class */
		ac = pc = *str++;
	else {
		Player_stat |= P_EXPR_ERR;
		return(-1);
	}

	switch (pc) {				/* xlate to pitch number */
		case 'a': pitchclass = 9; break;
		case 'b': pitchclass = 11; break;
		case 'c': pitchclass = 0; break;
		case 'd': pitchclass = 2; break;
		case 'e': pitchclass = 4; break;
		case 'f': pitchclass = 5; break;
		case 'g': pitchclass = 7;
	}
	ac = '\0';
	if (*str == 's' || *str == 'f' || *str == 'n' 
		|| *str == 'S' || *str == 'F') { /* get accidental */
		ac = *str++;
		set_ac(pitchclass, ac, ac_list);
		switch (ac) {
			case 's': acoff = 1; break;
			case 'f': acoff = -1; break;
			case 'S': acoff = 2; break;
			case 'F': acoff = -2; break;
			case 'n': 
			default:  acoff = 0; break;
		}
	}
	else
		acoff = get_ac(pitchclass, ac_list);

	pitchclass += acoff;
	if (pitchclass < 0) 			/* modulo table length */
		pitchclass += PITCHES;
	if (pitchclass >= PITCHES) 
		pitchclass -= PITCHES;
	if (*str != '\0') {
		while (*str == '-' || *str == '+') {	/* relative octave */
			if (*str == '-')
				*oct -= 1.0;
			if (*str == '+')
				*oct += 1.0;
			*str++;
		}
		if (*str != '\0')			/* absolute octave */
			*oct = atof(str) - 4.0;	/* home octave is c4-b4 */
	}
	return(pitchclass);
}

/*
 * Pitches -	evaluate cyclic list as Pitch fields
 * takes:	
 *	nl -	pointer to string of Pitch fields in cyclic list format
 * returns:	
 *	pitch in Hz of evaluated field
 * side effects:	
 *	increments pointer to next expression field
 * 	sets Player_stat:
 * 		to P_HOLD_SET if expr == HOLDSTR
 * 		to P_STOP_SET if expr == STOPSTR
 * 		to P_REST_SET if expr == RESTR
 * notes:	
 *	May stop Player if expr == STOPSTR.
 *	Returns last value if HOLDSTR is set and does not increment
 *		pointer to next expression field.
 */

double 
Pitches(nl)
	register char *nl;
{
	register double frequency;
	register int rtn;
	register char *c;
	register struct noteheader *h = getid(nl);

	_motive(h, nl);
skip:	c = h->np->datum;
	if (!strcmp(HOLDSTR, c)) {
		rtn = pc_oct(h->last_val, &h->oct, &h->ac_list);
		Player_stat |= P_HOLD_SET;
	} else {
		if ((rtn = pc_oct(c, &h->oct, &h->ac_list)) == -2) {
			if (nextnot(h) != 0)
				clr_ac(&h->ac_list);
			goto skip;
		}
		h->last_val = c;
		if (h->np->chord_element != 0)
			Self->chordstat = TRUE;
		else
			Self->chordstat = FALSE;
		if (nextnot(h) != 0) {	
			clr_ac(&h->ac_list);
		}
	}
	if (Player_stat & P_STOP_SET) {	/* stop this player */
		Self->runstat = P_STOPPED;
		return(1.0);	/* return something safe */
	}
	if (Player_stat & P_REST_SET) {	/* play a rest */
 		Self->reststat = TRUE;
		return(1.0);	/* return something safe */
	}
	if (Player_stat & P_EXPR_ERR) {	/* unknown note specification */
		fprintf(stderr, "Pitches: unknown note spec.:``%s''\n", c);
		return(1.0);
	}
	frequency = Scale[rtn] * pow(2.0, h->oct);
	return(frequency);
}

/*
 * Pitch -	evaluate string as Pitch field
 * takes:	
 *	str -	pointer to Pitch string
 * returns:	
 *	pitch in Hz of evaluated string
 * side effects: 
 *	sets Player_stat
 * 		to P_HOLD_SET if expr == HOLDSTR
 * 		to P_STOP_SET if expr == STOPSTR
 * 		to P_REST_SET if expr == RESTR
 * notes:	
 *	Does not stop Player on STOPSTR.
 *	Returns 1.0 if expression == HOLDSTR, RESTR, or STOPSTR.
 */

double
Pitch(str)
	register char *str;
{
	float oct;
	register int pc;
	
	Player_stat &= ~P_HOLD_SET;
	if (!strcmp(HOLDSTR, str))
		Player_stat |= P_HOLD_SET;

	pc = pc_oct(str, &oct, NULL);

	if (Player_stat & (P_STOP_SET|P_REST_SET|P_EXPR_ERR|P_HOLD_SET) == 0)
		return(Scale[pc] * pow(2.0, (double) oct));
	else
		return(1.0);
}

/*
 * Octave -	evaluate string as Pitch field, return octave
 * takes:	
 *	str - pointer to Pitch string
 * returns:	
 *	octave number of evaluated string
 * side effects: 
 *	sets Player_stat
 * 		to P_HOLD_SET if expr == HOLDSTR
 * 		to P_STOP_SET if expr == STOPSTR
 * 		to P_REST_SET if expr == RESTR
 * notes:	
 *	Does not stop Player on STOPSTR.
 *	Returns 1.0 if expression == HOLDSTR, RESTR, or STOPSTR.
 *	Returns -1.0 if no octave specified in string.
 */

double
Octave(str)
	register char *str;
{
	float oct = -1.0;

	Player_stat &= ~P_HOLD_SET;
	if (!strcmp(HOLDSTR, str))
		Player_stat |= P_HOLD_SET;

	(void) pc_oct(str, &oct, NULL);

	if (Player_stat & (P_STOP_SET|P_REST_SET|P_EXPR_ERR|P_HOLD_SET) == 0)
		return((double) oct);
	else
		return(1.0);
}

/*
 * Keys -	evaluate Pitch fields as piano key numbers
 * takes:	
 *	nl -	pointer to string of Pitch fields in cyclic list format
 * returns:	
 *	pitch in semitones above A0 of evaluated field
 * side effects:	
 *	Increments pointer to next expression field.
 * 	Sets Player_stat:
 * 		to P_HOLD_SET if expr == HOLDSTR
 * 		to P_STOP_SET if expr == STOPSTR
 * 		to P_REST_SET if expr == RESTR
 * notes:	
 *	May stop Player if expr == STOPSTR.
 *	Returns last value if HOLDSTR is set and does not increment
 *		pointer to next expr field.
 */

double
Keys(nl)
	register char *nl;
{
	struct noteheader *h;
	int pc;
	double pitch;
	char *c;

	h = getid(nl);
	if (_motive(h, nl) != 0)
		clr_ac(&h->ac_list);
skip:	c = h->np->datum;
	Player_stat &= ~P_HOLD_SET;
	if (!strcmp(HOLDSTR, c)) {
		pc = pc_oct(h->last_val, &h->oct, &h->ac_list);
		Player_stat |= P_HOLD_SET;
	} else {
		if ((pc = pc_oct(c, &h->oct, &h->ac_list)) == -2) {
			if (nextnot(h) != 0)
				clr_ac(&h->ac_list);
			goto skip;
		}
		h->last_val = c;
		if (h->np->chord_element != 0)
			Self->chordstat = TRUE;
		else
			Self->chordstat = FALSE;
		if (nextnot(h) != 0)
			clr_ac(&h->ac_list);
	}
	if (Player_stat & P_STOP_SET) {	/* stop this player */
		Self->runstat = P_STOPPED;
		return(1.0);	/* return something safe */
	}
	if (Player_stat & P_REST_SET) {
 		Self->reststat = TRUE;
		return(1.0);	/* return something safe */
	}
	if (Player_stat & P_EXPR_ERR) {	/* unknown note specification */
		fprintf(stderr, "Keys: unknown note spec.:``%s''\n");
		return(1.0);
	}
	pitch = pc + (PITCHES * (h->oct + 4.0));
	return(pitch);
}

/*
 * Key -	evaluate Pitch string as semitone index
 * takes:	
 *	str -	pointer to Pitch string
 * returns:	
 *	pitch in semitones above A0 of evaluated field
 * side effects: 
 *	Sets Player_stat:
 * 		to P_HOLD_SET if expr == HOLDSTR
 * 		to P_STOP_SET if expr == STOPSTR
 * 		to P_REST_SET if expr == RESTR
 * notes:	
 *	Does not stop Player on STOPSTR.
 *	Returns 1.0 if expression == HOLDSTR, RESTR, or STOPSTR.
 */

double
Key(str)
	register char *str;
{
	float oct;
	register int pc = pc_oct(str, &oct, NULL);

	if (Player_stat & (P_STOP_SET|P_REST_SET|P_EXPR_ERR|P_HOLD_SET) == 0)
		return(pc + (PITCHES * (oct + 4.0)));
	else
		return(1.0);
}
