/*
 * newwire - basic signal processing template program
 * 
 * This program reads its standard input for a stream of floating point
 * binary numbers called floatsams and writes samples on its standard
 * output.  The input floatsams are simply copied in binary form to the
 * standard output, on the assumption that they are destined as input to
 * another program that reads floatsams.  If the standard output is connected
 * to a terminal, the output sample stream is produced in ASCII.  A "-o"
 * flag will suppress sample output altogether.
 * 
 * Like an ideal bare wire, this program simply copies its input to its
 * output.  To get it to do something else, such as rectifying the input
 * signal, you could substitute the line
 * 
 * 	output = input > 0 ? input : -input;
 * 
 * for the output assignment statement(s) given in this program.  The two
 * possible output formats are handled separately to reduce the
 * computation required to produce each sample.
 * 
 * newwire uses the CARL procom facilities to interpret (optional) header
 * information contained in the input sample stream, and generates an
 * explicit output header so that the output sample stream may be
 * conveniently made different from the input stream.  The crack() routine
 * is used to digest command line arguments that refer to header
 * information.  If a flag is given without a value (such as "-R"), the
 * corresponding input header value in printed on the user's terminal; if
 * a value is supplied for the flag (such as "-R16K"), then the output
 * header value is forced to the supplied value, and both the input and
 * output values for that header value are reported on the user's
 * terminal.  The CARL expr() program is used to allow values of certain
 * command line flags to be expressions.  For example,
 * 
 *     sndin sfile1 | newwire -R | sndout sfile2
 * 
 * copies sfile1 to sfile2, and reports the sampling rate to the user,
 * while
 * 
 *     sndin sfile1 | newwire -R48K | sndout sfile2
 * 
 * makes an exact copy of sfile1 in sfile2, but sfile2 will contain a
 * header specifying a 48K sampling rate regardless of the sampling rate
 * of sfile1.
 * 
 * Further information about procom, crack, expr, getfloat, etc., can be
 * found in the CARL man pages and via the help program.
 */


#include <stdio.h>
#include <carl/carl.h>
#include <carl/procom.h>
#include <carl/defaults.h>
#include <math.h>

#define IGNORE 1
#define IFBAREFLAG(flag) arg_index = 0 ; if(arg_option) *arg_option = '\0' ;\
x = crack(argc,argv,flag,IGNORE) ; \
if ( x != NULL && x != EOF && *arg_option == '\0')
#define IFFLAG(flag) arg_index = 0 ; if(arg_option) *arg_option = '\0' ;\
x = crack(argc,argv,flag,IGNORE) ; \
if ( x != NULL && x != EOF && *arg_option != '\0')
#define FPR(msg) fprintf( stderr, "%s\n", msg )
#define ERROR(msg){FPR(msg);exit(-1);}

/* cc -O newwire.c -lcarl -lm -o newwire */

main( argc, argv ) char *argv[] ; {
 PROP *ihp ;
 int outflag = 1 ;
 char x, *R, *c, *n, *f, *malloc() ;
 float input, output ;

    if ( argc == 1 && isatty(0) ) {
	FPR( "Usage: newwire [-#[Rcnf]] <floatsams\n" ) ;
	FPR( "where -#X inquires as to the header value of X\n" ) ;
	FPR( "R = SRATE\n" ) ;
	FPR( "c = NCHANS\n" ) ;
	FPR( "n = FILENAME\n" ) ;
	FPR( "f = FORMAT\n" ) ;
	exit(0) ;
    }
    if ( isatty(0) ) 
	ERROR( "NEWWIRE: standard input must be a file or pipe" ) ;
    ihp = getheader(stdin) ;
    noautocp() ;
    IFBAREFLAG( "R|" )
	fprintf( stderr, "Input SRATE = %s\n", getprop( ihp, "SRATE" ) ) ;
    IFBAREFLAG( "c|" )
	fprintf( stderr, "Input NCHANS = %s\n", getprop( ihp, "NCHANS" ) ) ;
    IFBAREFLAG( "n|" )
	fprintf( stderr, "Input FILENAME = %s\n", getprop( ihp, "FILENAME" ) ) ;
    IFBAREFLAG( "f|" )
	fprintf( stderr, "Input FORMAT = %s\n", getprop( ihp, "FORMAT" ) ) ;

    IFFLAG( "n|" ) { 
	n = malloc( 100 ) ; 
	strcpy( n, arg_option ) ;
	fprintf( stderr, 
	    "Input:Output FILENAME = %s:%s\n", getprop( ihp, "FILENAME" ), n ) ;
    } else
	n = getprop( ihp, "FILENAME" ) ;
    IFFLAG( "R|" ) {
	R = malloc( 100 ) ;
	sprintf( R, "%f", expr( arg_option ) ) ;
	fprintf( stderr, 
	    "Input:Output SRATE = %s:%s\n", getprop( ihp, "SRATE" ), R ) ;
    } else
	R = getprop( ihp, "SRATE" ) ;
    IFFLAG( "c|" ) {
	c = malloc( 100 ) ;
	sprintf( c, "%.0f", expr( arg_option ) ) ;
	fprintf( stderr, 
	    "Input:Output NCHANS = %s:%s\n", getprop( ihp, "NCHANS" ), c ) ;
    } else
	c = getprop( ihp, "NCHANS" ) ;
    IFFLAG( "f|" ) {
	f = malloc( 100 ) ;
	strcpy( f, arg_option ) ;
	fprintf( stderr, 
	    "Input:Output FORMAT = %s:%s\n", getprop( ihp, "FORMAT" ), f ) ;
    } else
	f = getprop( ihp, "FORMAT" ) ;

    stdheader( stdout, n, R, c, f ) ;

    arg_index = 0 ;
    if ( crack( argc, argv, "o", IGNORE ) == 'o' )
	outflag = 0 ;

    if ( isatty(1) ) {

	while ( getfloat( &input ) > 0 ) {
	    output = input ;
	    if ( outflag )
		printf( "%f\n", output ) ;
	}
    }
    else {

	while ( getfloat( &input ) > 0 ) {
	    output = input ;
	    if ( outflag )
		putfloat( &output ) ;
	}

	flushfloat() ;
    }
}
