#include <qpainter.h>

#include "QJScoreWidget.h"
#include "QJScoreTimeMarker.h"
#include "QJPhraseEditor.h"
#include "QJTrackScore.h"
#include "QJEditCntrl.h"

#include "jam/JContext.h"
#include "jam/JFocus.h"
#include "jam/JVoiceRecorder.h"
#include "jam/JTrack.h"
#include "jam/JTrackUpdate.h"
#include "rep/JLRRep.h"
#include "jam/JCursor.h"
#include "jam/JBeatToX.h"
#include "jam/JBPT.h"
#include "jam/JConductor.h"

QJScoreWidget::QJScoreWidget(JContext *context,
			     QWidget *parent)
  :QFrame(parent,"QJScoreWidget"),
   _track(0),
   _conductor(context->conductor()),
   _focus(context->focus())

{

  _beatToX = new JBeatToX(60,
			  x(),
			  width());
    

  _phraseEditor = new QJPhraseEditor(context); 

  context->recorder()->attach(_phraseEditor);

  
  _editCntrl    = new QJEditCntrl(this,_phraseEditor);

  _trackScore   = new QJTrackScore(_beatToX,
				   context,
				   this,"TrackScore");

  //  _layout->addWidget(_trackScore);

  _timeMarker = new QJScoreTimeMarker(this);

  _leftMarker = new QJScoreTimeMarker(this);
  _rightMarker = new QJScoreTimeMarker(this);

  _timeMarker->setPalette(QPalette(QColor(100,100,100)));
  _leftMarker->setPalette(QPalette(QColor(0,255,0)));
  _rightMarker->setPalette(QPalette(QColor(255,0,0)));

  context->cursor()->attach(this);

  _rightMarker->raise();
  _leftMarker->raise();
  _timeMarker->raise();

  connect(_trackScore,SIGNAL(aMousePressEvent(QMouseEvent *)),
	  SLOT(aMousePressEvent(QMouseEvent *)));

  connect(_trackScore,SIGNAL(aMouseMoveEvent(QMouseEvent *)),
	  SLOT(aMouseMoveEvent(QMouseEvent *)));

  connect(_trackScore,SIGNAL(aKeyPressEvent(QKeyEvent *)),
	  _phraseEditor,SLOT(keyPress(QKeyEvent *)));

  

}

QJScoreWidget::~QJScoreWidget()
{
  delete _beatToX;
  delete _phraseEditor;
}

void 
QJScoreWidget::playTimeUpdate(JBeat beat)
{

  if (_beatToX->setFocus(beat,false)){
    _trackScore->rebuild();
  }

  //TODO: should this be here ?

//   if (_focus->cursor()->beat() != beat) {
//     _focus->cursor()->setBeat(beat);
//     _trackScore->assertCursor();
//   }

  setTimeMarkers();
}

void 
QJScoreWidget::handle(const JBPT *bpt)
{

  JBeat beat=bpt->beat();

  if (bpt->isNamed("left")) {
    _conductor->setLeft(beat);
  } else {
    _conductor->setRight(beat);
  }

  if (_beatToX->setFocus(beat)){
    //    debug(" QJScoreWidget Time Update %f ", _beatToX->leftBeat().rep());
    _trackScore->rebuild();
  }

  setTimeMarkers();
}

void
QJScoreWidget::setTimeMarkers()
{


  int x1=_beatToX->x(_conductor->currentBeat());
  _timeMarker->move(x1,0);

  int x2=_beatToX->x(_conductor->left());
  if (x2 == x1) x2--;
  _leftMarker->move(x2,0);

  
  int x3=_beatToX->x(_conductor->right());
  if (x3 == x1) x3++;
  _rightMarker->move(x3,0);

}

void
QJScoreWidget::handle(const JFocus *focus)
{

  //  debug(" QJScoreWidget FOCUS ");

  if (_focus->track() == 0) {
    _trackScore->setTrack(0);
    return;
  }

  //  if (focus->track() != _track) {
  JTrackUpdate update(focus->track());
  handle(update);

}



void 
QJScoreWidget::handle(const JCursor *cursor)
{

  //  debug(" QJScoreWidget CURSOR ");

  if (_beatToX->setFocus(cursor->beat())){
    _trackScore->rebuild();
  } else {
    _trackScore->assertCursor();
  }

  setTimeMarkers();

}


void
QJScoreWidget::handle(JTrackUpdate  update)
{

  // debug(" QJScoreWidget UPDATE TRACK ");

  const JTrack *track = update.track();

  JBeat focusBeat=_conductor->currentBeat();

  if (track != _track) {
    JObserver<JTrackUpdate,void>::safeDetachFrom((JTrack *)_track);
    _track = track;
    ((JTrack *)_track)->attach(this);
    _trackScore->setTrack(track);
    _beatToX->setFocus(focusBeat);
    _trackScore->rebuild();
  } else { 
    assert(track != 0);
    if (_beatToX->setFocus(focusBeat)) {
      _trackScore->rebuild();
    } else {
      JLRRep ud(update.lrRep().left()-JDuration(1),update.lrRep().right()+JDuration(1));
      _trackScore->update(ud);
    }
  }

  setTimeMarkers();
}



void
QJScoreWidget::aMouseMoveEvent(QMouseEvent *e)
{

  JBeat beat =_beatToX->beatAt(e->x());

  if (_beatToX->setFocus(beat,false)) {
    _trackScore->rebuild();
  }

  beat.quantize(12);

  if (e->state() & MidButton)  {
    _conductor->setCurrentBeat(beat);
  } 

  if (e->state() & LeftButton) {
    _conductor->setLeft(beat);
  } 

  if (e->state() & RightButton) {
    _conductor->setRight(beat);
  }


  int x=_beatToX->x(beat);

  QPoint lP(x,e->y());
  QPoint gp=mapToGlobal(lP);
  
  QCursor::setPos(gp);

  setTimeMarkers();
}


void
QJScoreWidget::aMouseReleaseEvent(QMouseEvent *)
{
}


void
QJScoreWidget::aMousePressEvent(QMouseEvent *e)
{

  // debug(" QJScoreWidget mouse click");

  JBeat beat =_beatToX->beatAt(e->x());
  beat.quantize(12);

  int x=e->x();

  if (e->state() & ShiftButton) {
    if (e->button()==MidButton) {
      _conductor->setCurrentBeat(beat);
    } else if (e->button()==LeftButton) {
      _conductor->setLeft(beat);
    } else if (e->button()==RightButton) {
      _conductor->setRight(beat);
    }

  } else {
  
    if (e->button()==MidButton) {
      if (_beatToX->setFocusAt(_conductor->currentBeat(),x))
	_trackScore->rebuild();
    } else if (e->button()==LeftButton) {
      if (_beatToX->setFocusAt(_conductor->left(),x))
	_trackScore->rebuild();
    } else if (e->button()==RightButton) {
      _editCntrl->popup(e->pos());
      return;

//       if (_beatToX->setFocusAt(_conductor->right(),x))
// 	_trackScore->rebuild();
    }
  }

  setTimeMarkers();
}


void 
QJScoreWidget::resizeEvent(QResizeEvent *)
{

  //  debug(" QJScoreWidget  paint %d %d ",width(),height());

  //  debug(" QJScoreWidget resize 1");

  _beatToX->set(0,width());

  _trackScore->resize(size());

  // debug(" QJScoreWidget resize 2");

  _trackScore->rebuild();
  _timeMarker->resize(1,height());
  _leftMarker->resize(1,height());
  _rightMarker->resize(1,height());
}
















