/*---------------------------------------------------------------------------
Configure.m -- Copyright (c) 1991 Rex Pruess

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 1, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA, or send
   electronic mail to the the author.

This routine loads & displays the configuration window.  It also contains the
methods for handling the user requests generated from within the configuration
window.

Rex Pruess <rpruess@umaxc.weeg.uiowa.edu>

$Header: /rpruess/apps/Remotes-2.0/RCS/Configure.m,v 2.0 91/01/22 15:32:00 rpruess Exp $
-----------------------------------------------------------------------------
$Log:	Configure.m,v $
Revision 2.0  91/01/22  15:32:00  rpruess
Remotes-2.0 was upgraded for NeXT System Release 2.0 (standard or extended).
Remotes-2.0 supports the NeXT supplied Terminal application and the Stuart
shareware product.

Revision 1.2  90/05/17  15:03:17  rpruess
Minor changes to comments.

Revision 1.1  90/04/10  14:25:10  rpruess
Initial revision

-----------------------------------------------------------------------------*/

/* Standard C header files */
#include <libc.h>
#include <stdio.h>
#include <strings.h>

/* Configure class header files */
#import "Configure.h"
#import "BoxViewSwitcher.h"
#import "HostListManager.h"

/* Appkit header files */
#import <appkit/Application.h>
#import <appkit/Button.h>
#import <appkit/Cell.h>
#import <appkit/defaults.h>
#import <appkit/Form.h>
#import <appkit/Matrix.h>
#import <appkit/Menu.h>
#import <appkit/PopUpList.h>
#import <appkit/View.h>

@implementation Configure

/*---------------------------------------------------------------------------
Create the Configure object.
-----------------------------------------------------------------------------*/
+ new
{
   self = [super new];
   [NXApp loadNibSection:"Configure.nib" owner:self withNames:NO];
   [configWindow setMiniwindowIcon:"app"];

   configurePUL = [PopUpList new];

   theBVS = [BoxViewSwitcher new];
   [theBVS setBoxView:boxView];
   [theBVS setBoxViewWindow:boxViewWindow];
   [theBVS setControlButton:appsButton];
   [theBVS setDelegate:self];
   
   return self;
}

/*---------------------------------------------------------------------------
Initialize the fields in the configuration window.
-----------------------------------------------------------------------------*/
- initConfigWindow:sender
{
   const char     *string;

   [configWindow disableDisplay];

   /* Clear some fields */
   [buttonForm setStringValue:""];
   [hostForm setStringValue:""];

   /* Set user ID form value */
   [idForm setStringValue:NXUserName ()];

   /* Unselect autoStart switch */
   [autoStartButton setIntValue:NO];

   /* Set protocol/app values */
   [protocolMatrix selectCellWithTag:RLOGIN];
   [self rloginSet:self];

   /* Set font form size value */
   string = NXGetDefaultValue ("Terminal", "NXFixedPitchFontSize");
   if (string != NULL)
      [fontForm setStringValue:string];
   else
      [fontForm setStringValue:""];

   /* Set font marix value */
   string = NXGetDefaultValue ("Terminal", "NXFixedPitchFont");
   if (strcmp (string, "Courier") == 0) {
      [fontMatrix selectCellWithTag:COURIER];
      [self courierSet:self];
   }
   else {
      [fontMatrix selectCellWithTag:OHLFS];
      [self ohlfsSet:self];
   }

   /* Set X & Y default values */
   [xCell setIntValue:XDEFAULT];
   [yCell setIntValue:YDEFAULT];

   /* Set columns form value */
   string = NXGetDefaultValue ("Terminal", "Columns");
   if (string != NULL)
      [colsCell setStringValue:string];

   /* Set lines form value */
   string = NXGetDefaultValue ("Terminal", "Rows");
   if (string != NULL)
      [linesCell setStringValue:string];

   /* Set default Terminal matrix items */
   [self initMatrixFlags:terminalMatrix];
   [[terminalMatrix selectCellWithTag:TTRANSLAT] setIntValue:YES];
   [[terminalMatrix selectCellWithTag:TAUTOWRAP] setIntValue:YES];
   [[terminalMatrix selectCellWithTag:TSCROLLBK] setIntValue:YES];

   [terminalMetaMatrix selectCellWithTag:METAESC];
   
   appType = TERMINAL;
   [theBVS selectBoxViewTitle:TERMLAB];

   /* Set default Stuart matrix items */
   [self initMatrixFlags:stuartMatrix];
   [[stuartMatrix selectCellWithTag:SSCROLLBK] setIntValue:YES];
   [[stuartMatrix selectCellWithTag:STRANSLAT] setIntValue:YES];
   [[stuartMatrix selectCellWithTag:STERMSPAC] setIntValue:YES];

   [stuartMetaMatrix selectCellWithTag:METAESC];

   [self launchState:NO];

   /* Position cursor to the button form field */
   [buttonForm selectTextAt:0];

   [configWindow reenableDisplay];
   [configWindow displayIfNeeded];
   
   return self;
}

/*---------------------------------------------------------------------------
Add a host into the internal linked list and into the popup/pulldown lists.
-----------------------------------------------------------------------------*/
- addNewItem:sender
{
   id              aCell;
   id              aMatrix;
   id              aMetaMatrix;
   char            buttonName[MAXPOPLEN + 1];
   char            errMsg[256];
   unsigned int    flagValues;
   int             i;
   int             j;
   int             nCols;
   int             nRows;
   int             slotNum;     /* Slot position of host in linked list */
   
   if ([self validHostFields:self] != 0)
      return self;

   strcpy (buttonName,[buttonForm stringValueAt:0]);

   slotNum = [theHLM getHostSlotNum:buttonName];
   if (slotNum >= 0) {
      [buttonForm selectTextAt:0];

      strcpy (errMsg, "The button name '%s' is already listed under Remote ");
      strcat (errMsg, "Systems.  If you want to update it, click Change; ");
      strcat (errMsg, "otherwise rename '%s' and click Add.");

      NXRunAlertPanel (NULL, errMsg, NULL, NULL, NULL, buttonName, buttonName);
      return self;
   }

   if (appType == TERMINAL)
      aMetaMatrix = terminalMetaMatrix;
   else
      aMetaMatrix = stuartMetaMatrix;
   
   slotNum = [theHLM addHost:buttonName
      fullHostName:[hostForm stringValueAt:0]
      loginName:[idForm stringValueAt:0]
      application:appType
      protocol:protocolType
      locX:[xCell intValue]
      locY:[yCell intValue]
      nLines:[linesCell intValue]
      nCols:[colsCell intValue]
      font:fontType
      fontSize:[fontForm intValueAt:0]
      autoStart:[autoStartButton intValue]
      flags:[self getMatrixFlags:self]
      meta:[[aMetaMatrix selectedCell] tag]   
      updateConfigFile:YES];

   if (slotNum < 0)
      return self;

   /* Insert host into popup & pulldown lists. */
   [theHostsPUL insertItem:buttonName at:slotNum];
   [configurePUL insertItem:buttonName at:slotNum + 1];

   [self launchState:YES];

   /* Position to button form to indicate success. */
   [buttonForm selectTextAt:0];
   return self;
}

/*---------------------------------------------------------------------------
Change the host's values for the specified button.
-----------------------------------------------------------------------------*/
- changeOldItem:sender
{
   id              aCell;
   id              aMetaMatrix;
   char            buttonName[MAXPOPLEN + 1];
   char            errMsg[256];
   unsigned int    flagValues;
   short int       fontNum;
   int             slotNum;     /* Slot position of host in linked list */

   if ([self validHostFields:self] != 0)
      return self;

   strcpy (buttonName,[buttonForm stringValueAt:0]);

   fontNum = [fontForm intValueAt:0];

   slotNum = [theHLM getHostSlotNum:buttonName];
   if (slotNum < 0) {
      [buttonForm selectTextAt:0];

      strcpy (errMsg, "The button name '%s' is not listed under Remote ");
      strcat (errMsg, "Systems.  If you would like to add '%s' to the ");
      strcat (errMsg, "Remote Systems list, click Add.");

      NXRunAlertPanel (NULL, errMsg, NULL, NULL, NULL, buttonName, buttonName);
      return self;
   }

   if (appType == TERMINAL)
      aMetaMatrix = terminalMetaMatrix;
   else
      aMetaMatrix = stuartMetaMatrix;

   slotNum = [theHLM changeHost:[buttonForm stringValueAt:0]
      fullHostName:[hostForm stringValueAt:0]
      loginName:[idForm stringValueAt:0]
      application:appType
      protocol:protocolType
      locX:[xCell intValue]
      locY:[yCell intValue]
      nLines:[linesCell intValue]
      nCols:[colsCell intValue]
      font:fontType
      fontSize:fontNum
      autoStart:[autoStartButton intValue]
      flags:[self getMatrixFlags:self]
      meta:[[aMetaMatrix selectedCell] tag]];

   if (slotNum < 0)
      return self;

   [self launchState:YES];

   /* Position to button form to indicate success. */
   [buttonForm selectTextAt:0];

   return self;
}

/*---------------------------------------------------------------------------
Delete the host entry from internal list & popup/pulldown lists.
-----------------------------------------------------------------------------*/
- deleteOldItem:sender
{
   char            buttonName[MAXPOPLEN + 1];
   char            errMsg[256];
   int             slotNum;     /* Slot position of host in linked list */

   strcpy (buttonName,[buttonForm stringValueAt:0]);

   slotNum = [theHLM getHostSlotNum:buttonName];
   if (slotNum < 0) {
      strcpy (errMsg, "The button name '%s' is not listed under Remote ");
      strcat (errMsg, "Systems.  Nothing has been deleted.");

      NXRunAlertPanel (NULL, errMsg, NULL, NULL, NULL, buttonName);
      return self;
   }

   if (NXRunAlertPanel (NULL, "OK to delete '%s'?", "OK", "Cancel", NULL,
         buttonName) != NX_ALERTDEFAULT)
      return self;

   slotNum = [theHLM deleteHost:slotNum];

   if (slotNum < 0)
      return self;

   [[theHostsPUL removeItemAt:slotNum] free];
   [[configurePUL removeItemAt:slotNum + 1] free];

   if (strcmp (buttonName,[theHostsButton title]) == 0) {
      slotNum--;
      if (slotNum < 0 && [theHLM getNumHosts:sender] > 0)
         slotNum = 0;
      if (slotNum < 0)
         [theHostsButton setTitle:""];
      else
         [theHostsButton setTitle:[theHLM getButtonName:slotNum]];
   }

   [self launchState:NO];

   /* Position to button form to indicate success. */
   [buttonForm selectTextAt:0];
   return self;
}

/*---------------------------------------------------------------------------
Display the host information for this button.
-----------------------------------------------------------------------------*/
- display:sender
{
   char            buttonName[MAXPOPLEN + 1];
   unsigned int    flagValues;
   int             font;
   int             protocol;
   int             slotNum;     /* Slot position of host in linked list */

   if ([sender selectedRow] == 0)
      return self;

   strcpy (buttonName,[[sender selectedCell] title]);

   slotNum = [theHLM getHostSlotNum:buttonName];

   if (slotNum < 0) {
      NXRunAlertPanel (NULL, "The button name '%s' is not in the Remotes data base.",
		       NULL, NULL, NULL, buttonName);
      return self;
   }

   [configWindow disableDisplay];

   [autoStartButton setIntValue:[theHLM getAutoStart:slotNum]];
   [buttonForm setStringValue:buttonName at:0];
   [colsCell setIntValue:[theHLM getColumns:slotNum]];
   [fontForm setIntValue:[theHLM getFontSize:slotNum] at:0];
   [hostForm setStringValue:[theHLM getFullHostName:slotNum] at:0];
   [idForm setStringValue:[theHLM getUserID:slotNum] at:0];
   [linesCell setIntValue:[theHLM getLines:slotNum]];
   [xCell setIntValue:[theHLM getX:slotNum]];
   [yCell setIntValue:[theHLM getY:slotNum]];

   protocol = [theHLM getProtocolType:slotNum];
   [protocolMatrix selectCellWithTag:protocol];
   if (protocol == TELNET)
      [self telnetSet:self];
   else if (protocol == TN3270)
      [self tn3270Set:self];
   else
      [self rloginSet:self];
   
   appType = [theHLM getAppType:slotNum];

   flagValues = [theHLM getFlags:slotNum];
   
   if (appType == STUART) {
      [[stuartMatrix selectCellWithTag:SSTRICTEM] setIntValue:(flagValues & SSTRICTEM)];
      [[stuartMatrix selectCellWithTag:SKEYPAD] setIntValue:(flagValues & SKEYPAD)];
      [[stuartMatrix selectCellWithTag:SSCROLLBK] setIntValue:(flagValues & SSCROLLBK)];
      [[stuartMatrix selectCellWithTag:STRANSLAT] setIntValue:(flagValues & STRANSLAT)];
      [[stuartMatrix selectCellWithTag:SREVERSE] setIntValue:(flagValues & SREVERSE)];
      [[stuartMatrix selectCellWithTag:SKEYBDFOC] setIntValue:(flagValues & SKEYBDFOC)];
      [[stuartMatrix selectCellWithTag:STERMSPAC] setIntValue:(flagValues & STERMSPAC)];
      [[stuartMatrix selectCellWithTag:SMOUSEFOC] setIntValue:(flagValues & SMOUSEFOC)];
      [[stuartMatrix selectCellWithTag:SSOURCEDL] setIntValue:(flagValues & SSOURCEDL)];
      [[stuartMatrix selectCellWithTag:STESTEXIT] setIntValue:(flagValues & STESTEXIT)];

      [stuartMetaMatrix selectCellWithTag:[theHLM getMeta:slotNum]];

      [theBVS selectBoxViewTitle:STUARTLAB];
   }
   else {
      [[terminalMatrix selectCellWithTag:TTRANSLAT] setIntValue:(flagValues & TTRANSLAT)];
      [[terminalMatrix selectCellWithTag:TKEYPAD] setIntValue:(flagValues & TKEYPAD)];
      [[terminalMatrix selectCellWithTag:TSTRICTEM] setIntValue:(flagValues & TSTRICTEM)];
      [[terminalMatrix selectCellWithTag:TAUTOWRAP] setIntValue:(flagValues & TAUTOWRAP)];
      [[terminalMatrix selectCellWithTag:TSCROLLBK] setIntValue:(flagValues & TSCROLLBK)];
      [[terminalMatrix selectCellWithTag:TAUTOFOCS] setIntValue:(flagValues & TAUTOFOCS)];
      [[terminalMatrix selectCellWithTag:TSOURCEDL] setIntValue:(flagValues & TSOURCEDL)];

      [terminalMetaMatrix selectCellWithTag:[theHLM getMeta:slotNum]];
       
      [theBVS selectBoxViewTitle:TERMLAB];
   }

   font = [theHLM getFontType:slotNum];
   [fontMatrix selectCellWithTag:font];
   if (font == COURIER)
      [self courierSet:self];
   else
      [self ohlfsSet:self];

   [self launchState:YES];

   [configWindow reenableDisplay];
   [configWindow displayIfNeeded];
   
   return self;
}

/*---------------------------------------------------------------------------
Get the Terminal or Stuart flag values.
-----------------------------------------------------------------------------*/
- (unsigned int)getMatrixFlags:sender
{
   id              aCell;
   id              aMatrix;
   unsigned int    flagValues;
   int             i;
   int             j;
   int             nCols;
   int             nRows;

   if (appType == TERMINAL)
      aMatrix = terminalMatrix;
   else
      aMatrix = stuartMatrix;

   [aMatrix getNumRows:&nRows numCols:&nCols];

   flagValues = 0;
   for (j = 0; j < nCols; j++)
      for (i = 0; i < nRows; i++) {
	 aCell = [aMatrix cellAt:i:j];
	 if ([aCell isEnabled] && [aCell intValue] == YES) 
	    flagValues |= [[aMatrix cellAt:i:j] tag];
      }

   return flagValues;
}   

/*---------------------------------------------------------------------------
Initialize the flags in the Terminal or Stuart matrix.
-----------------------------------------------------------------------------*/
- initMatrixFlags:aMatrix
{
   id              aCell;
   int             i;
   int             j;
   int             nCols;
   int             nRows;

   [aMatrix getNumRows:&nRows numCols:&nCols];
   for (j = 0; j < nCols; j++)
      for (i = 0; i < nRows; i++) {
	 aCell = [aMatrix cellAt:i:j];
	 if ([aCell isEnabled])
	    [aCell setIntValue:NO];
      }

   return self;
}

/*---------------------------------------------------------------------------
Set configuration values based on chosen application (boxView).
-----------------------------------------------------------------------------*/
- boxViewDidSwitch:sender
{
   if ( strcmp ([[sender boxView] title], STUARTLAB) == 0)
      [self stuart:self];
   else
      [self terminal:self];
   
   return self;
}

/*---------------------------------------------------------------------------
Set variables and enable/disable fields based on the application type.
-----------------------------------------------------------------------------*/
- stuart:sender
{
   appType = STUART;
   [self launchState:NO];
   return self;
}

- terminal:sender
{
   appType = TERMINAL;
   [self launchState:NO];
   return self;
}

/*---------------------------------------------------------------------------
Set variables and enable/disable fields based on the font style.
-----------------------------------------------------------------------------*/
- courierFont:sender
{
   [self courierSet:self];
   [self launchState:NO];
   return self;
}

- courierSet:sender
{
   fontType = COURIER;
   return self;
}

- ohlfsFont:sender
{
   [self ohlfsSet:self];
   [self launchState:NO];
   return self;
}

- ohlfsSet:sender
{
   fontType = OHLFS;
   return self;
}

/*---------------------------------------------------------------------------
Set variables and enable/disable fields based on the protocol.
-----------------------------------------------------------------------------*/
- rlogin:sender
{
   [self rloginSet:self];
   [self launchState:NO];
   return self;
}

- rloginSet:sender
{
   protocolType = RLOGIN;
   [idForm setEnabled:YES];
   return self;
}

- telnet:sender
{
   [self telnetSet:self];
   [self launchState:NO];
   return self;
}

- telnetSet:sender
{
   protocolType = TELNET;
   [idForm setEnabled:NO];
   return self;
}

- tn3270:sender
{
   [self tn3270Set:self];
   [self launchState:NO];
   return self;
}

- tn3270Set:sender
{
   protocolType = TN3270;
   [idForm setEnabled:NO];
   return self;
}

/*---------------------------------------------------------------------------
Handle launch request from the configuration window.  Must login to a host.
-----------------------------------------------------------------------------*/
- launch:sender
{
   [theHLM loginToHost:[buttonForm stringValueAt:0] activate:YES];
   return self;
}

/*---------------------------------------------------------------------------
Disable the launch button.  This method is connected to some of the boxes
in the configuration window via the Interface Builder.  This method is not
called internally by this program.
-----------------------------------------------------------------------------*/
- launchDisable:sender
{
   [self launchState:NO];
   return self;
}

/*---------------------------------------------------------------------------
Enable or disable the launch button.  This routine is called by various
methods within this program.  It provides a single focal point for
controlling the status of the launch button.
-----------------------------------------------------------------------------*/
- launchState:(BOOL) flag 
{
   [launchButton setEnabled:flag];
   launchFlag =flag; return self;
}

/*---------------------------------------------------------------------------
The user pressed the menu revert button.
-----------------------------------------------------------------------------*/
- menuRevert:sender
{
   if ([configWindow isKeyWindow] == YES)
      [self initConfigWindow:self];
   return self;
}

/*---------------------------------------------------------------------------
Slap that configuration window onto the screen.
-----------------------------------------------------------------------------*/
- showConfigWindow:sender
{
   [configWindow makeKeyAndOrderFront:sender];
   return self;
}

/*---------------------------------------------------------------------------
Validate the fields in the configuration window.
-----------------------------------------------------------------------------*/
- (int)validHostFields:sender
{
   char            errorStr[81];/* Error string */
   int             length;      /* Temporary variable */
   NXSize          screenSize;  /* Screen width/height */
   int             xMin;        /* Minimum X value allowed */
   int             xMax;        /* Maximum X value allowed */
   int             yMin;        /* Minimum Y value allowed */
   int             yMax;        /* Maximum Y value allowed */

   /* Button name validation */
   length = strlen ([buttonForm stringValueAt:0]);
   if (length > MAXPOPLEN || length <= 0) {
      [buttonForm selectTextAt:0];
      NXRunAlertPanel (NULL, "Button name length must be between 1 & %d.", NULL, NULL, NULL, MAXPOPLEN);
      return -1;
   }

   /* Host name validation */
   length = strlen ([hostForm stringValueAt:0]);
   if (length > HOSTENTRYLEN || length <= 0) {
      [hostForm selectTextAt:0];
      NXRunAlertPanel (NULL, "Hostname length must be between 1 & %d.", NULL, NULL, NULL, HOSTENTRYLEN);
      return -1;
   }

   if (index ([hostForm stringValueAt:0], ' ') != 0) {
      [hostForm selectTextAt:0];
      NXRunAlertPanel (NULL, "Blanks are not allowed in host names.", NULL, NULL, NULL);
      return -1;
   }

   /* User ID validation */
   length = strlen ([idForm stringValueAt:0]);
   if (length > MAXIDLEN) {
      [idForm selectTextAt:0];
      NXRunAlertPanel (NULL, "User ID length can not exceed %d.", NULL, NULL, NULL, MAXIDLEN);
      return -1;
   }

   if (index ([idForm stringValueAt:0], ' ') != 0) {
      [idForm selectTextAt:0];
      NXRunAlertPanel (NULL, "Blanks are not allowed in user IDs.", NULL, NULL, NULL);
      return -1;
   }

   /* Font size validation */
   length = [fontForm intValueAt:0];
   if (length < MINFONTSIZE || length > MAXFONTSIZE) {
      [fontForm selectTextAt:0];
      NXRunAlertPanel (NULL, "Font size must be between %d & %d.", NULL, NULL, NULL, MINFONTSIZE, MAXFONTSIZE);
      return -1;
   }

   /* Get the screen size & establish maximum/minimum values allowed. */
   [NXApp getScreenSize:&screenSize];
   xMin = 0;
   xMax = screenSize.width - (screenSize.width *.08);
   yMin = 100;
   yMax = screenSize.height;

   /* X validation */
   length = [xCell intValue];
   if (length < xMin || length > xMax) {
      [windowMatrix selectCell:xCell];
      NXRunAlertPanel (NULL, "X position must be between %d & %d pixels.", NULL, NULL, NULL, xMin, xMax);
      return -1;
   }

   /* Y validation */
   length = [yCell intValue];
   if (length < yMin || length > yMax) {
      [windowMatrix selectCell:yCell];
      NXRunAlertPanel (NULL, "Y position must be between %d & %d pixels.", NULL, NULL, NULL, yMin, yMax);
      return -1;
   }

   length = [linesCell intValue];
   if (length < MINLINES || length > MAXLINES) {
      [windowMatrix selectCell:linesCell];
      NXRunAlertPanel (NULL, "X size (lines) must be between %d & %d.", NULL, NULL, NULL, MINLINES, MAXLINES);
      return -1;
   }

   length = [colsCell intValue];
   if (length < MINCOLS || length > MAXCOLS) {
      [windowMatrix selectCell:colsCell];
      NXRunAlertPanel (NULL, "Y size (columns) must be between %d & %d.", NULL, NULL, NULL, MINCOLS, MAXCOLS);
      return -1;
   }

   return 0;
}

/*---------------------------------------------------------------------------
Text changed in configuration window so disable the launch button.
-----------------------------------------------------------------------------*/
- text:sender isEmpty:(BOOL) flag
{
   [self launchState:NO];
   return self;
}

/*---------------------------------------------------------------------------
Establish the outlets.
-----------------------------------------------------------------------------*/
- setAppsButton:anObject
{
   appsButton = anObject;
   return self;
}

- setAutoStartButton:anObject
{
   autoStartButton = anObject;
   return self;
}

- setButtonForm:anObject
{
   buttonForm = anObject;
   return self;
}

- setColsCell:anObject
{
   colsCell = anObject;
   return self;
}

- setFontForm:anObject
{
   fontForm = anObject;
   return self;
}

- setHostForm:anObject
{
   hostForm = anObject;
   return self;
}

- setIdForm:anObject
{
   idForm = anObject;
   return self;
}

- setLaunchButton:anObject
{
   launchButton = anObject;
   return self;
}

- setLinesCell:anObject
{
   linesCell = anObject;
   return self;
}

- setXCell:anObject
{
   xCell = anObject;
   return self;
}

- setYCell:anObject
{
   yCell = anObject;
   return self;
}

- setFontMatrix:anObject
{
   fontMatrix = anObject;
   return self;
}

- setProtocolMatrix:anObject
{
   protocolMatrix = anObject;
   return self;
}

- setTerminalMatrix:anObject
{
   terminalMatrix = anObject;
   return self;
}

- setTerminalMetaMatrix:anObject
{
   terminalMetaMatrix = anObject;
   return self;
}

- setStuartMatrix:anObject
{
   stuartMatrix = anObject;
   return self;
}

- setStuartMetaMatrix:anObject
{
   stuartMetaMatrix = anObject;
   return self;
}

- setWindowMatrix:anObject
{
   windowMatrix = anObject;
   return self;
}

- setConfigWindow:anObject
{
   configWindow = anObject;
   return self;
}

- setBoxViewWindow:anObject
{
   boxViewWindow = anObject;
   return self;
}

- setBoxView:anObject
{
   boxView = anObject;
   return self;
}

/*---------------------------------------------------------------------------
Set up the configuration pulldown list and its associated button.
-----------------------------------------------------------------------------*/
- setConfigureButton:anObject
{
   configureButton = anObject;
   return self;
}

- initConfigButton:anObject
{
   [configurePUL addItem:"Remote Systems"];
   [theHLM loadPopUpList:configurePUL];

   [configureButton setTarget:configurePUL];
   [configureButton setAction:@selector (popUp:)];

   NXAttachPopUpList (configureButton, configurePUL);

   [configurePUL changeButtonTitle:NO];
   [configureButton setIcon:"pulldown" position:NX_ICONRIGHT];
   [configureButton setAltIcon:"pulldownH"];

   [configurePUL setTarget:self];
   [configurePUL setAction:@selector (display:)];
}

/*---------------------------------------------------------------------------
Messages from Remotes.m result in the execution of these sets.
-----------------------------------------------------------------------------*/
- setHostsButton:anObject
{
   theHostsButton = anObject;
   return self;
}

- setHostsPUL:anObject
{
   theHostsPUL = anObject;
   return self;
}

- setHLM:anObject
{
   theHLM = anObject;
   return self;
}

@end
