/*
 *	$Id: Pseudo.c,v 1.4 89/11/28 16:16:10 gregc Exp $
 *
 *	Copyright 1989 by the Regents of the University of California and
 *	the UCSF Computer Graphics Laboratory.  All rights reserved.
 *	Freely redistributable.
 *
 *	Pseudo:	 by Greg Couch, gregc@cgl.ucsf.edu, July 1989.
 *
 *		One of the main design defects of the Workspace manager
 *	and NeXT defaults database is that you can't specify command line
 *	arguments to applications invoked via the dock.  This is especially
 *	important if you wish to startup more than one terminal emulator
 *	without them appearing in the same place, same font, same ....
 *
 *		This program pretends to be an application, so it will
 *	get the -MachLaunch arguments to communicate back to the Workspace
 *	manager with.  All of its arguments are passed through to the
 *	real application being run.  To find out which application to run
 *	and the set of arguments to use, it checks the name that it was
 *	invoked with and uses that as the application name for finding
 *	variables in the defaults database.
 */

# include	<appkit/defaults.h>
# include	<stdlib.h>
# include	<string.h>
# include	<syslog.h>
# include	<sys/param.h>

extern int	execvp(const char *, const char **);
static char	*strend(char *, const char *, const char *);

# define	DEFAULT_SHELL	"/bin/sh"
# define	CMDBUF_END	(&cmdbuf[sizeof cmdbuf])

static char	cmdbuf[NCARGS];
static char const
		*shell[]	= { DEFAULT_SHELL, "-c", cmdbuf, NULL };

int
main(argc, argv)
	int	argc;
	char	*argv[];
{
	register char	*cmdend;
	register int	i;
	register char	*invoked_as;
		/* defaults */
	register const char	*Application;
	register const char	*Args;
	register const char	*Shell;

	if (argc <= 0)
		return 1;

	/* find out which application we're pretending to be */
	if ((invoked_as = strrchr(argv[0], '/')) != NULL)
		invoked_as += 1;
	else
		invoked_as = argv[0];

	openlog(invoked_as, LOG_PID, LOG_USER);

	Application = NXGetDefaultValue(invoked_as, "Application");
	if (Application == NULL) {
		syslog(LOG_ERR, "missing Application default");
		return 2;
	}

	Shell = NXGetDefaultValue(invoked_as, "Shell");
	if (Shell == NULL)
		Shell = getenv("SHELL");
	if (Shell != NULL)
		shell[0] = Shell;

	/*
	 * build command string
	 */

	cmdend = cmdbuf;
	cmdend = strend(cmdend, "exec ", CMDBUF_END);

	/* start with application */
	cmdend = strend(cmdend, Application, CMDBUF_END);
	cmdend = strend(cmdend, " ", CMDBUF_END);

	/* pass through command line arguments */
	for (i = 1; i < argc; i += 1) {
		register char	*s,  *t;

		cmdend = strend(cmdend, "\"", CMDBUF_END);
		for (s = argv[i]; (t = strchr(s, '"')) != NULL; s = t + 1) {
			*t = '\0';
			cmdend = strend(cmdend, s, CMDBUF_END);
			*t = '"';
		}
		cmdend = strend(cmdend, s, CMDBUF_END);
		cmdend = strend(cmdend, "\" ", CMDBUF_END);
	}

	/* add on specific arguments */
	Args = NXGetDefaultValue(invoked_as, "Args");
	if (Args != NULL)
		cmdend = strend(cmdend, Args, CMDBUF_END);

	if (cmdend == NULL) {
		syslog(LOG_ERR, "argument list too long");
		return 3;
	}

	(void) execvp(shell[0], shell);
	i = errno;
	syslog(LOG_ERR, "exec failed (%s): %m", shell[0]);
	syslog(LOG_ERR, "PATH = %s", getenv("PATH"));
	return i;
}

/*
 *	strend:
 *
 *		append src string to dest string and return the end of the
 *	dest string without going beyond the last position in the dest string.
 */

static	char	*
strend(char *dest, const char *src, const char *last)
{
	if (dest == NULL)
		return NULL;

	while (dest < last && *src)
		*dest++ = *src++;

	if (*src)
		return NULL;
	*dest = '\0';
	return dest;
}
