/* Load a task using the single server, and then run it
   as if we were the kernel.
   Copyright (C) 1993, 1994, 1995, 1996 Free Software Foundation, Inc.

This file is part of the GNU Hurd.

The GNU Hurd is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

The GNU Hurd is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with the GNU Hurd; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

/* Written by Michael I. Bushnell.  */

#include <mach.h>
#include <mach/notify.h>
#include <device/device.h>
#include <a.out.h>
#include <mach/message.h>
#include <mach/mig_errors.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <cthreads.h>
#include <varargs.h>
#include <fcntlbits.h>
#include <elf.h>
#include <mach/mig_support.h>
#include <mach/default_pager.h>

#include "notify_S.h"
#include "exec_S.h"
#include "ourdevice_S.h"
#include "io_S.h"
#include "device_reply_U.h"
#include "io_reply_U.h"
#include "term_S.h"
#include "bootstrap_S.h"
/* #include "tioctl_S.h" */

#include "boot_script.h"

#include <hurd/auth.h>

#ifdef UX

#include "ux.h"

#else  /* !UX */

#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <termios.h>
#include <error.h>
#include <hurd.h>

static struct termios orig_tty_state;

static void
init_termstate ()
{
  struct termios tty_state;

  if (tcgetattr (0, &tty_state) < 0)
    error (10, errno, "tcgetattr");

  orig_tty_state = tty_state;
  cfmakeraw (&tty_state);

  if (tcsetattr (0, 0, &tty_state) < 0)
    error (11, errno, "tcsetattr");
}

static void
restore_termstate ()
{
  tcsetattr (0, 0, &orig_tty_state);
}

#define host_fstat fstat
typedef struct stat host_stat_t;
#define host_exit exit

#endif /* UX */

mach_port_t privileged_host_port, master_device_port, defpager;
mach_port_t pseudo_master_device_port;
mach_port_t receive_set;
mach_port_t pseudo_console;
auth_t authserver;

spin_lock_t queuelock = SPIN_LOCK_INITIALIZER;
spin_lock_t readlock = SPIN_LOCK_INITIALIZER;

mach_port_t php_child_name, psmdp_child_name, taskname;

task_t child_task;
mach_port_t bootport;

int console_mscount;

vm_address_t fs_stack_base;
vm_size_t fs_stack_size;

void init_termstate ();
void restore_termstate ();

char *fsname;

char *bootstrap_args;
char *bootdevice = "sd0a";

void set_mach_stack_args ();

void safe_gets (char *buf, int buf_len)
{
  fgets (buf, buf_len, stdin);
}

char *useropen_dir;

int
useropen (const char *name, int flags, int mode)
{
  if (useropen_dir)
    {
      static int dlen;
      if (!dlen) dlen = strlen (useropen_dir);
      {
	int len = strlen (name);
	char try[dlen + 1 + len + 1];
	int fd;
	memcpy (try, useropen_dir, dlen);
	try[dlen] = '/';
	memcpy (&try[dlen + 1], name, len + 1);
	fd = open (try, flags, mode);
	if (fd >= 0)
	  return fd;
      }
    }
  return open (name, flags, mode);
}

int
request_server (mach_msg_header_t *inp,
		mach_msg_header_t *outp)
{
  extern int io_server (mach_msg_header_t *, mach_msg_header_t *);
  extern int device_server (mach_msg_header_t *, mach_msg_header_t *);
  extern int notify_server (mach_msg_header_t *, mach_msg_header_t *);
  extern int term_server (mach_msg_header_t *, mach_msg_header_t *);
/*  extern int tioctl_server (mach_msg_header_t *, mach_msg_header_t *); */
  extern int bootstrap_server (mach_msg_header_t *, mach_msg_header_t *);
  extern void bootstrap_compat ();

#if 0
  if (inp->msgh_local_port == bootport && boot_like_cmudef)
    {
      if (inp->msgh_id == 999999)
	{
	  bootstrap_compat (inp, outp);
	  return 1;
	}
      else
	return bootstrap_server (inp, outp);
    }
  else
#endif
   return (io_server (inp, outp)
	   || device_server (inp, outp)
	   || notify_server (inp, outp)
	   || term_server (inp, outp)
	   /*	    || tioctl_server (inp, outp) */);
}

vm_address_t
load_image (task_t t,
	    char *file)
{
  int fd;
  union
    {
      struct exec a;
      Elf32_Ehdr e;
    } hdr;
  char msg[] = ": cannot open bootstrap file\n";

  fd = useropen (file, O_RDONLY, 0);

  if (fd == -1)
    {
      write (2, file, strlen (file));
      write (2, msg, sizeof msg - 1);
      task_terminate (t);
      host_exit (1);
    }

  read (fd, &hdr, sizeof hdr);
  if (*(Elf32_Word *) hdr.e.e_ident == *(Elf32_Word *) "\177ELF")
    {
      Elf32_Phdr phdrs[hdr.e.e_phnum], *ph;
      lseek (fd, hdr.e.e_phoff, SEEK_SET);
      read (fd, phdrs, sizeof phdrs);
      for (ph = phdrs; ph < &phdrs[sizeof phdrs/sizeof phdrs[0]]; ++ph)
	if (ph->p_type == PT_LOAD)
	  {
	    vm_address_t buf;
	    vm_size_t offs = ph->p_offset & (ph->p_align - 1);
	    vm_size_t bufsz = round_page (ph->p_filesz + offs);

	    vm_allocate (mach_task_self (), &buf, bufsz, 1);

	    lseek (fd, ph->p_offset, SEEK_SET);
	    read (fd, (void *)(buf + offs), ph->p_filesz);

	    ph->p_memsz = ((ph->p_vaddr + ph->p_memsz + ph->p_align - 1)
			   & ~(ph->p_align - 1));
	    ph->p_vaddr &= ~(ph->p_align - 1);
	    ph->p_memsz -= ph->p_vaddr;

	    vm_allocate (t, (vm_address_t*)&ph->p_vaddr, ph->p_memsz, 0);
	    vm_write (t, ph->p_vaddr, buf, bufsz);
	    vm_deallocate (mach_task_self (), buf, bufsz);
	    vm_protect (t, ph->p_vaddr, ph->p_memsz, 0,
			((ph->p_flags & PF_R) ? VM_PROT_READ : 0) |
			((ph->p_flags & PF_W) ? VM_PROT_WRITE : 0) |
			((ph->p_flags & PF_X) ? VM_PROT_EXECUTE : 0));
	  }
      return hdr.e.e_entry;
    }
  else
    {
      /* a.out */
      int magic = N_MAGIC (hdr.a);
      int headercruft;
      vm_address_t base = 0x10000;
      int rndamount, amount;
      vm_address_t bsspagestart, bssstart;
      char *buf;

      headercruft = sizeof (struct exec) * (magic == ZMAGIC);

      amount = headercruft + hdr.a.a_text + hdr.a.a_data;
      rndamount = round_page (amount);
      vm_allocate (mach_task_self (), (u_int *)&buf, rndamount, 1);
      lseek (fd, sizeof hdr.a - headercruft, SEEK_SET);
      read (fd, buf, amount);
      vm_allocate (t, &base, rndamount, 0);
      vm_write (t, base, (u_int) buf, rndamount);
      if (magic != OMAGIC)
	vm_protect (t, base, trunc_page (headercruft + hdr.a.a_text),
		    0, VM_PROT_READ | VM_PROT_EXECUTE);
      vm_deallocate (mach_task_self (), (u_int)buf, rndamount);

      bssstart = base + hdr.a.a_text + hdr.a.a_data + headercruft;
      bsspagestart = round_page (bssstart);
      vm_allocate (t, &bsspagestart,
		   hdr.a.a_bss - (bsspagestart - bssstart), 0);

      return hdr.a.a_entry;
    }
}


void read_reply ();
void msg_thread ();

/* Callbacks for boot_script.c; see boot_script.h.  */

void *
boot_script_malloc (int size)
{
  return malloc (size);
}

void
boot_script_free (void *ptr, int size)
{
  free (ptr);
}

mach_port_t
boot_script_read_file (const char *filename)
{
  static const char msg[] = ": cannot open\n";
  int fd = useropen (filename, O_RDONLY, 0);
  host_stat_t st;
  error_t err;
  mach_port_t memobj;
  vm_address_t region;

  write (2, filename, strlen (filename));
  if (fd < 0)
    {
      write (2, msg, sizeof msg - 1);
      host_exit (1);
    }
  else
    write (2, msg + sizeof msg - 2, 1);

  host_fstat (fd, &st);

  err = default_pager_object_create (defpager, &memobj,
				     round_page (st.st_size));
  if (err)
    {
      static const char msg[] = "cannot create default-pager object\n";
      write (2, msg, sizeof msg - 1);
      host_exit (1);
    }

  region = 0;
  vm_map (mach_task_self (), &region, round_page (st.st_size),
	  0, 1, memobj, 0, 0, VM_PROT_ALL, VM_PROT_ALL, VM_INHERIT_NONE);
  read (fd, (char *) region, st.st_size);
  vm_deallocate (mach_task_self (), region, round_page (st.st_size));

  close (fd);
  return memobj;
}

int
boot_script_exec_cmd (mach_port_t task, char *path, int argc,
		      char **argv, char *strings, int stringlen)
{
  char *args, *p;
  int arg_len, i;
  unsigned reg_size;
  void *arg_pos;
  vm_offset_t stack_start, stack_end;
  vm_address_t startpc, str_start;
  thread_t thread;
  struct i386_thread_state regs;

  write (2, path, strlen (path));
  write (2, "\r\n", 2);

  startpc = load_image (task, path);
  thread_create (task, &thread);
  arg_len = stringlen + (argc + 2) * sizeof (char *) + sizeof (integer_t);
  arg_len += 5 * sizeof (int);
  stack_end = VM_MAX_ADDRESS;
  stack_start = VM_MAX_ADDRESS - 16 * 1024 * 1024;
  vm_allocate (task, &stack_start, stack_end - stack_start, FALSE);
  reg_size = i386_THREAD_STATE_COUNT;
  thread_get_state (thread, i386_THREAD_STATE,
		    (thread_state_t) &regs, &reg_size);
  regs.eip = (int) startpc;
  regs.uesp = (int) ((stack_end - arg_len) & ~(sizeof (int) - 1));
  thread_set_state (thread, i386_THREAD_STATE,
		    (thread_state_t) &regs, reg_size);
  arg_pos = (void *) regs.uesp;
  vm_allocate (mach_task_self (), (vm_address_t *) &args,
	       stack_end - trunc_page ((vm_offset_t) arg_pos), TRUE);
  str_start = ((vm_address_t) arg_pos
	       + (argc + 2) * sizeof (char *) + sizeof (integer_t));
  p = args + ((vm_address_t) arg_pos & (vm_page_size - 1));
  *((int *) p)++ = argc;
  for (i = 0; i < argc; i++)
    *((char **) p)++ = argv[i] - strings + (char *) str_start;
  *((char **) p)++ = 0;
  *((char **) p)++ = 0;
  memcpy (p, strings, stringlen);
  bzero (args, (vm_offset_t) arg_pos & (vm_page_size - 1));
  vm_write (task, trunc_page ((vm_offset_t) arg_pos), (vm_address_t) args,
	    stack_end - trunc_page ((vm_offset_t) arg_pos));
  vm_deallocate (mach_task_self (), (vm_address_t) args,
		 stack_end - trunc_page ((vm_offset_t) arg_pos));
  thread_resume (thread);
  mach_port_deallocate (mach_task_self (), thread);
  return 0;
}

int
main (int argc, char **argv, char **envp)
{
  mach_port_t foo;
  static const char usagemsg[]
    = "Usage: boot [-D dir] [SWITCHES] SCRIPT ROOT-DEVICE\n";
  char *buf = 0;
  char *bootscript;
  int i, len;
  char *newargs;

  get_privileged_ports (&privileged_host_port, &master_device_port);

  defpager = MACH_PORT_NULL;
  vm_set_default_memory_manager (privileged_host_port, &defpager);

  if (argc < 2 || (argv[1][0] == '-' && argc < 3))
    {
    usage:
      write (2, usagemsg, sizeof usagemsg);
      host_exit (1);
    }

  if (!strcmp (argv[1], "-D"))
    {
      if (argc < 4)
	goto usage;
      useropen_dir = argv[2];
      argv += 2;
    }

  if (argv[1][0] != '-')
    {
      bootstrap_args = "-x";
      bootscript = argv[1];
      bootdevice = argv[2];
    }
  else
    {
      bootstrap_args = argv[1];
      bootscript = argv[2];
      bootdevice = argv[3];
    }

  newargs = malloc (strlen (bootstrap_args) + 2);
  strcpy (newargs, bootstrap_args);
  strcat (newargs, "f");
  bootstrap_args = newargs;

  mach_port_allocate (mach_task_self (), MACH_PORT_RIGHT_PORT_SET,
		      &receive_set);

  mach_port_allocate (mach_task_self (), MACH_PORT_RIGHT_RECEIVE,
		      &pseudo_master_device_port);
  mach_port_insert_right (mach_task_self (),
			  pseudo_master_device_port,
			  pseudo_master_device_port,
			  MACH_MSG_TYPE_MAKE_SEND);
  mach_port_move_member (mach_task_self (), pseudo_master_device_port,
			 receive_set);
  mach_port_request_notification (mach_task_self (), pseudo_master_device_port,
				  MACH_NOTIFY_NO_SENDERS, 1,
				  pseudo_master_device_port,
				  MACH_MSG_TYPE_MAKE_SEND_ONCE, &foo);
  if (foo != MACH_PORT_NULL)
    mach_port_deallocate (mach_task_self (), foo);

  mach_port_allocate (mach_task_self (), MACH_PORT_RIGHT_RECEIVE,
		      &pseudo_console);
  mach_port_move_member (mach_task_self (), pseudo_console, receive_set);
  mach_port_request_notification (mach_task_self (), pseudo_console,
				  MACH_NOTIFY_NO_SENDERS, 1, pseudo_console,
				  MACH_MSG_TYPE_MAKE_SEND_ONCE, &foo);
  if (foo != MACH_PORT_NULL)
    mach_port_deallocate (mach_task_self (), foo);

  /* Initialize boot script variables.  */
  if (boot_script_set_variable ("host-port", VAL_PORT,
				(int) privileged_host_port)
      || boot_script_set_variable ("device-port", VAL_PORT,
				   (int) pseudo_master_device_port)
      || boot_script_set_variable ("root-device", VAL_STR, (int) bootdevice)
      || boot_script_set_variable ("boot-args", VAL_STR, (int) bootstrap_args))
    {
      static const char msg[] = "error setting variable";

      write (2, msg, strlen (msg));
      host_exit (1);
    }

  /* Parse the boot script.  */
  {
    char *p, *line;
    static const char filemsg[] = "Can't open boot script";
    int amt, fd, err;

    fd = open (bootscript, O_RDONLY, 0);
    if (fd < 0)
      {
	write (2, filemsg, sizeof (filemsg));
	host_exit (1);
      }
    p = buf = malloc (500);
    len = 500;
    amt = 0;
    while (1)
      {
	i = read (fd, p, len - (p - buf));
	if (i <= 0)
	  break;
	p += i;
	amt += i;
	if (p == buf + len)
	  {
	    char *newbuf;

	    len += 500;
	    newbuf = realloc (buf, len);
	    p = newbuf + (p - buf);
	    buf = newbuf;
	  }
      }
    line = p = buf;
    while (1)
      {
	while (p < buf + amt && *p != '\n')
	  p++;
	*p = '\0';
	err = boot_script_parse_line (line);
	if (err)
	  {
	    char *str;
	    int i;

	    str = boot_script_error_string (err);
	    i = strlen (str);
	    write (2, str, i);
	    write (2, " in `", 5);
	    write (2, line, strlen (line));
	    write (2, "'\n", 2);
	    host_exit (1);
	  }
	if (p == buf + amt)
	  break;
	line = ++p;
      }
  }

  if (index (bootstrap_args, 'd'))
    {
      static const char msg[] = "Pausing. . .";
      char c;
      write (2, msg, sizeof (msg) - 1);
      read (0, &c, 1);
    }

  init_termstate ();

  /* The boot script has now been parsed into internal data structures.
     Now execute its directives.  */
  {
    int err;

    err = boot_script_exec ();
    if (err)
      {
	char *str = boot_script_error_string (err);
	int i = strlen (str);

	write (2, str, i);
	host_exit (1);
      }
    free (buf);
  }

  mach_port_deallocate (mach_task_self (), pseudo_master_device_port);

  cthread_detach (cthread_fork ((cthread_fn_t) msg_thread, (any_t) 0));

  for (;;)
    {
      fd_set rmask;
      FD_ZERO (&rmask);
      FD_SET (0, &rmask);
      if (select (1, &rmask, 0, 0, 0) == 1)
	read_reply ();
      else
	{			/* We hosed */
	  perror ("select");
	  exit (5);
	}
    }

/*  mach_msg_server (request_server, __vm_page_size * 2, receive_set); */
}

/* Set up stack args the Mach way */
void
set_mach_stack_args (user_task,
		     user_thread,
		     startpc,
		     va_alist)
     task_t user_task;
     thread_t user_thread;
     char *startpc;
     va_dcl
{
  /* This code is lifted from .../mk/bootstrap/load.c. */
  va_list			argv_ptr;
  char *			arg_ptr;

  int			arg_len;
  int			arg_count;
  char *			arg_pos;
  unsigned int		arg_item_len;

	/*
	 * Calculate the size of the argument list.
	 */
	va_start(argv_ptr);
	arg_len = 0;
	arg_count = 0;
	for (;;) {
	    arg_ptr = va_arg(argv_ptr, char *);
	    if (arg_ptr == (char *)0)
		break;
	    arg_count++;
	    arg_len += strlen(arg_ptr) + 1;	/* space for '\0' */
	}
	va_end(argv_ptr);
	/*
	 * Add space for:
	 *    arg_count
	 *    pointers to arguments
	 *    trailing 0 pointer
	 *    dummy 0 pointer to environment variables
	 *    and align to integer boundary
	 */
	arg_len += sizeof(integer_t) + (2 + arg_count) * sizeof(char *);
	arg_len = (arg_len + (sizeof(integer_t) - 1)) & ~(sizeof(integer_t)-1);


  /* This small piece is from .../mk/bootstrap/i386/exec.c. */
  {
	vm_offset_t	stack_start;
	vm_offset_t	stack_end;
	struct i386_thread_state	regs;
	unsigned int		reg_size;

#define STACK_SIZE (1024 * 1024 * 16)

	/*
	 * Add space for 5 ints to arguments, for
	 * PS program. XXX
	 */
	arg_len += 5 * sizeof(int);

	/*
	 * Allocate stack.
	 */
	stack_end = VM_MAX_ADDRESS;
	stack_start = VM_MAX_ADDRESS - STACK_SIZE;
	(void)vm_allocate(user_task,
			  &stack_start,
			  (vm_size_t)(stack_end - stack_start),
			  FALSE);

	reg_size = i386_THREAD_STATE_COUNT;
	(void)thread_get_state(user_thread,
				i386_THREAD_STATE,
				(thread_state_t)&regs,
				&reg_size);

	regs.eip = (int) startpc;
	regs.uesp = (int)((stack_end - arg_len) & ~(sizeof(int)-1));

	(void)thread_set_state(user_thread,
				i386_THREAD_STATE,
				(thread_state_t)&regs,
				reg_size);

	arg_pos = (void *) regs.uesp;
      }

	/*
	 * Copy out the arguments.
	 */
	{
	    vm_offset_t	u_arg_start;
				/* user start of argument list block */
	    vm_offset_t	k_arg_start;
				/* kernel start of argument list block */
	    vm_offset_t u_arg_page_start;
				/* user start of args, page-aligned */
	    vm_size_t	arg_page_size;
				/* page_aligned size of args */
	    vm_offset_t	k_arg_page_start;
				/* kernel start of args, page-aligned */

	    register
	    char **	k_ap;	/* kernel arglist address */
	    char *	u_cp;	/* user argument string address */
	    register
	    char *	k_cp;	/* kernel argument string address */
	    register
	    int		i;

	    /*
	     * Get address of argument list in user space
	     */
	    u_arg_start = (vm_offset_t)arg_pos;

	    /*
	     * Round to page boundaries, and allocate kernel copy
	     */
	    u_arg_page_start = trunc_page(u_arg_start);
	    arg_page_size = (vm_size_t)(round_page(u_arg_start + arg_len)
					- u_arg_page_start);

	    vm_allocate(mach_task_self(),
				 &k_arg_page_start,
				 (vm_size_t)arg_page_size,
				 TRUE);

	    /*
	     * Set up addresses corresponding to user pointers
	     * in the kernel block
	     */
	    k_arg_start = k_arg_page_start + (u_arg_start - u_arg_page_start);

	    k_ap = (char **)k_arg_start;

	    /*
	     * Start the strings after the arg-count and pointers
	     */
	    u_cp = (char *)u_arg_start + arg_count * sizeof(char *)
					+ 2 * sizeof(char *)
					+ sizeof(integer_t);
	    k_cp = (char *)k_arg_start + arg_count * sizeof(char *)
					+ 2 * sizeof(char *)
					+ sizeof(integer_t);

	    /*
	     * first the argument count
	     */
	    *k_ap++ = (char *)(natural_t)arg_count;

	    /*
	     * Then the strings and string pointers for each argument
	     */
	    va_start(argv_ptr);
	    for (i = 0; i < arg_count; i++) {
		arg_ptr = va_arg(argv_ptr, char *);
		arg_item_len = strlen(arg_ptr) + 1; /* include trailing 0 */

		/* set string pointer */
		*k_ap++ = u_cp;

		/* copy string */
		bcopy(arg_ptr, k_cp, arg_item_len);
		k_cp += arg_item_len;
		u_cp += arg_item_len;
	    }
	    va_end(argv_ptr);

	    /*
	     * last, the trailing 0 argument and a null environment pointer.
	     */
	    *k_ap++ = (char *)0;
	    *k_ap   = (char *)0;

	    /*
	     * Now write all of this to user space.
	     */
	    (void) vm_write(user_task,
			    u_arg_page_start,
			    k_arg_page_start,
			    arg_page_size);

	    (void) vm_deallocate(mach_task_self(),
				 k_arg_page_start,
				 arg_page_size);
	}
}


void
msg_thread()
{
  while (1)
    mach_msg_server (request_server, 0, receive_set);
}


enum read_type
{
  DEV_READ,
  DEV_READI,
  IO_READ,
};
struct qr
{
  enum read_type type;
  mach_port_t reply_port;
  mach_msg_type_name_t reply_type;
  int amount;
  struct qr *next;
};
struct qr *qrhead, *qrtail;

/* Queue a read for later reply. */
void
queue_read (enum read_type type,
	    mach_port_t reply_port,
	    mach_msg_type_name_t reply_type,
	    int amount)
{
  struct qr *qr;

  spin_lock (&queuelock);

  qr = malloc (sizeof (struct qr));
  qr->type = type;
  qr->reply_port = reply_port;
  qr->reply_type = reply_type;
  qr->amount = amount;
  qr->next = 0;
  if (qrtail)
    qrtail->next = qr;
  else
    qrhead = qrtail = qr;

  spin_unlock (&queuelock);
}

/* TRUE if there's data available on stdin, which should be used to satisfy
   console read requests.  */
static int should_read = 0;

/* Reply to a queued read. */
void
read_reply ()
{
  int avail;
  struct qr *qr;
  char * buf;
  int amtread;

  /* By forcing SHOULD_READ to true before trying the lock, we ensure that
     either we get the lock ourselves or that whoever currently holds the
     lock will service this read when he unlocks it.  */
  should_read = 1;
  if (! spin_try_lock (&readlock))
    return;

  /* Since we're commited to servicing the read, no one else need do so.  */
  should_read = 0;

  ioctl (0, FIONREAD, &avail);
  if (!avail)
    {
      spin_unlock (&readlock);
      return;
    }

  spin_lock (&queuelock);

  if (!qrhead)
    {
      spin_unlock (&queuelock);
      spin_unlock (&readlock);
      return;
    }

  qr = qrhead;
  qrhead = qr->next;
  if (qr == qrtail)
    qrtail = 0;

  spin_unlock (&queuelock);

  if (qr->type == DEV_READ)
    vm_allocate (mach_task_self (), (vm_address_t *)&buf, qr->amount, 1);
  else
    buf = alloca (qr->amount);
  amtread = read (0, buf, qr->amount);

  spin_unlock (&readlock);

  switch (qr->type)
    {
    case DEV_READ:
      if (amtread >= 0)
	ds_device_read_reply (qr->reply_port, qr->reply_type, 0,
			      (io_buf_ptr_t) buf, amtread);
      else
	ds_device_read_reply (qr->reply_port, qr->reply_type, errno, 0, 0);
      break;

    case DEV_READI:
      if (amtread >= 0)
	ds_device_read_reply_inband (qr->reply_port, qr->reply_type, 0,
				     buf, amtread);
      else
	ds_device_read_reply_inband (qr->reply_port, qr->reply_type, errno,
				     0, 0);
      break;

    case IO_READ:
      if (amtread >= 0)
	io_read_reply (qr->reply_port, qr->reply_type, 0,
		       buf, amtread);
      else
	io_read_reply (qr->reply_port, qr->reply_type, errno, 0, 0);
      break;
    }

  free (qr);
}

/* Unlock READLOCK, and also service any new read requests that it was
   blocking.  */
static void
unlock_readlock ()
{
  spin_unlock (&readlock);
  while (should_read)
    read_reply ();
}

/*
 *	Handle bootstrap requests.
 */
/* These two functions from .../mk/bootstrap/default_pager.c. */

kern_return_t
do_bootstrap_privileged_ports(bootstrap, hostp, devicep)
	mach_port_t bootstrap;
	mach_port_t *hostp, *devicep;
{
	*hostp = privileged_host_port;
	*devicep = pseudo_master_device_port;
	return KERN_SUCCESS;
}

void
bootstrap_compat(in, out)
	mach_msg_header_t *in, *out;
{
	mig_reply_header_t *reply = (mig_reply_header_t *) out;
	mach_msg_return_t mr;

	struct imsg {
		mach_msg_header_t	hdr;
		mach_msg_type_t		port_desc_1;
		mach_port_t		port_1;
		mach_msg_type_t		port_desc_2;
		mach_port_t		port_2;
	} imsg;

	/*
	 * Send back the host and device ports.
	 */

	imsg.hdr.msgh_bits = MACH_MSGH_BITS_COMPLEX |
		MACH_MSGH_BITS(MACH_MSGH_BITS_REMOTE(in->msgh_bits), 0);
	/* msgh_size doesn't need to be initialized */
	imsg.hdr.msgh_remote_port = in->msgh_remote_port;
	imsg.hdr.msgh_local_port = MACH_PORT_NULL;
	/* msgh_seqno doesn't need to be initialized */
	imsg.hdr.msgh_id = in->msgh_id + 100;	/* this is a reply msg */

	imsg.port_desc_1.msgt_name = MACH_MSG_TYPE_COPY_SEND;
	imsg.port_desc_1.msgt_size = (sizeof(mach_port_t) * 8);
	imsg.port_desc_1.msgt_number = 1;
	imsg.port_desc_1.msgt_inline = TRUE;
	imsg.port_desc_1.msgt_longform = FALSE;
	imsg.port_desc_1.msgt_deallocate = FALSE;
	imsg.port_desc_1.msgt_unused = 0;

	imsg.port_1 = privileged_host_port;

	imsg.port_desc_2 = imsg.port_desc_1;

	imsg.port_desc_2.msgt_name = MACH_MSG_TYPE_MAKE_SEND;
	imsg.port_2 = pseudo_master_device_port;

	/*
	 * Send the reply message.
	 * (mach_msg_server can not do this, because the reply
	 * is not in standard format.)
	 */

	mr = mach_msg(&imsg.hdr, MACH_SEND_MSG,
		      sizeof imsg, 0, MACH_PORT_NULL,
		      MACH_MSG_TIMEOUT_NONE, MACH_PORT_NULL);
	if (mr != MACH_MSG_SUCCESS)
		(void) mach_port_deallocate(mach_task_self (),
					    imsg.hdr.msgh_remote_port);

	/*
	 * Tell mach_msg_server to do nothing.
	 */

	reply->RetCode = MIG_NO_REPLY;
}

/* Implementation of device interface */

kern_return_t
ds_device_open (mach_port_t master_port,
		mach_port_t reply_port,
		mach_msg_type_name_t reply_type,
		dev_mode_t mode,
		dev_name_t name,
		mach_port_t *device,
		mach_msg_type_name_t *devicetype)
{
  if (master_port != pseudo_master_device_port)
    return D_INVALID_OPERATION;

  if (!strcmp (name, "console"))
    {
#if 0
      mach_port_insert_right (mach_task_self (), pseudo_console,
			      pseudo_console, MACH_MSG_TYPE_MAKE_SEND);
      console_send_rights++;
#endif
      console_mscount++;
      *device = pseudo_console;
      *devicetype = MACH_MSG_TYPE_MAKE_SEND;
      return 0;
    }

  *devicetype = MACH_MSG_TYPE_MOVE_SEND;
  return device_open (master_device_port, mode, name, device);
}

kern_return_t
ds_device_close (device_t device)
{
  if (device != pseudo_console)
    return D_NO_SUCH_DEVICE;
  return 0;
}

kern_return_t
ds_device_write (device_t device,
		 mach_port_t reply_port,
		 mach_msg_type_name_t reply_type,
		 dev_mode_t mode,
		 recnum_t recnum,
		 io_buf_ptr_t data,
		 unsigned int datalen,
		 int *bytes_written)
{
  if (device != pseudo_console)
    return D_NO_SUCH_DEVICE;

#if 0
  if (console_send_rights)
    {
      mach_port_mod_refs (mach_task_self (), pseudo_console,
			  MACH_PORT_TYPE_SEND, -console_send_rights);
      console_send_rights = 0;
    }
#endif

  *bytes_written = write (1, data, datalen);

  return (*bytes_written == -1 ? D_IO_ERROR : D_SUCCESS);
}

kern_return_t
ds_device_write_inband (device_t device,
			mach_port_t reply_port,
			mach_msg_type_name_t reply_type,
			dev_mode_t mode,
			recnum_t recnum,
			io_buf_ptr_inband_t data,
			unsigned int datalen,
			int *bytes_written)
{
  if (device != pseudo_console)
    return D_NO_SUCH_DEVICE;

#if 0
  if (console_send_rights)
    {
      mach_port_mod_refs (mach_task_self (), pseudo_console,
			  MACH_PORT_TYPE_SEND, -console_send_rights);
      console_send_rights = 0;
    }
#endif

  *bytes_written = write (1, data, datalen);

  return (*bytes_written == -1 ? D_IO_ERROR : D_SUCCESS);
}

kern_return_t
ds_device_read (device_t device,
		mach_port_t reply_port,
		mach_msg_type_name_t reply_type,
		dev_mode_t mode,
		recnum_t recnum,
		int bytes_wanted,
		io_buf_ptr_t *data,
		unsigned int *datalen)
{
  int avail;

  if (device != pseudo_console)
    return D_NO_SUCH_DEVICE;

#if 0
  if (console_send_rights)
    {
      mach_port_mod_refs (mach_task_self (), pseudo_console,
			  MACH_PORT_TYPE_SEND, -console_send_rights);
      console_send_rights = 0;
    }
#endif

  spin_lock (&readlock);
  ioctl (0, FIONREAD, &avail);
  if (avail)
    {
      vm_allocate (mach_task_self (), (pointer_t *)data, bytes_wanted, 1);
      *datalen = read (0, *data, bytes_wanted);
      unlock_readlock ();
      return (*datalen == -1 ? D_IO_ERROR : D_SUCCESS);
    }
  else
    {
      unlock_readlock ();
      queue_read (DEV_READ, reply_port, reply_type, bytes_wanted);
      return MIG_NO_REPLY;
    }
}

kern_return_t
ds_device_read_inband (device_t device,
		       mach_port_t reply_port,
		       mach_msg_type_name_t reply_type,
		       dev_mode_t mode,
		       recnum_t recnum,
		       int bytes_wanted,
		       io_buf_ptr_inband_t data,
		       unsigned int *datalen)
{
  int avail;

  if (device != pseudo_console)
    return D_NO_SUCH_DEVICE;

#if 0
  if (console_send_rights)
    {
      mach_port_mod_refs (mach_task_self (), pseudo_console,
			  MACH_PORT_TYPE_SEND, -console_send_rights);
      console_send_rights = 0;
    }
#endif

  spin_lock (&readlock);
  ioctl (0, FIONREAD, &avail);
  if (avail)
    {
      *datalen = read (0, data, bytes_wanted);
      unlock_readlock ();
      return (*datalen == -1 ? D_IO_ERROR : D_SUCCESS);
    }
  else
    {
      unlock_readlock ();
      queue_read (DEV_READI, reply_port, reply_type, bytes_wanted);
      return MIG_NO_REPLY;
    }
}

kern_return_t
ds_xxx_device_set_status (device_t device,
			  dev_flavor_t flavor,
			  dev_status_t status,
			  u_int statu_cnt)
{
  if (device != pseudo_console)
    return D_NO_SUCH_DEVICE;
  return D_INVALID_OPERATION;
}

kern_return_t
ds_xxx_device_get_status (device_t device,
			  dev_flavor_t flavor,
			  dev_status_t status,
			  u_int *statuscnt)
{
  if (device != pseudo_console)
    return D_NO_SUCH_DEVICE;
  return D_INVALID_OPERATION;
}

kern_return_t
ds_xxx_device_set_filter (device_t device,
			  mach_port_t rec,
			  int pri,
			  filter_array_t filt,
			  unsigned int len)
{
  if (device != pseudo_console)
    return D_NO_SUCH_DEVICE;
  return D_INVALID_OPERATION;
}

kern_return_t
ds_device_map (device_t device,
	       vm_prot_t prot,
	       vm_offset_t offset,
	       vm_size_t size,
	       memory_object_t *pager,
	       int unmap)
{
  if (device != pseudo_console)
    return D_NO_SUCH_DEVICE;
  return D_INVALID_OPERATION;
}

kern_return_t
ds_device_set_status (device_t device,
		      dev_flavor_t flavor,
		      dev_status_t status,
		      unsigned int statuslen)
{
  if (device != pseudo_console)
    return D_NO_SUCH_DEVICE;
  return D_INVALID_OPERATION;
}

kern_return_t
ds_device_get_status (device_t device,
		      dev_flavor_t flavor,
		      dev_status_t status,
		      unsigned int *statuslen)
{
  if (device != pseudo_console)
    return D_NO_SUCH_DEVICE;
  return D_INVALID_OPERATION;
}

kern_return_t
ds_device_set_filter (device_t device,
		      mach_port_t receive_port,
		      int priority,
		      filter_array_t filter,
		      unsigned int filterlen)
{
  if (device != pseudo_console)
    return D_NO_SUCH_DEVICE;
  return D_INVALID_OPERATION;
}


/* Implementation of notify interface */
kern_return_t
do_mach_notify_port_deleted (mach_port_t notify,
			     mach_port_t name)
{
  return EOPNOTSUPP;
}

kern_return_t
do_mach_notify_msg_accepted (mach_port_t notify,
			     mach_port_t name)
{
  return EOPNOTSUPP;
}

kern_return_t
do_mach_notify_port_destroyed (mach_port_t notify,
			       mach_port_t port)
{
  return EOPNOTSUPP;
}

kern_return_t
do_mach_notify_no_senders (mach_port_t notify,
			   mach_port_mscount_t mscount)
{
  static int no_console;
  mach_port_t foo;
  if (notify == pseudo_master_device_port)
    {
      if (no_console)
	goto bye;
      pseudo_master_device_port = MACH_PORT_NULL;
      return 0;
    }
  if (notify == pseudo_console)
    {
      if (mscount == console_mscount &&
	  pseudo_master_device_port == MACH_PORT_NULL)
	{
	bye:
	  restore_termstate ();
	  write (2, "bye\n", 4);
	  host_exit (0);
	}
      else
	{
	  no_console = (mscount == console_mscount);

	  mach_port_request_notification (mach_task_self (), pseudo_console,
					  MACH_NOTIFY_NO_SENDERS,
					  console_mscount == mscount
					  ? mscount + 1
					  : console_mscount,
					  pseudo_console,
					  MACH_MSG_TYPE_MAKE_SEND_ONCE, &foo);
	  if (foo != MACH_PORT_NULL)
	    mach_port_deallocate (mach_task_self (), foo);
	}
    }

  return EOPNOTSUPP;
}

kern_return_t
do_mach_notify_send_once (mach_port_t notify)
{
  return EOPNOTSUPP;
}

kern_return_t
do_mach_notify_dead_name (mach_port_t notify,
			  mach_port_t name)
{
#if 0
  if (name == child_task && notify == bootport)
    host_exit (0);
#endif
  return EOPNOTSUPP;
}


/* Implementation of the Hurd I/O interface, which
   we support for the console port only. */

kern_return_t
S_io_write (mach_port_t object,
	    mach_port_t reply_port,
	    mach_msg_type_name_t reply_type,
	    char *data,
	    mach_msg_type_number_t datalen,
	    off_t offset,
	    mach_msg_type_number_t *amtwritten)
{
  if (object != pseudo_console)
    return EOPNOTSUPP;

#if 0
  if (console_send_rights)
    {
      mach_port_mod_refs (mach_task_self (), pseudo_console,
			  MACH_PORT_TYPE_SEND, -console_send_rights);
      console_send_rights = 0;
    }
#endif

  *amtwritten = write (1, data, datalen);
  return *amtwritten == -1 ? errno : 0;
}

kern_return_t
S_io_read (mach_port_t object,
	   mach_port_t reply_port,
	   mach_msg_type_name_t reply_type,
	   char **data,
	   mach_msg_type_number_t *datalen,
	   off_t offset,
	   mach_msg_type_number_t amount)
{
  mach_msg_type_number_t avail;

  if (object != pseudo_console)
    return EOPNOTSUPP;

#if 0
  if (console_send_rights)
    {
      mach_port_mod_refs (mach_task_self (), pseudo_console,
			  MACH_PORT_TYPE_SEND, -console_send_rights);
      console_send_rights = 0;
    }
#endif

  spin_lock (&readlock);
  ioctl (0, FIONREAD, &avail);
  if (avail)
    {
      if (amount > *datalen)
	vm_allocate (mach_task_self (), (vm_address_t *) data, amount, 1);
      *datalen = read (0, *data, amount);
      unlock_readlock ();
      return *datalen == -1 ? errno : 0;
    }
  else
    {
      unlock_readlock ();
      queue_read (IO_READ, reply_port, reply_type, amount);
      return MIG_NO_REPLY;
    }
}

kern_return_t
S_io_seek (mach_port_t object,
	   mach_port_t reply_port,
	   mach_msg_type_name_t reply_type,
	   off_t offset,
	   int whence,
	   off_t *newp)
{
  return object == pseudo_console ? ESPIPE : EOPNOTSUPP;
}

kern_return_t
S_io_readable (mach_port_t object,
	       mach_port_t reply_port,
	       mach_msg_type_name_t reply_type,
	       mach_msg_type_number_t *amt)
{
  if (object != pseudo_console)
    return EOPNOTSUPP;
  ioctl (0, FIONREAD, amt);
  return 0;
}

kern_return_t
S_io_set_all_openmodes (mach_port_t object,
			mach_port_t reply_port,
			mach_msg_type_name_t reply_type,
			int bits)
{
  return EOPNOTSUPP;
}

kern_return_t
S_io_get_openmodes (mach_port_t object,
		    mach_port_t reply_port,
		    mach_msg_type_name_t reply_type,
		    int *modes)
{
  *modes = O_READ | O_WRITE;
  return object == pseudo_console ? 0 : EOPNOTSUPP;
}

kern_return_t
S_io_set_some_openmodes (mach_port_t object,
			 mach_port_t reply_port,
			 mach_msg_type_name_t reply_type,
			 int bits)
{
  return EOPNOTSUPP;
}

kern_return_t
S_io_clear_some_openmodes (mach_port_t object,
			   mach_port_t reply_port,
			   mach_msg_type_name_t reply_type,
			   int bits)
{
  return EOPNOTSUPP;
}

kern_return_t
S_io_async (mach_port_t object,
	    mach_port_t reply_port,
	    mach_msg_type_name_t reply_type,
	    mach_port_t notify,
	    mach_port_t *id,
	    mach_msg_type_name_t *idtype)
{
  return EOPNOTSUPP;
}

kern_return_t
S_io_mod_owner (mach_port_t object,
		mach_port_t reply_port,
		mach_msg_type_name_t reply_type,
		pid_t owner)
{
  return EOPNOTSUPP;
}

kern_return_t
S_io_get_owner (mach_port_t object,
		mach_port_t reply_port,
		mach_msg_type_name_t reply_type,
		pid_t *owner)
{
  return EOPNOTSUPP;
}

kern_return_t
S_io_get_icky_async_id (mach_port_t object,
			mach_port_t reply_port,
			mach_msg_type_name_t reply_type,
			mach_port_t *id,
			mach_msg_type_name_t *idtype)
{
  return EOPNOTSUPP;
}

kern_return_t
S_io_select (mach_port_t object,
	     mach_port_t reply_port,
	     mach_msg_type_name_t reply_type,
	     int *type)
{
  fd_set r, w, x;
  int n;

  if (object != pseudo_console)
    return EOPNOTSUPP;

  FD_ZERO (&r);
  FD_ZERO (&w);
  FD_ZERO (&x);
  FD_SET (0, &r);
  FD_SET (0, &w);
  FD_SET (0, &x);

  n = select (1,
	      (*type & SELECT_READ) ? &r : 0,
	      (*type & SELECT_WRITE) ? &w : 0,
	      (*type & SELECT_URG) ? &x : 0,
	      0);
  if (n < 0)
    return errno;

  if (! FD_ISSET (0, &r))
    *type &= ~SELECT_READ;
  if (! FD_ISSET (0, &w))
    *type &= ~SELECT_WRITE;
  if (! FD_ISSET (0, &x))
    *type &= ~SELECT_URG;

  return 0;
}

kern_return_t
S_io_stat (mach_port_t object,
	   mach_port_t reply_port,
	   mach_msg_type_name_t reply_type,
	   struct stat *st)
{
  if (object != pseudo_console)
    return EOPNOTSUPP;

  bzero (st, sizeof (struct stat));
  st->st_blksize = 1024;
  return 0;
}

kern_return_t
S_io_reauthenticate (mach_port_t object,
		     mach_port_t reply_port,
		     mach_msg_type_name_t reply_type,
		     mach_port_t rend)
{
  uid_t *gu, *au;
  gid_t *gg, *ag;
  unsigned int gulen = 0, aulen = 0, gglen = 0, aglen = 0;

  if (! auth_server_authenticate (authserver,
				  rend, MACH_MSG_TYPE_COPY_SEND,
				  object, MACH_MSG_TYPE_MAKE_SEND,
				  &gu, &gulen,
				  &au, &aulen,
				  &gg, &gglen,
				  &ag, &aglen))
    {
      mig_deallocate ((vm_address_t) gu, gulen * sizeof *gu);
      mig_deallocate ((vm_address_t) au, aulen * sizeof *gu);
      mig_deallocate ((vm_address_t) gg, gglen * sizeof *gu);
      mig_deallocate ((vm_address_t) au, aulen * sizeof *gu);
    }
  mach_port_deallocate (mach_task_self (), rend);

  return 0;
}

kern_return_t
S_io_restrict_auth (mach_port_t object,
		    mach_port_t reply_port,
		    mach_msg_type_name_t reply_type,
		    mach_port_t *newobject,
		    mach_msg_type_name_t *newobjtype,
		    uid_t *uids,
		    u_int nuids,
		    uid_t *gids,
		    u_int ngids)
{
  if (object != pseudo_console)
    return EOPNOTSUPP;
  *newobject = pseudo_console;
  *newobjtype = MACH_MSG_TYPE_MAKE_SEND;
  console_mscount++;
  return 0;
}

kern_return_t
S_io_duplicate (mach_port_t object,
		mach_port_t reply_port,
		mach_msg_type_name_t reply_type,
		mach_port_t *newobj,
		mach_msg_type_name_t *newobjtype)
{
  if (object != pseudo_console)
    return EOPNOTSUPP;
  *newobj = pseudo_console;
  *newobjtype = MACH_MSG_TYPE_MAKE_SEND;
  console_mscount++;
  return 0;
}

kern_return_t
S_io_server_version (mach_port_t object,
		     mach_port_t reply_port,
		     mach_msg_type_name_t reply_type,
		     char *name,
		     int *maj,
		     int *min,
		     int *edit)
{
  return EOPNOTSUPP;
}

kern_return_t
S_io_map (mach_port_t obj,
	  mach_port_t reply_port,
	  mach_msg_type_name_t reply_type,
	  mach_port_t *rd,
	  mach_msg_type_name_t *rdtype,
	  mach_port_t *wr,
	  mach_msg_type_name_t *wrtype)
{
  return EOPNOTSUPP;
}

kern_return_t
S_io_map_cntl (mach_port_t obj,
	       mach_port_t reply_port,
	       mach_msg_type_name_t reply_type,
	       mach_port_t *mem,
	       mach_msg_type_name_t *memtype)
{
  return EOPNOTSUPP;
}

kern_return_t
S_io_get_conch (mach_port_t obj,
		mach_port_t reply_port,
		mach_msg_type_name_t reply_type)
{
  return EOPNOTSUPP;
}

kern_return_t
S_io_release_conch (mach_port_t obj,
		    mach_port_t reply_port,
		    mach_msg_type_name_t reply_type)
{
  return EOPNOTSUPP;
}

kern_return_t
S_io_eofnotify (mach_port_t obj,
		mach_port_t reply_port,
		mach_msg_type_name_t reply_type)

{
  return EOPNOTSUPP;
}

kern_return_t
S_io_prenotify (mach_port_t obj,
		mach_port_t reply_port,
		mach_msg_type_name_t reply_type,
		vm_offset_t start,
		vm_offset_t end)
{
  return EOPNOTSUPP;
}

kern_return_t
S_io_postnotify (mach_port_t obj,
		 mach_port_t reply_port,
		 mach_msg_type_name_t reply_type,
		 vm_offset_t start,
		 vm_offset_t end)
{
  return EOPNOTSUPP;
}

kern_return_t
S_io_readsleep (mach_port_t obj,
		mach_port_t reply_port,
		mach_msg_type_name_t reply_type)
{
  return EOPNOTSUPP;
}

kern_return_t
S_io_readnotify (mach_port_t obj,
		 mach_port_t reply_port,
		 mach_msg_type_name_t reply_type)
{
  return EOPNOTSUPP;
}


kern_return_t
S_io_sigio (mach_port_t obj,
	    mach_port_t reply_port,
	    mach_msg_type_name_t reply_type)
{
  return EOPNOTSUPP;
}


kern_return_t
S_io_pathconf (mach_port_t obj,
	       mach_port_t reply_port,
	       mach_msg_type_name_t reply_type,
	       int name, int *value)
{
  return EOPNOTSUPP;
}

kern_return_t
S_io_identity (mach_port_t obj,
	       mach_port_t reply,
	       mach_msg_type_name_t replytype,
	       mach_port_t *id,
	       mach_msg_type_name_t *idtype,
	       mach_port_t *fsid,
	       mach_msg_type_name_t *fsidtype,
	       int *fileno)
{
  return EOPNOTSUPP;
}



/* Implementation of the Hurd terminal driver interface, which we only
   support on the console device.  */

kern_return_t
S_termctty_open_terminal (mach_port_t object,
			  int flags,
			  mach_port_t *result,
			  mach_msg_type_name_t *restype)
{
  return EOPNOTSUPP;
}

kern_return_t
S_term_getctty (mach_port_t object,
		mach_port_t *cttyid, mach_msg_type_name_t *cttyPoly)
{
  static mach_port_t id = MACH_PORT_NULL;

  if (object != pseudo_console)
    return EOPNOTSUPP;

  if (id == MACH_PORT_NULL)
    mach_port_allocate (mach_task_self (), MACH_PORT_RIGHT_DEAD_NAME, &id);

  *cttyid = id;
  *cttyPoly = MACH_MSG_TYPE_COPY_SEND;
  return 0;
}


kern_return_t S_term_open_ctty
(
	io_t terminal,
	pid_t pid,
	pid_t pgrp,
	mach_port_t *newtty,
	mach_msg_type_name_t *newttytype
)
{ return EOPNOTSUPP; }

kern_return_t S_term_set_nodename
(
	io_t terminal,
	string_t name
)
{ return EOPNOTSUPP; }

kern_return_t S_term_get_nodename
(
	io_t terminal,
	string_t name
)
{ return EOPNOTSUPP; }

kern_return_t S_term_set_filenode
(
	io_t terminal,
	file_t filenode
)
{ return EOPNOTSUPP; }

kern_return_t S_term_get_bottom_type
(
	io_t terminal,
	int *ttype
)
{ return EOPNOTSUPP; }

kern_return_t S_term_on_machdev
(
	io_t terminal,
	mach_port_t machdev
)
{ return EOPNOTSUPP; }

kern_return_t S_term_on_hurddev
(
	io_t terminal,
	io_t hurddev
)
{ return EOPNOTSUPP; }

kern_return_t S_term_on_pty
(
	io_t terminal,
	io_t *ptymaster
)
{ return EOPNOTSUPP; }
