// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (c) 2023, Linaro Limited
 */

#include <linux/module.h>
#include <linux/slab.h>
#include <linux/platform_device.h>
#include <linux/device.h>
#include <linux/kernel.h>
#include <linux/component.h>
#include <linux/pm_runtime.h>
#include <linux/irq.h>
#include <linux/irqdomain.h>
#include <linux/of.h>
#include <linux/soundwire/sdw.h>
#include <linux/soundwire/sdw_type.h>
#include <linux/soundwire/sdw_registers.h>
#include <linux/regmap.h>
#include <sound/soc.h>
#include <sound/soc-dapm.h>
#include "wcd939x.h"

#define SWRS_SCP_HOST_CLK_DIV2_CTL_BANK(m) (0xE0 + 0x10 * (m))

static struct wcd939x_sdw_ch_info wcd939x_sdw_rx_ch_info[] = {
	WCD_SDW_CH(WCD939X_HPH_L, WCD939X_HPH_PORT, BIT(0)),
	WCD_SDW_CH(WCD939X_HPH_R, WCD939X_HPH_PORT, BIT(1)),
	WCD_SDW_CH(WCD939X_CLSH, WCD939X_CLSH_PORT, BIT(0)),
	WCD_SDW_CH(WCD939X_COMP_L, WCD939X_COMP_PORT, BIT(0)),
	WCD_SDW_CH(WCD939X_COMP_R, WCD939X_COMP_PORT, BIT(1)),
	WCD_SDW_CH(WCD939X_LO, WCD939X_LO_PORT, BIT(0)),
	WCD_SDW_CH(WCD939X_DSD_L, WCD939X_DSD_PORT, BIT(0)),
	WCD_SDW_CH(WCD939X_DSD_R, WCD939X_DSD_PORT, BIT(1)),
	WCD_SDW_CH(WCD939X_HIFI_PCM_L, WCD939X_HIFI_PCM_PORT, BIT(0)),
	WCD_SDW_CH(WCD939X_HIFI_PCM_R, WCD939X_HIFI_PCM_PORT, BIT(1)),
};

static struct wcd939x_sdw_ch_info wcd939x_sdw_tx_ch_info[] = {
	WCD_SDW_CH(WCD939X_ADC1, WCD939X_ADC_1_4_PORT, BIT(0)),
	WCD_SDW_CH(WCD939X_ADC2, WCD939X_ADC_1_4_PORT, BIT(1)),
	WCD_SDW_CH(WCD939X_ADC3, WCD939X_ADC_1_4_PORT, BIT(2)),
	WCD_SDW_CH(WCD939X_ADC4, WCD939X_ADC_1_4_PORT, BIT(3)),
	WCD_SDW_CH(WCD939X_DMIC0, WCD939X_DMIC_0_3_MBHC_PORT, BIT(0)),
	WCD_SDW_CH(WCD939X_DMIC1, WCD939X_DMIC_0_3_MBHC_PORT, BIT(1)),
	WCD_SDW_CH(WCD939X_MBHC, WCD939X_DMIC_0_3_MBHC_PORT, BIT(2)),
	WCD_SDW_CH(WCD939X_DMIC2, WCD939X_DMIC_0_3_MBHC_PORT, BIT(2)),
	WCD_SDW_CH(WCD939X_DMIC3, WCD939X_DMIC_0_3_MBHC_PORT, BIT(3)),
	WCD_SDW_CH(WCD939X_DMIC4, WCD939X_DMIC_3_7_PORT, BIT(0)),
	WCD_SDW_CH(WCD939X_DMIC5, WCD939X_DMIC_3_7_PORT, BIT(1)),
	WCD_SDW_CH(WCD939X_DMIC6, WCD939X_DMIC_3_7_PORT, BIT(2)),
	WCD_SDW_CH(WCD939X_DMIC7, WCD939X_DMIC_3_7_PORT, BIT(3)),
};

static struct sdw_dpn_prop wcd939x_rx_dpn_prop[WCD939X_MAX_RX_SWR_PORTS] = {
	{
		.num = WCD939X_HPH_PORT,
		.type = SDW_DPN_SIMPLE,
		.min_ch = 1,
		.max_ch = 2,
		.simple_ch_prep_sm = true,
	},
	{
		.num = WCD939X_CLSH_PORT,
		.type = SDW_DPN_SIMPLE,
		.min_ch = 1,
		.max_ch = 1,
		.simple_ch_prep_sm = true,
	},
	{
		.num = WCD939X_COMP_PORT,
		.type = SDW_DPN_SIMPLE,
		.min_ch = 1,
		.max_ch = 2,
		.simple_ch_prep_sm = true,
	},
	{
		.num = WCD939X_LO_PORT,
		.type = SDW_DPN_SIMPLE,
		.min_ch = 1,
		.max_ch = 1,
		.simple_ch_prep_sm = true,
	},
	{
		.num = WCD939X_DSD_PORT,
		.type = SDW_DPN_SIMPLE,
		.min_ch = 1,
		.max_ch = 2,
		.simple_ch_prep_sm = true,
	},
	{
		.num = WCD939X_HIFI_PCM_PORT,
		.type = SDW_DPN_SIMPLE,
		.min_ch = 1,
		.max_ch = 2,
		.simple_ch_prep_sm = true,
	}
};

static struct sdw_dpn_prop wcd939x_tx_dpn_prop[WCD939X_MAX_TX_SWR_PORTS] = {
	{
		.num = WCD939X_ADC_1_4_PORT,
		.type = SDW_DPN_SIMPLE,
		.min_ch = 1,
		.max_ch = 4,
		.simple_ch_prep_sm = true,
	},
	{
		.num = WCD939X_ADC_DMIC_1_2_PORT,
		.type = SDW_DPN_SIMPLE,
		.min_ch = 1,
		.max_ch = 4,
		.simple_ch_prep_sm = true,
	},
	{
		.num = WCD939X_DMIC_0_3_MBHC_PORT,
		.type = SDW_DPN_SIMPLE,
		.min_ch = 1,
		.max_ch = 4,
		.simple_ch_prep_sm = true,
	},
	{
		.num = WCD939X_DMIC_3_7_PORT,
		.type = SDW_DPN_SIMPLE,
		.min_ch = 1,
		.max_ch = 4,
		.simple_ch_prep_sm = true,
	}
};

struct device *wcd939x_sdw_device_get(struct device_node *np)
{
	return bus_find_device_by_of_node(&sdw_bus_type, np);
}
EXPORT_SYMBOL_GPL(wcd939x_sdw_device_get);

unsigned int wcd939x_swr_get_current_bank(struct sdw_slave *sdev)
{
	return FIELD_GET(SDW_SCP_STAT_CURR_BANK,
			 sdw_read(sdev, SDW_SCP_CTRL));
}
EXPORT_SYMBOL_GPL(wcd939x_swr_get_current_bank);

int wcd939x_sdw_hw_params(struct wcd939x_sdw_priv *wcd,
			  struct snd_pcm_substream *substream,
			  struct snd_pcm_hw_params *params,
			  struct snd_soc_dai *dai)
{
	struct sdw_port_config port_config[WCD939X_MAX_SWR_PORTS];
	unsigned long ch_mask;
	int i, j;

	wcd->sconfig.ch_count = 1;
	wcd->active_ports = 0;
	for (i = 0; i < WCD939X_MAX_SWR_PORTS; i++) {
		ch_mask = wcd->port_config[i].ch_mask;

		if (!ch_mask)
			continue;

		for_each_set_bit(j, &ch_mask, 4)
			wcd->sconfig.ch_count++;

		port_config[wcd->active_ports] = wcd->port_config[i];
		wcd->active_ports++;
	}

	wcd->sconfig.bps = 1;
	wcd->sconfig.frame_rate = params_rate(params);
	if (wcd->is_tx)
		wcd->sconfig.direction = SDW_DATA_DIR_TX;
	else
		wcd->sconfig.direction = SDW_DATA_DIR_RX;

	wcd->sconfig.type = SDW_STREAM_PCM;

	return sdw_stream_add_slave(wcd->sdev, &wcd->sconfig, &port_config[0],
				    wcd->active_ports, wcd->sruntime);
}
EXPORT_SYMBOL_GPL(wcd939x_sdw_hw_params);

int wcd939x_sdw_free(struct wcd939x_sdw_priv *wcd,
		     struct snd_pcm_substream *substream,
		     struct snd_soc_dai *dai)
{
	sdw_stream_remove_slave(wcd->sdev, wcd->sruntime);

	return 0;
}
EXPORT_SYMBOL_GPL(wcd939x_sdw_free);

int wcd939x_sdw_set_sdw_stream(struct wcd939x_sdw_priv *wcd,
			       struct snd_soc_dai *dai, void *stream,
			       int direction)
{
	wcd->sruntime = stream;

	return 0;
}
EXPORT_SYMBOL_GPL(wcd939x_sdw_set_sdw_stream);

struct regmap *wcd939x_swr_get_regmap(struct wcd939x_sdw_priv *wcd)
{
	if (wcd->regmap)
		return wcd->regmap;

	return ERR_PTR(-EINVAL);
}
EXPORT_SYMBOL_GPL(wcd939x_swr_get_regmap);

static int wcd9390_update_status(struct sdw_slave *slave,
				 enum sdw_slave_status status)
{
	struct wcd939x_sdw_priv *wcd = dev_get_drvdata(&slave->dev);

	if (wcd->regmap && status == SDW_SLAVE_ATTACHED) {
		/* Write out any cached changes that happened between probe and attach */
		regcache_cache_only(wcd->regmap, false);
		return regcache_sync(wcd->regmap);
	}

	return 0;
}

static int wcd9390_bus_config(struct sdw_slave *slave,
			      struct sdw_bus_params *params)
{
	sdw_write(slave, SWRS_SCP_HOST_CLK_DIV2_CTL_BANK(params->next_bank),
		  0x01);

	return 0;
}

/*
 * Handle Soundwire out-of-band interrupt event by triggering
 * the first irq of the slave_irq irq domain, which then will
 * be handled by the regmap_irq threaded irq.
 * Looping is to ensure no interrupts were missed in the process.
 */
static int wcd9390_interrupt_callback(struct sdw_slave *slave,
				      struct sdw_slave_intr_status *status)
{
	struct wcd939x_sdw_priv *wcd = dev_get_drvdata(&slave->dev);
	struct irq_domain *slave_irq = wcd->slave_irq;
	u32 sts1, sts2, sts3;

	do {
		handle_nested_irq(irq_find_mapping(slave_irq, 0));
		regmap_read(wcd->regmap, WCD939X_DIGITAL_INTR_STATUS_0, &sts1);
		regmap_read(wcd->regmap, WCD939X_DIGITAL_INTR_STATUS_1, &sts2);
		regmap_read(wcd->regmap, WCD939X_DIGITAL_INTR_STATUS_2, &sts3);

	} while (sts1 || sts2 || sts3);

	return IRQ_HANDLED;
}

static const struct reg_default wcd939x_defaults[] = {
	/* Default values except for Read-Only & Volatile registers */
	{ WCD939X_ANA_PAGE, 0x00 },
	{ WCD939X_ANA_BIAS, 0x00 },
	{ WCD939X_ANA_RX_SUPPLIES, 0x00 },
	{ WCD939X_ANA_HPH, 0x0c },
	{ WCD939X_ANA_EAR, 0x00 },
	{ WCD939X_ANA_EAR_COMPANDER_CTL, 0x02 },
	{ WCD939X_ANA_TX_CH1, 0x20 },
	{ WCD939X_ANA_TX_CH2, 0x00 },
	{ WCD939X_ANA_TX_CH3, 0x20 },
	{ WCD939X_ANA_TX_CH4, 0x00 },
	{ WCD939X_ANA_MICB1_MICB2_DSP_EN_LOGIC, 0x00 },
	{ WCD939X_ANA_MICB3_DSP_EN_LOGIC, 0x00 },
	{ WCD939X_ANA_MBHC_MECH, 0x39 },
	{ WCD939X_ANA_MBHC_ELECT, 0x08 },
	{ WCD939X_ANA_MBHC_ZDET, 0x00 },
	{ WCD939X_ANA_MBHC_BTN0, 0x00 },
	{ WCD939X_ANA_MBHC_BTN1, 0x10 },
	{ WCD939X_ANA_MBHC_BTN2, 0x20 },
	{ WCD939X_ANA_MBHC_BTN3, 0x30 },
	{ WCD939X_ANA_MBHC_BTN4, 0x40 },
	{ WCD939X_ANA_MBHC_BTN5, 0x50 },
	{ WCD939X_ANA_MBHC_BTN6, 0x60 },
	{ WCD939X_ANA_MBHC_BTN7, 0x70 },
	{ WCD939X_ANA_MICB1, 0x10 },
	{ WCD939X_ANA_MICB2, 0x10 },
	{ WCD939X_ANA_MICB2_RAMP, 0x00 },
	{ WCD939X_ANA_MICB3, 0x00 },
	{ WCD939X_ANA_MICB4, 0x00 },
	{ WCD939X_BIAS_CTL, 0x2a },
	{ WCD939X_BIAS_VBG_FINE_ADJ, 0x55 },
	{ WCD939X_LDOL_VDDCX_ADJUST, 0x01 },
	{ WCD939X_LDOL_DISABLE_LDOL, 0x00 },
	{ WCD939X_MBHC_CTL_CLK, 0x00 },
	{ WCD939X_MBHC_CTL_ANA, 0x00 },
	{ WCD939X_MBHC_ZDET_VNEG_CTL, 0x00 },
	{ WCD939X_MBHC_ZDET_BIAS_CTL, 0x46 },
	{ WCD939X_MBHC_CTL_BCS, 0x00 },
	{ WCD939X_MBHC_TEST_CTL, 0x00 },
	{ WCD939X_LDOH_MODE, 0x2b },
	{ WCD939X_LDOH_BIAS, 0x68 },
	{ WCD939X_LDOH_STB_LOADS, 0x00 },
	{ WCD939X_LDOH_SLOWRAMP, 0x50 },
	{ WCD939X_MICB1_TEST_CTL_1, 0x1a },
	{ WCD939X_MICB1_TEST_CTL_2, 0x00 },
	{ WCD939X_MICB1_TEST_CTL_3, 0xa4 },
	{ WCD939X_MICB2_TEST_CTL_1, 0x1a },
	{ WCD939X_MICB2_TEST_CTL_2, 0x00 },
	{ WCD939X_MICB2_TEST_CTL_3, 0x24 },
	{ WCD939X_MICB3_TEST_CTL_1, 0x9a },
	{ WCD939X_MICB3_TEST_CTL_2, 0x80 },
	{ WCD939X_MICB3_TEST_CTL_3, 0x24 },
	{ WCD939X_MICB4_TEST_CTL_1, 0x1a },
	{ WCD939X_MICB4_TEST_CTL_2, 0x80 },
	{ WCD939X_MICB4_TEST_CTL_3, 0x24 },
	{ WCD939X_TX_COM_ADC_VCM, 0x39 },
	{ WCD939X_TX_COM_BIAS_ATEST, 0xe0 },
	{ WCD939X_TX_COM_SPARE1, 0x00 },
	{ WCD939X_TX_COM_SPARE2, 0x00 },
	{ WCD939X_TX_COM_TXFE_DIV_CTL, 0x22 },
	{ WCD939X_TX_COM_TXFE_DIV_START, 0x00 },
	{ WCD939X_TX_COM_SPARE3, 0x00 },
	{ WCD939X_TX_COM_SPARE4, 0x00 },
	{ WCD939X_TX_1_2_TEST_EN, 0xcc },
	{ WCD939X_TX_1_2_ADC_IB, 0xe9 },
	{ WCD939X_TX_1_2_ATEST_REFCTL, 0x0b },
	{ WCD939X_TX_1_2_TEST_CTL, 0x38 },
	{ WCD939X_TX_1_2_TEST_BLK_EN1, 0xff },
	{ WCD939X_TX_1_2_TXFE1_CLKDIV, 0x00 },
	{ WCD939X_TX_3_4_TEST_EN, 0xcc },
	{ WCD939X_TX_3_4_ADC_IB, 0xe9 },
	{ WCD939X_TX_3_4_ATEST_REFCTL, 0x0b },
	{ WCD939X_TX_3_4_TEST_CTL, 0x38 },
	{ WCD939X_TX_3_4_TEST_BLK_EN3, 0xff },
	{ WCD939X_TX_3_4_TXFE3_CLKDIV, 0x00 },
	{ WCD939X_TX_3_4_TEST_BLK_EN2, 0xfb },
	{ WCD939X_TX_3_4_TXFE2_CLKDIV, 0x00 },
	{ WCD939X_TX_3_4_SPARE1, 0x00 },
	{ WCD939X_TX_3_4_TEST_BLK_EN4, 0xfb },
	{ WCD939X_TX_3_4_TXFE4_CLKDIV, 0x00 },
	{ WCD939X_TX_3_4_SPARE2, 0x00 },
	{ WCD939X_CLASSH_MODE_1, 0x40 },
	{ WCD939X_CLASSH_MODE_2, 0x3a },
	{ WCD939X_CLASSH_MODE_3, 0xf0 },
	{ WCD939X_CLASSH_CTRL_VCL_1, 0x7c },
	{ WCD939X_CLASSH_CTRL_VCL_2, 0x82 },
	{ WCD939X_CLASSH_CTRL_CCL_1, 0x31 },
	{ WCD939X_CLASSH_CTRL_CCL_2, 0x80 },
	{ WCD939X_CLASSH_CTRL_CCL_3, 0x80 },
	{ WCD939X_CLASSH_CTRL_CCL_4, 0x51 },
	{ WCD939X_CLASSH_CTRL_CCL_5, 0x00 },
	{ WCD939X_CLASSH_BUCK_TMUX_A_D, 0x00 },
	{ WCD939X_CLASSH_BUCK_SW_DRV_CNTL, 0x77 },
	{ WCD939X_CLASSH_SPARE, 0x80 },
	{ WCD939X_FLYBACK_EN, 0x4e },
	{ WCD939X_FLYBACK_VNEG_CTRL_1, 0x0b },
	{ WCD939X_FLYBACK_VNEG_CTRL_2, 0x45 },
	{ WCD939X_FLYBACK_VNEG_CTRL_3, 0x14 },
	{ WCD939X_FLYBACK_VNEG_CTRL_4, 0xdb },
	{ WCD939X_FLYBACK_VNEG_CTRL_5, 0x83 },
	{ WCD939X_FLYBACK_VNEG_CTRL_6, 0x98 },
	{ WCD939X_FLYBACK_VNEG_CTRL_7, 0xa9 },
	{ WCD939X_FLYBACK_VNEG_CTRL_8, 0x68 },
	{ WCD939X_FLYBACK_VNEG_CTRL_9, 0x66 },
	{ WCD939X_FLYBACK_VNEGDAC_CTRL_1, 0xed },
	{ WCD939X_FLYBACK_VNEGDAC_CTRL_2, 0xf8 },
	{ WCD939X_FLYBACK_VNEGDAC_CTRL_3, 0xa6 },
	{ WCD939X_FLYBACK_CTRL_1, 0x65 },
	{ WCD939X_FLYBACK_TEST_CTL, 0x02 },
	{ WCD939X_RX_AUX_SW_CTL, 0x00 },
	{ WCD939X_RX_PA_AUX_IN_CONN, 0x01 },
	{ WCD939X_RX_TIMER_DIV, 0x32 },
	{ WCD939X_RX_OCP_CTL, 0x1f },
	{ WCD939X_RX_OCP_COUNT, 0x77 },
	{ WCD939X_RX_BIAS_EAR_DAC, 0xa0 },
	{ WCD939X_RX_BIAS_EAR_AMP, 0xaa },
	{ WCD939X_RX_BIAS_HPH_LDO, 0xa9 },
	{ WCD939X_RX_BIAS_HPH_PA, 0xaa },
	{ WCD939X_RX_BIAS_HPH_RDACBUFF_CNP2, 0xca },
	{ WCD939X_RX_BIAS_HPH_RDAC_LDO, 0x88 },
	{ WCD939X_RX_BIAS_HPH_CNP1, 0x82 },
	{ WCD939X_RX_BIAS_HPH_LOWPOWER, 0x82 },
	{ WCD939X_RX_BIAS_AUX_DAC, 0xa0 },
	{ WCD939X_RX_BIAS_AUX_AMP, 0xaa },
	{ WCD939X_RX_BIAS_VNEGDAC_BLEEDER, 0x50 },
	{ WCD939X_RX_BIAS_MISC, 0x00 },
	{ WCD939X_RX_BIAS_BUCK_RST, 0x08 },
	{ WCD939X_RX_BIAS_BUCK_VREF_ERRAMP, 0x44 },
	{ WCD939X_RX_BIAS_FLYB_ERRAMP, 0x40 },
	{ WCD939X_RX_BIAS_FLYB_BUFF, 0xaa },
	{ WCD939X_RX_BIAS_FLYB_MID_RST, 0x14 },
	{ WCD939X_HPH_CNP_EN, 0x80 },
	{ WCD939X_HPH_CNP_WG_CTL, 0x9a },
	{ WCD939X_HPH_CNP_WG_TIME, 0x14 },
	{ WCD939X_HPH_OCP_CTL, 0x28 },
	{ WCD939X_HPH_AUTO_CHOP, 0x56 },
	{ WCD939X_HPH_CHOP_CTL, 0x83 },
	{ WCD939X_HPH_PA_CTL1, 0x46 },
	{ WCD939X_HPH_PA_CTL2, 0x50 },
	{ WCD939X_HPH_L_EN, 0x80 },
	{ WCD939X_HPH_L_TEST, 0xe0 },
	{ WCD939X_HPH_L_ATEST, 0x50 },
	{ WCD939X_HPH_R_EN, 0x80 },
	{ WCD939X_HPH_R_TEST, 0xe0 },
	{ WCD939X_HPH_R_ATEST, 0x50 },
	{ WCD939X_HPH_RDAC_CLK_CTL1, 0x80 },
	{ WCD939X_HPH_RDAC_CLK_CTL2, 0x0b },
	{ WCD939X_HPH_RDAC_LDO_CTL, 0x33 },
	{ WCD939X_HPH_RDAC_CHOP_CLK_LP_CTL, 0x00 },
	{ WCD939X_HPH_REFBUFF_UHQA_CTL, 0x00 },
	{ WCD939X_HPH_REFBUFF_LP_CTL, 0x8e },
	{ WCD939X_HPH_L_DAC_CTL, 0x20 },
	{ WCD939X_HPH_R_DAC_CTL, 0x20 },
	{ WCD939X_HPH_SURGE_COMP_SEL, 0x55 },
	{ WCD939X_HPH_SURGE_EN, 0x19 },
	{ WCD939X_HPH_SURGE_MISC1, 0xa0 },
	{ WCD939X_EAR_EN, 0x22 },
	{ WCD939X_EAR_PA_CON, 0x44 },
	{ WCD939X_EAR_SP_CON, 0xdb },
	{ WCD939X_EAR_DAC_CON, 0x80 },
	{ WCD939X_EAR_CNP_FSM_CON, 0xb2 },
	{ WCD939X_EAR_TEST_CTL, 0x00 },
	{ WCD939X_FLYBACK_NEW_CTRL_2, 0x00 },
	{ WCD939X_FLYBACK_NEW_CTRL_3, 0x00 },
	{ WCD939X_FLYBACK_NEW_CTRL_4, 0x44 },
	{ WCD939X_ANA_NEW_PAGE, 0x00 },
	{ WCD939X_HPH_NEW_ANA_HPH2, 0x00 },
	{ WCD939X_HPH_NEW_ANA_HPH3, 0x00 },
	{ WCD939X_SLEEP_CTL, 0x18 },
	{ WCD939X_SLEEP_WATCHDOG_CTL, 0x00 },
	{ WCD939X_MBHC_NEW_ELECT_REM_CLAMP_CTL, 0x00 },
	{ WCD939X_MBHC_NEW_CTL_1, 0x02 },
	{ WCD939X_MBHC_NEW_CTL_2, 0x05 },
	{ WCD939X_MBHC_NEW_PLUG_DETECT_CTL, 0xe9 },
	{ WCD939X_MBHC_NEW_ZDET_ANA_CTL, 0x0f },
	{ WCD939X_MBHC_NEW_ZDET_RAMP_CTL, 0x00 },
	{ WCD939X_TX_NEW_CH12_MUX, 0x11 },
	{ WCD939X_TX_NEW_CH34_MUX, 0x23 },
	{ WCD939X_DIE_CRACK_DET_EN, 0x00 },
	{ WCD939X_HPH_NEW_INT_RDAC_GAIN_CTL, 0x00 },
	{ WCD939X_HPH_NEW_INT_PA_GAIN_CTL_L, 0x00 },
	{ WCD939X_HPH_NEW_INT_RDAC_VREF_CTL, 0x08 },
	{ WCD939X_HPH_NEW_INT_RDAC_OVERRIDE_CTL, 0x00 },
	{ WCD939X_HPH_NEW_INT_PA_GAIN_CTL_R, 0x00 },
	{ WCD939X_HPH_NEW_INT_PA_MISC1, 0x32 },
	{ WCD939X_HPH_NEW_INT_PA_MISC2, 0x00 },
	{ WCD939X_HPH_NEW_INT_PA_RDAC_MISC, 0x00 },
	{ WCD939X_HPH_NEW_INT_TIMER1, 0xfe },
	{ WCD939X_HPH_NEW_INT_TIMER2, 0x02 },
	{ WCD939X_HPH_NEW_INT_TIMER3, 0x4e },
	{ WCD939X_HPH_NEW_INT_TIMER4, 0x54 },
	{ WCD939X_HPH_NEW_INT_PA_RDAC_MISC2, 0x0b },
	{ WCD939X_HPH_NEW_INT_PA_RDAC_MISC3, 0x00 },
	{ WCD939X_HPH_NEW_INT_RDAC_HD2_CTL_L, 0xa0 },
	{ WCD939X_HPH_NEW_INT_RDAC_HD2_CTL_R, 0xa0 },
	{ WCD939X_RX_NEW_INT_HPH_RDAC_BIAS_LOHIFI, 0x64 },
	{ WCD939X_RX_NEW_INT_HPH_RDAC_BIAS_ULP, 0x01 },
	{ WCD939X_RX_NEW_INT_HPH_RDAC_LDO_LP, 0x11 },
	{ WCD939X_MBHC_NEW_INT_MOISTURE_DET_DC_CTRL, 0x57 },
	{ WCD939X_MBHC_NEW_INT_MOISTURE_DET_POLLING_CTRL, 0x01 },
	{ WCD939X_MBHC_NEW_INT_MECH_DET_CURRENT, 0x00 },
	{ WCD939X_MBHC_NEW_INT_ZDET_CLK_AND_MOISTURE_CTL_NEW, 0x47 },
	{ WCD939X_EAR_INT_NEW_CHOPPER_CON, 0xa8 },
	{ WCD939X_EAR_INT_NEW_CNP_VCM_CON1, 0x42 },
	{ WCD939X_EAR_INT_NEW_CNP_VCM_CON2, 0x22 },
	{ WCD939X_EAR_INT_NEW_DYNAMIC_BIAS, 0x00 },
	{ WCD939X_SLEEP_INT_WATCHDOG_CTL_1, 0x0a },
	{ WCD939X_SLEEP_INT_WATCHDOG_CTL_2, 0x0a },
	{ WCD939X_DIE_CRACK_INT_DET_INT1, 0x02 },
	{ WCD939X_DIE_CRACK_INT_DET_INT2, 0x60 },
	{ WCD939X_TX_COM_NEW_INT_FE_DIVSTOP_L2, 0xff },
	{ WCD939X_TX_COM_NEW_INT_FE_DIVSTOP_L1, 0x7f },
	{ WCD939X_TX_COM_NEW_INT_FE_DIVSTOP_L0, 0x3f },
	{ WCD939X_TX_COM_NEW_INT_FE_DIVSTOP_ULP1P2M, 0x1f },
	{ WCD939X_TX_COM_NEW_INT_FE_DIVSTOP_ULP0P6M, 0x0f },
	{ WCD939X_TX_COM_NEW_INT_FE_ICTRL_STG1_L2L1, 0xd7 },
	{ WCD939X_TX_COM_NEW_INT_FE_ICTRL_STG1_L0, 0xc8 },
	{ WCD939X_TX_COM_NEW_INT_FE_ICTRL_STG1_ULP, 0xc6 },
	{ WCD939X_TX_COM_NEW_INT_FE_ICTRL_STG2MAIN_L2L1, 0x95 },
	{ WCD939X_TX_COM_NEW_INT_FE_ICTRL_STG2MAIN_L0, 0x6a },
	{ WCD939X_TX_COM_NEW_INT_FE_ICTRL_STG2MAIN_ULP, 0x05 },
	{ WCD939X_TX_COM_NEW_INT_FE_ICTRL_STG2CASC_L2L1L0, 0xa5 },
	{ WCD939X_TX_COM_NEW_INT_FE_ICTRL_STG2CASC_ULP, 0x13 },
	{ WCD939X_TX_COM_NEW_INT_ADC_SCBIAS_L2L1, 0x88 },
	{ WCD939X_TX_COM_NEW_INT_ADC_SCBIAS_L0ULP, 0x42 },
	{ WCD939X_TX_COM_NEW_INT_ADC_INT_L2, 0xff },
	{ WCD939X_TX_COM_NEW_INT_ADC_INT_L1, 0x64 },
	{ WCD939X_TX_COM_NEW_INT_ADC_INT_L0, 0x64 },
	{ WCD939X_TX_COM_NEW_INT_ADC_INT_ULP, 0x77 },
	{ WCD939X_DIGITAL_PAGE, 0x00 },
	{ WCD939X_DIGITAL_SWR_TX_CLK_RATE, 0x00 },
	{ WCD939X_DIGITAL_CDC_RST_CTL, 0x03 },
	{ WCD939X_DIGITAL_TOP_CLK_CFG, 0x00 },
	{ WCD939X_DIGITAL_CDC_ANA_CLK_CTL, 0x00 },
	{ WCD939X_DIGITAL_CDC_DIG_CLK_CTL, 0xf0 },
	{ WCD939X_DIGITAL_SWR_RST_EN, 0x00 },
	{ WCD939X_DIGITAL_CDC_PATH_MODE, 0x55 },
	{ WCD939X_DIGITAL_CDC_RX_RST, 0x00 },
	{ WCD939X_DIGITAL_CDC_RX0_CTL, 0xfc },
	{ WCD939X_DIGITAL_CDC_RX1_CTL, 0xfc },
	{ WCD939X_DIGITAL_CDC_RX2_CTL, 0xfc },
	{ WCD939X_DIGITAL_CDC_TX_ANA_MODE_0_1, 0x00 },
	{ WCD939X_DIGITAL_CDC_TX_ANA_MODE_2_3, 0x00 },
	{ WCD939X_DIGITAL_CDC_COMP_CTL_0, 0x00 },
	{ WCD939X_DIGITAL_CDC_ANA_TX_CLK_CTL, 0x1e },
	{ WCD939X_DIGITAL_CDC_HPH_DSM_A1_0, 0x00 },
	{ WCD939X_DIGITAL_CDC_HPH_DSM_A1_1, 0x01 },
	{ WCD939X_DIGITAL_CDC_HPH_DSM_A2_0, 0x63 },
	{ WCD939X_DIGITAL_CDC_HPH_DSM_A2_1, 0x04 },
	{ WCD939X_DIGITAL_CDC_HPH_DSM_A3_0, 0xac },
	{ WCD939X_DIGITAL_CDC_HPH_DSM_A3_1, 0x04 },
	{ WCD939X_DIGITAL_CDC_HPH_DSM_A4_0, 0x1a },
	{ WCD939X_DIGITAL_CDC_HPH_DSM_A4_1, 0x03 },
	{ WCD939X_DIGITAL_CDC_HPH_DSM_A5_0, 0xbc },
	{ WCD939X_DIGITAL_CDC_HPH_DSM_A5_1, 0x02 },
	{ WCD939X_DIGITAL_CDC_HPH_DSM_A6_0, 0xc7 },
	{ WCD939X_DIGITAL_CDC_HPH_DSM_A7_0, 0xf8 },
	{ WCD939X_DIGITAL_CDC_HPH_DSM_C_0, 0x47 },
	{ WCD939X_DIGITAL_CDC_HPH_DSM_C_1, 0x43 },
	{ WCD939X_DIGITAL_CDC_HPH_DSM_C_2, 0xb1 },
	{ WCD939X_DIGITAL_CDC_HPH_DSM_C_3, 0x17 },
	{ WCD939X_DIGITAL_CDC_HPH_DSM_R1, 0x4d },
	{ WCD939X_DIGITAL_CDC_HPH_DSM_R2, 0x29 },
	{ WCD939X_DIGITAL_CDC_HPH_DSM_R3, 0x34 },
	{ WCD939X_DIGITAL_CDC_HPH_DSM_R4, 0x59 },
	{ WCD939X_DIGITAL_CDC_HPH_DSM_R5, 0x66 },
	{ WCD939X_DIGITAL_CDC_HPH_DSM_R6, 0x87 },
	{ WCD939X_DIGITAL_CDC_HPH_DSM_R7, 0x64 },
	{ WCD939X_DIGITAL_CDC_EAR_DSM_A1_0, 0x00 },
	{ WCD939X_DIGITAL_CDC_EAR_DSM_A1_1, 0x01 },
	{ WCD939X_DIGITAL_CDC_EAR_DSM_A2_0, 0x96 },
	{ WCD939X_DIGITAL_CDC_EAR_DSM_A2_1, 0x09 },
	{ WCD939X_DIGITAL_CDC_EAR_DSM_A3_0, 0xab },
	{ WCD939X_DIGITAL_CDC_EAR_DSM_A3_1, 0x05 },
	{ WCD939X_DIGITAL_CDC_EAR_DSM_A4_0, 0x1c },
	{ WCD939X_DIGITAL_CDC_EAR_DSM_A4_1, 0x02 },
	{ WCD939X_DIGITAL_CDC_EAR_DSM_A5_0, 0x17 },
	{ WCD939X_DIGITAL_CDC_EAR_DSM_A5_1, 0x02 },
	{ WCD939X_DIGITAL_CDC_EAR_DSM_A6_0, 0xaa },
	{ WCD939X_DIGITAL_CDC_EAR_DSM_A7_0, 0xe3 },
	{ WCD939X_DIGITAL_CDC_EAR_DSM_C_0, 0x69 },
	{ WCD939X_DIGITAL_CDC_EAR_DSM_C_1, 0x54 },
	{ WCD939X_DIGITAL_CDC_EAR_DSM_C_2, 0x02 },
	{ WCD939X_DIGITAL_CDC_EAR_DSM_C_3, 0x15 },
	{ WCD939X_DIGITAL_CDC_EAR_DSM_R1, 0xa4 },
	{ WCD939X_DIGITAL_CDC_EAR_DSM_R2, 0xb5 },
	{ WCD939X_DIGITAL_CDC_EAR_DSM_R3, 0x86 },
	{ WCD939X_DIGITAL_CDC_EAR_DSM_R4, 0x85 },
	{ WCD939X_DIGITAL_CDC_EAR_DSM_R5, 0xaa },
	{ WCD939X_DIGITAL_CDC_EAR_DSM_R6, 0xe2 },
	{ WCD939X_DIGITAL_CDC_EAR_DSM_R7, 0x62 },
	{ WCD939X_DIGITAL_CDC_HPH_GAIN_RX_0, 0x55 },
	{ WCD939X_DIGITAL_CDC_HPH_GAIN_RX_1, 0xa9 },
	{ WCD939X_DIGITAL_CDC_HPH_GAIN_DSD_0, 0x3d },
	{ WCD939X_DIGITAL_CDC_HPH_GAIN_DSD_1, 0x2e },
	{ WCD939X_DIGITAL_CDC_HPH_GAIN_DSD_2, 0x01 },
	{ WCD939X_DIGITAL_CDC_EAR_GAIN_DSD_0, 0x00 },
	{ WCD939X_DIGITAL_CDC_EAR_GAIN_DSD_1, 0xfc },
	{ WCD939X_DIGITAL_CDC_EAR_GAIN_DSD_2, 0x01 },
	{ WCD939X_DIGITAL_CDC_HPH_GAIN_CTL, 0x00 },
	{ WCD939X_DIGITAL_CDC_EAR_GAIN_CTL, 0x00 },
	{ WCD939X_DIGITAL_CDC_EAR_PATH_CTL, 0x00 },
	{ WCD939X_DIGITAL_CDC_SWR_CLH, 0x00 },
	{ WCD939X_DIGITAL_SWR_CLH_BYP, 0x00 },
	{ WCD939X_DIGITAL_CDC_TX0_CTL, 0x68 },
	{ WCD939X_DIGITAL_CDC_TX1_CTL, 0x68 },
	{ WCD939X_DIGITAL_CDC_TX2_CTL, 0x68 },
	{ WCD939X_DIGITAL_CDC_TX_RST, 0x00 },
	{ WCD939X_DIGITAL_CDC_REQ_CTL, 0x01 },
	{ WCD939X_DIGITAL_CDC_RST, 0x00 },
	{ WCD939X_DIGITAL_CDC_AMIC_CTL, 0x0f },
	{ WCD939X_DIGITAL_CDC_DMIC_CTL, 0x04 },
	{ WCD939X_DIGITAL_CDC_DMIC1_CTL, 0x01 },
	{ WCD939X_DIGITAL_CDC_DMIC2_CTL, 0x01 },
	{ WCD939X_DIGITAL_CDC_DMIC3_CTL, 0x01 },
	{ WCD939X_DIGITAL_CDC_DMIC4_CTL, 0x01 },
	{ WCD939X_DIGITAL_EFUSE_PRG_CTL, 0x00 },
	{ WCD939X_DIGITAL_EFUSE_CTL, 0x2b },
	{ WCD939X_DIGITAL_CDC_DMIC_RATE_1_2, 0x11 },
	{ WCD939X_DIGITAL_CDC_DMIC_RATE_3_4, 0x11 },
	{ WCD939X_DIGITAL_PDM_WD_CTL0, 0x00 },
	{ WCD939X_DIGITAL_PDM_WD_CTL1, 0x00 },
	{ WCD939X_DIGITAL_PDM_WD_CTL2, 0x00 },
	{ WCD939X_DIGITAL_INTR_MODE, 0x00 },
	{ WCD939X_DIGITAL_INTR_MASK_0, 0xff },
	{ WCD939X_DIGITAL_INTR_MASK_1, 0xe7 },
	{ WCD939X_DIGITAL_INTR_MASK_2, 0x0e },
	{ WCD939X_DIGITAL_INTR_CLEAR_0, 0x00 },
	{ WCD939X_DIGITAL_INTR_CLEAR_1, 0x00 },
	{ WCD939X_DIGITAL_INTR_CLEAR_2, 0x00 },
	{ WCD939X_DIGITAL_INTR_LEVEL_0, 0x00 },
	{ WCD939X_DIGITAL_INTR_LEVEL_1, 0x00 },
	{ WCD939X_DIGITAL_INTR_LEVEL_2, 0x00 },
	{ WCD939X_DIGITAL_INTR_SET_0, 0x00 },
	{ WCD939X_DIGITAL_INTR_SET_1, 0x00 },
	{ WCD939X_DIGITAL_INTR_SET_2, 0x00 },
	{ WCD939X_DIGITAL_INTR_TEST_0, 0x00 },
	{ WCD939X_DIGITAL_INTR_TEST_1, 0x00 },
	{ WCD939X_DIGITAL_INTR_TEST_2, 0x00 },
	{ WCD939X_DIGITAL_TX_MODE_DBG_EN, 0x00 },
	{ WCD939X_DIGITAL_TX_MODE_DBG_0_1, 0x00 },
	{ WCD939X_DIGITAL_TX_MODE_DBG_2_3, 0x00 },
	{ WCD939X_DIGITAL_LB_IN_SEL_CTL, 0x00 },
	{ WCD939X_DIGITAL_LOOP_BACK_MODE, 0x00 },
	{ WCD939X_DIGITAL_SWR_DAC_TEST, 0x00 },
	{ WCD939X_DIGITAL_SWR_HM_TEST_RX_0, 0x40 },
	{ WCD939X_DIGITAL_SWR_HM_TEST_TX_0, 0x40 },
	{ WCD939X_DIGITAL_SWR_HM_TEST_RX_1, 0x00 },
	{ WCD939X_DIGITAL_SWR_HM_TEST_TX_1, 0x00 },
	{ WCD939X_DIGITAL_SWR_HM_TEST_TX_2, 0x00 },
	{ WCD939X_DIGITAL_PAD_CTL_SWR_0, 0x8f },
	{ WCD939X_DIGITAL_PAD_CTL_SWR_1, 0x06 },
	{ WCD939X_DIGITAL_I2C_CTL, 0x00 },
	{ WCD939X_DIGITAL_CDC_TX_TANGGU_SW_MODE, 0x00 },
	{ WCD939X_DIGITAL_EFUSE_TEST_CTL_0, 0x00 },
	{ WCD939X_DIGITAL_EFUSE_TEST_CTL_1, 0x00 },
	{ WCD939X_DIGITAL_PAD_CTL_PDM_RX0, 0xf1 },
	{ WCD939X_DIGITAL_PAD_CTL_PDM_RX1, 0xf1 },
	{ WCD939X_DIGITAL_PAD_CTL_PDM_TX0, 0xf1 },
	{ WCD939X_DIGITAL_PAD_CTL_PDM_TX1, 0xf1 },
	{ WCD939X_DIGITAL_PAD_CTL_PDM_TX2, 0xf1 },
	{ WCD939X_DIGITAL_PAD_INP_DIS_0, 0x00 },
	{ WCD939X_DIGITAL_PAD_INP_DIS_1, 0x00 },
	{ WCD939X_DIGITAL_DRIVE_STRENGTH_0, 0x00 },
	{ WCD939X_DIGITAL_DRIVE_STRENGTH_1, 0x00 },
	{ WCD939X_DIGITAL_DRIVE_STRENGTH_2, 0x00 },
	{ WCD939X_DIGITAL_RX_DATA_EDGE_CTL, 0x1f },
	{ WCD939X_DIGITAL_TX_DATA_EDGE_CTL, 0x80 },
	{ WCD939X_DIGITAL_GPIO_MODE, 0x00 },
	{ WCD939X_DIGITAL_PIN_CTL_OE, 0x00 },
	{ WCD939X_DIGITAL_PIN_CTL_DATA_0, 0x00 },
	{ WCD939X_DIGITAL_PIN_CTL_DATA_1, 0x00 },
	{ WCD939X_DIGITAL_DIG_DEBUG_CTL, 0x00 },
	{ WCD939X_DIGITAL_DIG_DEBUG_EN, 0x00 },
	{ WCD939X_DIGITAL_ANA_CSR_DBG_ADD, 0x00 },
	{ WCD939X_DIGITAL_ANA_CSR_DBG_CTL, 0x48 },
	{ WCD939X_DIGITAL_SSP_DBG, 0x00 },
	{ WCD939X_DIGITAL_SPARE_0, 0x00 },
	{ WCD939X_DIGITAL_SPARE_1, 0x00 },
	{ WCD939X_DIGITAL_SPARE_2, 0x00 },
	{ WCD939X_DIGITAL_TX_REQ_FB_CTL_0, 0x88 },
	{ WCD939X_DIGITAL_TX_REQ_FB_CTL_1, 0x88 },
	{ WCD939X_DIGITAL_TX_REQ_FB_CTL_2, 0x88 },
	{ WCD939X_DIGITAL_TX_REQ_FB_CTL_3, 0x88 },
	{ WCD939X_DIGITAL_TX_REQ_FB_CTL_4, 0x88 },
	{ WCD939X_DIGITAL_DEM_BYPASS_DATA0, 0x55 },
	{ WCD939X_DIGITAL_DEM_BYPASS_DATA1, 0x55 },
	{ WCD939X_DIGITAL_DEM_BYPASS_DATA2, 0x55 },
	{ WCD939X_DIGITAL_DEM_BYPASS_DATA3, 0x01 },
	{ WCD939X_DIGITAL_DEM_SECOND_ORDER, 0x03 },
	{ WCD939X_DIGITAL_DSM_CTRL, 0x00 },
	{ WCD939X_DIGITAL_DSM_0_STATIC_DATA_0, 0x00 },
	{ WCD939X_DIGITAL_DSM_0_STATIC_DATA_1, 0x00 },
	{ WCD939X_DIGITAL_DSM_0_STATIC_DATA_2, 0x00 },
	{ WCD939X_DIGITAL_DSM_0_STATIC_DATA_3, 0x00 },
	{ WCD939X_DIGITAL_DSM_1_STATIC_DATA_0, 0x00 },
	{ WCD939X_DIGITAL_DSM_1_STATIC_DATA_1, 0x00 },
	{ WCD939X_DIGITAL_DSM_1_STATIC_DATA_2, 0x00 },
	{ WCD939X_DIGITAL_DSM_1_STATIC_DATA_3, 0x00 },
	{ WCD939X_RX_TOP_PAGE, 0x00 },
	{ WCD939X_RX_TOP_TOP_CFG0, 0x00 },
	{ WCD939X_RX_TOP_HPHL_COMP_WR_LSB, 0x00 },
	{ WCD939X_RX_TOP_HPHL_COMP_WR_MSB, 0x00 },
	{ WCD939X_RX_TOP_HPHL_COMP_LUT, 0x00 },
	{ WCD939X_RX_TOP_HPHR_COMP_WR_LSB, 0x00 },
	{ WCD939X_RX_TOP_HPHR_COMP_WR_MSB, 0x00 },
	{ WCD939X_RX_TOP_HPHR_COMP_LUT, 0x00 },
	{ WCD939X_RX_TOP_DSD0_DEBUG_CFG1, 0x05 },
	{ WCD939X_RX_TOP_DSD0_DEBUG_CFG2, 0x08 },
	{ WCD939X_RX_TOP_DSD0_DEBUG_CFG3, 0x00 },
	{ WCD939X_RX_TOP_DSD0_DEBUG_CFG4, 0x00 },
	{ WCD939X_RX_TOP_DSD1_DEBUG_CFG1, 0x03 },
	{ WCD939X_RX_TOP_DSD1_DEBUG_CFG2, 0x08 },
	{ WCD939X_RX_TOP_DSD1_DEBUG_CFG3, 0x00 },
	{ WCD939X_RX_TOP_DSD1_DEBUG_CFG4, 0x00 },
	{ WCD939X_RX_TOP_HPHL_PATH_CFG0, 0x00 },
	{ WCD939X_RX_TOP_HPHL_PATH_CFG1, 0x00 },
	{ WCD939X_RX_TOP_HPHR_PATH_CFG0, 0x00 },
	{ WCD939X_RX_TOP_HPHR_PATH_CFG1, 0x00 },
	{ WCD939X_RX_TOP_PATH_CFG2, 0x00 },
	{ WCD939X_RX_TOP_HPHL_PATH_SEC0, 0x00 },
	{ WCD939X_RX_TOP_HPHL_PATH_SEC1, 0x00 },
	{ WCD939X_RX_TOP_HPHL_PATH_SEC2, 0x00 },
	{ WCD939X_RX_TOP_HPHL_PATH_SEC3, 0x00 },
	{ WCD939X_RX_TOP_HPHR_PATH_SEC0, 0x00 },
	{ WCD939X_RX_TOP_HPHR_PATH_SEC1, 0x00 },
	{ WCD939X_RX_TOP_HPHR_PATH_SEC2, 0x00 },
	{ WCD939X_RX_TOP_HPHR_PATH_SEC3, 0x00 },
	{ WCD939X_RX_TOP_PATH_SEC4, 0x00 },
	{ WCD939X_RX_TOP_PATH_SEC5, 0x00 },
	{ WCD939X_COMPANDER_HPHL_CTL0, 0x60 },
	{ WCD939X_COMPANDER_HPHL_CTL1, 0xdb },
	{ WCD939X_COMPANDER_HPHL_CTL2, 0xff },
	{ WCD939X_COMPANDER_HPHL_CTL3, 0x35 },
	{ WCD939X_COMPANDER_HPHL_CTL4, 0xff },
	{ WCD939X_COMPANDER_HPHL_CTL5, 0x00 },
	{ WCD939X_COMPANDER_HPHL_CTL7, 0x08 },
	{ WCD939X_COMPANDER_HPHL_CTL8, 0x00 },
	{ WCD939X_COMPANDER_HPHL_CTL9, 0x00 },
	{ WCD939X_COMPANDER_HPHL_CTL10, 0x06 },
	{ WCD939X_COMPANDER_HPHL_CTL11, 0x12 },
	{ WCD939X_COMPANDER_HPHL_CTL12, 0x1e },
	{ WCD939X_COMPANDER_HPHL_CTL13, 0x2a },
	{ WCD939X_COMPANDER_HPHL_CTL14, 0x36 },
	{ WCD939X_COMPANDER_HPHL_CTL15, 0x3c },
	{ WCD939X_COMPANDER_HPHL_CTL16, 0xc4 },
	{ WCD939X_COMPANDER_HPHL_CTL17, 0x00 },
	{ WCD939X_COMPANDER_HPHL_CTL18, 0x0c },
	{ WCD939X_COMPANDER_HPHL_CTL19, 0x16 },
	{ WCD939X_R_CTL0, 0x60 },
	{ WCD939X_R_CTL1, 0xdb },
	{ WCD939X_R_CTL2, 0xff },
	{ WCD939X_R_CTL3, 0x35 },
	{ WCD939X_R_CTL4, 0xff },
	{ WCD939X_R_CTL5, 0x00 },
	{ WCD939X_R_CTL7, 0x08 },
	{ WCD939X_R_CTL8, 0x00 },
	{ WCD939X_R_CTL9, 0x00 },
	{ WCD939X_R_CTL10, 0x06 },
	{ WCD939X_R_CTL11, 0x12 },
	{ WCD939X_R_CTL12, 0x1e },
	{ WCD939X_R_CTL13, 0x2a },
	{ WCD939X_R_CTL14, 0x36 },
	{ WCD939X_R_CTL15, 0x3c },
	{ WCD939X_R_CTL16, 0xc4 },
	{ WCD939X_R_CTL17, 0x00 },
	{ WCD939X_R_CTL18, 0x0c },
	{ WCD939X_R_CTL19, 0x16 },
	{ WCD939X_E_PATH_CTL, 0x00 },
	{ WCD939X_E_CFG0, 0x07 },
	{ WCD939X_E_CFG1, 0x3c },
	{ WCD939X_E_CFG2, 0x00 },
	{ WCD939X_E_CFG3, 0x00 },
	{ WCD939X_DSD_HPHL_PATH_CTL, 0x00 },
	{ WCD939X_DSD_HPHL_CFG0, 0x00 },
	{ WCD939X_DSD_HPHL_CFG1, 0x00 },
	{ WCD939X_DSD_HPHL_CFG2, 0x22 },
	{ WCD939X_DSD_HPHL_CFG3, 0x00 },
	{ WCD939X_DSD_HPHL_CFG4, 0x1a },
	{ WCD939X_DSD_HPHL_CFG5, 0x00 },
	{ WCD939X_DSD_HPHR_PATH_CTL, 0x00 },
	{ WCD939X_DSD_HPHR_CFG0, 0x00 },
	{ WCD939X_DSD_HPHR_CFG1, 0x00 },
	{ WCD939X_DSD_HPHR_CFG2, 0x22 },
	{ WCD939X_DSD_HPHR_CFG3, 0x00 },
	{ WCD939X_DSD_HPHR_CFG4, 0x1a },
	{ WCD939X_DSD_HPHR_CFG5, 0x00 },
};

static bool wcd939x_rdwr_register(struct device *dev, unsigned int reg)
{
	switch (reg) {
	case WCD939X_ANA_PAGE:
	case WCD939X_ANA_BIAS:
	case WCD939X_ANA_RX_SUPPLIES:
	case WCD939X_ANA_HPH:
	case WCD939X_ANA_EAR:
	case WCD939X_ANA_EAR_COMPANDER_CTL:
	case WCD939X_ANA_TX_CH1:
	case WCD939X_ANA_TX_CH2:
	case WCD939X_ANA_TX_CH3:
	case WCD939X_ANA_TX_CH4:
	case WCD939X_ANA_MICB1_MICB2_DSP_EN_LOGIC:
	case WCD939X_ANA_MICB3_DSP_EN_LOGIC:
	case WCD939X_ANA_MBHC_MECH:
	case WCD939X_ANA_MBHC_ELECT:
	case WCD939X_ANA_MBHC_ZDET:
	case WCD939X_ANA_MBHC_BTN0:
	case WCD939X_ANA_MBHC_BTN1:
	case WCD939X_ANA_MBHC_BTN2:
	case WCD939X_ANA_MBHC_BTN3:
	case WCD939X_ANA_MBHC_BTN4:
	case WCD939X_ANA_MBHC_BTN5:
	case WCD939X_ANA_MBHC_BTN6:
	case WCD939X_ANA_MBHC_BTN7:
	case WCD939X_ANA_MICB1:
	case WCD939X_ANA_MICB2:
	case WCD939X_ANA_MICB2_RAMP:
	case WCD939X_ANA_MICB3:
	case WCD939X_ANA_MICB4:
	case WCD939X_BIAS_CTL:
	case WCD939X_BIAS_VBG_FINE_ADJ:
	case WCD939X_LDOL_VDDCX_ADJUST:
	case WCD939X_LDOL_DISABLE_LDOL:
	case WCD939X_MBHC_CTL_CLK:
	case WCD939X_MBHC_CTL_ANA:
	case WCD939X_MBHC_ZDET_VNEG_CTL:
	case WCD939X_MBHC_ZDET_BIAS_CTL:
	case WCD939X_MBHC_CTL_BCS:
	case WCD939X_MBHC_TEST_CTL:
	case WCD939X_LDOH_MODE:
	case WCD939X_LDOH_BIAS:
	case WCD939X_LDOH_STB_LOADS:
	case WCD939X_LDOH_SLOWRAMP:
	case WCD939X_MICB1_TEST_CTL_1:
	case WCD939X_MICB1_TEST_CTL_2:
	case WCD939X_MICB1_TEST_CTL_3:
	case WCD939X_MICB2_TEST_CTL_1:
	case WCD939X_MICB2_TEST_CTL_2:
	case WCD939X_MICB2_TEST_CTL_3:
	case WCD939X_MICB3_TEST_CTL_1:
	case WCD939X_MICB3_TEST_CTL_2:
	case WCD939X_MICB3_TEST_CTL_3:
	case WCD939X_MICB4_TEST_CTL_1:
	case WCD939X_MICB4_TEST_CTL_2:
	case WCD939X_MICB4_TEST_CTL_3:
	case WCD939X_TX_COM_ADC_VCM:
	case WCD939X_TX_COM_BIAS_ATEST:
	case WCD939X_TX_COM_SPARE1:
	case WCD939X_TX_COM_SPARE2:
	case WCD939X_TX_COM_TXFE_DIV_CTL:
	case WCD939X_TX_COM_TXFE_DIV_START:
	case WCD939X_TX_COM_SPARE3:
	case WCD939X_TX_COM_SPARE4:
	case WCD939X_TX_1_2_TEST_EN:
	case WCD939X_TX_1_2_ADC_IB:
	case WCD939X_TX_1_2_ATEST_REFCTL:
	case WCD939X_TX_1_2_TEST_CTL:
	case WCD939X_TX_1_2_TEST_BLK_EN1:
	case WCD939X_TX_1_2_TXFE1_CLKDIV:
	case WCD939X_TX_3_4_TEST_EN:
	case WCD939X_TX_3_4_ADC_IB:
	case WCD939X_TX_3_4_ATEST_REFCTL:
	case WCD939X_TX_3_4_TEST_CTL:
	case WCD939X_TX_3_4_TEST_BLK_EN3:
	case WCD939X_TX_3_4_TXFE3_CLKDIV:
	case WCD939X_TX_3_4_TEST_BLK_EN2:
	case WCD939X_TX_3_4_TXFE2_CLKDIV:
	case WCD939X_TX_3_4_SPARE1:
	case WCD939X_TX_3_4_TEST_BLK_EN4:
	case WCD939X_TX_3_4_TXFE4_CLKDIV:
	case WCD939X_TX_3_4_SPARE2:
	case WCD939X_CLASSH_MODE_1:
	case WCD939X_CLASSH_MODE_2:
	case WCD939X_CLASSH_MODE_3:
	case WCD939X_CLASSH_CTRL_VCL_1:
	case WCD939X_CLASSH_CTRL_VCL_2:
	case WCD939X_CLASSH_CTRL_CCL_1:
	case WCD939X_CLASSH_CTRL_CCL_2:
	case WCD939X_CLASSH_CTRL_CCL_3:
	case WCD939X_CLASSH_CTRL_CCL_4:
	case WCD939X_CLASSH_CTRL_CCL_5:
	case WCD939X_CLASSH_BUCK_TMUX_A_D:
	case WCD939X_CLASSH_BUCK_SW_DRV_CNTL:
	case WCD939X_CLASSH_SPARE:
	case WCD939X_FLYBACK_EN:
	case WCD939X_FLYBACK_VNEG_CTRL_1:
	case WCD939X_FLYBACK_VNEG_CTRL_2:
	case WCD939X_FLYBACK_VNEG_CTRL_3:
	case WCD939X_FLYBACK_VNEG_CTRL_4:
	case WCD939X_FLYBACK_VNEG_CTRL_5:
	case WCD939X_FLYBACK_VNEG_CTRL_6:
	case WCD939X_FLYBACK_VNEG_CTRL_7:
	case WCD939X_FLYBACK_VNEG_CTRL_8:
	case WCD939X_FLYBACK_VNEG_CTRL_9:
	case WCD939X_FLYBACK_VNEGDAC_CTRL_1:
	case WCD939X_FLYBACK_VNEGDAC_CTRL_2:
	case WCD939X_FLYBACK_VNEGDAC_CTRL_3:
	case WCD939X_FLYBACK_CTRL_1:
	case WCD939X_FLYBACK_TEST_CTL:
	case WCD939X_RX_AUX_SW_CTL:
	case WCD939X_RX_PA_AUX_IN_CONN:
	case WCD939X_RX_TIMER_DIV:
	case WCD939X_RX_OCP_CTL:
	case WCD939X_RX_OCP_COUNT:
	case WCD939X_RX_BIAS_EAR_DAC:
	case WCD939X_RX_BIAS_EAR_AMP:
	case WCD939X_RX_BIAS_HPH_LDO:
	case WCD939X_RX_BIAS_HPH_PA:
	case WCD939X_RX_BIAS_HPH_RDACBUFF_CNP2:
	case WCD939X_RX_BIAS_HPH_RDAC_LDO:
	case WCD939X_RX_BIAS_HPH_CNP1:
	case WCD939X_RX_BIAS_HPH_LOWPOWER:
	case WCD939X_RX_BIAS_AUX_DAC:
	case WCD939X_RX_BIAS_AUX_AMP:
	case WCD939X_RX_BIAS_VNEGDAC_BLEEDER:
	case WCD939X_RX_BIAS_MISC:
	case WCD939X_RX_BIAS_BUCK_RST:
	case WCD939X_RX_BIAS_BUCK_VREF_ERRAMP:
	case WCD939X_RX_BIAS_FLYB_ERRAMP:
	case WCD939X_RX_BIAS_FLYB_BUFF:
	case WCD939X_RX_BIAS_FLYB_MID_RST:
	case WCD939X_HPH_CNP_EN:
	case WCD939X_HPH_CNP_WG_CTL:
	case WCD939X_HPH_CNP_WG_TIME:
	case WCD939X_HPH_OCP_CTL:
	case WCD939X_HPH_AUTO_CHOP:
	case WCD939X_HPH_CHOP_CTL:
	case WCD939X_HPH_PA_CTL1:
	case WCD939X_HPH_PA_CTL2:
	case WCD939X_HPH_L_EN:
	case WCD939X_HPH_L_TEST:
	case WCD939X_HPH_L_ATEST:
	case WCD939X_HPH_R_EN:
	case WCD939X_HPH_R_TEST:
	case WCD939X_HPH_R_ATEST:
	case WCD939X_HPH_RDAC_CLK_CTL1:
	case WCD939X_HPH_RDAC_CLK_CTL2:
	case WCD939X_HPH_RDAC_LDO_CTL:
	case WCD939X_HPH_RDAC_CHOP_CLK_LP_CTL:
	case WCD939X_HPH_REFBUFF_UHQA_CTL:
	case WCD939X_HPH_REFBUFF_LP_CTL:
	case WCD939X_HPH_L_DAC_CTL:
	case WCD939X_HPH_R_DAC_CTL:
	case WCD939X_HPH_SURGE_COMP_SEL:
	case WCD939X_HPH_SURGE_EN:
	case WCD939X_HPH_SURGE_MISC1:
	case WCD939X_EAR_EN:
	case WCD939X_EAR_PA_CON:
	case WCD939X_EAR_SP_CON:
	case WCD939X_EAR_DAC_CON:
	case WCD939X_EAR_CNP_FSM_CON:
	case WCD939X_EAR_TEST_CTL:
	case WCD939X_FLYBACK_NEW_CTRL_2:
	case WCD939X_FLYBACK_NEW_CTRL_3:
	case WCD939X_FLYBACK_NEW_CTRL_4:
	case WCD939X_ANA_NEW_PAGE:
	case WCD939X_HPH_NEW_ANA_HPH2:
	case WCD939X_HPH_NEW_ANA_HPH3:
	case WCD939X_SLEEP_CTL:
	case WCD939X_SLEEP_WATCHDOG_CTL:
	case WCD939X_MBHC_NEW_ELECT_REM_CLAMP_CTL:
	case WCD939X_MBHC_NEW_CTL_1:
	case WCD939X_MBHC_NEW_CTL_2:
	case WCD939X_MBHC_NEW_PLUG_DETECT_CTL:
	case WCD939X_MBHC_NEW_ZDET_ANA_CTL:
	case WCD939X_MBHC_NEW_ZDET_RAMP_CTL:
	case WCD939X_TX_NEW_CH12_MUX:
	case WCD939X_TX_NEW_CH34_MUX:
	case WCD939X_DIE_CRACK_DET_EN:
	case WCD939X_HPH_NEW_INT_RDAC_GAIN_CTL:
	case WCD939X_HPH_NEW_INT_PA_GAIN_CTL_L:
	case WCD939X_HPH_NEW_INT_RDAC_VREF_CTL:
	case WCD939X_HPH_NEW_INT_RDAC_OVERRIDE_CTL:
	case WCD939X_HPH_NEW_INT_PA_GAIN_CTL_R:
	case WCD939X_HPH_NEW_INT_PA_MISC1:
	case WCD939X_HPH_NEW_INT_PA_MISC2:
	case WCD939X_HPH_NEW_INT_PA_RDAC_MISC:
	case WCD939X_HPH_NEW_INT_TIMER1:
	case WCD939X_HPH_NEW_INT_TIMER2:
	case WCD939X_HPH_NEW_INT_TIMER3:
	case WCD939X_HPH_NEW_INT_TIMER4:
	case WCD939X_HPH_NEW_INT_PA_RDAC_MISC2:
	case WCD939X_HPH_NEW_INT_PA_RDAC_MISC3:
	case WCD939X_HPH_NEW_INT_RDAC_HD2_CTL_L:
	case WCD939X_HPH_NEW_INT_RDAC_HD2_CTL_R:
	case WCD939X_RX_NEW_INT_HPH_RDAC_BIAS_LOHIFI:
	case WCD939X_RX_NEW_INT_HPH_RDAC_BIAS_ULP:
	case WCD939X_RX_NEW_INT_HPH_RDAC_LDO_LP:
	case WCD939X_MBHC_NEW_INT_MOISTURE_DET_DC_CTRL:
	case WCD939X_MBHC_NEW_INT_MOISTURE_DET_POLLING_CTRL:
	case WCD939X_MBHC_NEW_INT_MECH_DET_CURRENT:
	case WCD939X_MBHC_NEW_INT_ZDET_CLK_AND_MOISTURE_CTL_NEW:
	case WCD939X_EAR_INT_NEW_CHOPPER_CON:
	case WCD939X_EAR_INT_NEW_CNP_VCM_CON1:
	case WCD939X_EAR_INT_NEW_CNP_VCM_CON2:
	case WCD939X_EAR_INT_NEW_DYNAMIC_BIAS:
	case WCD939X_SLEEP_INT_WATCHDOG_CTL_1:
	case WCD939X_SLEEP_INT_WATCHDOG_CTL_2:
	case WCD939X_DIE_CRACK_INT_DET_INT1:
	case WCD939X_DIE_CRACK_INT_DET_INT2:
	case WCD939X_TX_COM_NEW_INT_FE_DIVSTOP_L2:
	case WCD939X_TX_COM_NEW_INT_FE_DIVSTOP_L1:
	case WCD939X_TX_COM_NEW_INT_FE_DIVSTOP_L0:
	case WCD939X_TX_COM_NEW_INT_FE_DIVSTOP_ULP1P2M:
	case WCD939X_TX_COM_NEW_INT_FE_DIVSTOP_ULP0P6M:
	case WCD939X_TX_COM_NEW_INT_FE_ICTRL_STG1_L2L1:
	case WCD939X_TX_COM_NEW_INT_FE_ICTRL_STG1_L0:
	case WCD939X_TX_COM_NEW_INT_FE_ICTRL_STG1_ULP:
	case WCD939X_TX_COM_NEW_INT_FE_ICTRL_STG2MAIN_L2L1:
	case WCD939X_TX_COM_NEW_INT_FE_ICTRL_STG2MAIN_L0:
	case WCD939X_TX_COM_NEW_INT_FE_ICTRL_STG2MAIN_ULP:
	case WCD939X_TX_COM_NEW_INT_FE_ICTRL_STG2CASC_L2L1L0:
	case WCD939X_TX_COM_NEW_INT_FE_ICTRL_STG2CASC_ULP:
	case WCD939X_TX_COM_NEW_INT_ADC_SCBIAS_L2L1:
	case WCD939X_TX_COM_NEW_INT_ADC_SCBIAS_L0ULP:
	case WCD939X_TX_COM_NEW_INT_ADC_INT_L2:
	case WCD939X_TX_COM_NEW_INT_ADC_INT_L1:
	case WCD939X_TX_COM_NEW_INT_ADC_INT_L0:
	case WCD939X_TX_COM_NEW_INT_ADC_INT_ULP:
	case WCD939X_DIGITAL_PAGE:
	case WCD939X_DIGITAL_SWR_TX_CLK_RATE:
	case WCD939X_DIGITAL_CDC_RST_CTL:
	case WCD939X_DIGITAL_TOP_CLK_CFG:
	case WCD939X_DIGITAL_CDC_ANA_CLK_CTL:
	case WCD939X_DIGITAL_CDC_DIG_CLK_CTL:
	case WCD939X_DIGITAL_SWR_RST_EN:
	case WCD939X_DIGITAL_CDC_PATH_MODE:
	case WCD939X_DIGITAL_CDC_RX_RST:
	case WCD939X_DIGITAL_CDC_RX0_CTL:
	case WCD939X_DIGITAL_CDC_RX1_CTL:
	case WCD939X_DIGITAL_CDC_RX2_CTL:
	case WCD939X_DIGITAL_CDC_TX_ANA_MODE_0_1:
	case WCD939X_DIGITAL_CDC_TX_ANA_MODE_2_3:
	case WCD939X_DIGITAL_CDC_COMP_CTL_0:
	case WCD939X_DIGITAL_CDC_ANA_TX_CLK_CTL:
	case WCD939X_DIGITAL_CDC_HPH_DSM_A1_0:
	case WCD939X_DIGITAL_CDC_HPH_DSM_A1_1:
	case WCD939X_DIGITAL_CDC_HPH_DSM_A2_0:
	case WCD939X_DIGITAL_CDC_HPH_DSM_A2_1:
	case WCD939X_DIGITAL_CDC_HPH_DSM_A3_0:
	case WCD939X_DIGITAL_CDC_HPH_DSM_A3_1:
	case WCD939X_DIGITAL_CDC_HPH_DSM_A4_0:
	case WCD939X_DIGITAL_CDC_HPH_DSM_A4_1:
	case WCD939X_DIGITAL_CDC_HPH_DSM_A5_0:
	case WCD939X_DIGITAL_CDC_HPH_DSM_A5_1:
	case WCD939X_DIGITAL_CDC_HPH_DSM_A6_0:
	case WCD939X_DIGITAL_CDC_HPH_DSM_A7_0:
	case WCD939X_DIGITAL_CDC_HPH_DSM_C_0:
	case WCD939X_DIGITAL_CDC_HPH_DSM_C_1:
	case WCD939X_DIGITAL_CDC_HPH_DSM_C_2:
	case WCD939X_DIGITAL_CDC_HPH_DSM_C_3:
	case WCD939X_DIGITAL_CDC_HPH_DSM_R1:
	case WCD939X_DIGITAL_CDC_HPH_DSM_R2:
	case WCD939X_DIGITAL_CDC_HPH_DSM_R3:
	case WCD939X_DIGITAL_CDC_HPH_DSM_R4:
	case WCD939X_DIGITAL_CDC_HPH_DSM_R5:
	case WCD939X_DIGITAL_CDC_HPH_DSM_R6:
	case WCD939X_DIGITAL_CDC_HPH_DSM_R7:
	case WCD939X_DIGITAL_CDC_EAR_DSM_A1_0:
	case WCD939X_DIGITAL_CDC_EAR_DSM_A1_1:
	case WCD939X_DIGITAL_CDC_EAR_DSM_A2_0:
	case WCD939X_DIGITAL_CDC_EAR_DSM_A2_1:
	case WCD939X_DIGITAL_CDC_EAR_DSM_A3_0:
	case WCD939X_DIGITAL_CDC_EAR_DSM_A3_1:
	case WCD939X_DIGITAL_CDC_EAR_DSM_A4_0:
	case WCD939X_DIGITAL_CDC_EAR_DSM_A4_1:
	case WCD939X_DIGITAL_CDC_EAR_DSM_A5_0:
	case WCD939X_DIGITAL_CDC_EAR_DSM_A5_1:
	case WCD939X_DIGITAL_CDC_EAR_DSM_A6_0:
	case WCD939X_DIGITAL_CDC_EAR_DSM_A7_0:
	case WCD939X_DIGITAL_CDC_EAR_DSM_C_0:
	case WCD939X_DIGITAL_CDC_EAR_DSM_C_1:
	case WCD939X_DIGITAL_CDC_EAR_DSM_C_2:
	case WCD939X_DIGITAL_CDC_EAR_DSM_C_3:
	case WCD939X_DIGITAL_CDC_EAR_DSM_R1:
	case WCD939X_DIGITAL_CDC_EAR_DSM_R2:
	case WCD939X_DIGITAL_CDC_EAR_DSM_R3:
	case WCD939X_DIGITAL_CDC_EAR_DSM_R4:
	case WCD939X_DIGITAL_CDC_EAR_DSM_R5:
	case WCD939X_DIGITAL_CDC_EAR_DSM_R6:
	case WCD939X_DIGITAL_CDC_EAR_DSM_R7:
	case WCD939X_DIGITAL_CDC_HPH_GAIN_RX_0:
	case WCD939X_DIGITAL_CDC_HPH_GAIN_RX_1:
	case WCD939X_DIGITAL_CDC_HPH_GAIN_DSD_0:
	case WCD939X_DIGITAL_CDC_HPH_GAIN_DSD_1:
	case WCD939X_DIGITAL_CDC_HPH_GAIN_DSD_2:
	case WCD939X_DIGITAL_CDC_EAR_GAIN_DSD_0:
	case WCD939X_DIGITAL_CDC_EAR_GAIN_DSD_1:
	case WCD939X_DIGITAL_CDC_EAR_GAIN_DSD_2:
	case WCD939X_DIGITAL_CDC_HPH_GAIN_CTL:
	case WCD939X_DIGITAL_CDC_EAR_GAIN_CTL:
	case WCD939X_DIGITAL_CDC_EAR_PATH_CTL:
	case WCD939X_DIGITAL_CDC_SWR_CLH:
	case WCD939X_DIGITAL_SWR_CLH_BYP:
	case WCD939X_DIGITAL_CDC_TX0_CTL:
	case WCD939X_DIGITAL_CDC_TX1_CTL:
	case WCD939X_DIGITAL_CDC_TX2_CTL:
	case WCD939X_DIGITAL_CDC_TX_RST:
	case WCD939X_DIGITAL_CDC_REQ_CTL:
	case WCD939X_DIGITAL_CDC_RST:
	case WCD939X_DIGITAL_CDC_AMIC_CTL:
	case WCD939X_DIGITAL_CDC_DMIC_CTL:
	case WCD939X_DIGITAL_CDC_DMIC1_CTL:
	case WCD939X_DIGITAL_CDC_DMIC2_CTL:
	case WCD939X_DIGITAL_CDC_DMIC3_CTL:
	case WCD939X_DIGITAL_CDC_DMIC4_CTL:
	case WCD939X_DIGITAL_EFUSE_PRG_CTL:
	case WCD939X_DIGITAL_EFUSE_CTL:
	case WCD939X_DIGITAL_CDC_DMIC_RATE_1_2:
	case WCD939X_DIGITAL_CDC_DMIC_RATE_3_4:
	case WCD939X_DIGITAL_PDM_WD_CTL0:
	case WCD939X_DIGITAL_PDM_WD_CTL1:
	case WCD939X_DIGITAL_PDM_WD_CTL2:
	case WCD939X_DIGITAL_INTR_MODE:
	case WCD939X_DIGITAL_INTR_MASK_0:
	case WCD939X_DIGITAL_INTR_MASK_1:
	case WCD939X_DIGITAL_INTR_MASK_2:
	case WCD939X_DIGITAL_INTR_CLEAR_0:
	case WCD939X_DIGITAL_INTR_CLEAR_1:
	case WCD939X_DIGITAL_INTR_CLEAR_2:
	case WCD939X_DIGITAL_INTR_LEVEL_0:
	case WCD939X_DIGITAL_INTR_LEVEL_1:
	case WCD939X_DIGITAL_INTR_LEVEL_2:
	case WCD939X_DIGITAL_INTR_SET_0:
	case WCD939X_DIGITAL_INTR_SET_1:
	case WCD939X_DIGITAL_INTR_SET_2:
	case WCD939X_DIGITAL_INTR_TEST_0:
	case WCD939X_DIGITAL_INTR_TEST_1:
	case WCD939X_DIGITAL_INTR_TEST_2:
	case WCD939X_DIGITAL_TX_MODE_DBG_EN:
	case WCD939X_DIGITAL_TX_MODE_DBG_0_1:
	case WCD939X_DIGITAL_TX_MODE_DBG_2_3:
	case WCD939X_DIGITAL_LB_IN_SEL_CTL:
	case WCD939X_DIGITAL_LOOP_BACK_MODE:
	case WCD939X_DIGITAL_SWR_DAC_TEST:
	case WCD939X_DIGITAL_SWR_HM_TEST_RX_0:
	case WCD939X_DIGITAL_SWR_HM_TEST_TX_0:
	case WCD939X_DIGITAL_SWR_HM_TEST_RX_1:
	case WCD939X_DIGITAL_SWR_HM_TEST_TX_1:
	case WCD939X_DIGITAL_SWR_HM_TEST_TX_2:
	case WCD939X_DIGITAL_PAD_CTL_SWR_0:
	case WCD939X_DIGITAL_PAD_CTL_SWR_1:
	case WCD939X_DIGITAL_I2C_CTL:
	case WCD939X_DIGITAL_CDC_TX_TANGGU_SW_MODE:
	case WCD939X_DIGITAL_EFUSE_TEST_CTL_0:
	case WCD939X_DIGITAL_EFUSE_TEST_CTL_1:
	case WCD939X_DIGITAL_PAD_CTL_PDM_RX0:
	case WCD939X_DIGITAL_PAD_CTL_PDM_RX1:
	case WCD939X_DIGITAL_PAD_CTL_PDM_TX0:
	case WCD939X_DIGITAL_PAD_CTL_PDM_TX1:
	case WCD939X_DIGITAL_PAD_CTL_PDM_TX2:
	case WCD939X_DIGITAL_PAD_INP_DIS_0:
	case WCD939X_DIGITAL_PAD_INP_DIS_1:
	case WCD939X_DIGITAL_DRIVE_STRENGTH_0:
	case WCD939X_DIGITAL_DRIVE_STRENGTH_1:
	case WCD939X_DIGITAL_DRIVE_STRENGTH_2:
	case WCD939X_DIGITAL_RX_DATA_EDGE_CTL:
	case WCD939X_DIGITAL_TX_DATA_EDGE_CTL:
	case WCD939X_DIGITAL_GPIO_MODE:
	case WCD939X_DIGITAL_PIN_CTL_OE:
	case WCD939X_DIGITAL_PIN_CTL_DATA_0:
	case WCD939X_DIGITAL_PIN_CTL_DATA_1:
	case WCD939X_DIGITAL_DIG_DEBUG_CTL:
	case WCD939X_DIGITAL_DIG_DEBUG_EN:
	case WCD939X_DIGITAL_ANA_CSR_DBG_ADD:
	case WCD939X_DIGITAL_ANA_CSR_DBG_CTL:
	case WCD939X_DIGITAL_SSP_DBG:
	case WCD939X_DIGITAL_SPARE_0:
	case WCD939X_DIGITAL_SPARE_1:
	case WCD939X_DIGITAL_SPARE_2:
	case WCD939X_DIGITAL_TX_REQ_FB_CTL_0:
	case WCD939X_DIGITAL_TX_REQ_FB_CTL_1:
	case WCD939X_DIGITAL_TX_REQ_FB_CTL_2:
	case WCD939X_DIGITAL_TX_REQ_FB_CTL_3:
	case WCD939X_DIGITAL_TX_REQ_FB_CTL_4:
	case WCD939X_DIGITAL_DEM_BYPASS_DATA0:
	case WCD939X_DIGITAL_DEM_BYPASS_DATA1:
	case WCD939X_DIGITAL_DEM_BYPASS_DATA2:
	case WCD939X_DIGITAL_DEM_BYPASS_DATA3:
	case WCD939X_DIGITAL_DEM_SECOND_ORDER:
	case WCD939X_DIGITAL_DSM_CTRL:
	case WCD939X_DIGITAL_DSM_0_STATIC_DATA_0:
	case WCD939X_DIGITAL_DSM_0_STATIC_DATA_1:
	case WCD939X_DIGITAL_DSM_0_STATIC_DATA_2:
	case WCD939X_DIGITAL_DSM_0_STATIC_DATA_3:
	case WCD939X_DIGITAL_DSM_1_STATIC_DATA_0:
	case WCD939X_DIGITAL_DSM_1_STATIC_DATA_1:
	case WCD939X_DIGITAL_DSM_1_STATIC_DATA_2:
	case WCD939X_DIGITAL_DSM_1_STATIC_DATA_3:
	case WCD939X_RX_TOP_PAGE:
	case WCD939X_RX_TOP_TOP_CFG0:
	case WCD939X_RX_TOP_HPHL_COMP_WR_LSB:
	case WCD939X_RX_TOP_HPHL_COMP_WR_MSB:
	case WCD939X_RX_TOP_HPHL_COMP_LUT:
	case WCD939X_RX_TOP_HPHR_COMP_WR_LSB:
	case WCD939X_RX_TOP_HPHR_COMP_WR_MSB:
	case WCD939X_RX_TOP_HPHR_COMP_LUT:
	case WCD939X_RX_TOP_DSD0_DEBUG_CFG1:
	case WCD939X_RX_TOP_DSD0_DEBUG_CFG2:
	case WCD939X_RX_TOP_DSD0_DEBUG_CFG3:
	case WCD939X_RX_TOP_DSD0_DEBUG_CFG4:
	case WCD939X_RX_TOP_DSD1_DEBUG_CFG1:
	case WCD939X_RX_TOP_DSD1_DEBUG_CFG2:
	case WCD939X_RX_TOP_DSD1_DEBUG_CFG3:
	case WCD939X_RX_TOP_DSD1_DEBUG_CFG4:
	case WCD939X_RX_TOP_HPHL_PATH_CFG0:
	case WCD939X_RX_TOP_HPHL_PATH_CFG1:
	case WCD939X_RX_TOP_HPHR_PATH_CFG0:
	case WCD939X_RX_TOP_HPHR_PATH_CFG1:
	case WCD939X_RX_TOP_PATH_CFG2:
	case WCD939X_RX_TOP_HPHL_PATH_SEC0:
	case WCD939X_RX_TOP_HPHL_PATH_SEC1:
	case WCD939X_RX_TOP_HPHL_PATH_SEC2:
	case WCD939X_RX_TOP_HPHL_PATH_SEC3:
	case WCD939X_RX_TOP_HPHR_PATH_SEC0:
	case WCD939X_RX_TOP_HPHR_PATH_SEC1:
	case WCD939X_RX_TOP_HPHR_PATH_SEC2:
	case WCD939X_RX_TOP_HPHR_PATH_SEC3:
	case WCD939X_RX_TOP_PATH_SEC4:
	case WCD939X_RX_TOP_PATH_SEC5:
	case WCD939X_COMPANDER_HPHL_CTL0:
	case WCD939X_COMPANDER_HPHL_CTL1:
	case WCD939X_COMPANDER_HPHL_CTL2:
	case WCD939X_COMPANDER_HPHL_CTL3:
	case WCD939X_COMPANDER_HPHL_CTL4:
	case WCD939X_COMPANDER_HPHL_CTL5:
	case WCD939X_COMPANDER_HPHL_CTL7:
	case WCD939X_COMPANDER_HPHL_CTL8:
	case WCD939X_COMPANDER_HPHL_CTL9:
	case WCD939X_COMPANDER_HPHL_CTL10:
	case WCD939X_COMPANDER_HPHL_CTL11:
	case WCD939X_COMPANDER_HPHL_CTL12:
	case WCD939X_COMPANDER_HPHL_CTL13:
	case WCD939X_COMPANDER_HPHL_CTL14:
	case WCD939X_COMPANDER_HPHL_CTL15:
	case WCD939X_COMPANDER_HPHL_CTL16:
	case WCD939X_COMPANDER_HPHL_CTL17:
	case WCD939X_COMPANDER_HPHL_CTL18:
	case WCD939X_COMPANDER_HPHL_CTL19:
	case WCD939X_R_CTL0:
	case WCD939X_R_CTL1:
	case WCD939X_R_CTL2:
	case WCD939X_R_CTL3:
	case WCD939X_R_CTL4:
	case WCD939X_R_CTL5:
	case WCD939X_R_CTL7:
	case WCD939X_R_CTL8:
	case WCD939X_R_CTL9:
	case WCD939X_R_CTL10:
	case WCD939X_R_CTL11:
	case WCD939X_R_CTL12:
	case WCD939X_R_CTL13:
	case WCD939X_R_CTL14:
	case WCD939X_R_CTL15:
	case WCD939X_R_CTL16:
	case WCD939X_R_CTL17:
	case WCD939X_R_CTL18:
	case WCD939X_R_CTL19:
	case WCD939X_E_PATH_CTL:
	case WCD939X_E_CFG0:
	case WCD939X_E_CFG1:
	case WCD939X_E_CFG2:
	case WCD939X_E_CFG3:
	case WCD939X_DSD_HPHL_PATH_CTL:
	case WCD939X_DSD_HPHL_CFG0:
	case WCD939X_DSD_HPHL_CFG1:
	case WCD939X_DSD_HPHL_CFG2:
	case WCD939X_DSD_HPHL_CFG3:
	case WCD939X_DSD_HPHL_CFG4:
	case WCD939X_DSD_HPHL_CFG5:
	case WCD939X_DSD_HPHR_PATH_CTL:
	case WCD939X_DSD_HPHR_CFG0:
	case WCD939X_DSD_HPHR_CFG1:
	case WCD939X_DSD_HPHR_CFG2:
	case WCD939X_DSD_HPHR_CFG3:
	case WCD939X_DSD_HPHR_CFG4:
	case WCD939X_DSD_HPHR_CFG5:
		return true;
	}

	return false;
}

static bool wcd939x_readable_register(struct device *dev, unsigned int reg)
{
	/* Read-Only Registers */
	switch (reg) {
	case WCD939X_ANA_MBHC_RESULT_1:
	case WCD939X_ANA_MBHC_RESULT_2:
	case WCD939X_ANA_MBHC_RESULT_3:
	case WCD939X_MBHC_MOISTURE_DET_FSM_STATUS:
	case WCD939X_TX_1_2_SAR2_ERR:
	case WCD939X_TX_1_2_SAR1_ERR:
	case WCD939X_TX_3_4_SAR4_ERR:
	case WCD939X_TX_3_4_SAR3_ERR:
	case WCD939X_HPH_L_STATUS:
	case WCD939X_HPH_R_STATUS:
	case WCD939X_HPH_SURGE_STATUS:
	case WCD939X_EAR_STATUS_REG_1:
	case WCD939X_EAR_STATUS_REG_2:
	case WCD939X_MBHC_NEW_FSM_STATUS:
	case WCD939X_MBHC_NEW_ADC_RESULT:
	case WCD939X_DIE_CRACK_DET_OUT:
	case WCD939X_DIGITAL_CHIP_ID0:
	case WCD939X_DIGITAL_CHIP_ID1:
	case WCD939X_DIGITAL_CHIP_ID2:
	case WCD939X_DIGITAL_CHIP_ID3:
	case WCD939X_DIGITAL_INTR_STATUS_0:
	case WCD939X_DIGITAL_INTR_STATUS_1:
	case WCD939X_DIGITAL_INTR_STATUS_2:
	case WCD939X_DIGITAL_SWR_HM_TEST_0:
	case WCD939X_DIGITAL_SWR_HM_TEST_1:
	case WCD939X_DIGITAL_EFUSE_T_DATA_0:
	case WCD939X_DIGITAL_EFUSE_T_DATA_1:
	case WCD939X_DIGITAL_PIN_STATUS_0:
	case WCD939X_DIGITAL_PIN_STATUS_1:
	case WCD939X_DIGITAL_MODE_STATUS_0:
	case WCD939X_DIGITAL_MODE_STATUS_1:
	case WCD939X_DIGITAL_EFUSE_REG_0:
	case WCD939X_DIGITAL_EFUSE_REG_1:
	case WCD939X_DIGITAL_EFUSE_REG_2:
	case WCD939X_DIGITAL_EFUSE_REG_3:
	case WCD939X_DIGITAL_EFUSE_REG_4:
	case WCD939X_DIGITAL_EFUSE_REG_5:
	case WCD939X_DIGITAL_EFUSE_REG_6:
	case WCD939X_DIGITAL_EFUSE_REG_7:
	case WCD939X_DIGITAL_EFUSE_REG_8:
	case WCD939X_DIGITAL_EFUSE_REG_9:
	case WCD939X_DIGITAL_EFUSE_REG_10:
	case WCD939X_DIGITAL_EFUSE_REG_11:
	case WCD939X_DIGITAL_EFUSE_REG_12:
	case WCD939X_DIGITAL_EFUSE_REG_13:
	case WCD939X_DIGITAL_EFUSE_REG_14:
	case WCD939X_DIGITAL_EFUSE_REG_15:
	case WCD939X_DIGITAL_EFUSE_REG_16:
	case WCD939X_DIGITAL_EFUSE_REG_17:
	case WCD939X_DIGITAL_EFUSE_REG_18:
	case WCD939X_DIGITAL_EFUSE_REG_19:
	case WCD939X_DIGITAL_EFUSE_REG_20:
	case WCD939X_DIGITAL_EFUSE_REG_21:
	case WCD939X_DIGITAL_EFUSE_REG_22:
	case WCD939X_DIGITAL_EFUSE_REG_23:
	case WCD939X_DIGITAL_EFUSE_REG_24:
	case WCD939X_DIGITAL_EFUSE_REG_25:
	case WCD939X_DIGITAL_EFUSE_REG_26:
	case WCD939X_DIGITAL_EFUSE_REG_27:
	case WCD939X_DIGITAL_EFUSE_REG_28:
	case WCD939X_DIGITAL_EFUSE_REG_29:
	case WCD939X_DIGITAL_EFUSE_REG_30:
	case WCD939X_DIGITAL_EFUSE_REG_31:
	case WCD939X_RX_TOP_HPHL_COMP_RD_LSB:
	case WCD939X_RX_TOP_HPHL_COMP_RD_MSB:
	case WCD939X_RX_TOP_HPHR_COMP_RD_LSB:
	case WCD939X_RX_TOP_HPHR_COMP_RD_MSB:
	case WCD939X_RX_TOP_DSD0_DEBUG_CFG5:
	case WCD939X_RX_TOP_DSD0_DEBUG_CFG6:
	case WCD939X_RX_TOP_DSD1_DEBUG_CFG5:
	case WCD939X_RX_TOP_DSD1_DEBUG_CFG6:
	case WCD939X_COMPANDER_HPHL_CTL6:
	case WCD939X_R_CTL6:
		return true;
	}

	return wcd939x_rdwr_register(dev, reg);
}

static bool wcd939x_volatile_register(struct device *dev, unsigned int reg)
{
	switch (reg) {
	case WCD939X_ANA_MBHC_RESULT_1:
	case WCD939X_ANA_MBHC_RESULT_2:
	case WCD939X_ANA_MBHC_RESULT_3:
	case WCD939X_MBHC_MOISTURE_DET_FSM_STATUS:
	case WCD939X_TX_1_2_SAR2_ERR:
	case WCD939X_TX_1_2_SAR1_ERR:
	case WCD939X_TX_3_4_SAR4_ERR:
	case WCD939X_TX_3_4_SAR3_ERR:
	case WCD939X_HPH_L_STATUS:
	case WCD939X_HPH_R_STATUS:
	case WCD939X_HPH_SURGE_STATUS:
	case WCD939X_EAR_STATUS_REG_1:
	case WCD939X_EAR_STATUS_REG_2:
	case WCD939X_MBHC_NEW_FSM_STATUS:
	case WCD939X_MBHC_NEW_ADC_RESULT:
	case WCD939X_DIE_CRACK_DET_OUT:
	case WCD939X_DIGITAL_INTR_STATUS_0:
	case WCD939X_DIGITAL_INTR_STATUS_1:
	case WCD939X_DIGITAL_INTR_STATUS_2:
	case WCD939X_DIGITAL_SWR_HM_TEST_0:
	case WCD939X_DIGITAL_SWR_HM_TEST_1:
	case WCD939X_DIGITAL_PIN_STATUS_0:
	case WCD939X_DIGITAL_PIN_STATUS_1:
	case WCD939X_DIGITAL_MODE_STATUS_0:
	case WCD939X_DIGITAL_MODE_STATUS_1:
	case WCD939X_RX_TOP_HPHL_COMP_RD_LSB:
	case WCD939X_RX_TOP_HPHL_COMP_RD_MSB:
	case WCD939X_RX_TOP_HPHR_COMP_RD_LSB:
	case WCD939X_RX_TOP_HPHR_COMP_RD_MSB:
	case WCD939X_RX_TOP_DSD0_DEBUG_CFG5:
	case WCD939X_RX_TOP_DSD0_DEBUG_CFG6:
	case WCD939X_RX_TOP_DSD1_DEBUG_CFG5:
	case WCD939X_RX_TOP_DSD1_DEBUG_CFG6:
	case WCD939X_COMPANDER_HPHL_CTL6:
	case WCD939X_R_CTL6:
		return true;
	}
	return false;
}

static bool wcd939x_writeable_register(struct device *dev, unsigned int reg)
{
	return wcd939x_rdwr_register(dev, reg);
}

static const struct regmap_config wcd939x_regmap_config = {
	.name = "wcd939x_csr",
	.reg_bits = 32,
	.val_bits = 8,
	.cache_type = REGCACHE_MAPLE,
	.reg_defaults = wcd939x_defaults,
	.num_reg_defaults = ARRAY_SIZE(wcd939x_defaults),
	.max_register = WCD939X_MAX_REGISTER,
	.readable_reg = wcd939x_readable_register,
	.writeable_reg = wcd939x_writeable_register,
	.volatile_reg = wcd939x_volatile_register,
};

static const struct sdw_slave_ops wcd9390_slave_ops = {
	.update_status = wcd9390_update_status,
	.interrupt_callback = wcd9390_interrupt_callback,
	.bus_config = wcd9390_bus_config,
};

static int wcd939x_sdw_component_bind(struct device *dev, struct device *master,
				      void *data)
{
	pm_runtime_set_autosuspend_delay(dev, 3000);
	pm_runtime_use_autosuspend(dev);
	pm_runtime_mark_last_busy(dev);
	pm_runtime_set_active(dev);
	pm_runtime_enable(dev);

	return 0;
}

static void wcd939x_sdw_component_unbind(struct device *dev,
					 struct device *master, void *data)
{
	pm_runtime_disable(dev);
	pm_runtime_set_suspended(dev);
	pm_runtime_dont_use_autosuspend(dev);
}

static const struct component_ops wcd939x_sdw_component_ops = {
	.bind = wcd939x_sdw_component_bind,
	.unbind = wcd939x_sdw_component_unbind,
};

static int wcd9390_probe(struct sdw_slave *pdev, const struct sdw_device_id *id)
{
	struct device *dev = &pdev->dev;
	struct wcd939x_sdw_priv *wcd;
	int ret;

	wcd = devm_kzalloc(dev, sizeof(*wcd), GFP_KERNEL);
	if (!wcd)
		return -ENOMEM;

	/*
	 * Port map index starts with 0, however the data port for this codec
	 * are from index 1
	 */
	if (of_property_read_bool(dev->of_node, "qcom,tx-port-mapping")) {
		wcd->is_tx = true;
		ret = of_property_read_u32_array(dev->of_node,
						 "qcom,tx-port-mapping",
						 &pdev->m_port_map[1],
						 WCD939X_MAX_TX_SWR_PORTS);
	} else {
		ret = of_property_read_u32_array(dev->of_node,
						 "qcom,rx-port-mapping",
						 &pdev->m_port_map[1],
						 WCD939X_MAX_RX_SWR_PORTS);
	}

	if (ret < 0)
		dev_info(dev, "Static Port mapping not specified\n");

	wcd->sdev = pdev;
	dev_set_drvdata(dev, wcd);

	pdev->prop.scp_int1_mask = SDW_SCP_INT1_IMPL_DEF |
				   SDW_SCP_INT1_BUS_CLASH | SDW_SCP_INT1_PARITY;
	pdev->prop.lane_control_support = true;
	pdev->prop.simple_clk_stop_capable = true;
	if (wcd->is_tx) {
		pdev->prop.source_ports = GENMASK(WCD939X_MAX_TX_SWR_PORTS - 1, 0);
		pdev->prop.src_dpn_prop = wcd939x_tx_dpn_prop;
		wcd->ch_info = &wcd939x_sdw_tx_ch_info[0];
		pdev->prop.wake_capable = true;
	} else {
		pdev->prop.sink_ports = GENMASK(WCD939X_MAX_RX_SWR_PORTS - 1, 0);
		pdev->prop.sink_dpn_prop = wcd939x_rx_dpn_prop;
		wcd->ch_info = &wcd939x_sdw_rx_ch_info[0];
	}

	if (wcd->is_tx) {
		/*
		 * Do not use devres here since devres_release_group() could
		 * be called by component_unbind() id the aggregate device
		 * fails to bind.
		 */
		wcd->regmap = regmap_init_sdw(pdev, &wcd939x_regmap_config);
		if (IS_ERR(wcd->regmap))
			return dev_err_probe(dev, PTR_ERR(wcd->regmap),
					     "Regmap init failed\n");

		/* Start in cache-only until device is enumerated */
		regcache_cache_only(wcd->regmap, true);
	}

	ret = component_add(dev, &wcd939x_sdw_component_ops);
	if (ret)
		return ret;

	/* Set suspended until aggregate device is bind */
	pm_runtime_set_suspended(dev);

	return 0;
}

static int wcd9390_remove(struct sdw_slave *pdev)
{
	struct device *dev = &pdev->dev;
	struct wcd939x_sdw_priv *wcd = dev_get_drvdata(dev);

	component_del(dev, &wcd939x_sdw_component_ops);

	if (wcd->regmap)
		regmap_exit(wcd->regmap);

	return 0;
}

static const struct sdw_device_id wcd9390_slave_id[] = {
	SDW_SLAVE_ENTRY(0x0217, 0x10e, 0), /* WCD9390 & WCD9390 RX/TX Device ID */
	{},
};
MODULE_DEVICE_TABLE(sdw, wcd9390_slave_id);

static int __maybe_unused wcd939x_sdw_runtime_suspend(struct device *dev)
{
	struct wcd939x_sdw_priv *wcd = dev_get_drvdata(dev);

	if (wcd->regmap) {
		regcache_cache_only(wcd->regmap, true);
		regcache_mark_dirty(wcd->regmap);
	}

	return 0;
}

static int __maybe_unused wcd939x_sdw_runtime_resume(struct device *dev)
{
	struct wcd939x_sdw_priv *wcd = dev_get_drvdata(dev);

	if (wcd->regmap) {
		regcache_cache_only(wcd->regmap, false);
		regcache_sync(wcd->regmap);
	}

	return 0;
}

static const struct dev_pm_ops wcd939x_sdw_pm_ops = {
	SET_RUNTIME_PM_OPS(wcd939x_sdw_runtime_suspend, wcd939x_sdw_runtime_resume, NULL)
};

static struct sdw_driver wcd9390_codec_driver = {
	.probe = wcd9390_probe,
	.remove = wcd9390_remove,
	.ops = &wcd9390_slave_ops,
	.id_table = wcd9390_slave_id,
	.driver = {
		.name = "wcd9390-codec",
		.pm = &wcd939x_sdw_pm_ops,
	}
};
module_sdw_driver(wcd9390_codec_driver);

MODULE_DESCRIPTION("WCD939X SDW codec driver");
MODULE_LICENSE("GPL");
