/*
 * llc_s_ac.c - actions performed during sap state transition.
 *
 * Description :
 *   Functions in this module are implementation of sap component actions.
 *   Details of actions can be found in IEEE-802.2 standard document.
 *   All functions have one sap and one event as input argument. All of
 *   them return 0 On success and 1 otherwise.
 *
 * Copyright (c) 1997 by Procom Technology, Inc.
 *		 2001 by Arnaldo Carvalho de Melo <acme@conectiva.com.br>
 *
 * This program can be redistributed or modified under the terms of the
 * GNU General Public License as published by the Free Software Foundation.
 * This program is distributed without any warranty or implied warranty
 * of merchantability or fitness for a particular purpose.
 *
 * See the GNU General Public License for more details.
 */
#include <linux/netdevice.h>
#include <net/llc_if.h>
#include <net/llc_sap.h>
#include <net/llc_main.h>
#include <net/llc_s_ev.h>
#include <net/llc_pdu.h>
#include <net/llc_mac.h>

/**
 *	llc_sap_action_unit_data_ind - forward UI PDU to network layer
 *	@sap: SAP
 *	@ev: the event to forward
 *
 *	Received a UI PDU from MAC layer; forward to network layer as a
 *	UNITDATA INDICATION; verify our event is the kind we expect
 */
int llc_sap_action_unitdata_ind(struct llc_sap *sap,
				struct llc_sap_state_ev *ev)
{
	llc_sap_rtn_pdu(sap, ev->data.pdu.skb, ev);
	return 0;
}

/**
 *	llc_sap_action_send_ui - sends UI PDU resp to UNITDATA REQ to MAC layer
 *	@sap: SAP
 *	@ev: the event to send
 *
 *	Sends a UI PDU to the MAC layer in response to a UNITDATA REQUEST
 *	primitive from the network layer. Verifies event is a primitive type of
 *	event. Verify the primitive is a UNITDATA REQUEST.
 */
int llc_sap_action_send_ui(struct llc_sap *sap, struct llc_sap_state_ev *ev)
{
	struct llc_prim_if_block *prim = ev->data.prim.data;
	struct llc_prim_unit_data *prim_data = &prim->data->udata;
	struct sk_buff *skb = prim->data->udata.skb;
	int rc;

	llc_pdu_header_init(skb, LLC_PDU_TYPE_U, prim_data->saddr.lsap,
			    prim_data->daddr.lsap, LLC_PDU_CMD);
	rc = llc_pdu_init_as_ui_cmd(skb);
	if (rc)
		goto out;
	rc = lan_hdrs_init(skb, prim_data->saddr.mac, prim_data->daddr.mac);
	if (!rc)
		llc_sap_send_pdu(sap, skb);
out:
	return rc;
}

/**
 *	llc_sap_action_send_xid_c - send XID PDU as response to XID REQ
 *	@sap: SAP
 *	@ev: the event to send
 *
 *	Send a XID command PDU to MAC layer in response to a XID REQUEST
 *	primitive from the network layer. Verify event is a primitive type
 *	event. Verify the primitive is a XID REQUEST.
 */
int llc_sap_action_send_xid_c(struct llc_sap *sap, struct llc_sap_state_ev *ev)
{
	struct llc_prim_if_block *prim = ev->data.prim.data;
	struct llc_prim_xid *prim_data = &prim->data->xid;
	struct sk_buff *skb = prim_data->skb;
	int rc;

	llc_pdu_header_init(skb, LLC_PDU_TYPE_U, prim_data->saddr.lsap,
			    prim_data->daddr.lsap, LLC_PDU_CMD);
	rc = llc_pdu_init_as_xid_cmd(skb, LLC_XID_NULL_CLASS_2, 0);
	if (rc)
		goto out;
	rc = lan_hdrs_init(skb, prim_data->saddr.mac, prim_data->daddr.mac);
	if (!rc)
		llc_sap_send_pdu(sap, skb);
out:
	return rc;
}

/**
 *	llc_sap_action_send_xid_r - send XID PDU resp to MAC for received XID
 *	@sap: SAP
 *	@ev: the event to send
 *
 *	Send XID response PDU to MAC in response to an earlier received XID
 *	command PDU. Verify event is a PDU type event
 */
int llc_sap_action_send_xid_r(struct llc_sap *sap, struct llc_sap_state_ev *ev)
{
	u8 mac_da[ETH_ALEN], mac_sa[ETH_ALEN], dsap;
	int rc = 1;
	struct sk_buff *ev_skb = ev->data.pdu.skb;
	struct sk_buff *skb;

	llc_pdu_decode_sa(ev_skb, mac_da);
	llc_pdu_decode_da(ev_skb, mac_sa);
	llc_pdu_decode_ssap(ev_skb, &dsap);
	skb = llc_alloc_frame();
	if (!skb)
		goto out;
	skb->dev = ev_skb->dev;
	llc_pdu_header_init(skb, LLC_PDU_TYPE_U, sap->laddr.lsap, dsap,
			    LLC_PDU_RSP);
	rc = llc_pdu_init_as_xid_rsp(skb, LLC_XID_NULL_CLASS_2, 0);
	if (rc)
		goto out;
	rc = lan_hdrs_init(skb, mac_sa, mac_da);
	if (!rc)
		llc_sap_send_pdu(sap, skb);
out:
	return rc;
}

/**
 *	llc_sap_action_send_test_c - send TEST PDU to MAC in resp to TEST REQ
 *	@sap: SAP
 *	@ev: the event to send
 *
 *	Send a TEST command PDU to the MAC layer in response to a TEST REQUEST
 *	primitive from the network layer. Verify event is a primitive type
 *	event; verify the primitive is a TEST REQUEST.
 */
int llc_sap_action_send_test_c(struct llc_sap *sap, struct llc_sap_state_ev *ev)
{
	struct llc_prim_if_block *prim = ev->data.prim.data;
	struct llc_prim_test *prim_data = &prim->data->test;
	struct sk_buff *skb = prim_data->skb;
	int rc;

	llc_pdu_header_init(skb, LLC_PDU_TYPE_U, prim_data->saddr.lsap,
			    prim_data->daddr.lsap, LLC_PDU_CMD);
	rc = llc_pdu_init_as_test_cmd(skb);
	if (rc)
		goto out;
	rc = lan_hdrs_init(skb, prim_data->saddr.mac, prim_data->daddr.mac);
	if (!rc)
		llc_sap_send_pdu(sap, skb);
out:
	return rc;
}

int llc_sap_action_send_test_r(struct llc_sap *sap, struct llc_sap_state_ev *ev)
{
	u8 mac_da[ETH_ALEN], mac_sa[ETH_ALEN], dsap;
	int rc = 1;
	struct sk_buff *ev_skb = ev->data.pdu.skb;
	struct sk_buff *skb;

	llc_pdu_decode_sa(ev_skb, mac_da);
	llc_pdu_decode_da(ev_skb, mac_sa);
	llc_pdu_decode_ssap(ev_skb, &dsap);
	skb = llc_alloc_frame();
	if (!skb)
		goto out;
	skb->dev = ev_skb->dev;
	llc_pdu_header_init(skb, LLC_PDU_TYPE_U, sap->laddr.lsap, dsap,
			    LLC_PDU_RSP);
	rc = llc_pdu_init_as_test_rsp(skb, ev_skb);
	if (rc)
		goto out;
	rc = lan_hdrs_init(skb, mac_sa, mac_da);
	if (!rc)
		llc_sap_send_pdu(sap, skb);
out:
	return rc;
}

/**
 *	llc_sap_action_report_status - report data link status to layer mgmt
 *	@sap: SAP
 *	@ev: the event to send
 *
 *	Report data link status to layer management. Verify our event is the
 *	kind we expect.
 */
int llc_sap_action_report_status(struct llc_sap *sap,
				 struct llc_sap_state_ev *ev)
{
	return 0;
}

/**
 *	llc_sap_action_xid_ind - send XID PDU resp to net layer via XID IND
 *	@sap: SAP
 *	@ev: the event to send
 *
 *	Send a XID response PDU to the network layer via a XID INDICATION
 *	primitive.
 */
int llc_sap_action_xid_ind(struct llc_sap *sap, struct llc_sap_state_ev *ev)
{
	llc_sap_rtn_pdu(sap, ev->data.pdu.skb, ev);
	return 0;
}

/**
 *	llc_sap_action_test_ind - send TEST PDU to net layer via TEST IND
 *	@sap: SAP
 *	@ev: the event to send
 *
 *	Send a TEST response PDU to the network layer via a TEST INDICATION
 *	primitive. Verify our event is a PDU type event.
 */
int llc_sap_action_test_ind(struct llc_sap *sap, struct llc_sap_state_ev *ev)
{
	llc_sap_rtn_pdu(sap, ev->data.pdu.skb, ev);
	return 0;
}
